/*
 * This file is part of sp-response-time
 *
 * Copyright (C) 2005-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */ 

/**
   @file common.c

   Implementation of configuration operations
   <p>
*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <syslog.h>
#include <sys/time.h>
#include <X11/Xlib.h>

#include "common.h"
#include "config.h"

static FILE* f_log = 0;
static Bool log_needs_closing = False;
static Bool logging_initialized = False;	
static pthread_mutex_t log_mutex;

static int handle_xerror(Display* dpy, XErrorEvent* e) {
	/*
	 * Really only here for debugging, for gdb backtrace 
	 */
	char msg[255];
	XGetErrorText(dpy, e->error_code, msg, sizeof msg);
	log_error("X error (%#lx): %s (opcode: %i)\n",
		e->resourceid, msg, e->request_code);
	
	exit(1);
}

/** 
 * Perform simple logging with timestamp and diff from last log
 * 
 * @param time	Timestamp for log message. If 0 then timestamp is ignored. 
 * 				Timestamp can be what ever is wanted to be shown in log 
 * 				related to analysis report.
 * 
 * @param format Message with formating information. This is similar to printf.
 */
void log_message(const struct timeval timestamp, const char* format, ...) {

	pthread_mutex_lock(&log_mutex);

	va_list ap;
	char *tmp = NULL;
	
	va_start(ap,format);
	vasprintf(&tmp, format, ap);
	va_end(ap);
	
//TODO: Enable system log support.
//	syslog(LOG_INFO, "%s\n", tmp);
	
	if (timestamp.tv_sec == 0 && timestamp.tv_usec == 0) {
		fprintf(f_log, "------------- : %s\n", tmp);
	} else {
		fprintf(f_log, "%lu.%03lu : %s\n", timestamp.tv_sec, timestamp.tv_usec/1000, tmp);
	}

	fflush(f_log);
	if (tmp) free(tmp);
	
	pthread_mutex_unlock(&log_mutex);
}

void log_error(const char* format, ...) {

	int ret = 0;
	char* tmp = NULL;
	va_list	ap;
	struct timeval tv;
	
	ret = gettimeofday(&tv, NULL);
	
	va_start(ap,format);
	vasprintf(&tmp, format, ap);
	va_end(ap);
	
	fprintf(stderr, "ERROR: %s\n", tmp);

	if (tmp) free(tmp);
}

void log_analysis_report(const struct timeval timestamp,  const struct timeval event_occured, const struct timeval last_update_occured, Time update_time, const char *additional) {
	if (additional != NULL) {
		log_message(timestamp, "%lu.%03lu, %lu.%03lu, %lu, \"Event at %lu.%03lus. Update took %lums. %s\"",
			event_occured.tv_sec, event_occured.tv_usec/1000,
			last_update_occured.tv_sec, last_update_occured.tv_usec/1000,
			update_time,
			event_occured.tv_sec, event_occured.tv_usec/1000,
			update_time, additional);
	} else {
		log_message(timestamp, "%lu.%03lu, %lu.%03lu, %lu, \"Event at %lu.%03lus. Update took %lums.\"",
			event_occured.tv_sec, event_occured.tv_usec/1000,
			last_update_occured.tv_sec, last_update_occured.tv_usec/1000,
			update_time,
			event_occured.tv_sec, event_occured.tv_usec/1000,
			update_time);
	}
}

/** 
 * Set up Display connection, required extensions and req other X bits
 * 
 * @param dpy_name Display to which attach.
 * @return Display pointer to the display which was requested or NULL if fails.
 */
Display* setup_display(char* dpy_name) {
	Display* dpy;
	
	if ( dpy_name == NULL ) {
		dpy_name = strdup(":0");
		log_error("DISPLAY not set? Trying ':0' ...");
	}

	if ((dpy = XOpenDisplay(dpy_name)) == NULL) {
		log_error("Unable to connect to DISPLAY.");
		return NULL;
	}
	
	XSynchronize(dpy, True);
	XSetErrorHandler(handle_xerror);
	
	XSelectInput(dpy, DefaultRootWindow(dpy), StructureNotifyMask);
	
	return dpy;
}

void init_logging(int log_type, const char* log_path) {

	pthread_mutex_init(&log_mutex, NULL);

	pthread_mutex_lock(&log_mutex);

	f_log = stdout;
	
	if ( logging_initialized == True ) {
		return;
	}
		
	if ( log_type == USE_STDERR ) {
		f_log = stderr;
	} else if ( log_type == USE_FILE ) {
		FILE *f_new = 0;
		
		if( (f_new = fopen(log_path, "w")) != NULL ) {
			f_log = f_new;
			log_needs_closing = True;
		} else {
			log_error(0, "Can't open %s for writing log, using stdout.", log_path);
		}
	}

//TODO: Enable system log support.
//	openlog(PACKAGE, LOG_PID, LOG_USER);
	
	logging_initialized = True;
	
	fprintf(f_log, "\n%s\n\n", PACKAGE_STRING);
	
	fprintf(f_log,
		"Timestamp      : Info ([event(s), damage(s), update(ms)"
		", ]message) -------------\n");
		
	pthread_mutex_unlock(&log_mutex);
}

void end_logging() {

	if ( log_needs_closing == True ) {
		fflush(f_log);
		fclose(f_log);
	}
		
//TODO: Enable system log support.
//	closelog();

	pthread_mutex_destroy(&log_mutex);	
}
