/**
 * @file metadata.c Metadata storage API
 *
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <libxml/tree.h>
#include "htmlview.h"
#include "metadata.h"
#include "debug.h"
#define	IMG_START	"<img class=\"feed\" src=\""
#define IMG_END		"\"><br>"

static GHashTable *strtoattrib;

typedef enum {
    POS_HEADTABLE,
    POS_HEAD,
    POS_BODY,
    POS_FOOTTABLE
} output_position;

struct attribute {
    gchar *strid;

    renderHTMLFunc renderhtmlfunc;

    gpointer user_data;
};

struct pair {
    struct attribute *attrib;
    GSList *data;
};

struct str_attrib {
    output_position pos;
    gchar *prompt;
};

static void attribs_register_default_renderer(const gchar * strid);

void metadata_register_renderer(const gchar * strid,
                                renderHTMLFunc renderfunc,
                                gpointer user_data);

/************************************************************************/
/* PRIVATE FUNCTIONS */
/************************************************************************/

static void
metadata_render(struct attribute *attrib,
                struct displayset *displayset, gpointer data)
{
    g_assert(attrib != NULL);
    attrib->renderhtmlfunc(data, displayset, attrib->user_data);
}

/* Now comes the stuff to define particular attributes */

static void
attribs_render_str(gpointer data,
                   struct displayset *displayset, gpointer user_data)
{
    struct str_attrib *props = (struct str_attrib *) user_data;
    gchar *str = NULL;

    switch (props->pos) {
        case POS_HEADTABLE:
            str = g_strdup_printf(HEAD_LINE, (gchar *) data);
            addToHTMLBufferFast(&(displayset->headtable), str);
            g_free(str);
            break;
        case POS_HEAD:
            addToHTMLBufferFast(&(displayset->head), (gchar *) data);
            break;
        case POS_BODY:
            addToHTMLBufferFast(&(displayset->body), (gchar *) data);
            break;
        case POS_FOOTTABLE:
            FEED_FOOT_WRITE(displayset->foottable, props->prompt,
                            (gchar *) data);
            break;
    }
}

#define REGISTER_SIMPLE_ATTRIBUTE(position, strid, promptStr) do { \
 struct str_attrib *props = g_new(struct str_attrib, 1); \
 props->pos = (position); \
 props->prompt = (promptStr); \
 metadata_register_renderer(strid, attribs_render_str, props); \
} while (0);

static void attribs_register_default_renderer(const gchar * strid)
{
    gchar *str = g_strdup(strid);

    REGISTER_SIMPLE_ATTRIBUTE(POS_FOOTTABLE, str, str);
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

GSList *metadata_list_get(GSList * metadata, const gchar * strid)
{
    GSList *list = metadata;

    while (list != NULL) {
        struct pair *p = (struct pair *) list->data;
        if (0 == strcmp(p->attrib->strid, strid))
            return p->data;
        list = list->next;
    }
    return NULL;
}


void metadata_init()
{
    strtoattrib = g_hash_table_new(g_str_hash, g_str_equal);
}

void
metadata_register_renderer(const gchar * strid,
                           renderHTMLFunc renderfunc, gpointer user_data)
{
    struct attribute *attrib = g_new(struct attribute, 1);

    if (g_hash_table_lookup(strtoattrib, strid) != NULL) {
        g_warning("Duplicate attribute was attempted to be registered: %s",
                  strid);
        return;
    }

    attrib->strid = g_strdup(strid);
    attrib->renderhtmlfunc = renderfunc;
    attrib->user_data = user_data;

    g_hash_table_insert(strtoattrib, attrib->strid, attrib);
}

GSList *metadata_list_append_enclosure_attr(GSList * metadata,
                                            const gchar * strid,
                                            struct enclosure_attribute * data)
{
    struct attribute *attrib = NULL;
    GSList *iter = metadata;
    struct pair *p = NULL;

    if (NULL == (attrib = g_hash_table_lookup(strtoattrib, strid))) {
        /* g_warning("Encountered unknown attribute type \"%s\".", strid); */
        attribs_register_default_renderer(strid);
        attrib = g_hash_table_lookup(strtoattrib, strid);
    }

    while (iter != NULL) {
        p = (struct pair *) iter->data;
        if (p->attrib == attrib) {
            p->data = g_slist_append(p->data, data);
            return metadata;
        }
        iter = iter->next;
    }
    p = g_new(struct pair, 1);
    p->attrib = attrib;
    p->data = g_slist_append(NULL, data);
    metadata = g_slist_append(metadata, p);
    return metadata;
}


GSList *metadata_list_append(GSList * metadata, const gchar * strid,
                             const gchar * data)
{
    struct attribute *attrib = NULL;
    GSList *iter = metadata;
    struct pair *p = NULL;

    if (NULL == (attrib = g_hash_table_lookup(strtoattrib, strid))) {
        /* g_warning("Encountered unknown attribute type \"%s\".", strid); */
        attribs_register_default_renderer(strid);
        attrib = g_hash_table_lookup(strtoattrib, strid);
    }

    while (iter != NULL) {
        p = (struct pair *) iter->data;
        if (p->attrib == attrib) {
            if (strcmp(strid, "enclosure") == 0)
                p->data =
                    g_slist_append(p->data,
                                   enclosure_dup_enclosure((struct
                                                            enclosure_attribute
                                                            *) data));
            else
                p->data = g_slist_append(p->data, g_strdup(data));
            return metadata;
        }
        iter = iter->next;
    }
    p = g_new(struct pair, 1);
    p->attrib = attrib;
    if (strcmp(strid, "enclosure") == 0)
        p->data =
            g_slist_append(NULL,
                           enclosure_dup_enclosure((struct enclosure_attribute
                                                    *) data));
    else
        p->data = g_slist_append(NULL, g_strdup(data));
    metadata = g_slist_append(metadata, p);
    return metadata;
}

void
metadata_list_set(GSList ** metadata, const gchar * strid, const gchar * data)
{
    struct attribute *attrib = NULL;
    GSList *iter = *metadata;
    struct pair *p = NULL;

    if (NULL == (attrib = g_hash_table_lookup(strtoattrib, strid))) {
        attribs_register_default_renderer(strid);
        attrib = g_hash_table_lookup(strtoattrib, strid);
    }

    while (iter != NULL) {
        p = (struct pair *) iter->data;
        if (p->attrib == attrib) {
            if (NULL != p->data) {
                /* exchange old value */
                g_free(((GSList *) p->data)->data);
                ((GSList *) p->data)->data = g_strdup(data);
            } else {
                p->data = g_slist_append(p->data, g_strdup(data));
            }
            return;
        }
        iter = iter->next;
    }
    p = g_new(struct pair, 1);
    p->attrib = attrib;
    p->data = g_slist_append(NULL, g_strdup(data));
    *metadata = g_slist_append(*metadata, p);
}

void metadata_list_render(GSList * metadata, struct displayset *displayset)
{
    GSList *list = metadata;

    while (list != NULL) {
        struct pair *p = (struct pair *) list->data;
        GSList *list2 = p->data;
        while (list2 != NULL) {
            metadata_render(p->attrib, displayset, (gchar *) list2->data);
            list2 = list2->next;
        }
        list = list->next;
    }
}

GSList *metadata_list_copy(GSList * from, GSList * to)
{
    GSList *list2 = NULL, *iter2 = NULL, *iter = from;
    struct pair *p = NULL;

    while (iter != NULL) {
        p = (struct pair *) iter->data;
        iter2 = list2 = p->data;
        while (iter2 != NULL) {
            to = metadata_list_append(to, (p->attrib)->strid, iter2->data);
            iter2 = iter2->next;
        }
        iter = iter->next;
    }

    return to;
}

void metadata_list_free(GSList * metadata)
{
    GSList *list2 = NULL, *iter2 = NULL, *iter = metadata;
    struct pair *p = NULL;
    gboolean enclosure = FALSE;

    while (iter != NULL) {
        p = (struct pair *) iter->data;
        list2 = p->data;
        iter2 = list2;
        enclosure = strcmp((p->attrib)->strid, "enclosure") == 0;
        while (iter2 != NULL) {
            if (enclosure) {
                ULOG_DEBUG("Freeing enclosure");
                enclosure_free_enclosure((struct enclosure_attribute *)
                                         iter2->data);
            } else
                g_free(iter2->data);
            iter2 = iter2->next;
        }
        g_slist_free(list2);
        g_free(p);
        iter = iter->next;
    }
    g_slist_free(metadata);
}

void metadata_add_xml_nodes(GSList * metadata, xmlNodePtr parentNode)
{
    GSList *list = metadata;
    xmlNodePtr attribute = NULL;
    xmlNodePtr metadataNode =
        xmlNewChild(parentNode, NULL, (const xmlChar *)"attributes", NULL);

    while (list != NULL) {
        struct pair *p = (struct pair *) list->data;
        GSList *list2 = p->data;
        while (list2 != NULL) {
            attribute =
                xmlNewTextChild(metadataNode, NULL, (const xmlChar *)"attribute", list2->data);
            xmlNewProp(attribute, (const xmlChar *)"name", (const xmlChar *)p->attrib->strid);
            list2 = list2->next;
        }
        list = list->next;
    }
}

GSList *metadata_parse_xml_nodes(xmlDocPtr doc, xmlNodePtr cur)
{
    xmlNodePtr attribute = cur->xmlChildrenNode;
    GSList *metadata = NULL;

    while (attribute != NULL) {
        if (attribute->type == XML_ELEMENT_NODE &&
            !xmlStrcmp(attribute->name, BAD_CAST "attribute")) {
            xmlChar *name = xmlGetProp(attribute, (const xmlChar *)"name");
            if (name != NULL) {
                gchar *value =
                    (gchar *)xmlNodeListGetString(doc, attribute->xmlChildrenNode,TRUE);
                if (value != NULL) {
                    metadata = metadata_list_append(metadata, (const char*)name, value);
                    xmlFree(value);
                }
                xmlFree(name);
            }
        }
        attribute = attribute->next;
    }
    return metadata;
}
