/**
 * @file feed.c common feed handling
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <string.h>
#include <time.h>
#include <unistd.h>             /* For unlink() */
#include <stdlib.h>
#include <hildon/hildon-banner.h>
#include <errno.h>

#include <osso-rss-feed-reader/settings.h>

#include "conf.h"
#include "osso-log.h"

#include "html.h"
#include "cdf_channel.h"
#include "rss_channel.h"
#include "atom10.h"
#include "pie_feed.h"
#include "ocs_dir.h"
#include "opml.h"
#include "vfolder.h"
#include "feed.h"
#include "favicon_cache.h"
#include "update.h"
#include "metadata.h"


#include "ui_feed_directory.h"

#include "debug_new.h"

extern long rss_unread;
extern time_t rss_updated;
extern AppData *app_data;
extern feedPtr searchFeed;

/* auto detection lookup table */
static GSList *feedhandlers;

struct feed_type {
    gint id_num;
    gchar *id_str;
};

gboolean conf_keep_feeds_in_memory = FALSE;

#define VERSION_2 2
#define VERSION_3 3

void download_image_link(gchar * link, gpointer user_data);

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

/* initializing function, only called upon startup */
void feed_init(void)
{

    feedhandlers = g_slist_append(feedhandlers, ocs_init_feed_handler());   /* Must 
                                                                             * come 
                                                                             * before 
                                                                             * RSS/RDF 
                                                                             */
    feedhandlers = g_slist_append(feedhandlers, rss_init_feed_handler());
    feedhandlers = g_slist_append(feedhandlers, cdf_init_feed_handler());
    /* tvh: import atom handler from liferea 1.0.4 */
    feedhandlers = g_slist_append(feedhandlers, atom10_init_feed_handler());    /* Must 
                                                                                 * be 
                                                                                 * before 
                                                                                 * pie 
                                                                                 */
    feedhandlers = g_slist_append(feedhandlers, pie_init_feed_handler());
    feedhandlers = g_slist_append(feedhandlers, opml_init_feed_handler());
    feedhandlers = g_slist_append(feedhandlers, vfolder_init_feed_handler());
}


/* ------------------------------------------------------------ */
/* feed type registration */
/* ------------------------------------------------------------ */

const gchar *feed_type_fhp_to_str(feedHandlerPtr fhp)
{
    if (fhp == NULL)
        return NULL;
    return fhp->typeStr;
}

feedHandlerPtr feed_type_str_to_fhp(const gchar * str)
{
    GSList *iter = NULL;
    feedHandlerPtr fhp = NULL;

    if (str == NULL)
        return NULL;

    for (iter = feedhandlers; iter != NULL; iter = iter->next) {
        fhp = (feedHandlerPtr) iter->data;
        if (!strcmp(str, fhp->typeStr))
            return fhp;
    }

    return NULL;
}

struct feed_parse_new_data {
    struct request *nreq;
    gchar *source;
    xmlDocPtr doc;
    feedHandlerPtr handler;
    feedPtr fp;
    gchar *data;
};

void feed_parse_new_download(struct request *nrequest)
{
    struct feed_parse_new_data *fpn_data =
        (struct feed_parse_new_data *) nrequest->fpn_data;
    struct request *old_req = nrequest->nreq;
    gchar *source = NULL;
    xmlDocPtr doc = NULL;
    feedHandlerPtr handler = NULL;
    feedPtr fp = NULL;
    gchar *data = NULL;

    old_req = fpn_data->nreq;
    source = fpn_data->source;
    doc = fpn_data->doc;
    handler = fpn_data->handler;
    fp = fpn_data->fp;
    data = fpn_data->data;

    g_free(fpn_data);

    if (NULL != nrequest->data) {
        debug0(DEBUG_UPDATE, "feed link download successful!");
        feed_set_source(fp, source);
        handler = feed_parse(fp, nrequest->data, nrequest->size, FALSE, NULL);
    } else {
        /* if the download fails we do nothing except
         * unsetting the handler so the original source
         * will get a "unsupported type" error */
        handler = NULL;
        debug0(DEBUG_UPDATE, "feed link download failed!");
    }
    g_free(source);
    download_request_free(&nrequest);



    if (doc != NULL)
        xmlFreeDoc(doc);

    ULOG_DEBUG("Exiting parser");
    debug_exit("feed_parse");
    g_free(data);
    old_req->fhp = handler;
    ui_feed_process_update_result(old_req);

}

/**
 * General feed source parsing function. Parses the passed feed source
 * and tries to determine the source type. If the type is HTML and 
 * autodiscover is TRUE the function tries to find a feed, tries to
 * download it and parse the feed's source instead of the passed source.
 *
 * @param fp        the feed structure to be filled
 * @param data        the feed source
 * @param dataLength the length of the 'data' string
 * @param autodiscover    TRUE if auto discovery should be possible
 */
feedHandlerPtr feed_parse(feedPtr fp, gchar * data, size_t dataLength,
                          gboolean autodiscover, struct request *nreq)
{
    gchar *source = NULL;
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL;
    GSList *handlerIter = NULL;
    gboolean handled = FALSE;
    feedHandlerPtr handler = NULL;

    debug_enter("feed_parse");
    // ULOG_DEBUG("String before: %s\n", data);
#ifdef feed_eliminate_end_sign_used
    data = g_strdup(feed_eliminate_end_sign(g_strdup(data)));
#else
    /* 
     * feed_eliminate_end_sign_used was modified with "return" as first stmt,
     * turning it into dead code and creating memory leak.
     * I enclosed that function (in common.c) 
     * in #ifdef feed_eliminate_end_sign_used
     * (NB which is not defined now!)
     *
     * calling it as above:g_strdup(feed_eliminate_end_sign(g_strdup(data)));
     * would result in effective call g_strdup(g_strdup(data));
     * causing leaked result of first strdup.
     * (that comment added by Olev Kartau while looking for memory leaks 2007-04-20)
     */
    data = g_strdup(data);
#endif                          /* feed_eliminate_end_sign_used */
    // ULOG_DEBUG("String: %s\n", data);

    if (fp != NULL)
        g_free(fp->parseErrors);
    else
    {
        ULOG_DEBUG("Exiting parser, feed pointer was NULL");
        g_free(data);
        return handler;
    }

    /* try to parse buffer with XML and to create a DOM tree */
    do {
        if (NULL == (doc = parseBuffer(data, dataLength, &(fp->parseErrors)))) {
            gchar *msg =
                g_strdup_printf
                ("<p>XML error occurred. Unable to load feed. \"%s\"</p>",
                 fp->source);
            addToHTMLBuffer(&(fp->parseErrors), msg);
            g_free(msg);
            break;
        }
        if (NULL == (cur = xmlDocGetRootElement(doc))) {
            addToHTMLBuffer(&(fp->parseErrors), "<p>Empty document</p>");
            break;
        }
        while (cur && xmlIsBlankNode(cur)) {
            cur = cur->next;
        }
        if (!cur || (NULL == cur->name)) {
            addToHTMLBuffer(&(fp->parseErrors), "<p>Invalid XML</p>");
            break;
        }

        /* determine the syndication format */
        handlerIter = feedhandlers;
        while (handlerIter != NULL) {
            handler = (feedHandlerPtr) (handlerIter->data);
            if (handler != NULL && handler->checkFormat != NULL
                && (*(handler->checkFormat)) (doc, cur)) {

                /* free old temp. parsing data, don't free right after
                 * parsing because it can be used until the last feed request 
                 * is finished, move me to the place where the last request
                 * in list otherRequests is finished :-) */
                g_hash_table_destroy(fp->tmpdata);
                fp->tmpdata =
                    g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
                                          g_free);

                /* we always drop old metadata */
                metadata_list_free(fp->metadata);
                fp->metadata = NULL;

                if (FALSE == handler->merge)
                    feed_clear_item_list(fp);   /* for directories */

                if (strcmp((handler->typeStr), "opml") == 0) {
                    newdialog_destroy();
                    folderPtr folder = ui_feedlist_get_target_folder();

                    ULOG_DEBUG
                        ("feed_parse: creating the feed dir dialog now");
                    ui_feed_directory_from_add_feed_dialog(doc, cur, folder,
                                                           (gchar *)
                                                           feed_get_source
                                                           (fp));

                    // free stuff and just return
                    // g_free(source);
                    if (doc != NULL)
                        xmlFreeDoc(doc);
                    g_free(data);

                    return handler;
                }

                (*(handler->feedParser)) (fp, doc, cur);    /* parse it */
                handled = TRUE;
                break;
            }
            handlerIter = handlerIter->next;
        }
    } while (0);

    if (handled)
        ULOG_DEBUG("handled %d: fp->title: %s", handled, fp->title);

    if (!handled) {
        handler = NULL;

        //    ULOG_DEBUG("testing if we have a html page");

        /* test if we have a HTML page */
        if (autodiscover && ((NULL != strstr(data, "<html>"))
                             || (NULL != strstr(data, "<HTML>"))
                             || (NULL != strstr(data, "<html "))
                             || (NULL != strstr(data, "<HTML "))
            )) {
            /* if yes we should scan for links */
            debug1(DEBUG_UPDATE,
                   "HTML detected, starting feed auto discovery (%s)",
                   feed_get_source(fp));
            if (NULL !=
                (source =
                 html_auto_discover_feed(data, feed_get_source(fp)))) {
                /* now download the first feed link found */
                struct request *request = download_request_new();
                debug1(DEBUG_UPDATE, "feed link found: %s", source);
                request->source = g_strdup(source);
                struct feed_parse_new_data *fpn_data =
                    g_new0(struct feed_parse_new_data, 1);
                fpn_data->fp = fp;
                fpn_data->source = source;
                fpn_data->nreq = nreq;
                fpn_data->doc = doc;
                fpn_data->data = data;
                fpn_data->fp = fp;
                if (nreq)
                    nreq->jump = TRUE;
                request->jump = TRUE;
                request->fpn_data = fpn_data;
                if (download_process(request)) {    //downloading in progress...trying later on
                    update_try_download(request);
                }
                metadata_list_free(fpn_data->fp->metadata);
                return NULL;
            } else {
                debug0(DEBUG_UPDATE, "no feed link found!");
            }
        } else {
            handler = NULL;
            feed_set_available(fp, FALSE);
            if (!app_data->app_ui_data->start_on_background)
                hildon_banner_show_information(GTK_WIDGET
                                               (app_data->app_ui_data->
                                                main_view), NULL,
                                               _
                                               ("rss_ni_errors_while_parsing_feed"));
            addToHTMLBuffer(&(fp->parseErrors),
                            "<p>Unable to specify the feed type. Check feed type and ensure it is <a href="
                            "\"http://feedvalidator.org\">valid</a> and in a <a href=\"http://liferea."
                            "sourceforge.net/index.php#supported_formats\">supported format</a>.</p>");
        }
    } else {
        debug1(DEBUG_UPDATE, "discovered feed format: %s",
               feed_type_fhp_to_str(handler));
    }

    //    ULOG_DEBUG("Freeing document");
    if (doc != NULL)
        xmlFreeDoc(doc);

    ULOG_DEBUG("Exiting parser");
    debug_exit("feed_parse");
    g_free(data);

    return handler;
}



/**
 * method to be called to schedule a feed to be updated
 */
void feed_schedule_update(feedPtr fp, gint flags)
{
    const gchar *source = NULL;
    struct request *request = NULL;
    g_assert(NULL != fp);

    ULOG_DEBUG("Scheduling %s to be updated", feed_get_title(fp));

    if (feed_get_discontinued(fp)) {
        ULOG_DEBUG
            ("The feed was discontinued. Liferea won't update it anymore!");
        return;
    }

    if (NULL == (source = feed_get_source(fp))) {
        g_warning
            ("Feed source is NULL! This should never happen - cannot update!");
        return;
    }
    request = download_request_new();
    request->callback = ui_feed_process_update_result;
    request->user_data = fp;
    request->feed = fp;
    /* prepare request url (strdup because it might be changed on permanent
     * HTTP redirection in netio.c) */
    request->source = g_strdup(source);
    request->lastmodified = feed_get_lastmodified(fp);
    if (feed_get_etag(fp) != NULL)
        request->etag = g_strdup(feed_get_etag(fp));
    request->flags = flags;
    request->priority = (flags & FEED_REQ_PRIORITY_HIGH) ? 1 : 0;
    if (feed_get_filter(fp) != NULL)
        request->filtercmd = g_strdup(feed_get_filter(fp));

    download_queue(request);

}

gboolean feed_add_item_ignore_old(feedPtr fp, itemPtr new_ip)
{
    GSList *old_items = NULL;
    itemPtr old_ip = NULL;
    gboolean found = FALSE, equal = FALSE;

    g_assert(NULL != fp);
    g_assert(NULL != new_ip);
    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));

    if (FST_VFOLDER != feed_get_type(fp)) {
        /* determine if we should add it... */
        debug1(DEBUG_VERBOSE, "check new item for merging: \"%s\"",
               item_get_title(new_ip));

        /* compare to every existing item in this feed */
        found = FALSE;
        old_items = feed_get_item_list(fp);
        while (NULL != old_items) {
            old_ip = old_items->data;

            /* try to compare the two items */

            /* trivial case: one item has item the other doesn't -> they
             * can't be equal */
            if (((item_get_id(old_ip) == NULL)
                 && (item_get_id(new_ip) != NULL))
                || ((item_get_id(old_ip) != NULL)
                    && (item_get_id(new_ip) == NULL))) {
                /* cannot be equal (different ids) so compare to next old
                 * item */
                ULOG_DEBUG
                    ("feed_add_item_ignore_old Comparing %s vs %s, id mismatch",
                     item_get_title(old_ip), item_get_title(new_ip));
                old_items = g_slist_next(old_items);
                continue;
            }

            /* just for the case there are no ids: compare titles and HTML
             * descriptions */
            equal = TRUE;

            if (((item_get_title(old_ip) != NULL)
                 && (item_get_title(new_ip) != NULL))
                && (0 !=
                    strcmp(item_get_title(old_ip), item_get_title(new_ip))))
                equal = FALSE;

            if (((item_get_description(old_ip) != NULL)
                 && (item_get_description(new_ip) != NULL))
                && (0 !=
                    strcmp(item_get_description(old_ip),
                           item_get_description(new_ip))))
                equal = FALSE;

            /* best case: they both have ids (position important: id check is 
             * useless without knowing if the items are different!) */
            if (NULL != item_get_id(old_ip)) {
                if (0 == strcmp(item_get_id(old_ip), item_get_id(new_ip))) {
                    found = TRUE;
                    break;
                }
            }

            if (equal) {
                found = TRUE;
                break;
            }

            old_items = g_slist_next(old_items);
        }

        if (!found) {
            if (item_get_time(new_ip) <= feed_get_newest_post(fp)) {
                item_free(new_ip);
                return FALSE;
            }

            /* uispec0.3: automatic removal changed */
            /* set all new items as on_server */
            g_debug("item->on_server = TRUE: %s\n", item_get_title(new_ip));
            new_ip->on_server = TRUE;

            if (0 == new_ip->nr) {
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            /* ensure that the feed last item nr is at maximum */
            if (new_ip->nr > fp->lastItemNr)
                fp->lastItemNr = new_ip->nr;

            /* ensure that the item nr's are unique */
            if (NULL != feed_lookup_item_by_nr(fp, new_ip->nr)) {
                g_warning
                    ("The item number to be added is not unique! Feed (%s) Item (%s) (%ld)\n",
                     fp->id, new_ip->title, new_ip->nr);
                new_ip->nr = ++(fp->lastItemNr);
                fp->needsCacheSave = TRUE;
            }

            if (FALSE == item_get_read_status(new_ip))
                feed_increase_unread_counter(fp);
            if (TRUE == item_get_mark(new_ip))
                feed_increase_mark_counter(fp);
            if (TRUE == new_ip->newStatus) {
                fp->newCount++;
            }
            fp->items = g_slist_prepend(fp->items, (gpointer) new_ip);
            new_ip->fp = fp;

            /* check if the item matches any vfolder rules */
            vfolder_check_item(new_ip);

            debug0(DEBUG_VERBOSE, "-> item added to feed itemlist");

        } else {

            /* uispec0.3: automatic removal changed */
            /* mark existing item as on_server */
            old_ip->on_server = TRUE;
            g_debug("item->on_server = TRUE: %s\n", item_get_title(old_ip));

            /* if the item was found but has other contents -> update
             * contents */
            if (!equal) {
                /* no item_set_new_status() - we don't treat changed items as 
                 * new items! */
                item_set_title(old_ip, item_get_title(new_ip));
                item_set_description(old_ip, item_get_description(new_ip));
                item_set_time(old_ip, item_get_time(new_ip));
                item_set_unread(old_ip);    /* FIXME: needed? */
                metadata_list_free(old_ip->metadata);
                old_ip->metadata = new_ip->metadata;
		new_ip->metadata = NULL;
                vfolder_update_item(old_ip);
                debug0(DEBUG_VERBOSE,
                       "-> item already existing and was updated");
            } else {
                debug0(DEBUG_VERBOSE, "-> item already exists");
            }
            item_free(new_ip);
        }
    } else {
        /* vfolder/search - just add the item */
        if (FALSE == item_get_read_status(new_ip))
            feed_increase_unread_counter(fp);
        fp->items = g_slist_append(fp->items, (gpointer) new_ip);
        new_ip->fp = fp;
        /* item number should already be set by item_copy() */

        ui_itemlist_search_append(new_ip);

        found = TRUE;
    }
    return !found;
}

void feed_remove_item(feedPtr fp, itemPtr ip)
{

    fp->items = g_slist_remove(fp->items, ip);
    vfolder_remove_item(ip);    /* remove item copies */
    remove_links_from_cache(ip);
    item_free(ip);              /* remove the item */
    fp->needsCacheSave = TRUE;
}

itemPtr feed_lookup_item_by_nr(feedPtr fp, gulong nr)
{
    GSList *items = NULL;
    itemPtr ip = NULL;

    g_assert((0 != fp->loaded) || (FST_VFOLDER == feed_get_type(fp)));
    items = fp->items;
    while (NULL != items) {
        ip = (itemPtr) (items->data);
        if (ip->nr == nr)
            return ip;
        items = g_slist_next(items);
    }

    return NULL;
}

/* Looks for ip that has fp as the parent fp But also its original fp matches 
 * orig_fp id (== sourceFeed. Look at item_copy We need the 3rd param for
 * virtual feed search One virtual feed can have 2 items that have the same
 * id due to one feed can be added twice. Or even when the items from
 * different feeds can accidentally have the same id
 * 
 * TODO: NOW CHANGED BY itemPtr feed_lookup_item_by_nr_extended() */

// itemPtr feed_lookup_item_by_id(feedPtr fp, gchar *id, gchar * orig_fp_id)
itemPtr
feed_lookup_item_by_nr_extended(feedPtr fp, gulong nr, gchar * orig_fp_id)
{
    GSList *items = NULL;
    itemPtr ip = NULL;

    if (FST_VFOLDER == feed_get_type(fp))
        ULOG_DEBUG("feed_lookup_item_by_id: Looking in a vfolder ");

    if ((0 == fp->loaded) && (FST_VFOLDER != feed_get_type(fp)))
        // feed_load(fp);
        ULOG_DEBUG
            ("feed_lookup_item_by_id: feed not loaded and feed not vfolder. \n\
                    something is wrong. Should we have loaded the feed first? ");

    items = fp->items;
    while (NULL != items) {
        ip = (itemPtr) (items->data);
        // ULOG_DEBUG("ip->id = %s, id = %s", ip->id, id);
        // ULOG_DEBUG("ip->id = %d, id = %d", ip->nr, nr);
        // just to make sure id is not NULL
        /* 
         * if (NULL == ip->id) { if ( ip->source == NULL)
         * ULOG_DEBUG("feed_lookup_item_by_id: This feed is fcuked, both \ id 
         * and source are NULL"); else item_set_id(ip, item_get_source(ip));
         * } */
        /* if we're searching for item from a virtual feed Then there's a
         * risk of there are 2 items found which are from 2 different feeds
         * but exactly the same feed. (like a feed is added twice */
        if (FST_VFOLDER == feed_get_type(fp)) {
            // if ( ( strcmp(ip->sourceFeed->id, orig_fp_id)==0) &&
            // (strcmp(ip->id, id) == 0) )
            if ((strcmp(ip->sourceFeed->id, orig_fp_id) == 0)
                && (ip->nr == nr))

                return ip;
        }
        /* normal feed */
        else {
            // if (strcmp(ip->id, id) == 0)
            if (ip->nr == nr)
                return ip;
        }
        items = g_slist_next(items);
    }

    return NULL;
}


gchar *feed_render(feedPtr fp)
{
    struct displayset displayset;
    gchar *buffer = NULL;
    gchar *tmp = NULL, *tmp2 = NULL;
    gboolean migration = FALSE;

    g_assert(0 != fp);
    g_assert(0 != fp->loaded);
    displayset.headtable = NULL;
    displayset.head = NULL;
    displayset.body = g_strdup(feed_get_description(fp));
    displayset.foot = NULL;
    displayset.foottable = NULL;

    /* I hope this is unique enough... */
    if ((NULL != displayset.body)
        && (NULL != strstr(displayset.body, "class=\"itemhead\"")))
        migration = TRUE;

    if (FALSE == migration) {
        metadata_list_render(fp->metadata, &displayset);

        /* Error description */
        if (NULL != (tmp = feed_get_error_description(fp))) {
            addToHTMLBufferFast(&buffer, tmp);
            g_free(tmp);
        }

        /* Head table */
        addToHTMLBufferFast(&buffer, HEAD_START);
        /* -- Feed line */
        if (feed_get_html_url(fp) != NULL)
            tmp = g_strdup_printf("<a href=\"%s\">%s</a>",
                                  feed_get_html_url(fp), feed_get_title(fp));
        else
            tmp = g_strdup(feed_get_title(fp));

        tmp2 = g_strdup_printf(HEAD_LINE, tmp);
        g_free(tmp);
        addToHTMLBufferFast(&buffer, tmp2);
        g_free(tmp2);

        /* -- Source line */
        if ((NULL != feed_get_source(fp))
            && (FST_VFOLDER != feed_get_type(fp))) {
            tmp =
                g_strdup_printf("<a href=\"%s\">%s</a>",
                                feed_get_source(fp), feed_get_source(fp));

            tmp2 = g_strdup_printf(HEAD_LINE, tmp);
            g_free(tmp);
            addToHTMLBufferFast(&buffer, tmp2);
            g_free(tmp2);
        }

        addToHTMLBufferFast(&buffer, displayset.headtable);
        g_free(displayset.headtable);
        addToHTMLBufferFast(&buffer, HEAD_END);

        /* Head */
        if (displayset.head != NULL) {
            addToHTMLBufferFast(&buffer, displayset.head);
            g_free(displayset.head);
        }

        /* feed/channel image */
        if (NULL != feed_get_image_url(fp)) {
            addToHTMLBufferFast(&buffer, "<img class=\"feed\" src=\"");
            addToHTMLBufferFast(&buffer, feed_get_image_url(fp));
            addToHTMLBufferFast(&buffer, "\"><br>");
        }
    }

    /* Body */
    if (displayset.body != NULL) {
        addToHTMLBufferFast(&buffer, displayset.body);
        g_free(displayset.body);
    }

    if (FALSE == migration) {
        /* Foot */
        if (displayset.foot != NULL) {
            addToHTMLBufferFast(&buffer, displayset.foot);
            g_free(displayset.foot);
        }

        if (displayset.foottable != NULL) {
            addToHTMLBufferFast(&buffer, FEED_FOOT_TABLE_START);
            addToHTMLBufferFast(&buffer, displayset.foottable);
            addToHTMLBufferFast(&buffer, FEED_FOOT_TABLE_START);
            g_free(displayset.foottable);
        }
    }
    return buffer;
}

void feed_free_requests(nodePtr ptr)
{
    GSList *iter;
    feedPtr fp = (feedPtr) ptr;
    g_assert(NULL != ptr);


    /* Don't free active feed requests here, because they might still be
     * processed in the update queues! Abandoned requests are free'd in
     * feed_process. They must be freed in the main thread for locking
     * reasons. */
    if (fp->request != NULL)
        fp->request->callback = NULL;

    /* same goes for other requests */
    iter = fp->otherRequests;
    while (NULL != iter) {
        ((struct request *) iter->data)->callback = NULL;
        iter = g_slist_next(iter);
    }
    g_slist_free(fp->otherRequests);
}

/* method to totally erase a feed, remove it from the config, etc.... */
gboolean feed_free(feedPtr fp, gboolean savechanges)
{
    gchar *filename = NULL;
    gboolean res = TRUE;
    /* This is not strictly necessary. It just speeds deletion of an entire
     * itemlist. */
    if (displayed_node == (nodePtr) fp) {
        ULOG_DEBUG("feed: feed_free");
        ui_mainwindow_show_statistics();
    }
    if (FST_VFOLDER == fp->type) {
        vfolder_free(fp);       /* some special preparations for vfolders */
    } else {
        // g_assert(fp->type == FST_FEED || fp->type == FST_FEED_DELETED);
    }

    /* free items */
    feed_remove_items(fp);

    if (fp->id && fp->id[0] != '\0')
        filename =
            common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
                                         fp->id, NULL);

    /* FIXME: Move this to a better place. The cache file does not need to
     * always be deleted, for example when freeing a feedstruct used for
     * updating. */
    if (filename && 0 != unlink(filename))
        /* Oh well.... Can't do anything about it. 99% of the time, this is
         * spam anyway. */ ;
    g_free(filename);

    feed_free_requests((nodePtr) fp);

    if (fp->icon != NULL)
        g_object_unref(fp->icon);

    if (fp->id) {
        favicon_remove(fp);
        conf_feedlist_save_config();
        g_free(fp->id);
    }
    if (fp->title)
        g_free(fp->title);
    if (fp->description)
        g_free(fp->description);
    if (fp->errorDescription)
        g_free(fp->errorDescription);
    if (fp->source)
        g_free(fp->source);
    if (fp->filtercmd)
        g_free(fp->filtercmd);
    if (fp->htmlUrl)
        g_free(fp->htmlUrl);
    if (fp->imageUrl)
        g_free(fp->imageUrl);
    if (fp->parseErrors)
        g_free(fp->parseErrors);
    if (fp->etag)
        g_free(fp->etag);
    metadata_list_free(fp->metadata);
    g_hash_table_destroy(fp->tmpdata);
    g_free(fp);
    return res;
}

void feed_set_new_subscription(feedPtr fp, gboolean new_subscription)
{
    g_assert(fp != NULL);

    fp->new_subscription = new_subscription;
    if (new_subscription == FALSE) {
        /*
         * a bit hackish, but this is the point where new
         * feed has been added to ui_feed structures
         * first time, so that save_config can write it to file
         */
        conf_feedlist_save_config();
    }
}

gboolean feed_get_new_subscription(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->new_subscription;
}

void feed_set_feed_directory(feedPtr fp, gboolean feed_directory)
{
    g_assert(fp != NULL);

    fp->feed_directory = feed_directory;
}

gboolean feed_get_feed_directory(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->feed_directory;
}

gboolean feed_get_parse_errors(feedPtr fp)
{
    g_assert(fp != NULL);

    return fp->err;
}

void download_image_link(gchar * link, gpointer user_data)
{
    // ULOG_DEBUG("Adding image link to download: <%s>", link);
    download_queue_image(link, NULL, FALSE);
}

void feed_download_images(feedPtr fp)
{
    g_assert(fp != NULL);

    GSList *iter = NULL;

    feed_load(fp);
    iter = fp->items;
    while (iter) {
        if (!item_get_hidden((itemPtr) iter->data))
            find_image_links((gchar *)
                             item_get_description((itemPtr) iter->data),
                             ((itemPtr) iter->data)->source,
                             download_image_link, NULL);
        iter = g_slist_next(iter);
    }
    feed_unload(fp);
}
