/**
 * @file html.c HTML file handling / feed auto discovery
 * 
 * Copyright (C) 2004 ahmed el-helw <ahmedre@cc.gatech.edu>
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE
#include <stdlib.h>
#include "common.h"
#include "debug.h"
#include "html.h"

#include <osso-log.h>
#include <string.h>
#include <strings.h>
#include <libgnomevfs/gnome-vfs.h>


/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

static gchar *checkLinkRef(const gchar * str)
{
    gchar *res = NULL;
    gchar *tmp = NULL, *tmp2 = NULL;

    tmp = strstr(str, "href=");
    if (NULL == tmp)
        tmp = strstr(str, "HREF=");
    if (NULL == tmp)
        return NULL;
    /* FIXME: single quotes support */
    tmp2 = strchr(tmp, '\"');
    if (NULL == tmp2)
        return NULL;
    tmp = strchr(tmp2 + 1, '\"');
    if (tmp == NULL)
        return NULL;

    res = g_strndup(tmp2 + 1, tmp - tmp2 - 1);

    if ((strstr(str, "alternate") != NULL) &&
        ((strstr(str, "text/xml") != NULL) ||
         (strstr(str, "rss+xml") != NULL) ||
         (strstr(str, "rdf+xml") != NULL) ||
         (strstr(str, "atom+xml") != NULL)))
        return res;

    g_free(res);
    return NULL;
}

static gchar *checkLinkRefForFavicon(const gchar * str)
{
    gchar *res = NULL;
    gchar *tmp = NULL, *tmp2 = NULL;

    tmp = strstr(str, "href=");
    if (NULL == tmp)
        tmp = strstr(str, "HREF=");
    if (NULL == tmp)
        return NULL;

    /* FIXME: single quotes support */
    tmp2 = strchr(tmp, '\"');
    if (NULL == tmp2)
        return NULL;

    tmp = strchr(tmp2 + 1, '\"');
    if (tmp == NULL)
        return NULL;
    
    res = g_strndup(tmp2 + 1, tmp - tmp2 - 1);

    if ((strstr(str, "shortcut icon") != NULL || strstr(str, "icon") != NULL))
        return res;

    g_free(res);

    return NULL;
}

static gchar *checkNormalLink(const gchar * str)
{
    gchar *res = NULL, *tmp = NULL, *tmp2 = NULL;

    tmp = strstr(str, "href=");
    if (NULL == tmp)
        tmp = strstr(str, "HREF=");
    if (NULL == tmp)
        return NULL;
    /* FIXME: single quotes support */
    tmp2 = strchr(tmp, '\"');
    if (NULL == tmp2)
        return NULL;
    tmp = strchr(tmp2 + 1, '\"');
    if (tmp == NULL)
        return NULL;
    res = g_strndup(tmp2 + 1, tmp - tmp2 - 1);

    if ((strstr(res, "rdf")) || (strstr(res, "xml")) || (strstr(res, "rss")))
        return res;
    g_free(res);
    return NULL;
}

static gchar *search_links(const gchar * data, int type, gboolean favicon)
{
    gchar *ptr = NULL;
    const gchar *tmp = data;
    gchar *result = NULL;
    gchar *res = NULL;
    gchar *tstr = NULL;
    gchar *endptr = NULL;

    while (1) {
        ptr = strstr(tmp, ((type == 0) ? "<link " : "<a "));
        if (NULL == ptr)
            ptr = strstr(tmp, ((type == 0) ? "<LINK " : "<A "));
        if (NULL == ptr)
            break;

        endptr = strchr(ptr, '>');
        if (endptr == NULL)
            return NULL;
	tstr = g_strndup(ptr, endptr - ptr);

        res = ((type == 0) ?
               (favicon ? checkLinkRefForFavicon(tstr) : checkLinkRef(tstr))
               : checkNormalLink(tstr));
        g_free(tstr);
        if (res != NULL) {
            result = res;
            break;
        }
        tmp = endptr;
    }
    return result;
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

gchar *html_auto_discover_feed(const gchar * data, const gchar * baseUri)
{
    gchar *res = NULL, *tmp = NULL;

    res = search_links(data, 0, FALSE);
    if (res == NULL) {
        res = search_links(data, 1, FALSE);
    }

    if (res != NULL) {
        /* turn relative URIs into absolute URIs */
        tmp = res;
        res = (gchar *)common_build_url((const gchar *)res, baseUri);
        g_free(tmp);
    }

    return res;
}

gchar *html_discover_favicon(const gchar * data, const gchar * baseUri)
{
    gchar *res = NULL, *tmp = NULL;

    res = search_links(data, 0, TRUE);

    if (res != NULL) {
        /* turn relative URIs into absolute URIs */
        tmp = res;
        res = (char *)common_build_url((const char *)res, baseUri);
        g_free(tmp);
    }

    return res;
}


static gchar *_getsrc(gchar *tag) {
    gchar *startpoint = NULL;
    gchar *tmp= NULL;
    gchar *endpoint = NULL;
    gchar *image = NULL;
    gint len = 0;

    tmp = (gchar *)strcasestr(tag,"src");
    if (tmp != NULL) {
        startpoint = (gchar *) strcasestr(tmp,"\"");
        if (startpoint != NULL) {
            startpoint++;
            /* weed out urls that have http:// in them .. they are covered in another scan */
            if (strcasestr(startpoint,"http") == NULL) {
                endpoint = (gchar *) strcasestr( (gchar *) (startpoint+1),"\\\"");
                if (endpoint == NULL) {
                    endpoint = (gchar *) strcasestr( (gchar *) (startpoint+1),"\"");
                }
    
                if (endpoint != NULL) {
                    len = (gint) (endpoint-startpoint);
                    if (len>0) {
                        image = g_strndup(startpoint, len);                        
                    }
                }
            }
        }
    }
    return image;
}



static gchar *_combine_urls(gchar *base, gchar *name) {
    return (gchar *)common_build_url((const char*)name,(const char*)base );
}



void find_image_links(gchar * buff, gchar *baseUrl, HtmlImageCallback callback,
                      gpointer user_data)
{
    gchar *p = NULL;
    gchar *n = NULL;
    gchar *imgptr = NULL;
    gchar *link = NULL;
    gchar *buf = NULL;
    gchar *end = NULL;
    gint size;
    gchar oldchar;

    if (!buff)
        return;

    buf = buff;
    while ((p = strcasestr(buff, "http"))) {
        if (((strncmp(p + 4, "://", 3) == 0) ||
             (strncasecmp(p + 4, "s://", 4) == 0)) &&
            (n = strpbrk(p + 7, "\"'")) &&
            (imgptr = memrchr(buff, '<', p - buff))) {
            oldchar = *p;
            *p = 0;
            imgptr = strcasestr(imgptr, "img");
            *p = oldchar;

            if (imgptr) {
                size = n - p;
        		link = g_strndup(p, size);
                callback(link, user_data);
                g_free(link);
            }
        }

        buff = p + 1;
    }
  
    /* This is a hack. For later time these two whiles should be incorporated together */ 
    while ( (p =  (gchar *) strcasestr(buf,"<img")) ) {
        buf = p + 1;
        end = (gchar *) strcasestr(p,">");
        if (end != NULL) {
            size = (int) (end-p)+1;
            if (size>1) {
                link = g_strndup(p,size);
                if (link != NULL) {
                    imgptr = _getsrc(link);
                    if (imgptr != NULL) {
                        n = _combine_urls(baseUrl,imgptr); 
                        if ( n != NULL ) {
                            callback(n,user_data);
                            xmlFree(n);
                        }   
                        g_free(imgptr);
                    }
                    g_free(link);
                }
            }
        }
    }
 



}
