/**
 * @file cache.c File cache
 *
 * Copyright (c) 2005-06 Nokia Corporation. All rights reserved.
 * Contact: Ouyang Qi <qi.ouyang@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Initial developer(s): Zsolt Simon
 */

#define _GNU_SOURCE
#include <stdlib.h>
#include <sys/vfs.h>
#include <fcntl.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <osso-log.h>
#include <glib.h>
#include <glib-object.h>
#include <libgnomevfs/gnome-vfs.h>

#include "cache.h"

const gint MAGICHEAD = 0xBC2002FC;

/**
 * Global variables
 */

static GObjectClass *parent_class = NULL;

/**
 * Internal functions forward declarations 
 */

static void file_cache_finalize(GObject * object);

void file_cache_fill(FileCache * cache);

void cache_entry_destroy(gpointer data);
gint cache_entry_cmp(gconstpointer a, gconstpointer b, gpointer user_data);
static gchar *strip_url(const gchar * url);
gboolean remove_file(gpointer key, gpointer value, gpointer user_data);
gboolean remove_by_id(gpointer key, gpointer value, gpointer user_data);
void remove_unwanted_entries(FileCache * cache, gint size);
static gboolean write_str_safe(gint fd, const gchar * buffer);
static gboolean write_safe(gint fd, const void *buffer, const gint buffer_size);
static gboolean read_str_safe(gint fd, gchar ** result, guint max_size);
static gboolean read_safe(gint fd, void *buffer, gint buffer_size);
void file_cache_init(FileCache * self);
/**
 * Object related stuff
 */

//Class constructor
static void file_cache_class_init(FileCacheClass * c)
{
    GObjectClass *object_class;

    object_class = G_OBJECT_CLASS(c);

    parent_class = g_type_class_ref(G_TYPE_OBJECT);
    object_class->finalize = file_cache_finalize;
}

//Destructor
static void file_cache_finalize(GObject * object)
{
    FileCache *cache = FILE_CACHE(object);

    //maybe need to ensure that no threads are using the cache anymore
    g_free(cache->cache_dir);
    g_queue_free(cache->queue);
    g_hash_table_destroy(cache->hash);


    parent_class->finalize(object);
}

//Object constructor
void file_cache_init(FileCache * self)
{
    self->max_size = 64 * 1024 * 1024;  //64M by default
    self->cache_dir = g_strdup_printf("%s" G_DIR_SEPARATOR_S ".cache",
                                      g_get_home_dir());
    self->hash = g_hash_table_new_full(g_str_hash, g_str_equal,
                                       NULL, cache_entry_destroy);
    self->queue = g_queue_new();
}

//Object type
GType file_cache_get_type(void)
{
    static GType type = 0;

    if (type == 0) {
        static const GTypeInfo type_info = {
            sizeof(FileCacheClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc) file_cache_class_init, /* class_init */
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof(FileCache),
            0,                  /* n_preallocs */
            (GInstanceInitFunc) file_cache_init /* instance_init */
        };

        type = g_type_register_static(G_TYPE_OBJECT, "FileCacheType",
                                      &type_info, 0);
    }

    return type;
}

FileCache *file_cache_new(const gchar * cache_dir, size_t max_size)
{
    FileCache *cache = g_object_new(FILE_CACHE_TYPE, 0);

    cache->max_size = max_size;
    if (cache_dir) {
        g_free(cache->cache_dir);
        cache->cache_dir = g_strdup(cache_dir);
    }

    g_mkdir_with_parents(cache->cache_dir, 0755);

    file_cache_fill(cache);

    return cache;
}

void file_cache_destroy(FileCache * cache)
{
    g_assert(IS_FILE_CACHE(cache));

    g_object_unref(G_OBJECT(cache));
}

static gboolean read_safe(gint fd, void *buffer, gint buffer_size)
{
    return read(fd, buffer, buffer_size) == buffer_size;
}

static gboolean read_str_safe(gint fd, gchar ** result, guint max_size)
{
    guint len = 0;

    if ( result ) *result = NULL;
    else return FALSE;

    if (read_safe(fd, &len, sizeof(len)) && len < max_size) {
        if (0 == len) {
            return TRUE;
        }
        else {
            gchar* str = g_new(gchar, len + 1);
            if (str) {
                if (read_safe(fd, str, len)) {
                    str[len] = 0;
                    *result  = str;
                    return TRUE;
                }
                g_free(str);
            }
        }
    }

    return FALSE;
}

/*
 * Fill the cache object from the specified cache directory
 */
void file_cache_fill(FileCache * cache)
{
    DIR *dir;
    struct dirent *dentry;
    CacheEntry *entry;
    gchar *fn;
    gint fd;
    gint fsize;
    gint magic;
    gchar *url;

    dir = opendir(cache->cache_dir);
    while (dir && (dentry = readdir(dir))) {
        //we intrested only in files
        if (dentry->d_type == 8 /*DT_REG */ ) {
            url = NULL;
            entry = NULL;

            fn = g_strconcat(cache->cache_dir, G_DIR_SEPARATOR_S,dentry->d_name, NULL);
            if (-1 != (fd = open(fn, O_RDONLY)))
            {
                fsize = lseek(fd, 0, SEEK_END);
                lseek(fd, 0, SEEK_SET);
                if (read_safe(fd, &magic, sizeof(magic)) && (magic == MAGICHEAD)
                    && read_str_safe(fd, &url, fsize)) {
                    if (url) {
                        entry = g_new0(CacheEntry, 1);
                        if (read_str_safe(fd, &entry->id, fsize) &&
                            read_safe(fd, &entry->refcount,
                                      sizeof(gint) + sizeof(GTimeVal))) {
                            entry->url = url;
                            entry->filename = fn;
                            entry->size = fsize;
                            cache->current_size += fsize;
                        } else {
                            g_free(entry->id);
                            g_free(entry);
                            entry = NULL;
                        }
                    } else
                        g_free(url);
                }
                close(fd);
            }

            if (entry) {
                //maybe search in the cache first to remove duplicate files
                g_hash_table_insert(cache->hash, url, entry);
                g_queue_push_tail(cache->queue, entry);
                entry->link = g_queue_peek_tail_link(cache->queue);
            } else {
                //remove unwanted items from the cache directory
                unlink(fn);
                g_free(fn);
            }
        }
    }

    if (dir)
        closedir(dir);
    g_queue_sort(cache->queue, cache_entry_cmp, NULL);
}

static gboolean
write_safe(gint fd, const void *buffer, const gint buffer_size)
{
    return buffer_size == write(fd, buffer, buffer_size);
}

static gboolean write_str_safe(gint fd, const gchar * buffer)
{
    gint len = buffer ? strlen(buffer) : 0;
    return write_safe(fd, &len, sizeof(len)) &&
        len ? write_safe(fd, buffer, len) : TRUE;
}

void remove_unwanted_entries(FileCache * cache, gint size)
{
    CacheEntry *entry = NULL;
    while (size > 0 && (entry = CACHE_ENTRY(g_queue_pop_head(cache->queue)))) {
	size -= entry->size;
	cache->current_size -= entry->size;
	unlink(entry->filename);
	//TODO: Check why the entry->url could be NULL in some reason
	if (cache->hash && entry->url) {
	    g_hash_table_remove(cache->hash, entry->url);
	}
    }
}

gboolean
file_cache_add(FileCache * cache, const gchar * lurl, const gchar * id,
               const gpointer buffer, const size_t buffer_size)
{
    CacheEntry *entry = NULL;
    struct statfs info;
    gchar *tmp;
    gint fd;
    size_t size;
    gchar *url = strip_url(lurl);

    g_assert(IS_FILE_CACHE(cache));

    if (url && cache->max_size && buffer && buffer_size) {

        if ((entry = g_hash_table_lookup(cache->hash, url))) {
            unlink(entry->filename);
            cache->current_size -= entry->size;
            g_queue_unlink(cache->queue, entry->link);
            g_hash_table_remove(cache->hash, url);
        }

        if (statfs(cache->cache_dir, &info) == 0) {
            //calculate the total size
            size = strlen(url) + sizeof(gint) +
                (id ? strlen(id) : 0) + sizeof(gint) +
                sizeof(gint) + sizeof(GTimeVal) + buffer_size;

            //try to free up the cache a little bit, if it is full
            if (cache->current_size + size > cache->max_size &&
                size < cache->max_size)
                remove_unwanted_entries(cache, size);

            //check if there is enough space
            if (cache->current_size + size <= cache->max_size &&
                info.f_bavail * info.f_bsize > size) {
                tmp = g_strconcat(cache->cache_dir,G_DIR_SEPARATOR_S, "chXXXXXX", NULL);
                fd = mkstemp(tmp);

                if (fd) {
                    entry = g_new0(CacheEntry, 1);
                    g_get_current_time(&entry->time);

                    //write the header + content
                    if (write_safe(fd, &MAGICHEAD, sizeof(MAGICHEAD)) &&
                        write_str_safe(fd, url) &&
                        write_str_safe(fd, id) &&
                        write_safe(fd, &entry->refcount,
                                   sizeof(gint) + sizeof(GTimeVal)) &&
                        write_safe(fd, buffer, buffer_size)) {
                        entry->id = g_strdup(id);
                        entry->filename = tmp;
                        entry->size = lseek(fd, 0, SEEK_CUR);
                        cache->current_size += entry->size;
                        entry->url = g_strdup(url);

                        close(fd);
                        g_hash_table_insert(cache->hash, entry->url, entry);
                        g_queue_push_tail(cache->queue, entry);
                        entry->link = g_queue_peek_tail_link(cache->queue);


                        g_free(url);
                        return TRUE;
                    } else {
                        //there were not enough space, some application was
                        //faster then us :)
                        close(fd);
                        unlink(tmp);

                        g_free(tmp);
                        g_free(entry);
                    }
                }
            }
        }

    }

    g_free(url);

    return FALSE;
}

gboolean file_cache_remove(FileCache * cache, const gchar * lurl)
{
    CacheEntry *entry = NULL;
    gchar *url = strip_url(lurl);

    g_assert(IS_FILE_CACHE(cache));


    if (url && cache->max_size
        && (entry = g_hash_table_lookup(cache->hash, url))) {
        unlink(entry->filename);

        cache->current_size -= entry->size;
        g_queue_unlink(cache->queue, entry->link);
        g_hash_table_remove(cache->hash, url);


        g_free(url);

        return TRUE;
    }


    g_free(url);

    return FALSE;
}

typedef struct _RemoveByIdData RemoveByIdData;
struct _RemoveByIdData {
    FileCache *cache;
    gchar *id;
};

gboolean remove_by_id(gpointer key, gpointer value, gpointer user_data)
{
    CacheEntry *entry = CACHE_ENTRY(value);
    RemoveByIdData *d = (RemoveByIdData *) user_data;

    if (entry->id && (strcmp(entry->id, d->id) == 0)) {
        g_queue_unlink(d->cache->queue, entry->link);
        d->cache->current_size -= entry->size;
        unlink(entry->filename);
        return TRUE;
    }

    return FALSE;
}

void file_cache_remove_by_id(FileCache * cache, gchar * id)
{
    RemoveByIdData d;

    g_assert(IS_FILE_CACHE(cache));


    if (id) {
        d.id = id;
        d.cache = cache;
        g_hash_table_foreach_remove(cache->hash, remove_by_id,
                                    (gpointer) & d);
    }

}

gboolean remove_file(gpointer key, gpointer value, gpointer user_data)
{
    CacheEntry *entry = CACHE_ENTRY(value);
    unlink(entry->filename);
    return TRUE;
}

void file_cache_clear(FileCache * cache)
{
    g_assert(IS_FILE_CACHE(cache));


    while (g_queue_pop_head(cache->queue)) ;
    g_hash_table_foreach_remove(cache->hash, remove_file, NULL);

    cache->current_size = 0;

}

gboolean file_cache_can_resize(FileCache * cache, size_t new_size)
{
    g_assert(IS_FILE_CACHE(cache));

    gboolean result = FALSE;

    if (new_size > cache->max_size) {
        struct statfs info;

        result =
            (statfs(cache->cache_dir, &info) == 0) &&
            (info.f_bavail * info.f_bsize > new_size - cache->max_size);
    } else
        result = TRUE;


    return result;
}

void file_cache_resize(FileCache * cache, size_t new_size)
{
    g_assert(IS_FILE_CACHE(cache));

    if (!new_size) {
        //remove the entire cache
        while (g_queue_pop_head(cache->queue)) ;
        g_hash_table_foreach_remove(cache->hash, remove_file, NULL);
    } else {
        //remove the excess elements from the cache if needed
        if (cache->current_size > new_size)
            remove_unwanted_entries(cache,
                                    cache->current_size - new_size);
    }
    cache->max_size = new_size;
}

size_t file_cache_size(FileCache * cache)
{
    g_assert(IS_FILE_CACHE(cache));

    return cache->max_size;
}

static gchar *strip_url(const gchar * url)
{
    if (!url)
        return NULL;

    gchar *result = NULL;
    gchar *p = strcasestr(url, "//www.");
    if (p && p - url < 8) {
        gchar buf[10];
        strncpy(buf, url, p - url + 2);
        buf[p - url + 2] = 0;
        result = g_strconcat(buf, p + 6, NULL);
    } else
        result = g_strdup(url);

    return result;
}


gpointer
file_cache_find(FileCache * cache, const gchar * lurl, size_t * size,
                gboolean increase_statistic)
{
    gpointer result = NULL;
    CacheEntry *entry = NULL;
    gint fsize;
    gint cpos;
    gint fd = 0;
    gint len;
    gchar *url = strip_url(lurl);

    g_assert(IS_FILE_CACHE(cache));

    //    ULOG_DEBUG("\n\n\n\n#####Cache find: %s, %d", url, cache->max_size);

    if (url && cache->max_size
        && (entry = g_hash_table_lookup(cache->hash, url))) {
        entry->refcount++;
	fd = open(entry->filename, O_RDWR);
        //    ULOG_DEBUG("#####Cache find: found");
        if (fd != -1) {
            fsize = lseek(fd, 0, SEEK_END);
            lseek(fd, sizeof(MAGICHEAD) +
                  sizeof(gint) + strlen(url) +
                  sizeof(gint) + (entry->id ? strlen(entry->id) : 0),
                  SEEK_SET);
            //write the usage count back to the file
            if (((!increase_statistic)
                 &&
                 ((cpos =
                   lseek(fd, sizeof(GTimeVal) + sizeof(gint),
                         SEEK_CUR)) <= fsize))
                || (write_safe(fd, &entry->refcount, sizeof(gint))
                    && ((cpos = lseek(fd, sizeof(GTimeVal), SEEK_CUR)) <=
                        fsize))) {
                len = fsize - cpos;
                if (size)
                    *size = len;
                result = (gpointer) g_try_malloc(len);
                if (result) {
                    if (!read_safe(fd, result, len))
                    {
                        ULOG_DEBUG("read failed");
                    }
                    close(fd);
                    //    ULOG_DEBUG("#####Cache find: loaded");
                    g_free(url);
                    return result;
                }
            }

        }
        //    ULOG_DEBUG("#####Cache find: something wrong");
        //damaged hash entry ... remove it
        g_queue_unlink(cache->queue, entry->link);
        g_hash_table_remove(cache->hash, url);
    }

    if (size)
        *size = 0;

    g_free(url);
    return NULL;
}

gboolean file_cache_exists(FileCache * cache, const gchar * lurl)
{
    g_assert(IS_FILE_CACHE(cache));

    gchar *url = strip_url(lurl);
    gboolean result = g_hash_table_lookup(cache->hash, url) != NULL;
    g_free(url);
    return result;
}

void cache_entry_destroy(gpointer data)
{
    CacheEntry *entry = CACHE_ENTRY(data);

    g_free(entry->id);
    g_free(entry->filename);
    g_free(entry->url);

    g_free(entry);
}

gint cache_entry_cmp(gconstpointer a, gconstpointer b, gpointer user_data)
{
    const CacheEntry *c1 = a;
    const CacheEntry *c2 = b;

    if (a == b)
        return 0;
    if (c1->time.tv_sec == c2->time.tv_sec)
        return c1->time.tv_usec - c2->time.tv_usec;
    return c1->time.tv_sec - c2->time.tv_sec;
}
