/**
	@file callbacks.c
	Various callbacks for dealing with the GUI events in Date & Time
	Settings CPA.

	Copyright (c) 2004-2007 Nokia Corporation.
	Parts of code are from Liferea.
	Liferea (C) 2003,2004 Lars Lindner <lars.lindner@gmx.net>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License 
	version 2 or later, as published by the Free Software Foundation. 

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
	02110-1301 USA
*/

#include <sys/vfs.h>
#include <assert.h>
#include <stdlib.h>
#include <osso-log.h>
#include <gdk/gdkkeysyms.h>
#include <hildon/hildon-banner.h>
#include <dbus/dbus-glib.h>

#include "interface.h"
#include "settings_database.h"
#include "callbacks.h"
#include "settings.h"
#include "i18n.h"
#include "debug.h"

/* Local interface */

gchar *
device_name (void);

/* Implementation */
gboolean dialog_key_press(GtkWidget* widget, GdkEventKey* event, gpointer user_data) 
{
    if (event->keyval == GDK_Escape)
    {
        /* Emit response from the dialog */
        gtk_dialog_response(GTK_DIALOG(widget), GTK_RESPONSE_CANCEL);
        return TRUE;
    }
  
    return FALSE;
}


void combobox_changed(GtkWidget *widget, gpointer user_data)
{
    Value value;
    gchar* key = (gchar*)user_data;
    
    value.type = TYPE_INTEGER;
    value.integer = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
    
    /* Set value for the given key in SettingsDatabase */
    if (!settings_set_value(key, &value))
    {
        ULOG_ERR("Error in setting value %s.", key);
    }
}


void checkbox_toggled(GtkToggleButton* togglebutton, gpointer user_data)
{
    Value value;
    gchar* key = (gchar*)user_data;
    
    value.type = TYPE_BOOLEAN;
   // value.boolean = gtk_toggle_button_get_active(togglebutton);
     value.boolean = hildon_check_button_get_active(togglebutton);
    
    /* Set value for the given key in SettingsDatabase */
    if (!settings_set_value(key, &value))
    {
        ULOG_ERR("Error in setting value %s.", key);
    }

    ui_enable_widgets(value.boolean);
}

gchar *
device_name (void)
{
    DBusGConnection *connection;
    GError *error;
    DBusGProxy *proxy;
    gchar *name;

    g_type_init ();

    error = NULL;
    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM,
                                 &error);
    if (connection == NULL)
    {
        /*printf ("Failed to open connection to bus: %s\n",
                    error->message);*/
        g_error_free (error);
        return "";
    }

    proxy = dbus_g_proxy_new_for_name (connection,
                                       "org.bluez",
                                       "/org/bluez/hci0",
                                       "org.bluez.Adapter");

    error = NULL;
    if (!dbus_g_proxy_call (proxy, "GetName", &error, G_TYPE_INVALID,
                            G_TYPE_STRING, &name, G_TYPE_INVALID))
    {
        g_error_free (error);
        return NULL;
    }

    printf("Device name: %s\n", name);

    g_object_unref (proxy);

    return name;
}

void on_dialog_response(GtkWidget * prefdialog, gint responseId,
        gpointer user_data)
{
    gboolean close = TRUE;
    Iphb_cb_data *iphb_data = (Iphb_cb_data *)user_data;

    if (responseId == GTK_RESPONSE_CANCEL ||
        responseId == GTK_RESPONSE_DELETE_EVENT) 
    {
        ui_destroy();
        ULOG_ERR("Settings dialog was canceled");
        return;
    }

    Value *v;
    size_t new_size = 0;
    v = settings_get_value(S_CACHE_SIZE);
    if (v && v->type == TYPE_INTEGER && 
        ((new_size = all_cache_size[v->integer]) > 
                all_cache_size[cache_size]))
    {
        struct statfs info;
        gboolean can_resize = ((statfs("/home/", &info) == 0) && 
            (info.f_bavail * info.f_bsize > 
                (new_size - all_cache_size[cache_size]) * 1024 * 1024));

        ULOG_DEBUG("new_size = %u, cache_size = %u, all_cache_size[cache_size] = %u",new_size,cache_size, all_cache_size[cache_size]);
        if (!can_resize)
        {
            ui_reset_cache(cache_size);
            
	    gchar *tmp = NULL;
            const gchar *fmt = _("rss_ni_unable_increase_img_cache");
            gchar *name = device_name();
	    if(NULL == name || name == ".")
	    {
            	tmp = g_strdup_printf(fmt,"device");
	    }
	    else
	    {
            	//gchar *tmp = g_strdup_printf(fmt, name ? name : "");
            	tmp = g_strdup_printf(fmt, name);
	    }
            hildon_banner_show_information(GTK_WIDGET(prefdialog),NULL, tmp);
            g_free(name);
            g_free(tmp);
            
            close = FALSE;
        }
    }
    
    if (close)
    {
		DMSG("Entering close\n");
        gboolean modified = FALSE;
        
        Value *v;
        v = settings_get_value(S_AUTOMATIC_UPDATES);
        if (v && v->type == TYPE_BOOLEAN)
        {
            modified = gconf_client_get_bool(gconf, 
                RSS_SETTINGS_AUTOMATIC_UPDATES, NULL) != v->boolean;
            gconf_client_set_bool(gconf, 
                        RSS_SETTINGS_AUTOMATIC_UPDATES, v->boolean, NULL);
        }
		DMSG("Modified value: %d\n",modified);
        v = settings_get_value(S_CACHE_SIZE);
        if (v && v->type == TYPE_INTEGER)
            gconf_client_set_int(gconf, RSS_SETTINGS_IMAGE_CACHE_SIZE,
                all_cache_size[v->integer], NULL);

        v = settings_get_value(S_CONN_INTERVAL);
        if (v && v->type == TYPE_INTEGER)
        {
            modified |= gconf_client_get_int(gconf, 
                RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL, NULL) != v->integer;
            gconf_client_set_int(gconf, RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
                v->integer, NULL);
        }
		DMSG("Modified value 2: %d\n",modified);
        v = settings_get_value(S_CONN_TYPE);
        if (v && v->type == TYPE_INTEGER)
        {
            modified |= gconf_client_get_int(gconf, 
                RSS_SETTINGS_AUTOMATIC_UPDATES_TYPE, NULL) != v->integer;
            gconf_client_set_int(gconf, RSS_SETTINGS_AUTOMATIC_UPDATES_TYPE,
                v->integer, NULL);
        }
		DMSG("Modified value 3: %d\n",modified);
        
#if 0
	v = settings_get_value(S_AUTOMATIC_SCROLLING);
        if (v && v->type == TYPE_BOOLEAN)
        {
            gconf_client_set_bool(gconf, 
                        RSS_SETTINGS_AUTOMATIC_SCROLLING, v->boolean, NULL);
        }
#endif
	
	DMSG("Modified value 4: %d\n",modified);
        v = settings_get_value(S_OPEN_FEEDS_TO);
        if (v && v->type == TYPE_INTEGER)
        {
            gconf_client_set_int(gconf, RSS_SETTINGS_OPEN_FEEDS_TO,
                v->integer, NULL);
        }

        DMSG("--========== Alarm setup (%d) =========--\n", modified);
        if (modified) {
            gconf_client_clear_cache(gconf);
	    close_iphb_handle((Iphb_cb_data)(*iphb_data));
	    remove_iphb_callback(iphb_data->source_id);
	    iphb_data->source_id = 0;
            setup_alarm_event(TRUE, iphb_data);
	    iphb_data->source_id = setup_iphb_callback(iphb_data);
        }
        
        ui_destroy();
    }
}
