/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include "gmozillaconnectivity.h"
#include "common.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"
#include "gmozillatransferitem.h"
#include <string.h>
#include "gmozillacppwrapper.h"

#ifdef USE_GCONF
#include <gconf/gconf-client.h>
#endif

#ifdef USE_GCONF
#define GCONF_PROXY_AUTOCONFIG_URL  "/system/proxy/autoconfig_url"
#define GCONF_PROXY_IGNORE_HOSTS    "/system/proxy/ignore_hosts"
#define GCONF_HTTP_PROXY_HOST       "/system/http_proxy/host"
#define GCONF_HTTP_PROXY_PORT       "/system/http_proxy/port"
#define GCONF_SECURE_PROXY_HOST     "/system/proxy/secure_host"
#define GCONF_SECURE_PROXY_PORT     "/system/proxy/secure_port"
#define GCONF_FTP_PROXY_HOST        "/system/proxy/ftp_host"
#define GCONF_FTP_PROXY_PORT        "/system/proxy/ftp_port"
#define GCONF_SYSTEM_PROXY_MODE     "/system/proxy/mode"
#endif // USE_GCONF
#define GCONF_USE_HTTP_PROXY        "/system/http_proxy/use_http_proxy"

static void
g_mozilla_connectivity_conf_handler_func (GConfClient * client, guint cnxn_id, GConfEntry * entry, GMozillaWeb *self)
{
   g_debug("Changed GConf proxy settings\n");
   g_timeout_add(1000, g_mozilla_connectivity_set_proxy, self);
}

void
g_mozilla_connectivity_set_global (GMozillaWeb *self)
{
    TRACE_LOG();
    const char *val = GCONF_USE_HTTP_PROXY;
    GConfClient *client = gconf_client_get_default();
    gconf_client_add_dir(client, val, GCONF_CLIENT_PRELOAD_NONE, NULL);
    guint id = gconf_client_notify_add(client, val,
                                     (GConfClientNotifyFunc) g_mozilla_connectivity_conf_handler_func, 
                                     self, NULL, NULL);
    g_object_set_data(G_OBJECT(self), "gconf_notify_id", (void*)id);
}

void
g_mozilla_connectivity_unset_global (GMozillaWeb *self)
{
    TRACE_LOG();
    GConfClient *client = gconf_client_get_default();
    gconf_client_notify_remove(client, (guint)g_object_steal_data(G_OBJECT(self), "gconf_notify_id"));
}

static gboolean
g_mozilla_parse_env_proxy (const gchar *env_name, const gchar *moz_proto)
{
    g_return_val_if_fail(env_name && moz_proto, FALSE);
    TRACE_LOG(">>>>>>>>>>.Env: %s > %s", env_name, moz_proto);
    const gchar * env = getenv(env_name);
    if (!env || !strlen(env))
        return FALSE;
    
    // match the variable value against URL pattern
    GRegex* regex = g_regex_new("((\\w+)://)?(?P<host>[^:/]+)(:(?P<port>\\d+))?(/.*)?$", 0, 0, NULL);
    GMatchInfo* match_info;
    gboolean matched = g_regex_match(regex, env, 0, &match_info);
    if (matched) {
        // set proxy hostname
        gchar* host = g_match_info_fetch_named(match_info, "host");
        gchar* proxy_param = g_strdup_printf(G_MOZILLA_NETWORK_PROXY_PARAM, moz_proto);
        g_mozilla_engine_set_pref(G_TYPE_STRING, proxy_param, host);
        g_free(proxy_param);
        g_free(host);
        // set proxy port
        gchar* port_string = g_match_info_fetch_named(match_info, "port");
        int port = port_string? atoi(port_string) : 0;
        gchar* proxy_param_port = g_strdup_printf(G_MOZILLA_NETWORK_PROXY_PARAM_PORT, moz_proto);
        g_mozilla_engine_set_pref(G_TYPE_INT, proxy_param_port, &port);
        g_free(proxy_param_port);
        g_free(port_string);
    }

    g_match_info_free(match_info);
    g_regex_unref(regex);    
    return matched;
}

gboolean g_mozilla_connectivity_set_proxy (void)
{
    TRACE_LOG();
    gchar *proxytype = NULL, *autoconf = NULL;
    gchar *proxy_http = NULL, *proxy_https = NULL, *proxy_ftp = NULL;
    gint proxy_http_port, proxy_https_port, proxy_ftp_port;
    gchar *omit_proxy = NULL;
    gint gval = 0;
#ifdef USE_GCONF
    GConfClient *gconf_client;
    gconf_client = gconf_client_get_default();
    proxytype = gconf_client_get_string(gconf_client,
                                        GCONF_SYSTEM_PROXY_MODE,
                                        NULL);
#endif
    gval = 1;
    g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    if (getenv(G_MOZILLA_HTTP_PROXY)) {
        if (g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_HTTP)) {
            if (!g_mozilla_parse_env_proxy (G_MOZILLA_FTP_PROXY, G_MOZILLA_FTP))
                g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_FTP);
            if (!g_mozilla_parse_env_proxy (G_MOZILLA_HTTPS_PROXY, G_MOZILLA_SSL))
                g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_SSL);
            gval = 1;
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, G_MOZILLA_LOCAL_HOST_IP);
        }
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_NONE) == 0) {
        gval = 0;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_AUTO) == 0) {
        gval = 2;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        autoconf = gconf_client_get_string(gconf_client,
                                           GCONF_PROXY_AUTOCONFIG_URL,
                                           NULL);
#endif
        //g_print ("%s\n", autoconf);
        if (autoconf) {
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_AUTOCONFIG_URL_PROXY, autoconf);
            EAL_GFREE(autoconf);
        }
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_MANUAL) == 0) {
        gval = 1;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        proxy_http = gconf_client_get_string(gconf_client,
                                             GCONF_HTTP_PROXY_HOST,
                                             NULL);
#endif
        //printf("HTTP PROXY: %s\n", proxy_http);
        if (proxy_http) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_HTTP_PROXY,
                                          g_str_has_prefix(proxy_http, G_MOZILLA_URL_PREFIX_HTTP) ? &proxy_http[7] : proxy_http);
            EAL_GFREE(proxy_http);
#ifdef USE_GCONF
            proxy_http_port = gconf_client_get_int(gconf_client,
                                                   GCONF_HTTP_PROXY_PORT,
                                                   NULL);
            //printf("HTTP PORT: %d\n", proxy_http_port);
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTP_PROXY_PORT, &proxy_http_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_HTTP);

#ifdef USE_GCONF
        proxy_https = gconf_client_get_string(gconf_client,
                                              GCONF_SECURE_PROXY_HOST,
                                              NULL);
#endif
        //printf("HTTPS PROXY: %s\n", proxy_http);
        if (proxy_https) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_HTTPS_PROXY,
                                          g_str_has_prefix(proxy_https, G_MOZILLA_URL_PREFIX_HTTPS) ? &proxy_https[8] : proxy_https);
            EAL_GFREE(proxy_https);
#ifdef USE_GCONF
            proxy_https_port = gconf_client_get_int(gconf_client,
                                                GCONF_SECURE_PROXY_PORT,
                                                NULL);
            //printf("HTTPS PORT: %d\n", proxy_https_port);
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTPS_PROXY_PORT, &proxy_https_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_HTTPS_PROXY, G_MOZILLA_SSL);

#ifdef USE_GCONF
        proxy_ftp = gconf_client_get_string(gconf_client,
                                            GCONF_FTP_PROXY_HOST,
                                            NULL);
        //printf("FTP PROXY: %s\n", proxy_ftp);
#endif
        if (proxy_ftp) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_FTP_PROXY,
                                          g_str_has_prefix(proxy_ftp, G_MOZILLA_URL_PREFIX_FTP) ? &proxy_ftp[6] : proxy_ftp);
            EAL_GFREE(proxy_ftp);
#ifdef USE_GCONF
            proxy_ftp_port = gconf_client_get_int(gconf_client,
                                                  GCONF_FTP_PROXY_PORT,
                                                  NULL);
            //printf("FTP PORT: %d\n", proxy_ftp_port);
#endif
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_FTP_PROXY_PORT, &proxy_ftp_port);
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_FTP_PROXY, G_MOZILLA_FTP);


#ifdef USE_GCONF

        omit_proxy = gconf_client_get_string(gconf_client,
                                             GCONF_PROXY_IGNORE_HOSTS,
                                             NULL);
#endif
        //printf("OMITED_HOSTS: %s\n", omit_proxy);
        if (omit_proxy)
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, omit_proxy);
    }
    g_mozilla_engine_save_prefs ();
    EAL_IF_GFREE(proxytype);
    EAL_IF_GFREE(omit_proxy);
    TRACE_LOG();
    return FALSE;
}
