/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
/**
 * Derived from GContentHandler http://landfill.mozilla.org/mxr-test/gnome/source/galeon/mozilla/ContentHandler.cpp
 */

#include <stdio.h>
#include "common.h"
#include "MicrobEalDownloadMgr.h"

#include "nsIDOMWindow.h"
#include "nsCOMPtr.h"
#include "nsStringGlue.h"
#include "nsIURI.h"
#include "nsIExternalHelperAppService.h"
#include "nsNetUtil.h"
#include "nsITransfer.h"
#include "nsIDownloadManager.h"
#include "nsDirectoryServiceUtils.h"
#include "nsDirectoryServiceDefs.h"
#include "nsIDownloadProgressListener.h"
#include "nsIWebBrowserPersist.h"
#include "nsIDOMNode.h"
#include "nsIDOMDocument.h"
#include "nsIWebBrowser.h"

#include "MicrobEalUtils.h"


#include "gmozillatransferitem.h"
#include "gmozillaweb.h"

class MicrobEmbedDownloadListener : public nsIDownloadProgressListener,
      public nsICancelable
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIDOWNLOADPROGRESSLISTENER
  NS_DECL_NSICANCELABLE

  static MicrobEmbedDownloadListener* GetDownloadListener();
  static void DestroyDownloadListener();
  void SetWebGlobal(void *aWebGlobal) {
    mWebGlobal = aWebGlobal;
  };

  MicrobEmbedDownloadListener() : mSpecialCase(PR_FALSE), mWebGlobal(0) {};

//  nsCOMPtr<nsIMIMEInfo> mMimeInfo;

  PRBool mSpecialCase;
  nsCString mSpecialTargetName;

private:
  ~MicrobEmbedDownloadListener() {};
  void *GetTransferByDownload(nsIDownload *aDownload);

protected:
  /* additional members */
  void *mWebGlobal;
};

static MicrobEmbedDownloadListener* sListener = nsnull;

MicrobEmbedDownloadListener*
MicrobEmbedDownloadListener::GetDownloadListener()
{
  if (!sListener) {
    sListener = new MicrobEmbedDownloadListener();
  }
  return sListener;
}

void
MicrobEmbedDownloadListener::DestroyDownloadListener()
{
  if (!sListener) return;

  sListener->Release();
  sListener = nsnull;
}

void *
MicrobEmbedDownloadListener::GetTransferByDownload(nsIDownload *aDownload)
{
  if (!aDownload) return nsnull;
  if (!mWebGlobal) return nsnull;
  PRUint32 id;
  aDownload->GetId(&id);
  for (int i = 0; i < (int) g_list_length(((GMozillaWeb*)mWebGlobal)->ongoing_download_list); i++) {
    GMozillaTransferItem *item = (GMozillaTransferItem *)g_list_nth_data(((GMozillaWeb*)mWebGlobal)->ongoing_download_list, i);
    if (!item) continue;
    if (item->download_id == (int) id)
      return item;
  }
  nsCOMPtr<nsIURI> source;
  aDownload->GetSource(getter_AddRefs(source));
  NS_ENSURE_TRUE(source, nsnull);
  nsCString sourceC;
  source->GetSpec(sourceC);
  if (sourceC.IsEmpty()) return nsnull;
  for (int i = 0; i < (int) g_list_length(((GMozillaWeb*)mWebGlobal)->ongoing_download_list); i++) {
    GMozillaTransferItem *item = (GMozillaTransferItem *)g_list_nth_data(((GMozillaWeb*)mWebGlobal)->ongoing_download_list, i);
    if (!item) continue;
    if (sourceC.Equals(item->server) && item->download_id < 0) {
      item->download_id = id;
      return item;
    }
  }

  return nsnull;
}

/* Implementation file */
NS_IMPL_ISUPPORTS2(MicrobEmbedDownloadListener, nsIDownloadProgressListener, nsICancelable)

/* attribute nsIDOMDocument document; */
NS_IMETHODIMP MicrobEmbedDownloadListener::GetDocument(nsIDOMDocument * *aDocument)
{
  //g_debug(">>>>>>Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  return NS_ERROR_NOT_IMPLEMENTED;
}
NS_IMETHODIMP MicrobEmbedDownloadListener::SetDocument(nsIDOMDocument * aDocument)
{
  //g_debug(">>>>>>Func:%s::%d\n", __PRETTY_FUNCTION__, __LINE__);
  return NS_ERROR_NOT_IMPLEMENTED;
}

#ifdef DEBUG
static void printDownloadInfo(nsIDownload *aDownload)
{
  nsCOMPtr<nsILocalFile> aTargetFile;
  PRInt32 aPercentComplete;
  PRInt64 aAmountTransferred;
  PRInt64 aSize;
  PRUint32 aId;
  PRInt16 aState;
  nsCOMPtr<nsIURI> aReferrer;
  PRBool aResumable;
  PRInt64 aStartTime;
  double aSpeed;
  nsCOMPtr<nsIURI> aSource;
  nsCOMPtr<nsIURI> aTarget;
  nsCOMPtr<nsICancelable> aCancelable;
  nsString aDisplayName;
  nsCOMPtr<nsIMIMEInfo> aMIMEInfo;
  aDownload->GetTargetFile(getter_AddRefs(aTargetFile));
  aDownload->GetPercentComplete(&aPercentComplete);
  aDownload->GetAmountTransferred(&aAmountTransferred);
  aDownload->GetSize(&aSize);
  aDownload->GetSource(getter_AddRefs(aSource));
  aDownload->GetTarget(getter_AddRefs(aTarget));
  aDownload->GetCancelable(getter_AddRefs(aCancelable));
  aDownload->GetDisplayName(aDisplayName);
  aDownload->GetStartTime(&aStartTime);
  aDownload->GetSpeed(&aSpeed);
  aDownload->GetMIMEInfo(getter_AddRefs(aMIMEInfo));
  aDownload->GetId(&aId);
  aDownload->GetState(&aState);
  aDownload->GetReferrer(getter_AddRefs(aReferrer));
  aDownload->GetResumable(&aResumable);
  nsCString aSourceC, aTargetC, aReferrerC;
  if (aSource)
    aSource->GetSpec(aSourceC);
  if (aTarget)
    aTarget->GetSpec(aTargetC);
  if (aReferrer)
    aReferrer->GetSpec(aReferrerC);
  const char *state_str = nsnull;
  switch (aState) {
  case nsIDownloadManager::DOWNLOAD_TYPE_DOWNLOAD: /* 0 */
    state_str = "nsIDownloadManager::DOWNLOAD_TYPE_DOWNLOAD";
    break;
  case nsIDownloadManager::DOWNLOAD_FINISHED: /* 1 */
    state_str = "nsIDownloadManager::DOWNLOAD_FINISHED";
    break;
  case nsIDownloadManager::DOWNLOAD_FAILED: /* 2 */
    state_str = "nsIDownloadManager::DOWNLOAD_FAILED";
    break;
  case nsIDownloadManager::DOWNLOAD_CANCELED: /* 3 */
    state_str = "nsIDownloadManager::DOWNLOAD_CANCELED";
    break;
  case nsIDownloadManager::DOWNLOAD_PAUSED: /* 4 */
    state_str = "nsIDownloadManager::DOWNLOAD_PAUSED";
    break;
  case nsIDownloadManager::DOWNLOAD_QUEUED: /* 5 */
    state_str = "nsIDownloadManager::DOWNLOAD_QUEUED";
    break;
  case nsIDownloadManager::DOWNLOAD_BLOCKED_PARENTAL: /* 6 */
    state_str = "nsIDownloadManager::DOWNLOAD_BLOCKED_PARENTAL";
    break;
  case nsIDownloadManager::DOWNLOAD_SCANNING: /* 7 */
    state_str = "nsIDownloadManager::DOWNLOAD_SCANNING";
    break;
  case nsIDownloadManager::DOWNLOAD_DIRTY: /* 8 */
    state_str = "nsIDownloadManager::DOWNLOAD_DIRTY";
    break;
  case nsIDownloadManager::DOWNLOAD_BLOCKED_POLICY: /* 9 */
    state_str = "nsIDownloadManager::DOWNLOAD_BLOCKED_POLICY";
    break;
  default:
    break;
  }
  printf("Download: id:%i, prc:%i, transf:%i, size:%i, state:%i[%s], resum:%i, speed:%g\n dspname:%s, src:%s, targ:%s, ref:%s\n",
          aId, aPercentComplete, (int)aAmountTransferred, (int)aSize, (int)aState, state_str, aResumable, aSpeed,
          NS_ConvertUTF16toUTF8(aDisplayName).get(), aSourceC.get(), aTargetC.get(), aReferrerC.get());
}
#endif

/* void onDownloadStateChange (in short aState, in nsIDownload aDownload); */
NS_IMETHODIMP MicrobEmbedDownloadListener::OnDownloadStateChange(PRInt16 aState, nsIDownload *aDownload)
{
  PRInt16 state;
  aDownload->GetState(&state);
#ifdef DEBUG
  printDownloadInfo(aDownload);
#endif
  switch (state) {
  case nsIDownloadManager::DOWNLOAD_TYPE_DOWNLOAD: /* 0 */
    GetTransferByDownload(aDownload);
    break;
  case nsIDownloadManager::DOWNLOAD_FINISHED: /* 1 */
    g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_COMPLETED);
    break;
  case nsIDownloadManager::DOWNLOAD_FAILED: /* 2 */
    g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_ERROR);
    break;
  case nsIDownloadManager::DOWNLOAD_CANCELED: /* 3 */
    g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_ERROR);
    break;
  case nsIDownloadManager::DOWNLOAD_PAUSED: /* 4 */
    //g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_ERROR);
    break;
  case nsIDownloadManager::DOWNLOAD_QUEUED: /* 5 */
    g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_STARTED);
    break;
  default:
    break;
  }
  return NS_OK;
}

/* void onStateChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in unsigned long aStateFlags, in nsresult aStatus, in nsIDownload aDownload); */
NS_IMETHODIMP MicrobEmbedDownloadListener::OnStateChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, PRUint32 aStateFlags, nsresult aStatus, nsIDownload *aDownload)
{
  //printf(">>>>>>Func:%s::%d, state:%d, status:%x\n", __PRETTY_FUNCTION__, __LINE__, aStateFlags, aStatus);
  return NS_ERROR_NOT_IMPLEMENTED;
}

/* void onProgressChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long long aCurSelfProgress, in long long aMaxSelfProgress, in long long aCurTotalProgress, in long long aMaxTotalProgress, in nsIDownload aDownload); */
NS_IMETHODIMP MicrobEmbedDownloadListener::OnProgressChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, PRInt64 aCurSelfProgress, PRInt64 aMaxSelfProgress, PRInt64 aCurTotalProgress, PRInt64 aMaxTotalProgress, nsIDownload *aDownload)
{
  g_signal_emit_by_name(G_OBJECT (GetTransferByDownload(aDownload)), G_WEB_TRANSFER_ITEM_SIGNAL_PROGRESS, (PRInt32)aCurSelfProgress, (PRInt32)aMaxSelfProgress, 1);
  return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in unsigned long aState, in nsIDownload aDownload); */
NS_IMETHODIMP MicrobEmbedDownloadListener::OnSecurityChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, PRUint32 aState, nsIDownload *aDownload)
{
  //printf(">>>>>>Func:%s::%d, aState:%i\n", __PRETTY_FUNCTION__, __LINE__, aState);
  return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP MicrobEmbedDownloadListener::Cancel(nsresult aReason)
{
  //printf("IMPL ME:>>>>>>Func:%s::%d, cancelReason:%x\n", __PRETTY_FUNCTION__, __LINE__, aReason);
  return NS_OK;
}

NS_IMPL_ISUPPORTS1(MicrobEalDownloadMgr, nsIHelperAppLauncherDialog);

MicrobEalDownloadMgr::MicrobEalDownloadMgr(void)
{
  nsresult rv;
  rv = NS_GetSpecialDirectory(NS_OS_HOME_DIR, getter_AddRefs(mHomeDir));
}

MicrobEalDownloadMgr::~MicrobEalDownloadMgr(void)
{
}

NS_IMETHODIMP
MicrobEalDownloadMgr::SetupDownloadListener(void **aDownloadListener, void *aWebGlobal)
{
  nsresult rv;

  nsCOMPtr<nsIDownloadManager> dm = do_GetService("@mozilla.org/download-manager;1", &rv);
  if (!dm) return rv;
  if (aDownloadListener && *aDownloadListener) {
    rv = dm->RemoveListener(*(nsIDownloadProgressListener**)aDownloadListener);
    MicrobEmbedDownloadListener::DestroyDownloadListener();
    return rv;
  }

  MicrobEmbedDownloadListener* downloadListener = MicrobEmbedDownloadListener::GetDownloadListener();
  if (!downloadListener) return NS_ERROR_OUT_OF_MEMORY;
  rv = dm->AddListener(downloadListener);

  if (aDownloadListener) {
    NS_ADDREF(*(nsIDownloadProgressListener**)aDownloadListener = downloadListener);
  }
  if (aWebGlobal)
    downloadListener->SetWebGlobal(aWebGlobal);

  return rv;
}

NS_IMETHODIMP
MicrobEalDownloadMgr::Show(nsIHelperAppLauncher* aLauncher,
                           nsISupports* aContext,
                           PRUint32 aForced)
{
  NS_ENSURE_ARG_POINTER(aLauncher);
  nsCOMPtr<nsIDOMWindow> parentDOMWindow = do_QueryInterface(aContext);
  if (!parentDOMWindow) {
    NS_WARNING("Failed to get nsIDOMWindow from context.");
  }
  GMozillaEngine *engine = GetGMozEngineForDOMWindow(parentDOMWindow);
  NS_ENSURE_TRUE(engine, NS_ERROR_FAILURE);
  /* File type */
  nsresult rv = aLauncher->GetMIMEInfo(getter_AddRefs(mMimeInfo));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCAutoString mimeType;
  rv = mMimeInfo->GetMIMEType(mimeType);
  NS_ENSURE_SUCCESS(rv, rv);

  /* File name */
  nsCAutoString tempFileName;
  nsAutoString suggestedFileName;
  rv = aLauncher->GetSuggestedFileName(suggestedFileName);

  NS_ENSURE_SUCCESS(rv, rv);

  tempFileName = NS_ConvertUTF16toUTF8(suggestedFileName);

  /* Complete source URL */
  rv = aLauncher->GetSource(getter_AddRefs(mLastUri));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCAutoString spec;
  rv = mLastUri->GetPrePath(spec);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCString fname;
  mLastUri->GetPath(fname);
  spec += fname;

  PRInt64 mContentLength = 0;
  rv = aLauncher->GetContentLength(&mContentLength);
  NS_ENSURE_SUCCESS(rv, rv);

  gulong file_size = (mContentLength > 0) ? mContentLength : 0;

  int mode = 1;

  char *location = gtk_moz_embed_get_location(GTK_MOZ_EMBED(engine->engine));
  if (sListener && location) {
    sListener->mSpecialCase = spec.Equals(location);
    if (!sListener->mSpecialCase) {
      gchar *file_name = g_path_get_basename(spec.get());
      if (file_name) {
        sListener->mSpecialCase = (g_strcasecmp(file_name, tempFileName.get()) != 0);
        EAL_GFREE(file_name);
      }
    }
    if (sListener->mSpecialCase)
      mode = 3; 
    EAL_GFREE(location);
  }

  g_signal_emit_by_name (G_OBJECT(engine), G_WEBWIDGET_SIGNAL_DOWNLOAD_REQUEST, spec.get(), tempFileName.get(), mimeType.get(), (gulong)file_size, mode);

//  MicrobEmbedDownloadListener* downloadListener = MicrobEmbedDownloadListener::GetDownloadListener();
//  if (downloadListener)
//    rv = aLauncher->GetMIMEInfo(getter_AddRefs(downloadListener->mMimeInfo));

  nsCOMPtr<nsICancelable> cancel = do_QueryInterface(aLauncher);
  if (!(sListener && sListener->mSpecialCase)) {
    if (cancel)
      rv = cancel->Cancel(NS_BINDING_ABORTED);
  } else if (sListener && !sListener->mSpecialTargetName.IsEmpty()) {
    if (StringBeginsWith(sListener->mSpecialTargetName, NS_LITERAL_CSTRING("file://")))
      sListener->mSpecialTargetName.Cut(0, 7); // Remove file:// from

    nsCOMPtr<nsILocalFile> mFinalFileDestination;
    NS_NewNativeLocalFile(sListener->mSpecialTargetName, PR_TRUE, getter_AddRefs(mFinalFileDestination));

    rv = aLauncher->SaveToDisk(mFinalFileDestination, PR_FALSE);
    sListener->mSpecialTargetName.Truncate();
  } else if (cancel)
    rv = cancel->Cancel(NS_BINDING_ABORTED);
  
  if (sListener)
    sListener->mSpecialCase = PR_FALSE;
  return rv;
}

NS_IMETHODIMP
MicrobEalDownloadMgr::InitTransferItem(const char *aUrl, const char *aDestFile, PRUint32 *aId, nsIDownload **aDownload)
{
  if (sListener && sListener->mSpecialCase) {
    if (aId) *aId = -1;
    sListener->mSpecialTargetName.Assign(aDestFile);
    if (StringBeginsWith(sListener->mSpecialTargetName, NS_LITERAL_CSTRING("file://")))
      sListener->mSpecialTargetName.Cut(0, 7); // Remove file:// from
    return NS_OK;
  }

  nsresult rv;

  nsCAutoString filePath;
  filePath.Assign(aDestFile);

  if (StringBeginsWith(filePath, NS_LITERAL_CSTRING("file://")))
    filePath.Cut(0, 7); // Remove file:// from
  nsCOMPtr<nsILocalFile> mFinalFileDestination;
  NS_NewNativeLocalFile(filePath, PR_TRUE, getter_AddRefs(mFinalFileDestination));

  nsCOMPtr<nsIURI> target;
  rv = NS_NewFileURI(getter_AddRefs(target), mFinalFileDestination);
  if (NS_FAILED(rv)) return rv;

  nsCOMPtr<nsIFile> tempDir;
  rv = NS_GetSpecialDirectory(NS_OS_HOME_DIR, getter_AddRefs(tempDir));
  nsCOMPtr<nsILocalFile> lf(do_QueryInterface(tempDir));

  nsCOMPtr<nsIURI> lastUri;
  rv = NS_NewURI(getter_AddRefs(lastUri), nsDependentCString(aUrl));
  NS_ENSURE_SUCCESS(rv, rv);

  PRTime timeDownloadStarted = PR_Now();

  nsCOMPtr<nsIDownloadManager> dm = do_GetService("@mozilla.org/download-manager;1", &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr<nsIDownload> inner;

  nsCOMPtr<nsIWebBrowserPersist> wbp = do_CreateInstance("@mozilla.org/embedding/browser/nsWebBrowserPersist;1", &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = wbp->SetPersistFlags(nsIWebBrowserPersist::PERSIST_FLAGS_REPLACE_EXISTING_FILES |
                            nsIWebBrowserPersist::PERSIST_FLAGS_AUTODETECT_APPLY_CONVERSION);

  rv = dm->AddDownload(nsIDownloadManager::DOWNLOAD_TYPE_DOWNLOAD, lastUri,
                       target, EmptyString(), nsnull /* MIMEInfo */,
                       timeDownloadStarted, nsnull, wbp /* Cancelable */,
                       getter_AddRefs(inner));
  NS_ENSURE_SUCCESS(rv, rv);
  if (aDownload)
    NS_ADDREF(*aDownload = inner);
  if (aId)
    inner->GetId(aId);
  return rv;
}

NS_IMETHODIMP
MicrobEalDownloadMgr::StartDownload(PRUint32 aId, nsIDownload *aDownload, PRBool aIsDocument, void*aCtxDoc, PRBool aFullContent)
{
  nsresult rv;
  nsCOMPtr <nsIDownload> download = do_QueryInterface(aDownload, &rv);
  if (!download) {
    nsCOMPtr<nsIDownloadManager> dm = do_GetService("@mozilla.org/download-manager;1", &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = dm->GetDownload(aId, getter_AddRefs(download));
    NS_ENSURE_SUCCESS(rv, rv);
    PRInt16 aState = 0;
    download->GetState(&aState);
    if (aState != nsIDownloadManager::DOWNLOAD_QUEUED) {
      rv = dm->ResumeDownload(aId);
      if (NS_FAILED(rv)) {
        rv = dm->CancelDownload(aId);
        rv = dm->RetryDownload(aId);
      }
      return rv;
    }
  }

  nsCOMPtr <nsICancelable> cancelable;
  download->GetCancelable(getter_AddRefs(cancelable));
  nsCOMPtr <nsIWebBrowserPersist> wbp = do_QueryInterface(cancelable, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr <nsIURI> source;
  nsCOMPtr <nsIURI> target;
  download->GetSource(getter_AddRefs(source));
  download->GetTarget(getter_AddRefs(target));
  wbp->SetProgressListener(download);
  if (!aIsDocument) {
    rv = wbp->SaveURI(source, nsnull, nsnull, nsnull, nsnull, target);
    return rv;
  }

  nsCString contentFolderPath;
  target->GetSpec(contentFolderPath);
  contentFolderPath.Append("_content");
  //printf(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>GetNativePath=%s ", contentFolderPath.get());

  nsCOMPtr<nsIIOService> ios(do_GetService(NS_IOSERVICE_CONTRACTID));
  nsCOMPtr<nsIURI> contentFolder;
  rv = ios->NewURI(contentFolderPath, "", nsnull, getter_AddRefs(contentFolder));

  nsCOMPtr<nsIDOMWindow> DOMWindow;
  nsCOMPtr<nsIWebBrowser> webBrowser = do_QueryInterface((nsISupports*)aCtxDoc, &rv);
  if (webBrowser)
    webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));

  if (aCtxDoc && !DOMWindow) {
    nsCOMPtr<nsIDOMNode> node = do_QueryInterface((nsISupports*)aCtxDoc, &rv);
    if (node)
      rv = GetDOMWindowByNode(node, getter_AddRefs(DOMWindow));
  }
  NS_ENSURE_TRUE(DOMWindow, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMDocument> doc;
  DOMWindow->GetDocument(getter_AddRefs(doc));
  NS_ENSURE_TRUE(doc, NS_ERROR_FAILURE);

  rv = wbp->SaveDocument(doc, target, aFullContent?contentFolder:nsnull, nsnull, 0, 0);
  return rv;
}

NS_IMETHODIMP
MicrobEalDownloadMgr::PromptForSaveToFile(nsIHelperAppLauncher* aLauncher,
    nsISupports* aWindowContext,
    const PRUnichar* aDefaultFileName,
    const PRUnichar* aSuggestedFileExtension,
    PRBool aForcePrompt,
    nsILocalFile** _retval)
{
  return NS_OK;
}

