/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Oleg Romashin.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */
#include "MicrobEalContextMenuInfo.h"
#include "MicrobEalUtils.h"

#include "gtkmicrob_context.h"
#include "gtkmozembed.h"
#include "gtkmozembed_internal.h"

#include "nsIContent.h"

#include "nsIDOMDocument.h"
#include "nsIDOMDocumentView.h"
#include "nsIDOMElement.h"
#include "nsIDOMEvent.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLAreaElement.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMHTMLTextAreaElement.h"
#include "nsIDOMMouseEvent.h"
#include "nsIDOMNSEvent.h"
#include "nsIDOMNSHTMLDocument.h"
#include "nsIDOMNSEditableElement.h"
#include "nsIDOMNode.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowCollection.h"
#include "nsIDOMNSHTMLInputElement.h"
#include "nsIDOMNSHTMLTextAreaElement.h"
#include "nsIDOMHTMLSelectElement.h"
#include "nsIDOMHTMLOptionElement.h"
#include "nsIDOMHTMLButtonElement.h"
#include "nsIDOMHTMLOptGroupElement.h"

#include "nsISelectionController.h"
#include "nsIFormControl.h"
#include "nsIWebBrowser.h"
#include "nsIWebNavigation.h"

// imglib2
#include "imgIRequest.h"
#include "imgIContainer.h"
#include "nsIImageLoadingContent.h"


#include "nsPIDOMWindow.h"
#include "nsIDocShell.h"
#include "nsIInterfaceRequestorUtils.h"
#include "nsIDocument.h"
#include "nsIPresShell.h"

// composer
#include "nsIEditingSession.h"

// editor
#include "nsIPlaintextEditor.h"
#include "nsIEditor.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIURI.h"


//#include ".h"
#include "nsIDOMNSElement.h"
#include "nsIDOMHTMLObjectElement.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMCSSStyleDeclaration.h"
#include "nsIDOMCSSValue.h"
#include "nsIDOMCSSPrimitiveValue.h"
#include "nsIDOMCSSValueList.h"
#include "nsIDOM3Document.h"
#include "nsIDOMViewCSS.h"

#ifndef MOZ_TRUNK_BUILD
#include "nsIDOMTextRectangle.h"
#else
#include "nsIDOMClientRect.h"
#endif

#include "nsIDOMNSElement.h"
#include "nsIDOMNSHTMLElement.h"
#include "nsIWebBrowserFocus.h"

#include "nsIWebBrowser.h"
#include "nsPIDOMWindow.h"
#include "nsICommandManager.h"
#include "nsIDocShell.h"
#include "nsIInterfaceRequestorUtils.h"

//Scrolling internal frames
#include "nsIContent.h"
#include "nsQueryFrame.h"
//#include "nsIFrame.h"
#include "nsIScrollableView.h"
#include "nsIScrollableViewProvider.h"
#include "nsIView.h"
#include "nsIDOMHTMLIFrameElement.h"
#include "nsIViewManager.h"

#include "nsITextControlFrame.h"
#include "nsGUIEvent.h"

#ifdef MOZ_XUL
#include "nsIDOMXULTextboxElement.h"
#endif

#include "nsIHTMLDocument.h"

//*****************************************************************************
// class MicrobEalContextMenuInfo
//*****************************************************************************

MicrobEalContextMenuInfo::MicrobEalContextMenuInfo(GtkMicroBContext *ctx, const void *aEmbed)
: mEventNode(nsnull),
  mCtx(ctx),
  mEmbed(aEmbed),
  mIsScrollableElement(PR_FALSE),
  mScrollableView(nsnull)
{
}

MicrobEalContextMenuInfo::~MicrobEalContextMenuInfo(void)
{
}

NS_IMPL_ADDREF(MicrobEalContextMenuInfo)
NS_IMPL_RELEASE(MicrobEalContextMenuInfo)
NS_INTERFACE_MAP_BEGIN(MicrobEalContextMenuInfo)
NS_INTERFACE_MAP_ENTRY(nsISupports)
NS_INTERFACE_MAP_END

static PRBool
IsXULNode(nsIDOMNode *aNode, PRUint32 *aType = 0)
{
  NS_ENSURE_ARG(aNode);

  const PRUint32 kThumb = 0;
  const PRUint32 kSlider = 2;
  const PRUint32 kScrollBarButton = 3;

  nsString sorigNode;
  aNode->GetNodeName(sorigNode);
  if (sorigNode.EqualsLiteral("#document"))
    return PR_FALSE;

  nsCOMPtr<nsIContent> targetContent = do_QueryInterface(aNode);
  NS_ENSURE_TRUE(targetContent, PR_FALSE);

  PRBool retval = targetContent->IsNodeOfType(nsINode::eXUL);

  NS_ENSURE_TRUE(aType, retval);
  if (sorigNode.EqualsLiteral("xul:thumb") ||
          sorigNode.EqualsLiteral("xul:vbox") ||
          sorigNode.EqualsLiteral("xul:spacer"))
    *aType = kThumb;
  else if (sorigNode.EqualsLiteral("xul:slider"))
    *aType = kSlider;
  else if (sorigNode.EqualsLiteral("xul:scrollbarbutton"))
    *aType = kScrollBarButton;

  return retval;

}

static PRBool
IsXULEdit(nsIDOMNode *aNode, PRUint32 *aType = 0)
{
  NS_ENSURE_ARG(aNode);

  nsString sorigNode;
  aNode->GetNodeName(sorigNode);
  if (sorigNode.EqualsLiteral("#document"))
    return PR_FALSE;

  nsCOMPtr<nsIContent> targetContent = do_QueryInterface(aNode);
  NS_ENSURE_TRUE(targetContent, PR_FALSE);

  PRBool retval = targetContent->IsNodeOfType(nsINode::eXUL);

  if (retval && sorigNode.EqualsLiteral("textbox")) {
    retval = PR_TRUE;
    if (aType) {
      *aType |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT;
    }
  }

  return retval;
}

nsresult
MicrobEalContextMenuInfo::SetFramePosition(void)
{
  nsresult rv;

  //FIXME canot get frame info without main window
  if (!GTK_IS_MOZ_EMBED(mEmbed)) return NS_ERROR_FAILURE;

  nsCOMPtr<nsIDOMWindow> mainWindow;
  {
    nsCOMPtr <nsIWebBrowser> webBrowser;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(mEmbed), getter_AddRefs(webBrowser));
    NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
    rv = webBrowser->GetContentDOMWindow(getter_AddRefs(mainWindow));
  }
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIDOMWindow> ctxWindow;
  rv = GetDOMWindowByNode(mEventNode, getter_AddRefs(ctxWindow));

  mCtx->docindex = -1;
  if (!ctxWindow || ctxWindow == mainWindow)
    return NS_OK;

  mCtx->type |= GTK_MICROB_EAL_CONTEXT_IFRAME;

  nsCOMPtr<nsIDOMWindowCollection> frames;
  rv = mainWindow->GetFrames(getter_AddRefs(frames));
  NS_ENSURE_SUCCESS(rv, rv);

  PRUint32 frameCount = 0;
  frames->GetLength(&frameCount);
  nsCOMPtr<nsIDOMWindow> currentWindow;
  NS_ENSURE_SUCCESS(rv, rv);
  for (unsigned int i= 0; i < frameCount; i++) {
    frames->Item(i, getter_AddRefs(currentWindow));
    if (currentWindow != ctxWindow) continue;
    mCtx->docindex = i;
    break;
  }
  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::CheckRichEdit(void)
{
  nsresult rv;
  nsCOMPtr<nsIDOMElement> targetDOMElement;
  rv = mCtxDocument->GetDocumentElement(getter_AddRefs(targetDOMElement));
  if (!targetDOMElement) return rv;

  nsCOMPtr<nsIDOMNSHTMLDocument> htmlDoc = do_QueryInterface(mCtxDocument, &rv);
  if (!htmlDoc) return rv;

  nsString DMode;
  htmlDoc->GetDesignMode(DMode);
  if (!DMode.EqualsLiteral("on")) return rv;

  mCtx->type |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT | GTK_MICROB_EAL_CONTEXT_RICHEDIT | GTK_MICROB_EAL_CONTEXT_MULTILINE;
  mIsScrollableElement = PR_FALSE;
  return rv;
}

nsresult
MicrobEalContextMenuInfo::CheckFormControl(nsIDOMNode* aNode)
{
  nsresult rv;
  PRBool rdonly = PR_FALSE;
  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  eventTarget = do_QueryInterface(aNode, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIContent> tgContent = do_QueryInterface(eventTarget, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  mCtxFormType = 0;
  if (!tgContent->IsNodeOfType(nsIContent::eHTML_FORM_CONTROL)) {
    unsigned short ns_node_type = 0;
    nsString name;
    
    aNode->GetNodeType (&ns_node_type);
    aNode->GetLocalName(name);
    nsCOMPtr<nsIDOMNSHTMLElement> nsel = do_QueryInterface (aNode);
    if (nsel) {
      nsString conteditable;
      nsel->GetContentEditable(conteditable);
      if (conteditable.EqualsLiteral("true")) {
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT | GTK_MICROB_EAL_CONTEXT_RICHEDIT;
        return NS_OK;
      }
    }
    return NS_ERROR_FAILURE;
  }

  nsCOMPtr<nsIFormControl> formControl(do_QueryInterface(tgContent, &rv));
  NS_ENSURE_SUCCESS(rv, rv);
  PRBool isDisabled = PR_FALSE;
  PRBool clickableForm = PR_FALSE;
  mCtxFormType = formControl->GetType();
  switch (mCtxFormType) {
  case NS_FORM_BUTTON_BUTTON:
  case NS_FORM_BUTTON_RESET:
  case NS_FORM_BUTTON_SUBMIT:
  {
    nsCOMPtr<nsIDOMHTMLButtonElement> disForm = do_QueryInterface(tgContent);
    if (disForm) disForm->GetDisabled(&isDisabled);
    clickableForm = PR_TRUE;
    break;
  }
  case NS_FORM_INPUT_BUTTON:
  case NS_FORM_INPUT_CHECKBOX:
  case NS_FORM_INPUT_FILE:
  case NS_FORM_INPUT_RESET:
  case NS_FORM_INPUT_RADIO:
  case NS_FORM_INPUT_SUBMIT:
  {
    nsCOMPtr<nsIDOMHTMLInputElement> disForm = do_QueryInterface(tgContent);
    if (disForm) disForm->GetDisabled(&isDisabled);
    clickableForm = PR_TRUE;
    break;
  }
  case NS_FORM_OPTION:
  {
    nsCOMPtr<nsIDOMHTMLOptionElement> disForm = do_QueryInterface(tgContent);
    if (disForm) disForm->GetDisabled(&isDisabled);
    clickableForm = PR_TRUE;
    break;
  }
  case NS_FORM_OPTGROUP:
  {
    nsCOMPtr<nsIDOMHTMLOptGroupElement> disForm = do_QueryInterface(tgContent);
    if (disForm) disForm->GetDisabled(&isDisabled);
    clickableForm = PR_TRUE;
    break;
  }
  case NS_FORM_INPUT_IMAGE:
  case NS_FORM_INPUT_HIDDEN:
    break;
  case NS_FORM_INPUT_PASSWORD:
  {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) {
      input->GetReadOnly(&rdonly);
      input->GetDisabled(&isDisabled);
    }
    if (!rdonly) {
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT;
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_IPASSWORD;
        clickableForm = PR_TRUE;
    }
    break;
  }
  case NS_FORM_INPUT_TEXT:
  {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) input->GetReadOnly(&rdonly);
    if (!rdonly) {
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT;
        clickableForm = PR_TRUE;
    }
    break;
  }
  case NS_FORM_LABEL:
  case NS_FORM_LEGEND:
    break;
  case NS_FORM_SELECT:
  {
    nsCOMPtr<nsIDOMHTMLSelectElement> disForm = do_QueryInterface(tgContent);
    if (disForm) disForm->GetDisabled(&isDisabled);
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_SELECT;
    clickableForm = PR_TRUE;
    break;
  }
  case NS_FORM_TEXTAREA:
  {
    nsCOMPtr<nsIDOMHTMLTextAreaElement> input;
    input = do_QueryInterface(mEventNode);
    if (input) {
      input->GetReadOnly(&rdonly);
      input->GetDisabled(&isDisabled);
    }
    if (!rdonly) {
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_INPUT_TEXT | GTK_MICROB_EAL_CONTEXT_MULTILINE;
        clickableForm = PR_TRUE;
    }
    break;
  }
  case NS_FORM_OBJECT:
    break;
  default:
    break;
  }
  if (clickableForm && !isDisabled)
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_FORM;

  if (mCtx->type & GTK_MICROB_EAL_CONTEXT_INPUT_TEXT)
    mIsScrollableElement = rdonly;

  return rv;
}

nsresult
MicrobEalContextMenuInfo::ResolveBaseURL (const nsAString &relurl, nsACString &url)
{
  nsCString cRelURL(NS_ConvertUTF16toUTF8(relurl).get());
  return mBaseURI->Resolve (cRelURL, url);
}

nsresult
MicrobEalContextMenuInfo::ScrollToSelectedNode(nsISupports *element)
{
  nsresult rv = NS_ERROR_FAILURE;
  nsCOMPtr<nsIDOMNSHTMLElement> nodeElement = do_QueryInterface(element);
  if (!nodeElement) {
    nsCOMPtr<nsIWebBrowser> webBrowser = do_QueryInterface(element);
    if (!webBrowser)
       return rv;
    nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(webBrowser));
    if (!focus)
      return rv;
    nsCOMPtr<nsIDOMElement> domElement;
    rv = focus->GetFocusedElement(getter_AddRefs(domElement));
    nodeElement = do_QueryInterface(domElement);
  }
  if (nodeElement) {
    // to using follow code, we need figure out why microb-eal can include nsIDocument.h
    nsCOMPtr<nsIDOMNode> node = do_QueryInterface(nodeElement, &rv);
    NS_ENSURE_TRUE(node, rv);
    nsCOMPtr<nsIDOMDocument> document;
    node->GetOwnerDocument(getter_AddRefs(document));
    NS_ENSURE_TRUE(document, rv);
    nsCOMPtr<nsIDocument> doc = do_QueryInterface(document, &rv);
    NS_ENSURE_TRUE(doc, rv);
    nsCOMPtr<nsIPresShell> presShell = doc->GetPrimaryShell();
    NS_ENSURE_TRUE(presShell, NS_ERROR_FAILURE);
    nsCOMPtr<nsIContent> content = do_QueryInterface(node, &rv);
    NS_ENSURE_TRUE(content, rv);
    presShell->ScrollContentIntoView(content, NS_PRESSHELL_SCROLL_ANYWHERE,
                                     NS_PRESSHELL_SCROLL_ANYWHERE);
  }
  return rv;
}

nsresult
MicrobEalContextMenuInfo::CheckDomHtmlNode(nsIDOMNode *aNode, PRBool &aHasImage)
{
  nsresult rv;

  nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aNode, &rv);
  NS_ENSURE_TRUE(node, NS_ERROR_FAILURE);

  nsString uTag;
  nsString tmp;
  PRUint16 dnode_type;
  rv = node->GetNodeType(&dnode_type);
  NS_ENSURE_SUCCESS(rv, rv);

  if (nsIDOMNode::ELEMENT_NODE != dnode_type)
    return rv;

  node->GetLocalName(uTag);
  if (uTag.LowerCaseEqualsLiteral("object")) {
    nsCOMPtr<nsIDOMHTMLObjectElement> object;
    object = do_QueryInterface (node);
    if (!object) return NS_ERROR_FAILURE;
    nsString value;
    object->GetType(value);
    if (StringBeginsWith(value, NS_LITERAL_STRING("/image"))) {
      nsString img;
      rv = object->GetData (img);
      if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
      nsCString cImg;
      rv = ResolveBaseURL (img, cImg);
      if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
      SetStringProperty ("image", cImg.get());
      mCtx->type |= GTK_MICROB_EAL_CONTEXT_IMAGE;
      aHasImage = PR_TRUE;
    } else
      mCtx->type |= GTK_MICROB_EAL_CONTEXT_EMBED;
    rv = NS_ERROR_FAILURE; // do not inspect parent nodes
  }
  if (uTag.LowerCaseEqualsLiteral("embed")) {
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_EMBED;
    rv = NS_ERROR_FAILURE; // do not inspect parent nodes
  }
  else if (uTag.LowerCaseEqualsLiteral("html")) {
    if(!aHasImage) { // do not overwrite image url if already set
      nsCOMPtr<nsIDOMNodeList> nodeList;
      rv = mCtxDocument->GetElementsByTagName (NS_LITERAL_STRING("body"),
                                               getter_AddRefs (nodeList));
      if (NS_SUCCEEDED (rv) && nodeList) {
        nsCOMPtr<nsIDOMNode> bodyNode;
        nodeList->Item (0, getter_AddRefs (bodyNode));
        nsString cssurl;
        rv = GetCSSBackground (bodyNode, cssurl);
        if (NS_SUCCEEDED (rv)) {
          nsCString bgimg;
          rv = ResolveBaseURL (cssurl, bgimg);
          if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
          SetStringProperty ("image", bgimg.get());
          mCtx->type |= GTK_MICROB_EAL_CONTEXT_IMAGE;
          aHasImage = PR_TRUE;
        }
      }
    }
    rv = NS_ERROR_FAILURE; // do not inspect parent nodes
  }
  else if (uTag.LowerCaseEqualsLiteral("a")) {

    nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(node, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    anchor->GetHref(tmp);

    if (StringBeginsWith(tmp, NS_LITERAL_STRING("mailto:"))) {
      nsCString href;
      NS_UTF16ToCString (tmp, NS_CSTRING_ENCODING_UTF8, href);
      href.Cut (0, 7);
      SetStringProperty ("email", href.get());
      mCtx->type |= GTK_MICROB_EAL_CONTEXT_MAILTO;
    }

    if (!tmp.IsEmpty()) {
      mCtx->type |= GTK_MICROB_EAL_CONTEXT_LINK;
      SetStringProperty ("link", tmp);
      if (tmp.LowerCaseEqualsLiteral("text/smartbookmark")) {

        nsCOMPtr<nsIDOMNode> childNode;
        node->GetFirstChild(getter_AddRefs(childNode));
        if (childNode) {
          PRInt32 width, height;
          tmp.Truncate();
          rv = CheckDomImageElement(node, tmp, width, height);
          if (NS_SUCCEEDED(rv)) {
            mCtx->type |= GTK_MICROB_EAL_CONTEXT_IMAGE;
            SetStringProperty ("image", tmp);
            aHasImage = PR_TRUE;
          }
        }
      } else {
        nsCOMPtr<nsIDOMNode> childNode;
        node->GetFirstChild(getter_AddRefs(childNode));
        if (childNode) {
          childNode->GetLocalName(uTag);
          mIsScrollableElement = !uTag.LowerCaseEqualsLiteral("div");
        }
      }
    }
    rv = NS_ERROR_FAILURE; // do not inspect parent nodes
  }
  else if (uTag.LowerCaseEqualsLiteral("option") || uTag.LowerCaseEqualsLiteral("optgroup")) {
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_NONE;
    rv = NS_OK;
  }
  else if (uTag.LowerCaseEqualsLiteral("select")) {
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_SELECT;
    mEventNode = aNode;
    rv = NS_OK;
  }
  else if (uTag.LowerCaseEqualsLiteral("area")) {
    nsCOMPtr<nsIDOMHTMLAreaElement> area = do_QueryInterface(node, &rv);
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_LINK;
    if (NS_SUCCEEDED(rv) && area) {
      PRBool aNoHref = PR_FALSE;
      rv = area->GetNoHref(&aNoHref);
      rv = aNoHref?area->GetTarget(tmp):area->GetHref(tmp);
      SetStringProperty ("link", tmp);
      rv = NS_OK;
    }
  }
  else if (uTag.LowerCaseEqualsLiteral("img")) {
    PRInt32 width, height;
    rv = CheckDomImageElement(node, tmp, width, height);
    if (NS_SUCCEEDED(rv)) {
      mCtx->type |= GTK_MICROB_EAL_CONTEXT_IMAGE;
      SetStringProperty ("image", tmp);
      aHasImage = PR_TRUE;
    }
    rv = NS_OK;
  }

  if (!aHasImage)
  {
    nsString cssurl;
    if (NS_SUCCEEDED (GetCSSBackground (node, cssurl)))
    {
      nsCString bgimg;
      if (!NS_FAILED (ResolveBaseURL (cssurl, bgimg)))
      {
        SetStringProperty ("image", bgimg.get());
        mCtx->type |= GTK_MICROB_EAL_CONTEXT_IMAGE;
        aHasImage = PR_TRUE;
      }
    }
  }
  return rv;
}

nsresult
MicrobEalContextMenuInfo::SetIntProperty (const char *name, int value)
{
  NS_ENSURE_ARG_POINTER(name);
  GValue *val = g_new0 (GValue, 1);
  g_value_init (val, G_TYPE_INT);
  g_value_set_int (val, value);
  NS_ENSURE_TRUE(gtk_microb_context_set_prop(mCtx, name, val), NS_ERROR_FAILURE);
  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::SetStringProperty (const char *name, const char *value)
{
  NS_ENSURE_ARG_POINTER(name);
  NS_ENSURE_ARG_POINTER(value);
  GValue *val = g_new0 (GValue, 1);
  g_value_init (val, G_TYPE_STRING);
  g_value_set_string (val, value);
  NS_ENSURE_TRUE(gtk_microb_context_set_prop(mCtx, name, val), NS_ERROR_FAILURE);
  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::SetStringProperty (const char *name, const nsAString &value)
{
  NS_ENSURE_ARG_POINTER(name);
  nsCString cValue(NS_ConvertUTF16toUTF8(value).get());
  return SetStringProperty (name, cValue.get());
}

/* static */
nsresult
MicrobEalContextMenuInfo::GetCSSBackground (nsIDOMNode *node, nsAString& url)
{

  nsresult rv;

  NS_ENSURE_TRUE (mViewCSS, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMElement> element = do_QueryInterface (node);
  NS_ENSURE_TRUE (element, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMCSSStyleDeclaration> decl;
  mViewCSS->GetComputedStyle (element, nsString(),
                              getter_AddRefs (decl));
  NS_ENSURE_TRUE (decl, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMCSSValue> CSSValue;
  decl->GetPropertyCSSValue (NS_LITERAL_STRING("background-image"), getter_AddRefs (CSSValue));
  NS_ENSURE_TRUE (CSSValue, NS_ERROR_FAILURE);

  PRUint16 valueType = 0;
  rv = CSSValue->GetCssValueType(&valueType);
  NS_ENSURE_SUCCESS(rv, rv);

  if (nsIDOMCSSValue::CSS_VALUE_LIST == valueType) {
    nsCOMPtr<nsIDOMCSSValueList> valueList = do_QueryInterface (CSSValue);
    if (valueList) {
      rv = valueList->Item(0, getter_AddRefs (CSSValue));
      NS_ENSURE_SUCCESS(rv, rv);
    }
  }
 
  nsCOMPtr<nsIDOMCSSPrimitiveValue> primitiveValue =
    do_QueryInterface (CSSValue);
  if (!primitiveValue) return NS_ERROR_FAILURE;

  PRUint16 type;
  rv = primitiveValue->GetPrimitiveType (&type);
  NS_ENSURE_SUCCESS (rv, NS_ERROR_FAILURE);

  if (type != nsIDOMCSSPrimitiveValue::CSS_URI) return NS_ERROR_FAILURE;

  rv = primitiveValue->GetStringValue (url);
  NS_ENSURE_SUCCESS (rv, NS_ERROR_FAILURE);

  return NS_OK;
}

/* static */
nsresult
MicrobEalContextMenuInfo::GetImageRequest(imgIRequest **aRequest, nsIDOMNode *aDOMNode)
{
  NS_ENSURE_ARG(aDOMNode);
  NS_ENSURE_ARG_POINTER(aRequest);

  // Get content
  nsCOMPtr<nsIImageLoadingContent> content(do_QueryInterface(aDOMNode));
  NS_ENSURE_TRUE(content, NS_ERROR_FAILURE);

  return content->GetRequest(nsIImageLoadingContent::CURRENT_REQUEST,
                             aRequest);
}

/* static */
nsresult
MicrobEalContextMenuInfo::CheckDomImageElement(nsIDOMNode *node, nsString& aHref,
                                           PRInt32 &aWidth, PRInt32 &aHeight, PRUint32 *aSize)
{
  nsresult rv;
  nsCOMPtr<nsIDOMHTMLImageElement> image =
    do_QueryInterface(node, &rv);
  NS_ENSURE_TRUE(image, rv);
  rv = image->GetSrc(aHref);
  NS_ENSURE_SUCCESS(rv, rv);
  image->GetWidth(&aWidth);
  image->GetHeight(&aHeight);
  if (!aSize) return rv;
  nsCOMPtr<imgIRequest> request;
  GetImageRequest(getter_AddRefs(request), node);
  NS_ENSURE_TRUE(request, rv);
  nsCOMPtr<imgIContainer> imgCont;
  request->GetImage(getter_AddRefs(imgCont));
  NS_ENSURE_TRUE(imgCont, rv);
  PRUint32 currentFrameIndex = 0;
  imgCont->GetCurrentFrameIndex(&currentFrameIndex);
  rv = imgCont->GetFrameImageDataLength(currentFrameIndex, aSize);
  return rv;
}

/* static */
nsresult
MicrobEalContextMenuInfo::GetJSHandledInfo(nsIDOMEvent *aEvent, PRInt32 *aIsJSHandled)
{
  if (!aEvent && !aIsJSHandled)
    return NS_ERROR_FAILURE;
  nsresult rv = NS_ERROR_FAILURE;
#ifdef MOZ_JS_HANDLER_INFO
  if (aEvent) {
    PRUint32 flags;
    rv = aEvent->GetPrivateFlags(&flags);
    if (NS_SUCCEEDED(rv) && (flags & NS_PRIV_EVENT_FLAG_SCRIPT)) {
      mIsJSHandledElement = aIsJSHandled?*aIsJSHandled:1;
    } else {
      mIsJSHandledElement = 0;
      rv = NS_ERROR_FAILURE;
    }
  }
  if (aIsJSHandled)
    *aIsJSHandled = mIsJSHandledElement;
#endif
  return rv;
}

nsresult
MicrobEalContextMenuInfo::InsertTextToNode(nsIDOMNode *aDOMNode, nsAString &aString, PRInt32 surr1, PRInt32 surr2)
{
  nsresult rv;

  nsCOMPtr<nsIDOMNode> targetNode = do_QueryInterface(aDOMNode, &rv);
  if (targetNode) {
    if (!(!mEventNode || mEventNode == targetNode) && mInsertNode != targetNode) {
      rv = CheckFormControl(aDOMNode);
      if (NS_SUCCEEDED(rv))
        mInsertNode = targetNode;
    }
  }
  else
    targetNode = do_QueryInterface(mEventNode, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

#ifdef MOZ_XUL
  nsCOMPtr<nsIDOMXULTextBoxElement> text = do_QueryInterface(targetNode, &rv);
  if (text) {
    rv = text->GetInputField(getter_AddRefs(targetNode));
    NS_ENSURE_SUCCESS(rv, rv);
    rv = CheckFormControl(targetNode);
    NS_ENSURE_SUCCESS(rv, rv);
  }
#endif

  nsString nodeName, buffer;
  PRInt32 selectionStart = 0, selectionEnd = 0;
  targetNode->GetNodeName(nodeName);
  if (mCtxFormType == NS_FORM_TEXTAREA) {
    nsCOMPtr<nsIDOMHTMLTextAreaElement> input;
    input = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    PRBool rdonly = PR_FALSE;
    input->GetReadOnly(&rdonly);
    if (rdonly) return NS_ERROR_FAILURE;

    nsCOMPtr<nsIDOMNSHTMLTextAreaElement> nsinput;
    nsinput = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = input->GetValue(buffer);
    if (!buffer.IsEmpty()) {
      NS_ENSURE_SUCCESS(rv, rv);
      rv = input->GetValue(buffer);
      nsinput->GetSelectionStart(&selectionStart);
      nsinput->GetSelectionEnd(&selectionEnd);

      if (selectionStart != selectionEnd)
        buffer.Cut(selectionStart, selectionEnd - selectionStart);
      if (surr1 != 0 || surr2 != 0) {
        buffer.Cut(selectionStart + surr1, surr2);
        buffer.Insert(aString, selectionStart + surr1);
      } else
        buffer.Insert(aString, selectionStart);
    } else
      buffer.Assign(aString);

    nsCOMPtr<nsIDOMNSEditableElement> nseditable;
    nseditable = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    nseditable->SetUserInput(buffer);

    rv = SendFormInputEvent(targetNode);
    NS_ENSURE_SUCCESS(rv, rv);
   
    int len = aString.Length() + surr1;
    nsinput->SetSelectionRange(selectionStart + len, selectionStart + len);

    nsCOMPtr<nsIEditor> theEditor;
    nseditable->GetEditor(getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_OK);

    nsCOMPtr<nsISelectionController> selectionController;
    theEditor->GetSelectionController(getter_AddRefs(selectionController));
    PRBool sync = PR_TRUE;
    short selectionType = nsISelectionController::SELECTION_NORMAL;
    selectionController->ScrollSelectionIntoView(selectionType, selectionType, sync);
  }
  else if ((mCtx->type & GTK_MICROB_EAL_CONTEXT_INPUT_TEXT) && !(mCtx->type & GTK_MICROB_EAL_CONTEXT_RICHEDIT)) {
    nsCOMPtr<nsIDOMHTMLInputElement> input;
    input = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    PRBool rdonly = PR_FALSE;
    input->GetReadOnly(&rdonly);
    if (rdonly) return NS_ERROR_FAILURE;

    nsCOMPtr<nsIDOMNSHTMLInputElement> nsinput;
    nsinput = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = input->GetValue(buffer);
    if (!buffer.IsEmpty()) {
      nsinput->GetSelectionStart(&selectionStart);
      nsinput->GetSelectionEnd(&selectionEnd);

      if (selectionStart != selectionEnd)
        buffer.Cut(selectionStart, selectionEnd - selectionStart);

      if (surr1 != 0 || surr2 != 0) {
        buffer.Cut(selectionStart + surr1, surr2);
        buffer.Insert(aString, selectionStart + surr1);
      } else
        buffer.Insert(aString, selectionStart);
    } else
      buffer.Assign(aString);

    nsCOMPtr<nsIDOMNSEditableElement> nseditable;
    nseditable = do_QueryInterface(targetNode, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    nsCOMPtr<nsIEditor> theEditor;
    nseditable->GetEditor(getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_OK);

    if (mCtx->type & GTK_MICROB_EAL_CONTEXT_IPASSWORD)
      theEditor->SetPasswordPosition(selectionStart+1);
    else
      theEditor->SetPasswordPosition(0);

    nseditable->SetUserInput(buffer);
    rv = SendFormInputEvent(targetNode);
    NS_ENSURE_SUCCESS(rv, rv);
    nsinput->SetSelectionRange(selectionStart + surr1 + aString.Length(), selectionStart + surr1 + aString.Length());

    nsCOMPtr<nsISelectionController> selectionController;
    theEditor->GetSelectionController(getter_AddRefs(selectionController));
    PRBool sync = PR_TRUE;
    short selectionType = nsISelectionController::SELECTION_NORMAL;
    selectionController->ScrollSelectionIntoView(selectionType, selectionType, sync);
  }
  else {
    nsCOMPtr<nsIWebBrowser> webBrowser;
    gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(mEmbed), getter_AddRefs(webBrowser));
    NS_ENSURE_TRUE(webBrowser, NS_ERROR_FAILURE);
    nsCOMPtr<nsIEditingSession> editingSession = do_GetInterface(webBrowser);
    if (!editingSession)
      return NS_ERROR_FAILURE;

    nsCOMPtr<nsIDOMWindow> domWin;
    rv = GetDOMWindowByNode(aDOMNode, getter_AddRefs(domWin));
    NS_ENSURE_SUCCESS(rv, rv);
    nsCOMPtr<nsPIDOMWindow> piWin(do_QueryInterface(domWin));
    NS_ENSURE_TRUE(piWin, NS_ERROR_FAILURE);
    nsCOMPtr<nsIEditor> theEditor;
    editingSession->GetEditorForWindow(piWin, getter_AddRefs(theEditor));
    NS_ENSURE_TRUE(theEditor, NS_ERROR_FAILURE);

    nsCOMPtr<nsIPlaintextEditor> textEditor;
    textEditor = do_QueryInterface(theEditor, &rv);
    if (NS_FAILED(rv) || !textEditor)
      return NS_ERROR_FAILURE;
    if (surr1 != 0 || surr2 != 0) {
      DoCommand(aDOMNode?(nsISupports*)aDOMNode:(nsISupports*)webBrowser, "cmd_delete", surr2);
    }
    textEditor->InsertText(aString);
  }
  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::GetEventData(bool& aIsMouseEvent)
{
  nsresult rv;

  nsCOMPtr<nsIDOMMouseEvent> mouseEvent(do_QueryInterface(mCtxEvent, &rv));
  aIsMouseEvent = (mouseEvent != nsnull);
  if (!mouseEvent) {
    nsCOMPtr<nsIDOMNSElement> nselement;
    nselement = do_QueryInterface(mOrigNode, &rv);
    if (nselement) {
#ifndef MOZ_TRUNK_BUILD
      nsCOMPtr<nsIDOMTextRectangle> rect;
#else
      nsCOMPtr<nsIDOMClientRect> rect;
#endif
      nselement->GetBoundingClientRect(getter_AddRefs(rect));
      if (rect) {
        float l = 0, t = 0;
        //float r = 0, b = 0;
        rect->GetLeft((float*)&l);
        rect->GetTop((float*)&t);
        mCtx->x = (guint)trunc(l);
        mCtx->y = (guint)trunc(t);
      }
    }
  } else {
    PRUint16 btn = 1729;
    mouseEvent->GetButton (&btn);
    switch (btn) {
    case 0:
      mCtx->button = 1;
      break;
    case 1:
      mCtx->button = 2;
      break;
    case 2:
      mCtx->button = 3;
      break;
    case (PRUint16) -1:
    default:
      mCtx->button = 0;
      break;
    }
    if (mCtx->button != 0) {
      mouseEvent->GetClientX((PRInt32*)&mCtx->x);
      mouseEvent->GetClientY((PRInt32*)&mCtx->y);
      mouseEvent->GetScreenX((PRInt32*)&mCtx->x_root);
      mouseEvent->GetScreenY((PRInt32*)&mCtx->y_root);
    }
  }
  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::RecalculateContextData(nsIDOMNode* aDOMNode)
{
  NS_ENSURE_TRUE(mCtxEvent && aDOMNode, NS_ERROR_FAILURE);

  UpdateContextDataFromNode(aDOMNode, true);

  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::UpdateContextDataFromElement(nsIDOMNSElement* aDOMElement)
{
  nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(aDOMElement);

  return UpdateContextDataFromNode(eventNode.get(), false);
}

nsresult
MicrobEalContextMenuInfo::UpdateContextData(nsIDOMEvent *aDOMEvent)
{
  NS_ENSURE_ARG_POINTER(aDOMEvent);

  nsresult rv;

  if (mCtxEvent == aDOMEvent)
    return NS_OK;

  mCtxEvent = do_QueryInterface(aDOMEvent, &rv);
  NS_ENSURE_SUCCESS(rv , NS_OK);

/* Does not work with touch component
  PRUint16 eventphase;
  mCtxEvent->GetEventPhase(&eventphase);
  if (!eventphase) {
    mCtxEvent = nsnull;
    return NS_ERROR_FAILURE;
  }
*/

  // Get mEventNode
  nsCOMPtr<nsIDOMEventTarget> eventTarget;
  rv = mCtxEvent->GetTarget(getter_AddRefs(eventTarget));
  NS_ENSURE_SUCCESS(rv , NS_OK);
  nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget, &rv);
  NS_ENSURE_SUCCESS(rv , NS_OK);

  bool isMouseEvent = false;
  GetEventData(isMouseEvent);
  return UpdateContextDataFromNode(eventNode, false, isMouseEvent);
}

nsresult
MicrobEalContextMenuInfo::UpdateContextDataFromNode(nsIDOMNode *aDOMNode, bool aForce, bool aUpdateTitle)
{
  nsresult rv;

  if (aDOMNode == mOrigNode && !aForce)
    return NS_OK;
  mEventNode = mOrigNode = aDOMNode;

  mCtxDocument = nsnull;
  mCtx->docindex = -1;
  mCtx->type = GTK_MICROB_EAL_CONTEXT_NONE;

  if (IsXULNode(mOrigNode, &mIsScrollableElement)) {
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_XUL;
    IsXULEdit(mEventNode, &mCtx->type);
    return NS_OK;
  }

  mIsScrollableElement = PR_TRUE;

  // Get mCtxDocument
  {
    nsCOMPtr<nsIDOMDocument> domDoc;
    rv = mEventNode->GetOwnerDocument(getter_AddRefs(domDoc));
    if (!NS_SUCCEEDED(rv) || !domDoc)
      rv = mOrigNode->GetOwnerDocument(getter_AddRefs(domDoc));
    if (NS_SUCCEEDED(rv) && domDoc && mCtxDocument != domDoc) {
      mCtxDocument = domDoc;
      nsCOMPtr<nsIDOM3Document> docuri = do_QueryInterface(mCtxDocument);
      NS_ENSURE_TRUE (docuri, NS_ERROR_FAILURE);
      nsString uri;
      docuri->GetDocumentURI(uri);
      SetStringProperty("docuri", uri);
      nsCOMPtr<nsIDOMHTMLDocument> doc = do_QueryInterface(mCtxDocument);
      if (doc) {
        doc->GetTitle(uri);
        SetStringProperty("doctitle", uri);
      }
    }
  }
  SetStringProperty("link", "");
  SetStringProperty("image", "");

  nsCOMPtr<nsIDOMDocumentView> docView (do_QueryInterface (mCtxDocument));
  NS_ENSURE_TRUE (docView, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMAbstractView> abstractView;
  docView->GetDefaultView (getter_AddRefs (abstractView));
  NS_ENSURE_TRUE (abstractView, NS_ERROR_FAILURE);
  mViewCSS = do_QueryInterface (abstractView);
  NS_ENSURE_TRUE (mViewCSS, NS_ERROR_FAILURE);

  nsCOMPtr<nsIWebNavigation> webNav (do_GetInterface (abstractView, &rv));
  NS_ENSURE_SUCCESS (rv, rv);

  rv = webNav->GetCurrentURI (getter_AddRefs (mBaseURI));
  NS_ENSURE_SUCCESS (rv, rv);

  SetFramePosition();
  CheckRichEdit();
  CheckFormControl(aDOMNode);

  nsCOMPtr<nsIDOMNode> node;
  nsCOMPtr<nsIDOMNode> parentNode;

  nsCOMPtr<nsIDOMHTMLElement> element = do_QueryInterface(mEventNode, &rv);
  if (element)
    node = do_QueryInterface(mEventNode, &rv);
  else /* Fallback to XML/XSLT content */
    node = do_QueryInterface(mOrigNode, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  PRBool has_image = PR_FALSE;
  if (!(mCtx->type & GTK_MICROB_EAL_CONTEXT_FORM)) { // form controls do not have CSM
    while (node) {
      if (NS_FAILED(CheckDomHtmlNode(node, has_image)))
        break;
    
      node->GetParentNode(getter_AddRefs(parentNode));
      node = parentNode;
    }
    mCtx->type |= GTK_MICROB_EAL_CONTEXT_DOCUMENT;
  }

  // update tooltip title only from mouse events
  if (aUpdateTitle) {
    nsString elementTitle;
    if (NS_SUCCEEDED(GetElementTitle(aDOMNode, elementTitle)))
      SetStringProperty("title", elementTitle);
    else
      SetStringProperty("title", "");
  }

  return NS_OK;
}

nsresult
MicrobEalContextMenuInfo::GetContextText(nsAString &aString, PRBool aSelection, PRInt32 *aSelStart)
{
  nsresult rv;

  nsCOMPtr<nsIDOMNode> targetNode = do_QueryInterface(mEventNode, &rv);

#ifdef MOZ_XUL
  nsCOMPtr<nsIDOMXULTextBoxElement> text = do_QueryInterface(targetNode, &rv);
  if (text) {
    rv = text->GetInputField(getter_AddRefs(targetNode));
    if (targetNode)
      CheckFormControl(targetNode);
  }
#endif

  if (mCtxFormType != 0 && targetNode) {
    PRInt32 selStart = 0, selEnd = 0;
    if (mCtxFormType == NS_FORM_INPUT_TEXT
        || mCtxFormType == NS_FORM_INPUT_PASSWORD
        || mCtxFormType == NS_FORM_INPUT_FILE) {
      nsCOMPtr<nsIDOMHTMLInputElement> input = do_QueryInterface(targetNode, &rv);
      if (NS_SUCCEEDED(rv) && input)
        rv = input->GetValue(aString);
      if (aString.IsEmpty()) return NS_ERROR_FAILURE;
      nsCOMPtr<nsIDOMNSHTMLInputElement> nsinput = do_QueryInterface(targetNode, &rv);
      if (NS_SUCCEEDED(rv) && nsinput) {
        nsinput->GetSelectionEnd(&selEnd);
        nsinput->GetSelectionStart(&selStart);
      }
    } else if (mCtxFormType == NS_FORM_TEXTAREA) {
      nsCOMPtr<nsIDOMHTMLTextAreaElement> input = do_QueryInterface(targetNode, &rv);
      if (NS_SUCCEEDED(rv) && input)
        rv = input->GetValue(aString);
      if (aString.IsEmpty()) return NS_ERROR_FAILURE;
      nsCOMPtr<nsIDOMNSHTMLTextAreaElement> nsinput = do_QueryInterface(targetNode, &rv);
      if (NS_SUCCEEDED(rv) && nsinput) {
        nsinput->GetSelectionEnd(&selEnd);
        nsinput->GetSelectionStart(&selStart);
      }
    }
    if (selStart < selEnd && aSelection) {
      PRInt32 textLength = aString.Length();
      aString.Cut(0, selStart);
      aString.Cut(selEnd-selStart, textLength);
    }
    if (aSelStart) *aSelStart = selStart;
  } else if (mCtxDocument && aSelection) {
    nsCOMPtr<nsIDOMNSHTMLDocument> htmlDoc = do_QueryInterface(mCtxDocument, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    rv = htmlDoc->GetSelection(aString);
    if (NS_FAILED(rv) || aString.IsEmpty())
      return NS_ERROR_FAILURE;
  }
  return NS_OK;
}

//static
nsresult
MicrobEalContextMenuInfo::DoCommand(nsISupports *aContext, const char *aCommand, PRInt32 aCount)
{
  nsCOMPtr<nsIDOMWindow> DOMWindow = do_QueryInterface(aContext);
  if (!DOMWindow) {
    nsCOMPtr<nsIWebBrowser> webBrowser = do_QueryInterface(aContext);
    if (webBrowser)
      webBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));
  }
  if (!DOMWindow) {
    nsCOMPtr<nsIDOMNode> node = do_QueryInterface(aContext);
    if (node)
      GetDOMWindowByNode(node, getter_AddRefs(DOMWindow));
  }

  nsCOMPtr<nsPIDOMWindow> window(do_QueryInterface(DOMWindow));
  NS_ENSURE_TRUE(window, NS_ERROR_FAILURE);

  nsCOMPtr<nsICommandManager> cmdMgr(do_GetInterface(window->GetDocShell()));
  NS_ENSURE_TRUE(cmdMgr, NS_ERROR_FAILURE);

  nsresult rv = NS_OK;
  if (!aCount) {
    PRBool enabled = PR_FALSE;
    rv = cmdMgr->IsCommandEnabled(aCommand, DOMWindow, &enabled);
    if (enabled)
      return NS_OK;
    return NS_ERROR_FAILURE;
  }
  for (int i = 0; i < aCount && NS_SUCCEEDED(rv); i++)
    rv = cmdMgr->DoCommand(aCommand, nsnull, DOMWindow);
  return rv;
}

nsresult
MicrobEalContextMenuInfo::GetElementTitle(nsIDOMNode *aNode, nsString &title)
{
  NS_ENSURE_TRUE(aNode, NS_ERROR_FAILURE); 
  nsCOMPtr<nsIDOMNode> eventNode = aNode;

  nsCOMPtr<nsIDOMHTMLImageElement> image = do_QueryInterface(eventNode);
  if (image) {
    image->GetTitle(title);
    if (!title.IsEmpty())
      return NS_OK;
    image->GetAlt(title);
    if (!title.IsEmpty())
      return NS_OK;
  }

  nsCOMPtr<nsIDOMNode> parentNode = aNode;

  while (eventNode) {
    nsCOMPtr<nsIDOMHTMLAnchorElement> anchor = do_QueryInterface(eventNode); 
    if (anchor) {
      anchor->GetTitle(title);
      if (!title.IsEmpty())
        return NS_OK;
      anchor->GetHref(title);
      if (!title.IsEmpty())
        return NS_OK;
    }
    eventNode->GetParentNode(getter_AddRefs(parentNode));
    if (eventNode == parentNode)
      break;
    eventNode = parentNode;
  }

  image = do_QueryInterface(eventNode);
  if (image) {
    image->GetSrc(title);
    if (!title.IsEmpty())
      return NS_OK;
  }
  return NS_ERROR_FAILURE;
}

// Need to expose this API from DOMWindowUtils...
#include "nsIScrollableFrame.h"

static inline nsIFrame* GetFrameFor(nsIView *aView)
{ 
  NS_ENSURE_TRUE(aView, nsnull);
  return static_cast<nsIFrame*>(aView->GetClientData());
}

static inline nsIScrollableFrame*
GetScrollableFrameFor(nsIScrollableView *aScrollableView)
{
  NS_ENSURE_TRUE(aScrollableView, nsnull);
  nsIFrame *frame = GetFrameFor(aScrollableView->View()->GetParent());
  nsIScrollableFrame *sf = do_QueryFrame(frame);
  return sf;
}

nsPresContext::ScrollbarStyles
ScrollbarStylesOfView(nsIScrollableView *aScrollableView)
{
  if (aScrollableView) {
    nsIScrollableFrame *sf = GetScrollableFrameFor(aScrollableView);
    if (sf)
        return sf->GetScrollbarStyles();
  }
  return nsPresContext::ScrollbarStyles(NS_STYLE_OVERFLOW_HIDDEN,
                                        NS_STYLE_OVERFLOW_HIDDEN);
}

static nsIScrollableView*
GetNearestScrollingView(nsIView* aView, int aDirection = G_WEB_SCROLLABLE_ANY)
{
  // If aDirection is eEither, find first view with a scrolllable frame.
  // Otherwise, find the first view that has a scrollable frame whose
  // ScrollbarStyles is not NS_STYLE_OVERFLOW_HIDDEN in aDirection
  // and where there is something currently not visible
  // that can be scrolled to in aDirection.
  NS_ASSERTION(aView, "GetNearestScrollingView expects a non-null view");
  nsIScrollableView* scrollableView = nsnull;
  for (; aView; aView = aView->GetParent()) {
    scrollableView = aView->ToScrollableView();
    if (scrollableView) {
      nsPresContext::ScrollbarStyles ss =
        ScrollbarStylesOfView(scrollableView);
      nsIScrollableFrame *scrollableFrame = GetScrollableFrameFor(scrollableView);
      NS_ASSERTION(scrollableFrame, "Must have scrollable frame for view!");
      nsMargin margin = scrollableFrame->GetActualScrollbarSizes();
      // Get size of total scrollable area
      nscoord totalWidth, totalHeight;
      scrollableView->GetContainerSize(&totalWidth, &totalHeight);
      // Get size of currently visible area
      nsSize visibleSize = aView->GetBounds().Size();
      // aDirection can be eHorizontal, eVertical, or eEither
      // If scrolling in a specific direction, require visible scrollbars or
      // something to scroll to in that direction.
      bool ver_hidden = NS_STYLE_OVERFLOW_HIDDEN == ss.mVertical;
      bool hor_hidden = NS_STYLE_OVERFLOW_HIDDEN == ss.mHorizontal;
      PRBool up, down, left, right; // this is mouse direction
      scrollableView->CanScroll(PR_FALSE, PR_TRUE, up); // == aDy > 0
      scrollableView->CanScroll(PR_FALSE, PR_FALSE, down); // == aDy < 0
      scrollableView->CanScroll(PR_TRUE, PR_TRUE, left); // == aDx > 0
      scrollableView->CanScroll(PR_TRUE, PR_FALSE, right); // == aDx < 0
      if ((!ver_hidden || !hor_hidden) && aDirection & G_WEB_SCROLLABLE_ANY && (up || down || left || right))
        break;
      if (up && !ver_hidden && aDirection & G_WEB_SCROLLABLE_UP) break;
      if (down && !ver_hidden && aDirection & G_WEB_SCROLLABLE_DOWN) break;
      if (right && !hor_hidden && aDirection & G_WEB_SCROLLABLE_RIGHT) break;
      if (left && !hor_hidden && aDirection & G_WEB_SCROLLABLE_LEFT) break;
    }
  }
  return scrollableView;
}

nsresult
MicrobEalContextMenuInfo::GetViewToScroll(nsIDOMNode *aDOMNode, nsIScrollableView **aScrollView, int aDirection)
{
  NS_ENSURE_ARG_POINTER(aScrollView);
  nsresult rv;
  nsCOMPtr <nsIContent> content = do_QueryInterface(aDOMNode, &rv);
  NS_ENSURE_TRUE(content, rv);

  nsCOMPtr<nsIDOMDocument> nodeDoc;
  rv = aDOMNode->GetOwnerDocument(getter_AddRefs(nodeDoc));
  nsCOMPtr<nsIDocument> doc = do_QueryInterface(nodeDoc, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  nsIPresShell *presShell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(presShell, NS_ERROR_FAILURE);

  nsIScrollableView* scrollView = nsnull;
  nsIFrame* startFrame = presShell->GetPrimaryFrameFor(content);
  if (startFrame) {
    nsIScrollableViewProvider* svp = do_QueryFrame(startFrame);
    nsIScrollableView* sv;
    nsIView* startView = NULL;
    if (svp && (sv = svp->GetScrollableView()))
      startView = sv->View();
    else {
      nsIFrame* ancView = startFrame->GetAncestorWithViewExternal();
      if (ancView)
        startView = ancView->GetViewExternal();
    }
    NS_ASSERTION(startView, "No view to start searching for scrollable view from");
    scrollView = GetNearestScrollingView(startView, aDirection);
  }

  if (!scrollView) {
    nsIViewManager* viewManager = presShell->GetViewManager();
    if (viewManager)
      viewManager->GetRootScrollableView(&scrollView);
  }

  NS_ENSURE_TRUE(scrollView, NS_ERROR_FAILURE);
  if (aScrollView) {
    *aScrollView = scrollView;
    NS_ENSURE_TRUE(*aScrollView, NS_ERROR_FAILURE);
  }

  return NS_OK;
}

int
MicrobEalContextMenuInfo::GetScrollableType(PRBool aIgnoreHidden)
{
  if (!mScrollableView)
    UpdateScrollableView();
  return MicrobEalContextMenuInfo::GetScrollableType(mScrollableView, aIgnoreHidden);
}

int
MicrobEalContextMenuInfo::GetScrollableType(nsIScrollableView *aScrollView, PRBool aIgnoreHidden)
{
  int type = G_WEB_SCROLLABLE_NONE;
  NS_ENSURE_TRUE(aScrollView, type);
  nsPresContext::ScrollbarStyles ss = ScrollbarStylesOfView(aScrollView);
  bool ver_hidden = NS_STYLE_OVERFLOW_HIDDEN == ss.mVertical && !aIgnoreHidden;
  bool hor_hidden = NS_STYLE_OVERFLOW_HIDDEN == ss.mHorizontal && !aIgnoreHidden;
  PRBool up, down, left, right; // this is mouse direction
  aScrollView->CanScroll(PR_FALSE, PR_TRUE, up); // == aDy > 0
  aScrollView->CanScroll(PR_FALSE, PR_FALSE, down); // == aDy < 0
  aScrollView->CanScroll(PR_TRUE, PR_TRUE, left); // == aDx > 0
  aScrollView->CanScroll(PR_TRUE, PR_FALSE, right); // == aDx < 0
  if (up && !ver_hidden) type |= G_WEB_SCROLLABLE_UP;
  if (down && !ver_hidden) type |= G_WEB_SCROLLABLE_DOWN;
  if (right && !hor_hidden) type |= G_WEB_SCROLLABLE_RIGHT;
  if (left && !hor_hidden) type |= G_WEB_SCROLLABLE_LEFT;
  return type;
}

nsresult
MicrobEalContextMenuInfo::ScrollScrollableView(PRInt32 aDX, PRInt32 aDY, PRInt32 *aOverflowX, PRInt32 *aOverflowY)
{
  int type = GetScrollableType();
  NS_ENSURE_TRUE(mScrollableView, NS_ERROR_FAILURE);
  if (!aDY && aDX && type) {
    nsIScrollableView *temp = mScrollableView;
    if (aDX > 0 && (!(type & G_WEB_SCROLLABLE_RIGHT))) {
      UpdateScrollableView(nsnull, G_WEB_SCROLLABLE_RIGHT);
    } else if (aDX < 0 && (!(type & G_WEB_SCROLLABLE_LEFT))) {
      UpdateScrollableView(nsnull, G_WEB_SCROLLABLE_LEFT);
    }
    if (!mScrollableView || !GetScrollableType())
      mScrollableView = temp;
  }
  if (!aDX && aDY && type) {
    nsIScrollableView *temp = mScrollableView;
    if (aDY > 0 && (!(type & G_WEB_SCROLLABLE_DOWN))) {
      UpdateScrollableView(nsnull, G_WEB_SCROLLABLE_DOWN);
    } else if (aDY < 0 && (!(type & G_WEB_SCROLLABLE_UP))) {
      UpdateScrollableView(nsnull, G_WEB_SCROLLABLE_UP);
    }
    if (!mScrollableView || !GetScrollableType())
      mScrollableView = temp;
  }
  PRInt32 overflowX = 0, overflowY = 0;
  nsresult rv = mScrollableView->ScrollByPixels(-aDX, -aDY, overflowX, overflowY);
  if (aOverflowX) *aOverflowX = overflowX;
  if (aOverflowY) *aOverflowY = overflowY;
  return rv;
}

nsresult
MicrobEalContextMenuInfo::SendFormInputEvent(nsIDOMNode *aDOMNode)
{
  ULOG();

  nsresult rv;

  nsCOMPtr<nsIContent> content = do_QueryInterface(aDOMNode, &rv);
  NS_ENSURE_TRUE(content, rv);

  nsCOMPtr<nsIDOMDocument> nodeDoc;
  rv = aDOMNode->GetOwnerDocument(getter_AddRefs(nodeDoc));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIDocument> doc = do_QueryInterface(nodeDoc, &rv);
  NS_ENSURE_SUCCESS(rv, rv);

  nsIPresShell *presShell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(presShell, NS_ERROR_FAILURE);

  nsEventStatus status = nsEventStatus_eIgnore;
  nsInputEvent event(PR_TRUE, NS_FORM_INPUT, nsnull);
  return presShell->HandleEventWithTarget(&event, nsnull, content, &status);
}

nsresult
MicrobEalContextMenuInfo::UpdateScrollableView(nsIDOMNode *aNode, int aDirection)
{
  return GetViewToScroll(aNode ? aNode : mEventNode.get(), &mScrollableView, aDirection ? aDirection : G_WEB_SCROLLABLE_ANY);
}

nsresult
MicrobEalContextMenuInfo::GetDomWindowScrollInfo(nsIDOMWindow *aDOMWindow, PRInt32 *aW, PRInt32 *aH, PRInt32 *aScrMX, PRInt32 *aScrMY, PRInt32 *aScrX, PRInt32 *aScrY, PRInt32 *aBidiOffset)
{
  NS_ENSURE_ARG_POINTER(aDOMWindow);
  nsresult rv;
  nsCOMPtr<nsIDocument> doc;
  nsCOMPtr<nsIDOMDocument> domDoc;
  rv = aDOMWindow->GetDocument(getter_AddRefs(domDoc));
  doc = do_QueryInterface(domDoc, &rv);
  NS_ENSURE_TRUE(doc, NS_ERROR_FAILURE);

  nsIPresShell *shell = doc->GetPrimaryShell();
  NS_ENSURE_TRUE(shell, NS_ERROR_FAILURE);

  PRBool bidi = PR_FALSE;
  if (doc->GetBidiEnabled()) {
    // Internal API usage, should be implemented as part of CSSOM
    nsIFrame *frame = nsnull;
    nsIContent *root = doc->GetRootContent();
    nsCOMPtr<nsIHTMLDocument> htmlDoc = do_QueryInterface(doc);
    if (htmlDoc) {
      nsIContent *bodyContent = htmlDoc->GetBodyContentExternal();
      if (bodyContent)
        root = bodyContent; // we can trust the document to hold on to it
    }

    if (root) {
      nsIFrame *rootsFrame = shell->GetPrimaryFrameFor(root);
      if (rootsFrame)
        frame = rootsFrame;
    }
    if (frame)
      bidi = frame->GetStyleVisibility()->mDirection == NS_STYLE_DIRECTION_RTL;
  }

  nsCOMPtr<nsIViewManager> viewManager;
  viewManager = shell->GetViewManager();
  NS_ENSURE_TRUE(viewManager, NS_ERROR_FAILURE);
  nsIScrollableView *scrollView = nsnull;
  rv = viewManager->GetRootScrollableView(&scrollView);
  nsSize scrolledSize;
  if (!scrollView) {
    nsCOMPtr<nsIDOMWindowInternal> mainWinInt = do_QueryInterface(aDOMWindow, &rv);
    NS_ENSURE_SUCCESS(rv, rv); // we should have internal aDOMWindow
    rv = NS_OK;
    if (aW)
      rv |= mainWinInt->GetInnerWidth(aW);
    if (aH)
      rv |= mainWinInt->GetInnerHeight(aH);
    return rv;
  }

  nsRect portRect = scrollView->View()->GetBounds();
  if (aW)
    *aW = (PRInt32)floor(nsPresContext::AppUnitsToFloatCSSPixels(portRect.width));
  if (aH)
    *aH = (PRInt32)floor(nsPresContext::AppUnitsToFloatCSSPixels(portRect.height));

  rv = scrollView->GetContainerSize(&scrolledSize.width, &scrolledSize.height);
  NS_ENSURE_SUCCESS(rv, rv);

  PRInt32 scrMax = PR_MAX(0, (PRInt32)floor(nsPresContext::AppUnitsToFloatCSSPixels(scrolledSize.width - portRect.width)));
  if (aScrMX)
    *aScrMX = scrMax;
  if (aBidiOffset && bidi)
    *aBidiOffset = scrMax;
  if (aScrMY)
    *aScrMY = PR_MAX(0, (PRInt32)floor(nsPresContext::AppUnitsToFloatCSSPixels(scrolledSize.height - portRect.height)));

  if (!aScrX && !aScrY)
    return NS_OK;
  nscoord xPos, yPos;
  rv = scrollView->GetScrollPosition(xPos, yPos);
  NS_ENSURE_SUCCESS(rv, rv);
  if (aScrX) {
    *aScrX = nsPresContext::AppUnitsToIntCSSPixels(xPos);
    if (bidi)
      *aScrX += scrMax;
  }

  if (aScrY)
    *aScrY = nsPresContext::AppUnitsToIntCSSPixels(yPos);

  return NS_OK;
}


