/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Oleg Romashin <romaxa@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib.h>
#include "common.h"

#include "gmozillacppwrapper.h"
#include "MicrobEalObserver.h"
#include "nsCOMPtr.h"
#include "nsICategoryManager.h"
#include "nsStringGlue.h"
#include "nsISupportsPrimitives.h"
#include "nsIExtensionManager.h"
#include "nsIComponentRegistrar.h"
#include "nsIHttpAuthManager.h"
#include "nsICookieManager.h"
#include "nsICookie2.h"
#include "nsNetCID.h"
#include "MicrobEalCertificate.h"
#include <nsIComponentManager.h>
#include <nsComponentManagerUtils.h>
#include <nsILocalFile.h>

#include <unistd.h>
#include "MicrobEalUtils.h"
#include "MicrobEalXshmRenderer.h"
#include "nsIDOMHTMLSelectElement.h"

#include "nsIWindowWatcher.h"
#include "nsIDOMWindow.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIPrefBranch2.h"
#include "nsIPrefService.h"
#include "nsIDOMEvent.h"
#include "nsIDOMEventTarget.h"
#include "nsSHistoryListener.h"
#include "nsIAppStartup.h"

static NS_DEFINE_CID(kHttpAuthManagerCID,        NS_HTTPAUTHMANAGER_CID);

#define DOM_WINDOW_OPENED       "domwindowopened"
#define DOM_WINDOW_CLOSED       "domwindowclosed"
#define DELAYED_RSS_URL         "delayed-rss-url"
#define DOMLINK_MAYBE_FEED      "DOMLink_MAYBE_FEED"

#ifdef MOZEAL_LOGGING
static gboolean bmozeal_trace = FALSE;
void
pr_mozeal_trace (const char *fmt, ...)
{
    if (bmozeal_trace)
    {
        va_list vargs;
        va_start (vargs, fmt);
        vprintf (fmt, vargs);
        va_end (vargs);
    }
}
#endif

nsresult
MicrobEalObserver::Create(nsISupports *aOuter, REFNSIID aIID, void **aResult)
{
  NS_ENSURE_ARG_POINTER(aResult);
  TRACE_LOG();
  nsresult rv;

  if (aOuter) {
    rv = NS_ERROR_NO_AGGREGATION;
    return rv;
  }

  MicrobEalObserver* adapter = new MicrobEalObserver();
  if (!adapter) {
    rv = NS_ERROR_OUT_OF_MEMORY;
    return rv;
  }

  NS_ADDREF(adapter);
  rv = adapter->Init();
  if (NS_SUCCEEDED(rv)) {
    rv = adapter->QueryInterface(aIID, aResult);
  }
  NS_RELEASE(adapter);

  return rv;
}

nsresult
MicrobEalObserver::RegisterSelf(nsIComponentManager* aCompMgr,
                                    nsIFile* aPath,
                                    const char* aLoaderStr,
                                    const char* aType,
                                    const nsModuleComponentInfo *aInfo)
{
  TRACE_LOG();
  nsresult rv;
  nsCOMPtr<nsICategoryManager> catMan = do_GetService(NS_CATEGORYMANAGER_CONTRACTID, &rv);
  if (!catMan) return NS_ERROR_FAILURE;

  rv = catMan->AddCategoryEntry(G_MOZILLA_APP_STARTUP,
                                G_MOZILLA_MICROB_EAL_OBSERVER,
                                NS_EMBED_CLIENT_OBSERVER_CONTRACTID,
                                PR_TRUE, PR_TRUE, nsnull);

  return rv;
}

nsresult
MicrobEalObserver::UnregisterSelf(nsIComponentManager* aCompMgr,
                                      nsIFile* aPath,
                                      const char *registryLocation,
                                      const nsModuleComponentInfo *aInfo)
{
  TRACE_LOG();
  nsresult rv;
  nsCOMPtr<nsICategoryManager> catMan = do_GetService(NS_CATEGORYMANAGER_CONTRACTID, &rv);
  if (!catMan) return NS_ERROR_FAILURE;

  rv = catMan->DeleteCategoryEntry(G_MOZILLA_APP_STARTUP,
                                   G_MOZILLA_MICROB_EAL_OBSERVER,
                                   PR_TRUE);
  return rv;
}

MicrobEalObserver::MicrobEalObserver()
{
  mGWeb = nsnull;
  mOssoContext = nsnull;
}

MicrobEalObserver::~MicrobEalObserver()
{
  if (mOssoContext)
  {
    osso_hw_unset_event_cb(mOssoContext, nsnull);
    osso_deinitialize(mOssoContext);
    mOssoContext = nsnull;
  }
}

void
MicrobEalObserver::OssoDisplayCallback(osso_display_state_t state, gpointer data)
{
  nsCOMPtr<nsIObserverService> os = do_GetService("@mozilla.org/observer-service;1");
  if (!os)
      return;

  if (state == OSSO_DISPLAY_ON)
      os->NotifyObservers(nsnull, "system-display-on", nsnull);
  else
      os->NotifyObservers(nsnull, "system-display-dimmed-or-off", nsnull);
}

void
MicrobEalObserver::OssoHardwareCallback(osso_hw_state_t *state, gpointer data)
{
  NS_ASSERTION(state, "osso_hw_state_t must not be null.");
  NS_ASSERTION(data, "data must not be null.");

  osso_hw_state_t* ourState = (osso_hw_state_t*) data;

  if (state->shutdown_ind) {
    nsCOMPtr<nsIAppStartup> appService =  do_GetService("@mozilla.org/toolkit/app-startup;1");
    if (appService)
      appService->Quit(nsIAppStartup::eForceQuit);
    return;
  }

  if (state->memory_low_ind) {
      if (!ourState->memory_low_ind) {
      nsCOMPtr<nsIObserverService> os = do_GetService("@mozilla.org/observer-service;1");
      if (os)
        os->NotifyObservers(nsnull, "memory-pressure", NS_LITERAL_STRING("low-memory").get());
    }
  }

  if (state->system_inactivity_ind != ourState->system_inactivity_ind) {
      nsCOMPtr<nsIObserverService> os = do_GetService("@mozilla.org/observer-service;1");
      if (!os)
        return;

      if (state->system_inactivity_ind)
          os->NotifyObservers(nsnull, "system-idle", nsnull);
      else
          os->NotifyObservers(nsnull, "system-active", nsnull);
  }

  memcpy(ourState, state, sizeof(osso_hw_state_t));
}


nsresult
MicrobEalObserver::Init()
{
  TRACE_LOG();
  nsresult rv;
  mObserverService = do_GetService(G_MOZILLA_OBSERVER_SERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED(rv)) {
    mObserverService->AddObserver(this, G_MOZILLA_XPCOM_SHUTDOWN, PR_TRUE);
    mObserverService->AddObserver(this, G_MOZILLA_QUIT_APPLICATION, PR_TRUE);
    mObserverService->AddObserver(this, G_MOZILLA_QUIT_APPLICATION_REQUESTED, PR_TRUE);
    mObserverService->AddObserver(this, G_MOZILLA_QUIT_APPLICATION_GRANTED, PR_TRUE);
    mObserverService->AddObserver(this, DOM_WINDOW_OPENED, PR_TRUE);
    mObserverService->AddObserver(this, DOM_WINDOW_CLOSED, PR_TRUE);
  }

  nsCOMPtr<nsIPrefBranch2> prefBranch = do_GetService(NS_PREFSERVICE_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS(rv, rv);
  rv = prefBranch->AddObserver("microb.", this, PR_FALSE);
  NS_ENSURE_SUCCESS(rv, rv);

  memset(&mHwState, 0, sizeof(osso_hw_state_t));

  mOssoContext = osso_initialize("microb-engine", VERSION, PR_TRUE, nsnull);

  /* Check that initilialization was ok, if not the it is still ok */
  if (mOssoContext == nsnull)
      return NS_OK;

  osso_hw_set_event_cb(mOssoContext,
                       nsnull,
                       OssoHardwareCallback,
                       &mHwState);

  osso_hw_set_display_event_cb(mOssoContext,
                               OssoDisplayCallback,
                               nsnull);

  return rv;
}

nsresult
MicrobEalObserver::CleanAuthData ()
{
    nsCOMPtr<nsIHttpAuthManager> authManager = do_GetService(kHttpAuthManagerCID);
    if (authManager)
        authManager->ClearAll();

    nsresult rv;
    nsCOMPtr<nsICookieManager> cookieManager = do_GetService(NS_COOKIEMANAGER_CONTRACTID, &rv);
    NS_ENSURE_SUCCESS(rv, rv);
    nsCOMPtr<nsISimpleEnumerator> cookieEnumerator;
    rv = cookieManager->GetEnumerator(getter_AddRefs(cookieEnumerator));
    NS_ENSURE_SUCCESS(rv, rv);
    PRBool enumResult;
    for (cookieEnumerator->HasMoreElements(&enumResult);
            enumResult == PR_TRUE;
            cookieEnumerator->HasMoreElements(&enumResult))
    {
        nsCOMPtr<nsICookie> nsCookie;
        nsCOMPtr<nsICookie2> nsCookie2;
        rv = cookieEnumerator->GetNext(getter_AddRefs(nsCookie));
        nsCookie2 = do_QueryInterface(nsCookie, &rv);
        PRBool aIsSession = PR_FALSE;
        if (NS_SUCCEEDED(rv) && nsCookie)
            nsCookie2->GetIsSession(&aIsSession);
        if (aIsSession) {
            nsCString host,path,name;
            nsCookie->GetHost(host);
            nsCookie->GetPath(path);
            nsCookie->GetName(name);
            cookieManager->Remove(host, name, path, PR_FALSE);
        }
    }
    return NS_OK;
}

static MicrobEalXshmRenderer* EnsureXshmRenderer(GObject* aGWeb)
{
  MicrobEalXshmRenderer *mshm = (MicrobEalXshmRenderer*)g_object_get_data(aGWeb, G_MOZILLA_PAINT_LISTENER);

  if (!mshm) {
    ULOG_DEBUG_F("%s - Creating XSHMR", __PRETTY_FUNCTION__);

    mshm = new MicrobEalXshmRenderer();
    if (mshm) {
      g_object_set_data(G_OBJECT(aGWeb), G_MOZILLA_PAINT_LISTENER, mshm);
      mshm->Init(aGWeb);
    }
  }
  else {
    ULOG_DEBUG_F("%s - XSHMR Exists", __PRETTY_FUNCTION__);
  }

  return mshm;
}

static MicrobEalCertificate* EnsureCertListener(GObject* aGWeb)
{
  MicrobEalCertificate *mcert = (MicrobEalCertificate*)g_object_get_data(aGWeb, G_MOZILLA_CERTIFICATE_LISTENER);

  if (!mcert) {
    ULOG_DEBUG_F("%s - Creating CERT LIST", __PRETTY_FUNCTION__);
    mcert = new MicrobEalCertificate();
    g_object_set_data(G_OBJECT(aGWeb), G_MOZILLA_CERTIFICATE_LISTENER, mcert);
  }
  else {
    ULOG_DEBUG_F("%s - CERT LIST Exists", __PRETTY_FUNCTION__);
  }

  return mcert;
}

/**
 * Get the current HTML document title from DOM.
 * @returns The title as a newly allocated string, or NULL on failure.
 */
static gchar *
GetDocumentTitle ()
{
    nsString nsTitle;
    nsCOMPtr<nsIWindowWatcher> wwatch;
    nsCOMPtr<nsIDOMWindow> DOMWindow;
    nsCOMPtr<nsIDOMHTMLDocument> nsHtmlDoc;
    nsCOMPtr<nsIDOMDocument> nsDoc;

    wwatch = do_GetService (NS_WINDOWWATCHER_CONTRACTID);
    NS_ENSURE_TRUE (wwatch, NULL);

    wwatch->GetActiveWindow (getter_AddRefs(DOMWindow));
    NS_ENSURE_TRUE (DOMWindow, NULL);

    DOMWindow->GetDocument (getter_AddRefs(nsDoc));
    nsHtmlDoc = do_QueryInterface (nsDoc);
    NS_ENSURE_TRUE (nsHtmlDoc, NULL);

    nsHtmlDoc->GetTitle (nsTitle);

    return g_strdup (NS_ConvertUTF16toUTF8 (nsTitle).get());
}

/**
 * Called when the document finished loading or is stopped for some other
 * reason. This function sends forward the delayed RSS request in case the
 * user has opened a plain XML file with RSS content.
 * @param embed GtkMozEmbed instance.
 * @param data Ignored.
 */
static void
LoadFinishedCB (GtkMozEmbed * embed, gpointer data)
{
    (void)data;
    NS_ENSURE_TRUE (GTK_IS_MOZ_EMBED(embed), );

    // The URL was stored in the embed object in MicrobEalObserver::Observe.
    gchar * url = (gchar *)g_object_steal_data (G_OBJECT(embed), DELAYED_RSS_URL);
    if (!url) {
        url = gtk_moz_embed_get_location (embed);
        NS_ENSURE_TRUE (url, );
    }

    // NOTE: We're getting title from the engine since at this time GtkMozEmbed
    //       has not received the correct title yet...
    gchar * title = GetDocumentTitle ();
    if (!title || title[0] == '\0') {
        // It's OK if we didn't get any title, we'll just use the URL in that case.
        title = g_strdup (url);
    }

    g_signal_emit_by_name (GTK_OBJECT(embed), G_WEBWIDGET_SIGNAL_RSS_REQUEST, url, title);

    if (url)
        NS_Free (url);
    if (title)
        NS_Free (title);
}

NS_IMETHODIMP
MicrobEalObserver::Observe(nsISupports *aSubject, const char *aTopic, const PRUnichar *aData)
{
  ULOG_DEBUG_F("Func:MicrobEalObserver::%s:%d, subj:%p, aTopic:%s, aData:%s\n", __FUNCTION__, __LINE__, aSubject, aTopic, NS_ConvertUTF16toUTF8(aData).get());

  nsresult rv = NS_OK;

  if (!strcmp(aTopic, "notification:select")) {
    nsCOMPtr<nsIDOMHTMLSelectElement> select = do_QueryInterface(aSubject, &rv);
    NS_ENSURE_SUCCESS(rv, NS_ERROR_FAILURE);
    nsCOMPtr<nsIDOMWindow> win;
    rv = GetDOMWindowByNode(select, getter_AddRefs(win));
    NS_ENSURE_SUCCESS(rv, NS_ERROR_FAILURE);
    // Get Engine
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(win);
    return create_select_widget (engine, (void*)select, 0);
  }

  if (!strcmp(aTopic, G_MOZILLA_WEB_NAME_DEFAULT)) {
    // Set GMozilla Web
    mGWeb = aSubject;

    return NS_OK;
  }

  if (!strcmp(aTopic, G_MOZILLA_ENGINE_CLEAN_AUTH)) {
    return CleanAuthData();
  }

  if (!strcmp(aTopic, DOM_WINDOW_OPENED) && aSubject) {
    nsCOMPtr<nsIDOMWindow> win = do_QueryInterface(aSubject, &rv);
    NS_ENSURE_SUCCESS(rv, NS_OK);
    // Get Engine
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(win);
    // Check Engine
    if (engine) {
      // Get XSHM Renderer
      MicrobEalXshmRenderer *xshm = EnsureXshmRenderer(G_OBJECT(mGWeb));
      // Check XSHM Renderer
      if (xshm)
        // Setup XSHM Renderer
        xshm->SetUp(engine);

      MicrobEalCertificate *cert = EnsureCertListener(G_OBJECT(mGWeb));
      if (cert)
        cert->SetUp(aSubject);

      nsSHistoryListener *shistory = new nsSHistoryListener();
      if (shistory) {
        NS_ADDREF(shistory);
        shistory->Init(win, engine);
        engine->shistory_listener = shistory;
      }
    }
  }

  if (!strcmp(aTopic, DOM_WINDOW_CLOSED) && aSubject) {
    nsCOMPtr<nsIDOMWindow> win = do_QueryInterface(aSubject, &rv);
    NS_ENSURE_SUCCESS(rv, NS_OK);

    // Get XSHM Renderer
    MicrobEalXshmRenderer *xshm = EnsureXshmRenderer(G_OBJECT(mGWeb));
    // Check XSHM Renderer
    if (xshm)
      // UnSetup XSHM Renderer
      xshm->UnSetup(win);

    MicrobEalCertificate *cert = EnsureCertListener(G_OBJECT(mGWeb));
    if (cert)
      cert->UnSetup(aSubject);

    return NS_OK;
  }

  if (!strcmp(aTopic, G_MOZILLA_XPCOM_SHUTDOWN)) {
    return NS_OK;
  }

  if (!strcmp(aTopic, G_MOZILLA_QUIT_APPLICATION)) {
     if (NS_ConvertUTF16toUTF8(aData).Equals(G_MOZILLA_RESTART)) {
        nsCOMPtr<nsIExtensionManager> em(do_GetService(G_MOZILLA_EXTENSIONS_MANAGER_CONTRACTID));
        NS_ENSURE_TRUE(em, rv);
        PRBool needsRestart = PR_FALSE;
        rv = em->CheckForMismatches(&needsRestart);
        if (!needsRestart) {
          em->Start(&needsRestart);
        }
        nsCOMPtr<nsIComponentRegistrar> cr;
        rv = NS_GetComponentRegistrar(getter_AddRefs(cr));
        NS_ENSURE_SUCCESS(rv, rv);
        // HACK FIXME
        gchar *compreg = g_strdup_printf(G_MOZILLA_COMPREG_FILENAME_PARAM, getenv (G_MOZILLA_HOME));
        gchar *xpti = g_strdup_printf(G_MOZILLA_XPTI_FILENAME_PARAM, getenv (G_MOZILLA_HOME));
        unlink(compreg);
        unlink(xpti);
        cr->AutoRegister(nsnull);
        g_free(compreg);
        g_free(xpti);
        gint retval = 0;
        g_signal_emit_by_name (G_OBJECT (mGWeb),
                               G_WEBWIDGET_SIGNAL_ON_SERVICE_NOTIFY,
                               NULL,
                               G_MOZILLA_COMPONENTS,
                               G_MOZILLA_RESTART_REQUIRED,
                               &retval);

     }
     return NS_OK;
  }
  
  if (!strcmp(aTopic, DOMLINK_MAYBE_FEED)) {
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(nsnull);
    NS_ENSURE_TRUE(engine, NS_ERROR_FAILURE);
    GtkMozEmbed *embed = GTK_MOZ_EMBED(engine->engine);
    NS_ENSURE_TRUE(embed, NS_ERROR_FAILURE);

    char * url = 0;
    if (aData)
        url = g_strdup(NS_ConvertUTF16toUTF8(aData).get());
    else
        url = gtk_moz_embed_get_location(embed);

    // Delay G_MOZILLA_RSS_REQUEST until after we've loaded the page. This is
    // done to make sure that there's a sane name available. It's necessary
    // becayse GtkMozEmbed's SetTitle has not yet been called by this time..
    // [See NB#110702]
    g_object_set_data_full (G_OBJECT(embed), DELAYED_RSS_URL, (gpointer)url, (GDestroyNotify)g_free);
    g_signal_connect ((gpointer)embed, "net_stop", G_CALLBACK(LoadFinishedCB), NULL);

    return NS_OK;
  }
  
  if (!strcmp(aTopic, NS_PREFBRANCH_PREFCHANGE_TOPIC_ID)) {
    // Do something here
    return NS_OK;
  }

  if (!strcmp(aTopic, "widgetutils:longpress")) {
    nsCOMPtr<nsIDOMEvent> event = do_QueryInterface(aSubject, &rv);
    NS_ENSURE_SUCCESS(rv, NS_ERROR_FAILURE);
    nsCOMPtr<nsIDOMEventTarget> eventTarget;
    rv = event->GetTarget(getter_AddRefs(eventTarget));
    NS_ENSURE_SUCCESS(rv , NS_ERROR_FAILURE);
    nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget, &rv);
    NS_ENSURE_SUCCESS(rv , NS_ERROR_FAILURE);
    nsCOMPtr<nsIDOMWindow> win;
    rv = GetDOMWindowByNode(eventNode, getter_AddRefs(win));
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(win);
    NS_ENSURE_TRUE(engine , NS_ERROR_FAILURE);
    engine->mouse_down = TRUE;
    dom_mouse_long_press_cb(GTK_MOZ_EMBED(engine->engine), event, engine);
    return NS_OK;
  }

  int retval = 0;
  nsCOMPtr<nsIDOMWindow> win = do_QueryInterface(aSubject, &rv);
  GMozillaEngine* engine = win ? GetGMozEngineForDOMWindow(win) : nsnull;
  if (G_IS_OBJECT(engine)) {
    g_signal_emit_by_name (G_OBJECT(engine),
                           G_WEBWIDGET_SIGNAL_ON_SERVICE_NOTIFY,
                           aSubject,
                           aTopic,
                           aData ? NS_ConvertUTF16toUTF8(aData).get() : nsnull,
                           &retval);
  } else if (mGWeb) {
    g_signal_emit_by_name (G_OBJECT(mGWeb),
                           G_WEBWIDGET_SIGNAL_ON_SERVICE_NOTIFY,
                           mGWeb,
                           aTopic,
                           aData ? NS_ConvertUTF16toUTF8(aData).get() : nsnull,
                           &retval);
  }

  return NS_OK;
}

NS_IMPL_ISUPPORTS2(MicrobEalObserver, nsIObserver, nsISupportsWeakReference)

