/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#undef __FILE__
#define __FILE__ "MicrobEalXshmRenderer"

#include "MicrobEalXshmRenderer.h"

#include "common.h"
#include <stdio.h>
#include "nsCOMPtr.h"
#include "nsIDOMWindow.h"
#include "nsPIDOMWindow.h"
#include "nsIDOMDocument.h"
#include "nsIDOM3Document.h"
#include "nsIDOMNode.h"
#include "nsStringGlue.h"
#include "nsIDOMEventTarget.h"
#include "nsIWindowWatcher.h"
#include "nsServiceManagerUtils.h"
#include "nsIDOMEvent.h"
#include "nsIDOMNSEvent.h"
#include "nsIPrefBranch.h"
#include "nsIPrefService.h"
#include "nsIPrefLocalizedString.h"
#include "nsComponentManagerUtils.h"
#include "nsIObserverService.h"
#include "nsIDOMClientRect.h"
#include "nsIDOMNotifyPaintEvent.h"
#include "nsIWebProgress.h"
#include "nsIDOMBarProp.h"
#include "nsIWebBrowserChrome.h"

#include "gfxContext.h"
#include "gfxImageSurface.h"
#include "nsRect.h"
#include "nsIDeviceContext.h"
#include "nsIPresShell.h"
#include "nsIDocument.h"
#include "nsIDocShell.h"

#include "MicrobEalUtils.h"

#include "nsIDOMWindowInternal.h"
#include "nsIDOMNSElement.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMWindowCollection.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMNSHTMLElement.h"
#include "nsIDOMViewCSS.h"
#include "nsIDOMCSSStyleDeclaration.h"
#include "nsIDOMCSSValue.h"

#include "gtkmozembed_internal.h"

#include <gdk/gdkx.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <stdio.h>
#include <unistd.h>
#include "gmozillasupport.h"
#include "gmozillaweb.h"
#include "nsIDOMHTMLIFrameElement.h"
#include "nsIDOMHTMLFrameElement.h"
#include "nsIDOMClientRectList.h"
#include "nsIDOMDocumentView.h"
#include "nsIDOMAbstractView.h"
#include "MicrobEalContextMenuInfo.h"
#include "nsIDOMHTMLDivElement.h"
#include "nsIDocShellTreeNode.h"
#include "nsIDocShellTreeItem.h"
#include "nsIInterfaceRequestorUtils.h"

#include "gmozillacppwrapper.h"

struct timeval curr_time, prev_time, result;

#define kNotFound -1
#define MOZ_AFTER_PAINT_LITERAL "MozAfterPaint"
#define MOZ_OVERFLOW_LITERAL "overflow"

// SHMB Check Blocked Updates Timeout in milliseconds
#define SHMB_CHECK_BLOCKED_UPDATES_TIMEOUT  200

#define ENGINE_BLOCKED_UPDATES_TM 200
#define UI_BLOCKED_UPDATES_TM 100

typedef struct _UIUpdateRequest UIUpdateRequest;
struct _UIUpdateRequest
{
  void *self;
  int id;
  GdkRectangle rect;
  int zoom;
};

NS_IMPL_ISUPPORTS2(MicrobEalXshmRenderer, nsIDOMEventListener,
                   nsISupportsWeakReference)

gboolean
MicrobEalXshmRenderer::shmbshrinktimer_cb(MicrobEalXshmRenderer* self)
{
  NS_ENSURE_TRUE(self, FALSE);

  // Check Window Count & if SHMB Blocked
  if (self->GetWindowCount() && self->mShmData.IsBlocked()) {
    // Restart Timer - Wait Until It's Not Blocked
    return TRUE;
  }

  self->mShmShrinkTimerId = 0;
  self->CreateSharedImageData(0, 0, 0);
  return FALSE;
}

MicrobShmBufData::~MicrobShmBufData()
{
  if (blocked && id != -1)
    shmdt(blocked);
  id = -1;
  blocked = 0;
  data = 0;
}

static void
FlushLayoutForTree(nsIDOMWindow* aWindow, PRInt32 aDeep = 0)
{
  ULOG("win:%p, aDeep:%i", aWindow, aDeep);
  // We are mobile, and don't want to get deep recurion because of bad frames structure
  if (aDeep > 2) return;
  nsCOMPtr<nsPIDOMWindow> piWin = do_QueryInterface(aWindow);
  if (!piWin) return;

  nsCOMPtr<nsIDOMDocument> domDoc;
  aWindow->GetDocument(getter_AddRefs(domDoc));
  nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
  if (doc)
    doc->FlushPendingNotifications(Flush_Layout);

  nsCOMPtr<nsIDocShellTreeNode> node =
      do_QueryInterface(piWin->GetDocShell());
  if (node) {
    PRInt32 i = 0, i_end;
    node->GetChildCount(&i_end);
    for (; i < i_end; ++i) {
      nsCOMPtr<nsIDocShellTreeItem> item;
      node->GetChildAt(i, getter_AddRefs(item));
      nsCOMPtr<nsIDOMWindow> win = do_GetInterface((nsISupports*)item.get());
      if (win)
        FlushLayoutForTree(win, ++aDeep);
    }
  }
}

static nsresult
DisableScrollbars(nsIDOMWindow *aDOMWindow)
{
  nsresult rv;
  nsCOMPtr<nsIDOMBarProp> scrollbars;
  rv = aDOMWindow->GetScrollbars(getter_AddRefs(scrollbars));
  NS_ENSURE_SUCCESS(rv, rv);
  PRBool isVisible = PR_TRUE;
  rv = scrollbars->GetVisible(&isVisible);

  if (isVisible && !NS_FAILED(rv))
    scrollbars->SetVisible(PR_FALSE);

  return NS_OK;
}

static void
ScrollChanged(GMozillaEngine *engine,
              PRInt32 curWScrX, PRInt32 curWScrY,
              PRInt32 &aUserX, PRInt32 &aUserY,
              PRInt32 &aSysX, PRInt32 &aSysY)
{
  if (curWScrX != engine->mWindowSize.x)
    aSysX = curWScrX - engine->mWindowSize.x;
  if (curWScrY != engine->mWindowSize.y)
    aSysY = curWScrY - engine->mWindowSize.y;
  if (engine->mPrevScrollPos.x != engine->mWindowSize.x)
    aUserX = engine->mWindowSize.x - engine->mPrevScrollPos.x;
  if (engine->mPrevScrollPos.y != engine->mWindowSize.y)
    aUserY = engine->mWindowSize.y - engine->mPrevScrollPos.y;
}

/* static */
nsresult
MicrobEalXshmRenderer::DrawRectToXshm(nsIDOMWindow *aWindow,
                                      gfxRect &aRect,
                                      MicrobShmBufData &aData,
                                      int aReqestId,
                                      float aZoomValue)
{
    nsresult rv;

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aWindow);

    NS_ENSURE_TRUE(engine, NS_ERROR_NULL_POINTER);

    if (aData.IsBlocked()) {
      ULOG_INFO("Buffer problems");
      return NS_ERROR_FAILURE;
    }

    GMozillaWeb* web = G_MOZILLA_WEB(engine->global);
    if (!g_mozilla_web_check_bus(web))
      return NS_ERROR_FAILURE;

    gfxRect dRect(aRect);
    nsCOMPtr<nsIDOMWindowInternal> winint = do_QueryInterface(aWindow, &rv);
    NS_ENSURE_SUCCESS(rv, rv); // we should have internal window
    PRInt32 innerWidth = 0, innerHeight = 0;
    PRInt32 scrollMaxX = 0, scrollMaxY = 0;
    PRInt32 scrollX = 0, scrollY = 0;
    float innerSW = 0, innerSH = 0;

    // Get Inner dimensions to limit requested area
    rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(aWindow, &innerWidth, &innerHeight, &scrollMaxX, &scrollMaxY, &scrollX, &scrollY);

    // Send scroll position not equals to -1 only if it was changed by engine
    PRInt32 ux = 0, uy = 0, sx = 0, sy = 0;
    ScrollChanged(engine, scrollX, scrollY, ux, uy, sx, sy);
    if (scrollX == engine->mWindowSize.x)
      scrollX = -1;
    else
      engine->mPrevScrollPos.x = engine->mWindowSize.x = scrollX;
    if (scrollY == engine->mWindowSize.y)
      scrollY = -1;
    else
      engine->mPrevScrollPos.y = engine->mWindowSize.y = scrollY;

    innerSW = (float)(innerWidth + scrollMaxX) * aZoomValue;
    innerSH = (float)(innerHeight + scrollMaxY) * aZoomValue;

    if (innerWidth == 1 && innerHeight == 1) {
      // Fixme, sometime internal window is setted to 1x1, and even after set window size
      int w = engine->mMinWindowSize.width ? engine->mMinWindowSize.width : engine->mWindowSize.width;
      int h = engine->mMinWindowSize.height ? engine->mMinWindowSize.height : engine->mWindowSize.height;
      winint->ResizeTo(w, h);
    }

    // Limit our dimensions and position accirding to real doc pos and size
    if (dRect.pos.x < 0)
      dRect.pos.x = 0;
    if (dRect.pos.y < 0)
      dRect.pos.y = 0;
    if (dRect.XMost() > innerSW)
      dRect.size.width = innerSW - dRect.pos.x;
    if (dRect.YMost() > innerSH)
      dRect.size.height = innerSH - dRect.pos.y;

    // Get result dimensions after zooming
    int l = (int)roundf(dRect.pos.x * (float)nsIDeviceContext::AppUnitsPerCSSPixel() / aZoomValue);
    int t = (int)roundf(dRect.pos.y * (float)nsIDeviceContext::AppUnitsPerCSSPixel() / aZoomValue);
    int width = (int)roundf(dRect.size.width * (float)nsIDeviceContext::AppUnitsPerCSSPixel() / aZoomValue);
    int height = (int)roundf(dRect.size.height * (float)nsIDeviceContext::AppUnitsPerCSSPixel() / aZoomValue);

    float zooml = roundf(dRect.pos.x);
    float zoomt = roundf(dRect.pos.y);
    float zoomWidth = roundf(dRect.size.width);
    float zoomHeight = roundf(dRect.size.height);

    ULOG("or[%g,%g,%g,%g] rt[%i,%i,%i,%i], zrt[%g,%g,%g,%g], aReqestId:%i, inner[%i,%i], scrMax[%i,%i], scr[%i,%i] z:%g\n",
         dRect.pos.x, dRect.pos.y, dRect.size.width, dRect.size.height,
         l, t, width, height, zooml, zoomt, zoomWidth, zoomHeight, aReqestId,
         innerWidth, innerHeight, scrollMaxX, scrollMaxY, scrollX, scrollY, aZoomValue);

    NS_ENSURE_TRUE(aData.data && aData.data != (void*)-1, NS_ERROR_FAILURE);

    // Init Image Format
    gfxASurface::gfxImageFormat imageFormat = gfxASurface::ImageFormatRGB24;

    // Adjust Image Format
    if (aData.depth == 16) imageFormat = gfxASurface::ImageFormatRGB16;
    if (aData.depth == 32) imageFormat = gfxASurface::ImageFormatARGB32;
    nsRefPtr<gfxImageSurface> surface =
        new gfxImageSurface((unsigned char*)aData.data, gfxIntSize(aData.width, aData.height),
                            aData.bpl,
                            imageFormat);

    // Prepare context and shell for drawing
    nsRefPtr<gfxContext> ctx = new gfxContext(surface);

    nsCOMPtr<nsIDOMDocument> ddoc;
    // Get DOM Document
    rv = aWindow->GetDocument(getter_AddRefs(ddoc));
    NS_ENSURE_SUCCESS(rv, rv);

    // Get Document
    nsCOMPtr<nsIDocument> doc = do_QueryInterface(ddoc, &rv);
    NS_ENSURE_SUCCESS(rv, rv);

    // Get Primary Shell
    nsIPresShell *presShell = doc->GetPrimaryShell();
    NS_ENSURE_TRUE(presShell, NS_ERROR_FAILURE);

    nsRect nr(l, t, width, height);

    // Scale target context with zoom value
    ctx->Scale(aZoomValue, aZoomValue);

    PRUint32 renderDocFlags = nsIPresShell::RENDER_IGNORE_VIEWPORT_SCROLLING | nsIPresShell::RENDER_CARET;
    if (!engine->mPaintFixedFrames)
      renderDocFlags |= nsIPresShell::RENDER_NO_FIXED_FRAMES;

    ULOG("Block painting:%p:%i->%i", engine, aData.IsBlocked(), TRUE);
    aData.SetBlocked(true);
    // Render Document
    presShell->RenderDocument(nr, renderDocFlags, NS_RGB(255, 255, 255), ctx);

    ULOG("[%i,%i,%i,%i]:, nr[%i,%i,%i,%i]\n", (int)l, (int)t, (int)width, (int)height, nr.x, nr.y, nr.width, nr.height);

    PaintNotifyMessage msg =
        { (guint32) engine, aReqestId, (int)roundf(zooml), (int)roundf(zoomt),
          (int)roundf(zoomWidth), (int)roundf(zoomHeight), scrollMaxX+innerWidth,
          scrollMaxY+innerHeight, scrollX, scrollY, (int)roundf(aZoomValue*100) };
    ULOG("id:%i, r[%i,%i,%i,%i], z:%i, scr[%i,%i], retval:%i, pff:%i", aReqestId, (int)roundf(zooml), (int)roundf(zoomt), (int)roundf(zoomWidth), (int)roundf(zoomHeight), (int)roundf(aZoomValue*100), scrollX, scrollY, engine->mPaintFixedFrames);
    g_web_bus_send_message(web->web_bus, web->ui_bus_name, PAINT_NOTIFY_MSG, &msg, sizeof(msg));

    // Check If Engine Is Active
    if (engine->is_active) {
      // Get Self Pointer
      MicrobEalXshmRenderer *self = (MicrobEalXshmRenderer*)g_object_get_data(G_OBJECT(engine->global), G_MOZILLA_PAINT_LISTENER);
      // Restart SHMB Shrink Timer
      self->RestartShmbShrinkTimer();
    }

    return NS_OK;
}

int
MicrobEalXshmRenderer::byte_per_pixel(int depth)
{
  int byte_per_pix = 0;
  switch (depth) {
  case 8:
    byte_per_pix = 1;
    break;
  case 16:
    byte_per_pix = 2;
    break;
  case 24:
  case 32:
    byte_per_pix = 4;
    break;
  default:
    byte_per_pix = 0;
  }
  return byte_per_pix;
}

nsresult
MicrobEalXshmRenderer::UpdateUIShmInfo(void)
{
  // Get Mozilla Web
  GMozillaWeb* web = G_MOZILLA_WEB(mGWeb);
  // Check Mozilla Web
  NS_ENSURE_TRUE(web, NS_ERROR_FAILURE);
  NS_ENSURE_TRUE(web->web_bus, NS_ERROR_FAILURE);
  ShmInfoMessage msg = { 0, mShmData.id, mShmData.width, mShmData.height, mShmData.depth };
  g_web_bus_send_message(web->web_bus, G_MOZILLA_UI, SHM_INFO_MSG, &msg, sizeof(msg));
  return NS_OK;
}

nsresult
MicrobEalXshmRenderer::CreateSharedImageData(int width, int height, int depth, GMozillaEngine *aEngine)
{
  if (width == 0 && height == 0 && mShmData.id != -1) {
    // Detach Shared Memory Data
    ULOG("Detach from shm memory");
    if (mShmData.blocked)
      shmdt(mShmData.blocked);
    mShmData.id = -1;
    mShmData.blocked = 0;
    mShmData.data = 0;
    UpdateUIShmInfo();
    return NS_OK;
  }
  if (width == 0 && height == 0 && mShmData.id == -1) {
    ULOG("Attempt to destroy shm twice");
    return NS_OK;
  }

  // Init  Visual Depth
  gint cur_value = depth;

  // Get Engine Image Surface Depth
  int res = g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_IMAGE_SURFACE_DEPTH, &cur_value);

  // Check Result
  if (res)
    // Adjust Depth According To Image Surface Depth
    depth = cur_value;
  else
    // Adjust Depth According To Visual Depth
    depth = gdk_rgb_get_visual()->depth;

  // Get Engine GFX Depth
  const char *gfxDepth = getenv("ENGINE_IMG_BPP");

  // Check Engine GFX Depth
  if (gfxDepth)
    // Adjust Depth According To Engine GFX Depth
    depth = atoi(gfxDepth);

  ULOG("XSHMB depth: %i bpp", depth);

  if (mShmData.id != -1 && width <= mShmData.width && height <= mShmData.height && depth == mShmData.depth) {
    ULOG("SHM already created %i", mShmData.id);
    UpdateUIShmInfo();
    return NS_OK;
  }

  width = MAX(mDefBuffWidth, width);
  height = MAX(mDefBuffHeight, height);
  width = MIN(SHM_MAX_WIDTH, width);
  height = MIN(SHM_MAX_HEIGHT, height);

  ULOG("Created new shm buffer, [%i,%i]:%i, Recreate:%s", width, height, depth, mShmData.id != -1 ? "TRUE" : "FALSE");

  // Calculate Byte Per Line
  int bpl = width * byte_per_pixel(depth);
  bpl += bpl % sizeof (uint32_t);
  if (!bpl)
    return NS_ERROR_FAILURE;

  // Check Shared Memory Data ID
  if (mShmData.id != -1 && mShmData.blocked) {
    // Detach Shared Memory Data
    shmdt(mShmData.blocked);
    mShmData.id = -1;
    mShmData.blocked = 0;
  }

  int shmId = shmget(IPC_PRIVATE, (bpl * height) + sizeof(int16_t), IPC_CREAT | 0777);

  if (shmId == -1)
    return NS_ERROR_FAILURE;

  // Attach To Current Process
  int16_t *blocked = (int16_t*)shmat(shmId, 0, 0);
  int16_t *data = blocked;
  data += sizeof(int16_t);

  NS_ENSURE_TRUE(data, NS_ERROR_FAILURE);

  shmctl(shmId, IPC_RMID, 0);

  mShmData.width = width;
  mShmData.height = height;
  mShmData.id = shmId;
  mShmData.bpl = bpl;
  mShmData.depth = depth;
  mShmData.data = (void*)data;
  mShmData.blocked = (int16_t*)blocked;
  UpdateUIShmInfo();

  return NS_OK;
}

nsresult
MicrobEalXshmRenderer::Init(GObject *aGWeb)
{
  mGWeb = aGWeb;

  g_object_set_data(G_OBJECT(mGWeb), G_MOZILLA_PAINT_LISTENER, this);
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_EXPAND_FRAMES, &mDoFrameExpand);
  g_mozilla_engine_get_pref(G_TYPE_BOOLEAN, G_MOZILLA_PREF_SHRINK_BUFFER, &mEnableBufferShrink);
  g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_BUFFER_DEF_WIDTH, &mDefBuffWidth);
  g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_BUFFER_DEF_HEIGHT, &mDefBuffHeight);
  g_mozilla_engine_get_pref(G_TYPE_INT, G_MOZILLA_PREF_SHRINK_TIMEOUT, &mShrinkTimeout);

  return NS_OK;
}

MicrobEalXshmRenderer::MicrobEalXshmRenderer()
  : mGWeb(nsnull), mDoFrameExpand(PR_TRUE)
  , mEnableBufferShrink(PR_TRUE), mShmShrinkTimerId(0), mShrinkTimeout(8000)
  , mDefBuffWidth(800), mDefBuffHeight(1300)
{
  // Prevent automatic destroy when engine will destroy all event listeners
  NS_ADDREF(this);
}

MicrobEalXshmRenderer::~MicrobEalXshmRenderer()
{
  // All windows has been closed, need to reset gweb paint listener pointer
  if (mGWeb && G_IS_OBJECT(mGWeb))
    g_object_set_data(G_OBJECT(mGWeb), G_MOZILLA_PAINT_LISTENER, nsnull);

  // Disable SHM Buffer Shrink Timer
  EAL_IF_GFREE_FUNC(mShmShrinkTimerId, g_source_remove);
}

// Set up the event handler for "command"-events
nsresult MicrobEalXshmRenderer::SetUp(GMozillaEngine* engine)
{
  if (!engine)
    return NS_ERROR_FAILURE;

  if (!engine->xshm_rendering_enabled)
    return NS_OK;

  nsCOMPtr<nsIDOMWindow> window;
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));
  NS_ENSURE_TRUE(webBrowser, NS_OK);
  webBrowser->GetContentDOMWindow(getter_AddRefs(window));

  nsCOMPtr<nsPIDOMWindow> pidomWindow = do_QueryInterface(window);
  if (!pidomWindow) {
    printf("ERROR!: Failed to get nsPIDOMWindow from dom window!!!\n");
    return NS_ERROR_FAILURE;
  }

  nsCOMPtr<nsIDOMEventTarget> target = do_QueryInterface(pidomWindow->GetChromeEventHandler());
  nsCOMPtr<nsIDOMWindowInternal> winInt = do_QueryInterface(window);
  if (winInt) {
    g_object_set_data(G_OBJECT(engine->engine), "disable_resize", engine);
    ULOG("Default Window resize [%i,%i]", engine->mWindowSize.width, engine->mWindowSize.height);
    int w = engine->mMinWindowSize.width ? engine->mMinWindowSize.width : engine->mWindowSize.width;
    int h = engine->mMinWindowSize.height ? engine->mMinWindowSize.height : engine->mWindowSize.height;
    winInt->ResizeTo(w, h);
  }

  // Create Shared Image Data
  CreateSharedImageData(mDefBuffWidth, mDefBuffHeight, SHM_DEPTH, engine);

  target->AddEventListener(NS_LITERAL_STRING(MOZ_OVERFLOW_LITERAL), this,  PR_FALSE);

  return target->AddEventListener(NS_LITERAL_STRING(MOZ_AFTER_PAINT_LITERAL), this,  PR_FALSE);
}

nsresult MicrobEalXshmRenderer::UnSetup(nsIDOMWindow *aWindow)
{
  nsCOMPtr<nsPIDOMWindow> pidomWindow = do_QueryInterface(aWindow);

  if (!pidomWindow) {
    NS_WARNING("ERROR!: Failed to get nsPIDOMWindow from dom window!!!\n");
    return NS_ERROR_FAILURE;
  }

  // Disable SHM Buffer Shrink Timer
  EAL_IF_GFREE_FUNC (mShmShrinkTimerId, g_source_remove);

  nsCOMPtr<nsIDOMEventTarget> target = do_QueryInterface(pidomWindow->GetChromeEventHandler());
  NS_ENSURE_TRUE(target, NS_ERROR_FAILURE);
  target->RemoveEventListener(NS_LITERAL_STRING(MOZ_OVERFLOW_LITERAL), this,  PR_FALSE);

  return target->RemoveEventListener(NS_LITERAL_STRING(MOZ_AFTER_PAINT_LITERAL), this,  PR_FALSE);
}

/* static */ gboolean
MicrobEalXshmRenderer::draw_blocked_rect(GMozillaEngine *engine, int left, int top, int width, int height, int id)
{
  g_return_val_if_fail(engine != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(engine->engine), FALSE);

  nsCOMPtr <nsIDOMWindow> window;
  MicrobEalXshmRenderer *mshm = (MicrobEalXshmRenderer*)g_object_get_data(G_OBJECT(engine->global), G_MOZILLA_PAINT_LISTENER);

  if (!mshm)
    return FALSE;

 if (!mshm->mShmData.id)
     return FALSE;

  if (!engine->is_active || mshm->mShmData.IsBlocked())
    return FALSE;

  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));

  if (!webBrowser)
    return FALSE;

  webBrowser->GetContentDOMWindow(getter_AddRefs(window));

  if (!window)
    return FALSE;

  gfxRect r(left, top, width, height);
  if (NS_FAILED(mshm->CreateSharedImageData(width, height, SHM_DEPTH)))
    return FALSE;

  //ULOG("request[%g,%g,%g,%g], zoom:%i, id:%i, fz:%g\n", r.pos.x, r.pos.y, r.size.width, r.size.height, zoom, id, (float)zoom / 100.0);
  MicrobEalXshmRenderer::DrawRectToXshm(window, r, mshm->mShmData, id, engine->mZoomValue);
  return TRUE;
}

/* static */ gboolean
MicrobEalXshmRenderer::check_blocked_updates(void *data)
{
  // Get Engine
  GMozillaEngine *engine = static_cast<GMozillaEngine*>(data);

  // Check Engine & If It's Active
  if (!engine || !engine->is_active)
    return FALSE;

  ULOG("Check Blocked mBlockedUpdates:%p", engine->mBlockedUpdates);
  // Init Painted
  gboolean painted = TRUE;

  // Check Blocked Updates
  if (engine->mBlockedUpdates) {
    GdkRectangle r;
    // Get Region
    gdk_region_get_clipbox(engine->mBlockedUpdates, &r);
    // Draw Blocked Rect
    painted = draw_blocked_rect(engine, r.x, r.y, r.width, r.height, -1);
    ULOG("Paint Blocked %p rect[%i,%i,%i,%i], painted:%s", engine->mBlockedUpdates, r.x, r.y, r.width, r.height, painted?"TRUE":"FALSE");

    // Check If Painted
    if (painted) {
      ULOG("Cleanup blockeUpdate region");
      // All blocked updates were sended to UI
      EAL_IF_GFREE_FUNC(engine->mBlockedUpdates, gdk_region_destroy);
    }
  }

  return !painted;
}

static bool
ScrollNodeIsScrollable(nsIDOMNode *aNode, PRBool ignoreHidden = PR_TRUE)
{
  nsIScrollableView *scrView = nsnull;
  MicrobEalContextMenuInfo::GetViewToScroll(aNode, &scrView, G_WEB_SCROLLABLE_ANY);
  int type = MicrobEalContextMenuInfo::GetScrollableType(scrView, ignoreHidden);
  ULOG("aNode:%p, scrview:%p, type:%i", aNode, scrView, type);
  return type;
}

static void
SetFrameScrollbarsDisable(nsIDOMNode *aNode)
{
  nsresult rv;
  nsCString autoVal(NS_LITERAL_CSTRING("auto"));
  nsCString noVal(NS_LITERAL_CSTRING("no"));
  nsCString yesVal(NS_LITERAL_CSTRING("yes"));
  nsString sautoVal(NS_LITERAL_STRING("auto"));
  nsString snoVal(NS_LITERAL_STRING("no"));

  nsCOMPtr<nsIDOMHTMLFrameElement> nodeAsFrame = do_QueryInterface(aNode, &rv);
  if (nodeAsFrame) {
    nsString val;
    nodeAsFrame->GetScrolling(val);
    NS_ConvertUTF16toUTF8 lval(val);
    ToLowerCase(lval);
    if (!lval.Equals(autoVal) && !lval.Equals(noVal))
      nodeAsFrame->SetScrolling(sautoVal);
    return;
  }
  nsCOMPtr<nsIDOMHTMLIFrameElement> nodeAsIFrame = do_QueryInterface(aNode, &rv);
  NS_ENSURE_TRUE(nodeAsIFrame, );
  nsString val;
  nodeAsIFrame->GetScrolling(val);
  NS_ConvertUTF16toUTF8 lval(val);
  ToLowerCase(lval);
  if (!lval.Equals(autoVal) && !lval.Equals(noVal) && !lval.Equals(yesVal))
    nodeAsIFrame->SetScrolling(lval.Equals(yesVal) ? sautoVal : snoVal);
}

static nsresult
ExpandByDivElement(nsIDOMNode *aNode, nsIDOMWindow *aWindow, nsSize &cl, nsSize &scr)
{
  nsresult rv;

  nsCOMPtr<nsIDOMWindow> topWindow;
  aWindow->GetTop(getter_AddRefs(topWindow));
  if (!topWindow)
    topWindow = aWindow;

  PRInt32 scrollX = 0, scrollY = 0, scrMaxX = 0, scrMaxY = 0;
  nsSize winSize(0,0);
  MicrobEalContextMenuInfo::GetDomWindowScrollInfo(topWindow, &winSize.width, &winSize.height, &scrMaxX, &scrMaxY, &scrollX, &scrollY);
  winSize.width += scrMaxX;
  winSize.height += scrMaxY;
  float l = 0, t = 0, w = 0, h = 0;
  get_node_box_or_nearest(aNode, &l, &t, &w, &h, false);
  l += scrollX;
  t += scrollY;

  int wdiff = 0, hdiff = 0;
  if (!(0 == w || (winSize.width != l + w)))
    wdiff = MAX(0, scr.width - cl.width);
  if (!(0 == h || (winSize.height != t + h)))
    hdiff = MAX(0, scr.height - cl.height);

  ULOG("Divb[%g,%g,%g,%g] [%i,%i], [%i,%i], wind[%i,%i], wscr[%i,%i] max[%i,%i], diff[%i,%i]\n", l, t, w, h, cl.width, cl.height, scr.width, scr.height, winSize.width, winSize.height, scrollX, scrollY, scrMaxX, scrMaxY, wdiff, hdiff);
  if (wdiff <= 0 && hdiff <= 0)
    return NS_OK;

  if (!ScrollNodeIsScrollable(aNode, PR_FALSE))
    return NS_OK;

  nsCOMPtr<nsIDOMElement> element = do_QueryInterface(aNode, &rv);
  if (element) {
    nsString style;
    nsString styleProp(NS_LITERAL_STRING("style"));
    nsString styleHidden(NS_LITERAL_STRING("overflow: hidden;"));
    nsString styleAppend(NS_LITERAL_STRING("overflow: auto;"));
    element->GetAttribute(styleProp, style);

    PRInt32 pos = style.RFind(styleHidden);
    if (pos != kNotFound)
      return NS_OK;

    pos = style.RFind(styleAppend);
    if (pos == kNotFound) {
      style.Append(styleAppend);
      element->SetAttribute(styleProp, style);
    }
  }

  nsCOMPtr<nsIDOMWindowInternal> mainWinInt = do_QueryInterface(topWindow, &rv);
  if (mainWinInt)
    mainWinInt->ResizeBy(wdiff, hdiff); // FIXME We need to fix it properly

  GMozillaEngine* engine = GetGMozEngineForDOMWindow(topWindow);
  NS_ENSURE_TRUE(engine, NS_OK);
  if (wdiff > 0)
    engine->mMinWindowSize.width = winSize.width + wdiff;
  if (hdiff > 0)
    engine->mMinWindowSize.height = winSize.height + hdiff;

  return NS_OK;
}

static nsresult
GetWDiffElement(nsIDOMElement *aNode, nsIDOMWindow *aWindow, nsSize &cl, nsSize &winSize, nsSize &diff)
{
  nsresult rv;
  PRInt32 frameScrMaxX = 0, frameScrMaxY = 0;

  nsCOMPtr<nsIDOMWindow> topWindow;
  aWindow->GetTop(getter_AddRefs(topWindow));
  if (!topWindow)
    topWindow = aWindow;

  PRInt32 scrollX = 0, scrollY = 0, mainScrMaxX = 0, mainScrMaxY = 0;
  rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(topWindow, &winSize.width, &winSize.height, &mainScrMaxX, &mainScrMaxY, &scrollX, &scrollY);
  winSize.height += mainScrMaxY;
  winSize.width += mainScrMaxX;

  NS_ENSURE_SUCCESS(rv, rv);

  rv = MicrobEalContextMenuInfo::GetDomWindowScrollInfo(aWindow, nsnull, nsnull, &frameScrMaxX, &frameScrMaxY, nsnull, nsnull);
  float l = 0, t = 0, w = 0, h = 0;
  get_node_box_or_nearest(aNode, &l, &t, &w, &h, false);
  l += scrollX;
  t += scrollY;
  l = roundf(l);
  t = roundf(t);
  w = roundf(w);
  h = roundf(h);
  // Expand only by frames which are attached to the bottom of page
  diff.width = MAX(0, w + frameScrMaxX + l - winSize.width);
  diff.height = MAX(0, h + frameScrMaxY + t - winSize.height);
  ULOG("fr[%g,%g,%g,%g].scrM[%i,%i] clSiz[%i,%i], wind[%i,%i], wscr[%i,%i] max[%i,%i], diff[%i,%i]\n",
       l, t, w, h, frameScrMaxX, frameScrMaxY, cl.width, cl.height, winSize.width, winSize.height, scrollX, scrollY, mainScrMaxX, mainScrMaxY, diff.width, diff.height);

  if (0 == h || (winSize.height != h + t && t != 0))
    diff.height = 0;

  if ((0 == w) || (winSize.width != l + w && l != 0))
    diff.width = 0;

  return NS_OK;
}

static nsresult
ExpandByFrameElement(nsIDOMElement *aNode, nsIDOMWindow *aWindow, nsSize &cl, nsSize &scr)
{
  nsresult rv;

  nsCOMPtr<nsIDOMWindow> topWindow;
  aWindow->GetTop(getter_AddRefs(topWindow));
  if (!topWindow)
    topWindow = aWindow;

  nsCOMPtr<nsIDOMWindowInternal> mainWinInt = do_QueryInterface(topWindow, &rv);

  nsSize diff(0,0);
  nsSize winSize(0,0);

  bool resizedX = false, resizedY = false;
  // Allow repeated resize only for frame elemens (it may have % size)
  if (mainWinInt) {
    int iter = 0;
    GetWDiffElement(aNode, aWindow, cl, winSize, diff);
    if (diff.width > 0 || diff.height > 0) {
      nsCOMPtr<nsIDOMDocument> document;
      aWindow->GetDocument(getter_AddRefs(document));
      nsCOMPtr<nsIDOMElement> del;
      document->GetDocumentElement(getter_AddRefs(del));
      if (!ScrollNodeIsScrollable(del))
        return NS_OK;
      iter++;
      ULOG("ExpandByFrameElement:%i, nd:%p, win:%p, resize:%i,%i, win:%i,%i\n", iter, aNode, aWindow, diff.width, diff.height, winSize.width, winSize.height);
      mainWinInt->ResizeBy(diff.width, diff.height); // FIXME We need to fix it properly
      if (diff.width > 0) resizedX = true;
      if (diff.height > 0) resizedY = true;
    }

    nsCOMPtr<nsIDOMHTMLFrameElement> nodeAsFrame = do_QueryInterface(aNode, &rv);
    if (nodeAsFrame && iter) {
      do {
        if (diff.width > 0) resizedX = true;
        if (diff.height > 0) resizedY = true;
        diff.width = diff.height = 0;
        GetWDiffElement(aNode, aWindow, cl, winSize, diff);
        ULOG("ExpandByFrameElement:%i, nd:%p, win:%p, resize:%i,%i, win:%i,%i\n", iter, aNode, aWindow, diff.width, diff.height, winSize.width, winSize.height);
        if (diff.width <=0 && diff.height <= 0) break;
        mainWinInt->ResizeBy(diff.width, diff.height); // FIXME We need to fix it properly
      } while (iter++ < 5);
    }
  }

  if (resizedX || resizedY)
    SetFrameScrollbarsDisable(aNode);

  GMozillaEngine* engine = GetGMozEngineForDOMWindow(topWindow);
  NS_ENSURE_TRUE(engine, NS_OK);
  if (resizedX)
    engine->mMinWindowSize.width = winSize.width + diff.width;
  if (resizedY)
    engine->mMinWindowSize.height = winSize.height + diff.height;

  return NS_OK;
}

// If aRect size becomes bigger than aSize, then we should
// make it smaller and don't lose aArrRect area
static PRBool
LimitRectBySizeAroundRect(gfxRect &aRect, const gfxSize &aSize, const gfxRect &aArrRect)
{
  int vs_w_diff = aRect.size.width - aSize.width;
  int vs_h_diff = aRect.size.height - aSize.height;
  if (vs_w_diff > 0)
    aRect.Inset(vs_w_diff);
  if (vs_h_diff > 0)
    aRect.Inset(vs_h_diff);
  if (vs_w_diff > 0 || vs_h_diff > 0) {
    aRect = aRect.Union(aArrRect);
    return PR_TRUE;
  }
  return PR_FALSE;
}

gboolean
MicrobEalXshmRenderer::CheckVisibility(nsIDOMElement *aElement)
{
  nsCOMPtr<nsIDOMDocument> domDoc;
  aElement->GetOwnerDocument(getter_AddRefs(domDoc));

  nsCOMPtr<nsIDOMDocumentView> docView (do_QueryInterface (domDoc));
  NS_ENSURE_TRUE (docView, TRUE);

  nsCOMPtr<nsIDOMAbstractView> abstractView;
  docView->GetDefaultView (getter_AddRefs (abstractView));
  NS_ENSURE_TRUE (abstractView, TRUE);
  
  nsCOMPtr<nsIDOMViewCSS> viewCSS;
  viewCSS = do_QueryInterface (abstractView);
  NS_ENSURE_TRUE (viewCSS, TRUE);

  nsCOMPtr<nsIDOMCSSStyleDeclaration> decl;
  viewCSS->GetComputedStyle (aElement, nsString(), getter_AddRefs (decl));
  NS_ENSURE_TRUE (decl, TRUE);
  
  nsCOMPtr<nsIDOMCSSValue> cssValue;
  decl->GetPropertyCSSValue (NS_LITERAL_STRING("visibility"), getter_AddRefs (cssValue));
  NS_ENSURE_TRUE (cssValue, TRUE);

  nsString value;
  cssValue->GetCssText(value);

  if (value.LowerCaseEqualsLiteral("hidden"))
    return FALSE;

  return TRUE;
}

// The handler for "command"-events
NS_IMETHODIMP MicrobEalXshmRenderer::HandleEvent(nsIDOMEvent *aEvent)
{
  NS_ASSERTION(aEvent, "event must not be null");

  nsresult rv;
  nsString type;

  if (aEvent)
    aEvent->GetType(type);
  ULOG("Event:'%s'", NS_ConvertUTF16toUTF8(type).get());

  if (type.EqualsLiteral(MOZ_OVERFLOW_LITERAL) && mDoFrameExpand) {
    nsCOMPtr<nsIDOMEventTarget> eventTarget;
    rv = aEvent->GetTarget(getter_AddRefs(eventTarget));
    NS_ENSURE_SUCCESS(rv , NS_OK);
    nsCOMPtr<nsIDOMNode> eventNode = do_QueryInterface(eventTarget, &rv);
    NS_ENSURE_SUCCESS(rv , NS_OK);
    nsCOMPtr<nsIDOMWindow> window;
    rv = GetDOMWindowByNode(eventNode, getter_AddRefs(window));
    nsCOMPtr<nsIDOMWindowInternal> winInt = do_QueryInterface(window, &rv);
    NS_ENSURE_TRUE(winInt, NS_OK);

    nsCOMPtr<nsIDOMHTMLDivElement> divEl = do_QueryInterface(eventTarget, &rv);
    nsCOMPtr<nsIDOMElement> frameEl;
    winInt->GetFrameElement(getter_AddRefs(frameEl));
    if (!divEl && !frameEl)
      return NS_OK;

    if (divEl && !CheckVisibility(divEl))
      return NS_OK;
    if (frameEl && !CheckVisibility(frameEl))
      return NS_OK;

    nsCOMPtr<nsIDOMNSElement> nsel = do_QueryInterface(eventNode, &rv);
    NS_ENSURE_TRUE(nsel, NS_OK);
    nsSize cl(0,0), scr(0,0);
    nsel->GetClientHeight(&cl.height);
    nsel->GetClientWidth(&cl.width);
    nsel->GetScrollHeight(&scr.height);
    nsel->GetScrollWidth(&scr.width);
    if (frameEl)
      ExpandByFrameElement(frameEl, window, cl, scr);

    if (divEl)
      ExpandByDivElement(eventNode, window, cl, scr);
    return NS_OK;
  }

  if (!type.EqualsLiteral(MOZ_AFTER_PAINT_LITERAL))
    return NS_OK;

  nsCOMPtr <nsIDOMNotifyPaintEvent> pev = do_QueryInterface (aEvent, &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);

  nsCOMPtr <nsIDOMWindow> eventWin;
  nsCOMPtr <nsIDOMEventTarget> target;

  aEvent->GetTarget(getter_AddRefs(target));
  eventWin = do_QueryInterface (target, &rv);
  NS_ENSURE_SUCCESS(rv, NS_OK);

  nsCOMPtr <nsIDOMWindow> topWin;
  eventWin->GetTop(getter_AddRefs(topWin));
  if (eventWin != topWin)
    return NS_OK;

  GMozillaEngine* engine = GetGMozEngineForDOMWindow(eventWin);
  NS_ENSURE_TRUE(engine, NS_OK);

  DisableScrollbars(eventWin);

  nsCOMPtr <nsIDOMWindowInternal> intWin;
  intWin = do_QueryInterface (eventWin, &rv);
  nsCOMPtr <nsIDOMClientRect> rect;

  nsCOMPtr<nsIDOMClientRectList> clientRects;
  rv = pev->GetClientRects(getter_AddRefs(clientRects));
  NS_ENSURE_SUCCESS(rv, rv);

  PRInt32 scrollX = 0, scrollY = 0;
  MicrobEalContextMenuInfo::GetDomWindowScrollInfo(eventWin, nsnull, nsnull, nsnull, nsnull, &scrollX, &scrollY);
  FlushLayoutForTree(eventWin);

  PRUint32 aLength = 0;
  clientRects->GetLength(&aLength);
  gfxRect rc(0,0,0,0);
  gfxRect visrect(engine->mLastVisibleArea.x, engine->mLastVisibleArea.y,
                  engine->mLastVisibleArea.width, engine->mLastVisibleArea.height);
  gfxRect target_rect(engine->mLastTargetArea.x, engine->mLastTargetArea.y,
                      engine->mLastTargetArea.width, engine->mLastTargetArea.height);
  gfxRect resultRect(0,0,0,0);
  gfxRect intr(0,0,0,0);
  gfxRect intr_target(0,0,0,0);

  gfxRect visrectPrev(visrect);
  visrect = visrect.Union(gfxRect(scrollX, scrollY,
                                  engine->mWindowSize.width,
                                  engine->mWindowSize.height));
  LimitRectBySizeAroundRect(visrect, gfxSize(SHM_MAX_WIDTH, SHM_MAX_HEIGHT), visrectPrev);

  target_rect = target_rect.Union(visrect);

  LimitRectBySizeAroundRect(target_rect, gfxSize(SHM_MAX_WIDTH, SHM_MAX_HEIGHT), visrect);

  PRInt32 ux=0,uy=0,sx=0,sy=0;
  ScrollChanged(engine, scrollX, scrollY, ux, uy, sx, sy);
  PRBool scrollDidChange = (ux || uy);

  float l = 0, t = 0, w = 0, h = 0;
  for (PRUint32 kr = 0; kr < aLength; kr++) {
    rv = clientRects->Item(kr, getter_AddRefs(rect));
    if (NS_SUCCEEDED(rv)) {
      rect->GetLeft(&l);
      rect->GetTop(&t);
      rect->GetWidth(&w);
      rect->GetHeight(&h);
      rc.pos.x = l + scrollX; rc.pos.y = t + scrollY;
      rc.size.width = w; rc.size.height = h;
      ULOG("Rect[%i]:[%g,%g,%g,%g]", kr, rc.pos.x, rc.pos.y, rc.size.width, rc.size.height);
      intr = scrollDidChange ? rc : rc.Intersect(visrect);
      if (!intr.IsEmpty()) {
        intr_target = rc.Intersect(target_rect);
        if (!intr_target.IsEmpty()) {
          resultRect = resultRect.Union(intr_target);
        }
      }
    }
  }
  ULOG("ResRect:[%g,%g,%g,%g], scr[%i,%i]", resultRect.pos.x, resultRect.pos.y, resultRect.size.width, resultRect.size.height, scrollX, scrollY);

  intr_target.pos.x = resultRect.pos.x * engine->mZoomValue;
  intr_target.pos.y = resultRect.pos.y * engine->mZoomValue;
  intr_target.size.width = resultRect.size.width * engine->mZoomValue;
  intr_target.size.height = resultRect.size.height * engine->mZoomValue;

  if ((mShmData.IsBlocked() || !engine->is_active) && !resultRect.IsEmpty()) {
    if (!engine->mBlockedUpdates)
      engine->mBlockedUpdates = gdk_region_new();
    GdkRectangle r = { intr_target.pos.x, intr_target.pos.y, intr_target.size.width, intr_target.size.height };
    gdk_region_union_with_rect(engine->mBlockedUpdates, &r);
    ULOG("Painting is blocked:%p:%i, r[%i,%i,%i,%i]", engine, mShmData.IsBlocked(), r.x, r.y, r.width, r.height);

    EAL_IF_GFREE_FUNC(engine->mBlockedUpdatesSource, g_source_remove);
    engine->mBlockedUpdatesSource = g_timeout_add(ENGINE_BLOCKED_UPDATES_TM, check_blocked_updates, engine);
    return NS_OK;
  }

  if (engine->mBlockedUpdates) {
    GdkRectangle r;
    gdk_region_get_clipbox(engine->mBlockedUpdates, &r);
    ULOG("Union with blocked rect[%i,%i,%i,%i]", r.x, r.y, r.width, r.height);
    intr_target = intr_target.Union(gfxRect(r.x, r.y, r.width, r.height));
    EAL_IF_GFREE_FUNC(engine->mBlockedUpdates, gdk_region_destroy);
    EAL_IF_GFREE_FUNC(engine->mBlockedUpdatesSource, g_source_remove);
  }

  if (!resultRect.IsEmpty()) {
    // Check Width & Height
    if (NS_FAILED(CreateSharedImageData(intr_target.size.width, intr_target.size.height, SHM_DEPTH)))
      return NS_OK;

    // Draw Rect To SHMB
    MicrobEalXshmRenderer::DrawRectToXshm(eventWin, intr_target, mShmData, -1, engine->mZoomValue);
  }

  return NS_OK;
}

nsresult MicrobEalXshmRenderer::RestartShmbShrinkTimer()
{
  if (!mEnableBufferShrink) return NS_OK;

  // Disable SHM Buffer Shrink Timer
  EAL_IF_GFREE_FUNC (mShmShrinkTimerId, g_source_remove);
  // Restart SHM Buffer Shrink Timer
  mShmShrinkTimerId = g_timeout_add(mShrinkTimeout, (GSourceFunc) shmbshrinktimer_cb, (gpointer) this);

  return NS_OK;
}

guint MicrobEalXshmRenderer::GetWindowCount()
{
  NS_ENSURE_TRUE(mGWeb, 0);

  GMozillaWeb* web = G_MOZILLA_WEB(mGWeb);
  GSList* window_list = (GSList*)web->window_list;
  return g_slist_length(window_list);
}

static void
filter_requests_list(GMozillaEngine *engine, GdkRectangle *rect, int zoom)
{
    GList * ptr = (GList*)engine->mBlockedUIRequests;
    while(ptr)
    {
        gboolean delCand = FALSE;
        UIUpdateRequest *request = static_cast<UIUpdateRequest*>(ptr->data);
        GdkRectangle intersect = { 0, 0, 0, 0 };

        //Remove all requests with different zoom level (to avoid auto zoom)
        if (request->zoom != zoom)
          delCand = TRUE;

//        if (!delCand && request->id > 0) {
//          gboolean is_intersect = gdk_rectangle_intersect(&request->rect, &engine->mLastVisibleArea, &intersect);
//          if (!is_intersect)
//            is_intersect = gdk_rectangle_intersect(&request->rect, &engine->mLastTargetArea, &intersect);
//          delCand = !is_intersect;
//        }

        if (!delCand && rect) {
          intersect = request->rect;
          GdkRegion *reg = gdk_region_rectangle(&request->rect);
          GdkRegion *regPainted = gdk_region_rectangle(rect);
          gdk_region_subtract(reg, regPainted);
          gdk_region_get_clipbox(reg, &intersect);
          if (gdk_region_empty(reg)) {
            ULOG("Rectangle already painted");
            delCand = TRUE;
          } else
            request->rect = intersect;
          gdk_region_destroy(reg);
          gdk_region_destroy(regPainted);
        }

        GList * dptr = ptr;
        ptr = ptr->next;
        if (delCand) {
          EAL_IF_GFREE(dptr->data);
          engine->mBlockedUIRequests = g_list_delete_link(engine->mBlockedUIRequests, dptr);
        }
    }
}

/* static */ gboolean
MicrobEalXshmRenderer::check_blocked_ui_updates(void *data)
{
  // Get Engine
  GMozillaEngine *engine = static_cast<GMozillaEngine*>(data);

  // Check Engine & If It's Active
  if (!engine || !engine->is_active)
    return FALSE;

  ULOG("Check Blocked mBlockedUIRequests:%p, len:%i", engine->mBlockedUIRequests, g_list_length(engine->mBlockedUIRequests));

  // Init Painted
  gboolean painted = FALSE;

  // Check Blocked UI requests
  if (engine->mBlockedUIRequests) {
    GList *lptr = g_list_last(engine->mBlockedUIRequests);
    UIUpdateRequest *lr = static_cast<UIUpdateRequest*>(lptr->data);

    GdkRectangle intersect = lr->rect;
    gboolean is_intersect = TRUE;
/*
    if (lr->id != -2) {
      is_intersect = gdk_rectangle_intersect(&lr->rect, &engine->mLastVisibleArea, &intersect);
      if (!is_intersect)
        is_intersect = gdk_rectangle_intersect(&lr->rect, &engine->mLastTargetArea, &intersect);
    }
*/

    // Draw Blocked Rect
    GdkRectangle rect = lr->rect;
    int zoom = lr->zoom;
    if (is_intersect) {
      painted = draw_blocked_rect(engine, intersect.x, intersect.y, intersect.width, intersect.height, lr->id);
      if (painted) {
        EAL_IF_GFREE(lptr->data);
        engine->mBlockedUIRequests = g_list_delete_link(engine->mBlockedUIRequests, lptr);
        filter_requests_list(engine, &rect, zoom);
      }
    } else {
      EAL_IF_GFREE(lptr->data);
      engine->mBlockedUIRequests = g_list_delete_link(engine->mBlockedUIRequests, lptr);

    }

    ULOG("Painted rect[%i,%i,%i,%i], tr[%i,%i,%i,%i], vis[%i,%i,%i,%i], id:%i, painted:%s, ignored:%s, len:%i",
          rect.x, rect.y, rect.width, rect.height,
          engine->mLastTargetArea.x, engine->mLastTargetArea.y, engine->mLastTargetArea.width, engine->mLastTargetArea.height,
          engine->mLastVisibleArea.x, engine->mLastVisibleArea.y, engine->mLastVisibleArea.width, engine->mLastVisibleArea.height,
          lr->id, painted?"TRUE":"FALSE", is_intersect?"FALSE":"TRUE", g_list_length(engine->mBlockedUIRequests));
  }

  return g_list_length(engine->mBlockedUIRequests) != 0;
}

static void
DestroyForceUpdateTimeoutNotify(void *aData)
{
  UIUpdateRequest *request = static_cast<UIUpdateRequest *>(aData);
  NS_ENSURE_TRUE(request, );
  G_MOZILLA_ENGINE(request->self)->mForcedUpdateTimeout = 0;
  EAL_IF_GFREE(request);
}

gboolean
MicrobEalXshmRenderer::force_update_timeout(void *aData)
{
  UIUpdateRequest *request = static_cast<UIUpdateRequest *>(aData);
  NS_ENSURE_TRUE(request, FALSE);
  gboolean cur_active = G_MOZILLA_ENGINE(request->self)->is_active;
  G_MOZILLA_ENGINE(request->self)->is_active = TRUE;
  gboolean painted =
     MicrobEalXshmRenderer::draw_blocked_rect(G_MOZILLA_ENGINE(request->self),
                                              request->rect.x, request->rect.y,
                                              request->rect.width, request->rect.height,
                                              request->id);
  ULOG("Attempt to paint forced update :%i, en:%p", painted, request->self);
  G_MOZILLA_ENGINE(request->self)->is_active = cur_active;
  return !painted;
}

nsresult
MicrobEalXshmRenderer::PushUpdateRequest(GMozillaEngine* engine, GdkRectangle &aRect, int aRequestId, int zoom)
{
  NS_ENSURE_ARG(engine);
  NS_ENSURE_TRUE(engine->engine, NS_ERROR_FAILURE);
  nsCOMPtr <nsIWebBrowser> webBrowser;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(engine->engine), getter_AddRefs(webBrowser));

  if (!webBrowser)
    return NS_ERROR_FAILURE;

  nsCOMPtr <nsIDOMWindow> window;
  webBrowser->GetContentDOMWindow(getter_AddRefs(window));

  engine->mZoomValue = (float)zoom / 100.0;
  UIUpdateRequest *request = g_new0(UIUpdateRequest, 1);
  NS_ENSURE_TRUE(request, NS_ERROR_OUT_OF_MEMORY);
  request->self = engine;
  request->id = aRequestId;
  request->rect = aRect;
  request->zoom = zoom;

  if (aRequestId == -4) {
    ULOG("Do force engine update: %p", engine);
    EAL_IF_GFREE_FUNC(engine->mForcedUpdateTimeout, g_source_remove);
    engine->mForcedUpdateTimeout = g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, 25,
                                                      force_update_timeout, request,
                                                      DestroyForceUpdateTimeoutNotify);
    return NS_OK;
  }

  engine->mBlockedUIRequests = g_list_append(engine->mBlockedUIRequests, request);

  EAL_IF_GFREE_FUNC(engine->mBlockedUIRequestsSource, g_source_remove);
  gboolean not_done = check_blocked_ui_updates(engine);
  if (mShmData.IsBlocked() || not_done) {
    ULOG("Painting is blocked:%p:%i, r[%i,%i,%i,%i]", engine, mShmData.IsBlocked(), aRect.x, aRect.y, aRect.width, aRect.height);
    engine->mBlockedUIRequestsSource = g_timeout_add(UI_BLOCKED_UPDATES_TM, check_blocked_ui_updates, engine);
  }
  return NS_OK;
}

/* static */nsresult
MicrobEalXshmRenderer::ResumePendingUpdates(GMozillaEngine* engine)
{
  NS_ENSURE_ARG(engine);
  if (!engine->mBlockedUIRequestsSource
      && (engine->mBlockedUIRequests && g_list_length(engine->mBlockedUIRequests)))
    engine->mBlockedUIRequestsSource = g_timeout_add(UI_BLOCKED_UPDATES_TM, check_blocked_ui_updates, engine);

  if (!engine->mBlockedUpdatesSource && engine->mBlockedUpdates)
    engine->mBlockedUpdatesSource = g_timeout_add(ENGINE_BLOCKED_UPDATES_TM, check_blocked_updates, engine);

  return NS_OK;
}

/* static */nsresult
MicrobEalXshmRenderer::SuspendPendingUpdates(GMozillaEngine* engine)
{
  NS_ENSURE_ARG(engine);

  if (engine->mBlockedUIRequestsSource
      && (engine->mBlockedUIRequests && g_list_length(engine->mBlockedUIRequests)))
    EAL_GFREE_FUNC(engine->mBlockedUIRequestsSource, g_source_remove);

  if (engine->mBlockedUpdatesSource && engine->mBlockedUpdates)
    EAL_GFREE_FUNC(engine->mBlockedUpdatesSource, g_source_remove);

  return NS_OK;
}
