/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* vim:expandtab:shiftwidth=4:tabstop=4: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *  Brian Ryner <bryner@brianryner.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "MicrobEalPromptService.h"
#include "nsCOMPtr.h"
#include "nsServiceManagerUtils.h"
#include "nsIWindowWatcher.h"
#include "nsIWebBrowserChrome.h"
#include "nsIEmbeddingSiteWindow.h"
#include "nsIDOMWindow.h"
#include "nsStringGlue.h"
#include "nsEmbedString.h"
#include "nsIStringBundle.h"
#include "nsIAuthInformation.h"
#include "nsIProxiedChannel.h"
#include "nsIProxyInfo.h"
#include "nsIIDNService.h"
#include "nsNetCID.h"
#include "nsIURI.h"
#include "nsIPrefService.h"
#include "nsIPrefBranch.h"
#include "nsNetUtil.h"
#include "nsIPrefLocalizedString.h"
#include "nsPromptUtils.h"


#include "gmozillaweb.h"
#include "gmozillaengine.h"

#include "MicrobEalUtils.h"

// ---------------------
//
// STATIC HELPER METHODS
//
// ---------------------

// Helper to convert info from nsIChannel and nsIAuthInformation
// into messages for user
static void
MakeDialogTexts(nsIChannel* aChannel, nsIAuthInformation* aAuthInfo,
                nsEmbedString& aServer, nsEmbedString& aMessage)
{
  nsCAutoString host;
  PRInt32 port;
  NS_GetAuthHostPort(aChannel, aAuthInfo, PR_FALSE, host, &port);

  nsCOMPtr<nsIURI> uri;
  nsresult rv = aChannel ? aChannel->GetURI(getter_AddRefs(uri)) : NS_ERROR_FAILURE;
  if (NS_SUCCEEDED(rv)) {
    nsCAutoString uriScheme;
    rv = uri ? uri->GetScheme(uriScheme) : NS_ERROR_FAILURE;
    if (NS_SUCCEEDED(rv)) {
      uriScheme.AppendLiteral("://");
      host.Insert(uriScheme, 0);
    }
  }

  nsAutoString displayHost;
  CopyUTF8toUTF16(host, displayHost);

  nsAutoString realm;
  if (aAuthInfo)
    aAuthInfo->GetRealm(realm);
  // Trim obnoxiously long realms.
  if (realm.Length() > 150) {
    realm.SetLength(150);

    // Append "..." (or localized equivalent). Yay complexity.
    nsAutoString ellipsis;
    nsCOMPtr<nsIPrefBranch> prefs(do_GetService(NS_PREFSERVICE_CONTRACTID));
    if (prefs) {
      nsCOMPtr<nsIPrefLocalizedString> prefString;
      prefs->GetComplexValue("intl.ellipsis",
                             NS_GET_IID(nsIPrefLocalizedString),
                             getter_AddRefs(prefString));
      if (prefString)
        prefString->ToString(getter_Copies(ellipsis));
    }
    if (ellipsis.IsEmpty())
      ellipsis.AssignLiteral("...");

    realm.Append(ellipsis);
  }

  aServer.Append(displayHost);
  aMessage.Append(realm);
}

// --------------------
//
// CLASS IMPLEMENTATION
//
// --------------------

MicrobEalPromptService::MicrobEalPromptService()
{
}

MicrobEalPromptService::~MicrobEalPromptService()
{
}

NS_IMPL_ISUPPORTS4(MicrobEalPromptService, nsIPromptService, nsIPromptService2, nsINonBlockingAlertService, nsICookiePromptService)

NS_IMETHODIMP
MicrobEalPromptService::Alert(nsIDOMWindow* aParent, const PRUnichar* aDialogTitle, 
                          const PRUnichar* aDialogText)
{
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);
    if (!engine) {
        GMozillaWeb *web = g_mozilla_get_current_web();
        gint retval = 1;
        if (web)
            g_signal_emit_by_name(G_OBJECT(web), "on_service_notify", NULL, "alert", NS_ConvertUTF16toUTF8(aDialogText).get(), &retval);
        return retval ? NS_OK : NS_ERROR_FAILURE;
    }


    g_mozilla_engine_alert(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        PR_TRUE,
        nsnull,
        nsnull);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::ShowNonBlockingAlert(nsIDOMWindow* aParent, const PRUnichar* aDialogTitle,
                                         const PRUnichar* aDialogText)
{
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    g_mozilla_engine_alert(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        PR_FALSE,
        nsnull,
        nsnull);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::AlertCheck(nsIDOMWindow* aParent,
                               const PRUnichar* aDialogTitle,
                               const PRUnichar* aDialogText,
                               const PRUnichar* aCheckMsg, PRBool* aCheckValue)
{
    NS_ENSURE_ARG_POINTER(aCheckValue);

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    g_mozilla_engine_alert(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        PR_FALSE,
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::Confirm(nsIDOMWindow* aParent,
                            const PRUnichar* aDialogTitle,
                            const PRUnichar* aDialogText, PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(aConfirm);

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aConfirm = g_mozilla_engine_confirm(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        nsnull,
        nsnull);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::ConfirmCheck(nsIDOMWindow* aParent,
                                 const PRUnichar* aDialogTitle,
                                 const PRUnichar* aDialogText,
                                 const PRUnichar* aCheckMsg,
                                 PRBool* aCheckValue, PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(aCheckValue);
    NS_ENSURE_ARG_POINTER(aConfirm);

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aConfirm = g_mozilla_engine_confirm(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::ConfirmEx(nsIDOMWindow* aParent,
                              const PRUnichar* aDialogTitle,
                              const PRUnichar* aDialogText,
                              PRUint32 aButtonFlags,
                              const PRUnichar* aButton0Title,
                              const PRUnichar* aButton1Title,
                              const PRUnichar* aButton2Title,
                              const PRUnichar* aCheckMsg, PRBool* aCheckValue,
                              PRInt32* aRetVal)
{
    NS_ENSURE_ARG_POINTER(aCheckValue);
    NS_ENSURE_ARG_POINTER(aRetVal);

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aRetVal = g_mozilla_web_confirm_ex(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        aButtonFlags,
        NS_ConvertUTF16toUTF8(aButton0Title).get(),
        NS_ConvertUTF16toUTF8(aButton1Title).get(),
        NS_ConvertUTF16toUTF8(aButton2Title).get(),
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::Prompt(nsIDOMWindow* aParent, const PRUnichar* aDialogTitle,
                           const PRUnichar* aDialogText, PRUnichar** aValue,
                           const PRUnichar* aCheckMsg, PRBool* aCheckValue,
                           PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(aValue);
    NS_ENSURE_ARG_POINTER(aConfirm);

    gchar * gValue = ToNewCString(NS_ConvertUTF16toUTF8(*aValue));

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aConfirm = g_mozilla_web_prompt(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        &gValue,
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    if (*aConfirm) {
        if (*aValue)
            NS_Free(*aValue);
        *aValue = ToNewUnicode(NS_ConvertUTF8toUTF16(gValue));
    }
    g_free(gValue);
    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::PromptUsernameAndPassword(nsIDOMWindow* aParent,
                                              const PRUnichar* aDialogTitle,
                                              const PRUnichar* aDialogText,
                                              PRUnichar** aUsername,
                                              PRUnichar** aPassword,
                                              const PRUnichar* aCheckMsg,
                                              PRBool* aCheckValue,
                                              PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(aUsername);
    NS_ENSURE_ARG_POINTER(aPassword);
    NS_ENSURE_ARG_POINTER(aConfirm);

    gchar * gUsername = ToNewCString(NS_ConvertUTF16toUTF8(*aUsername));
    gchar * gPassword = ToNewCString(NS_ConvertUTF16toUTF8(*aPassword));

    GObject* mozEmbed = NULL;
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent, &mozEmbed);

    *aConfirm = g_mozilla_web_prompt_auth(engine, mozEmbed,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        &gUsername,
        &gPassword,
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    if (*aConfirm) {
        if (*aUsername)
            NS_Free(*aUsername);
        *aUsername = ToNewUnicode(NS_ConvertUTF8toUTF16(gUsername));

        if (*aPassword)
            NS_Free(*aPassword);
        *aPassword = ToNewUnicode(NS_ConvertUTF8toUTF16(gPassword));
    }
    g_free(gUsername);
    g_free(gPassword);
    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::PromptPassword(nsIDOMWindow* aParent,
                                   const PRUnichar* aDialogTitle,
                                   const PRUnichar* aDialogText,
                                   PRUnichar** aPassword,
                                   const PRUnichar* aCheckMsg,
                                   PRBool* aCheckValue, PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(aPassword);
    NS_ENSURE_ARG_POINTER(aCheckValue);
    NS_ENSURE_ARG_POINTER(aConfirm);

    gchar * gPassword = ToNewCString(NS_ConvertUTF16toUTF8(*aPassword));

    GObject* mozEmbed = NULL;
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent, &mozEmbed);

    *aConfirm = g_mozilla_web_prompt_auth(engine, mozEmbed,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        NULL,
        &gPassword,
        NS_ConvertUTF16toUTF8(aCheckMsg).get(),
        aCheckValue);

    if (*aConfirm) {
        if (*aPassword)
            NS_Free(*aPassword);
        *aPassword = ToNewUnicode(NS_ConvertUTF8toUTF16(gPassword));
    }
    g_free(gPassword);
    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::Select(nsIDOMWindow* aParent, const PRUnichar* aDialogTitle,
                           const PRUnichar* aDialogText, PRUint32 aCount,
                           const PRUnichar** aSelectList, PRInt32* outSelection,
                           PRBool* aConfirm)
{
    NS_ENSURE_ARG_POINTER(outSelection);
    NS_ENSURE_ARG_POINTER(aConfirm);

    GList * list = NULL;

    nsCString *itemList = new nsCString[aCount];
    NS_ENSURE_TRUE(itemList, NS_ERROR_OUT_OF_MEMORY);

    for (PRUint32 i = 0; i < aCount; ++i) {
        itemList[i] = ToNewCString(NS_ConvertUTF16toUTF8(aSelectList[i]));
        list = g_list_append(list, (gpointer)itemList[i].get());
    }

    guint outselection = *outSelection;

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aConfirm = g_mozilla_web_select(engine,
        NS_ConvertUTF16toUTF8(aDialogTitle).get(),
        NS_ConvertUTF16toUTF8(aDialogText).get(),
        list,
        &outselection);

    *outSelection = outselection;

    delete[] itemList;
    g_list_free(list);
    return NS_OK;
}

const PRUint32 kButton0 = 0;
const PRUint32 kButton1 = 8;
const PRUint32 kButton2 = 16;

NS_IMETHODIMP
MicrobEalPromptService::CookieDialog(nsIDOMWindow *aParent, nsICookie *aCookie, const nsACString & aHostname,
                                 PRInt32 aCookiesFromHost, PRBool aChangingCookie, PRBool *aRememberDecision, PRInt32 *aRetval)
{
    NS_ENSURE_ARG_POINTER(aRememberDecision);
    NS_ENSURE_ARG_POINTER(aRetval);
    NS_ENSURE_ARG_POINTER(aCookie);

    GMozillaEngine* engine = GetGMozEngineForDOMWindow(aParent);

    *aRememberDecision = PR_TRUE;          // "remember this decision" should be checked when we show the dialog
    PRInt32 buttonPressed = 0;

    nsCString hostName(aHostname);
    nsCString aName, aValue, aDomain, aPath;
    PRUint64 aExpires = 0;

    if (aCookie) {
        aCookie->GetName(aName);
        aCookie->GetValue(aValue);
        aCookie->GetHost(aDomain);
        aCookie->GetPath(aPath);
        aCookie->GetExpires(&aExpires);
    }

    g_mozilla_web_ask_cookie(engine,
        aRememberDecision,
        &buttonPressed,
        (gulong)aExpires,
        (const gchar*)hostName.get(),
        (const gchar*)aName.get(),
        (const gchar*)aValue.get(),
        (const gchar*)aDomain.get(),
        (const gchar*)aPath.get());

    // map return values for nsICookiePromptService
    switch (buttonPressed)
    {
    case 0: // allow button
        *aRetval = nsICookiePromptService::ACCEPT_COOKIE;
        break;
    case 1: // deny button
        *aRetval = nsICookiePromptService::DENY_COOKIE;
        break;
    case 2: // allow for session button
        *aRetval = nsICookiePromptService::ACCEPT_SESSION_COOKIE;
        break;
    }

    return NS_OK;
}

NS_IMETHODIMP
MicrobEalPromptService::PromptAuth(nsIDOMWindow *aParent, nsIChannel *aChannel,
                            PRUint32 aLevel, nsIAuthInformation *aAuthInfo,
                            const PRUnichar *aCheckboxLabel, PRBool *aCheckValue, PRBool *aRetval)
{
    nsEmbedString message;
    nsEmbedString server;
    MakeDialogTexts(aChannel, aAuthInfo, server, message);

    nsAutoString defaultUser, defaultDomain, defaultPass;
    PRUint32 flags = 0;

    if (aAuthInfo) {
        aAuthInfo->GetUsername(defaultUser);
        aAuthInfo->GetDomain(defaultDomain);
        aAuthInfo->GetPassword(defaultPass);
        aAuthInfo->GetFlags(&flags);
    }
    
    if ((flags & nsIAuthInformation::NEED_DOMAIN) && !defaultDomain.IsEmpty()) {
        defaultDomain.Append(PRUnichar('\\'));
        defaultUser.Insert(defaultDomain, 0);
    }
    
    // NOTE: Allocation failure is not fatal here (just default to empty string
    // if allocation fails)
    PRUnichar* user = ToNewUnicode(defaultUser);
    PRUnichar* pass = ToNewUnicode(defaultPass);

    nsresult rv;
    if (flags & nsIAuthInformation::ONLY_PASSWORD) {
        rv = PromptPassword(aParent, server.get(), message.get(),
                            &pass, aCheckboxLabel,
                            aCheckValue, aRetval);
    }
    else {
        rv = PromptUsernameAndPassword(aParent, server.get(), message.get(),
                                       &user, &pass, aCheckboxLabel,
                                       aCheckValue, aRetval);
    }
    
    nsEmbedString userStr(user);
    nsEmbedString passStr(pass);
    NS_SetAuthInfo(aAuthInfo, userStr, passStr);
    
    return rv;
}

// NOTE: Not implemented currently, asynchronous dialogs not supported
NS_IMETHODIMP
MicrobEalPromptService::AsyncPromptAuth(nsIDOMWindow *aParent, nsIChannel *aChannel, nsIAuthPromptCallback *aCallback, nsISupports *aContext,
                                    PRUint32 level, nsIAuthInformation *authInfo, const PRUnichar *checkboxLabel,
                                    PRBool *checkValue, nsICancelable **_retval)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}
