/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include "gmozillaconnectivity.h"
#include "common.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"
#include "gmozillatransferitem.h"
#include <string.h>
#include "gmozillacppwrapper.h"

#ifdef USE_GCONF
#include <gconf/gconf-client.h>
#endif

#ifdef USE_DBUS
#ifdef USE_CONIC
#include <osso-ic-dbus.h>
#endif
#include <dbus/dbus.h>
#endif

#ifdef USE_CONIC
#include <conic/conic.h>
#endif

static GMainLoop *gsloop = NULL;
static GObject *webglobal = NULL;
static GObject *transferitem = NULL;
static gboolean can_connect = TRUE;
static gboolean is_connecting = FALSE;
static gboolean connected = FALSE;  /**< If the engine is connected or not */

#ifdef USE_GCONF
#define GCONF_PROXY_AUTOCONFIG_URL  "/system/proxy/autoconfig_url"
#define GCONF_PROXY_IGNORE_HOSTS    "/system/proxy/ignore_hosts"
#define GCONF_HTTP_PROXY_HOST       "/system/http_proxy/host"
#define GCONF_HTTP_PROXY_PORT       "/system/http_proxy/port"
#define GCONF_SECURE_PROXY_HOST     "/system/proxy/secure_host"
#define GCONF_SECURE_PROXY_PORT     "/system/proxy/secure_port"
#define GCONF_FTP_PROXY_HOST        "/system/proxy/ftp_host"
#define GCONF_FTP_PROXY_PORT        "/system/proxy/ftp_port"
#define GCONF_SYSTEM_PROXY_MODE     "/system/proxy/mode"
#endif // USE_GCONF
#define GCONF_USE_HTTP_PROXY        "/system/http_proxy/use_http_proxy"

#ifdef USE_CONIC
static ConIcConnection *connection;
static GSList *connections = NULL;
static int connection_count = -1;        /* negative for the initial state */

#define USER_DATA_MAGIC 0xacdcacdc
#define OSSO_CON_IC_CONNECTING             0x05

#ifndef CON_IC_STATUS_NETWORK_UP
#define CON_IC_STATUS_NETWORK_UP 4
#endif

static gint g_mozilla_iap_strcmp(gconstpointer a, gconstpointer b)
{
    return strcmp(a, b);
}

static void g_mozilla_iap_free(gpointer data, gpointer u)
{
    g_free(data);
}

static void connection_cb(ConIcConnection *connection,
                          ConIcConnectionEvent *event,
                          gpointer user_data)
{
    const gchar *iap_id, *bearer;
    gchar *tmp_iap;
    GSList *tmp_list;
    ConIcConnectionStatus status;
    ConIcConnectionError error;

    g_assert(GPOINTER_TO_INT(user_data) == USER_DATA_MAGIC);
    g_assert(CON_IC_IS_CONNECTION_EVENT(event));

    status = con_ic_connection_event_get_status(event);
    error = con_ic_connection_event_get_error(event);
    iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
    bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));

    g_debug("connection_cb(%p, %s, %s, %i, %i) count %i",
            (void *)connection, iap_id, bearer, status, error,
            connection_count);
    
    if (connection_count < 0) {
        connection_count = 0;
        /*
         * If it is the initial connectivity signal and we are not
         * connected, don't change the internal state.
         */
        if (iap_id == NULL)
            return;
    }
    
    switch (status) {
    case CON_IC_STATUS_NETWORK_UP:
        /*
         * We have a network connection, but the IAP is not yet fully
         * connected. Pretend we are already connected in order to let
         * Connectivity invoke the browser in this state. We will get
         * a connected signal later when everything is ok.
         */
        status = CON_IC_STATUS_CONNECTED;
        /* fall through */
    case CON_IC_STATUS_CONNECTED:
        /*
         * If we are not already tracking the status of this iap,
         * i.e., we haven't received a CONNECTED signal about it, add
         * it to the list of tracked iaps. We may get multiple
         * CONNECTED signals for the same iap if we requested the
         * connection ourselves.
         */
        if (g_slist_find_custom(connections, iap_id, g_mozilla_iap_strcmp) == NULL) {
            g_debug("connection_cb(%p, %s, %s, %i, %i) connected", (void *)connection, iap_id, bearer, status, error);
            tmp_iap = g_strdup(iap_id);
            if (tmp_iap == NULL) {
                g_debug("connection_cb(): g_strdup out of memory!");
                /* XXX signal out of memory error */
                return;
            }
            tmp_list = g_slist_prepend(connections, tmp_iap);
            if (tmp_list == NULL) {
                g_debug("connection_cb(): g_slist_prepend out of memory!");
                g_free(tmp_iap);
                /* XXX signal out of memory error */
                return;
            }
            connections = tmp_list;
            connection_count++;
        } else {
            /*
             * We have already received the CONNECTED signal for this
             * iap, no need to do anything.
             */
            g_debug("connection_cb(%p, %s, %s, %i, %i) already connected",
                    (void *)connection, iap_id, bearer, status, error);
            return;
        }
        connected = TRUE;
        is_connecting = FALSE;
        g_signal_emit_by_name(G_OBJECT(webglobal), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, status);
        return;

    case CON_IC_STATUS_DISCONNECTED:
        /*
         * See if we are already tracking this iap.
         */
        if (iap_id != NULL && (tmp_list = g_slist_find_custom(connections, iap_id, g_mozilla_iap_strcmp)) != NULL) {
            g_debug("connection_cb(%p, %s, %s, %i, %i) disconnected", (void *)connection, iap_id, bearer, status, error);
            tmp_iap = tmp_list->data;
            connections = g_slist_remove(connections, tmp_iap);
            g_free(tmp_iap);
            --connection_count;
        }
#ifdef USE_DBUS
        // return;
#else
        break;
#endif
    case CON_IC_STATUS_DISCONNECTING:
        break;

    default:
        break;
    }
    
    /*
     * If we are still connected (e.g. in the change connection use
     * case we might get a connect signal first and a disconnected
     * signal later), don't signal disconnection.
     */
    if (connection_count > 0) {
        g_debug("connection_cb(%p, %s, %s, %i, %i) still connected (%d)",
                (void *)connection, iap_id, bearer, status, error,
                connection_count);
        return;
    }

    g_debug("connection_cb(%p, %s, %s, %i, %i) not connected (%d)",
            (void *)connection, iap_id, bearer, status, error,
            connection_count);
    connected = FALSE;
    g_signal_emit_by_name(G_OBJECT(webglobal), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, status);
    if (gsloop && g_main_loop_is_running(gsloop))
        g_main_loop_quit (gsloop);
    gsloop = NULL;
    is_connecting = FALSE;
}
#endif

gboolean
g_mozilla_connectivity_check_buggy()
{
    if (gsloop && g_main_loop_is_running(gsloop) && is_connecting)
    {
        g_main_loop_quit (gsloop);
        gsloop = NULL;
        connected = FALSE;
        is_connecting = FALSE;
        ULOG_INFO_F("\n*******************************\n\n"
               "\tBROWSER FILL SELF BAD :(\n"
               "\tOLD CONNECTIVITY API SUXXX, need dbus implementation :(\n"
               "\n*******************************\n");
        return TRUE;
    }
    return FALSE;
}


void g_mozilla_connectivity_handle (GMozillaEngine *self, gint event)
{
    TRACE_LOG();
    switch (event) {
#ifdef USE_CONIC
    case CON_IC_STATUS_CONNECTED:

        g_mozilla_engine_observe(G_MOZILLA_NETWORK_IO_SERVICE_CONTRACTID, NULL, G_MOZILLA_PROFILE_CHANGE_NET_TEARDOWN, NULL);
        g_mozilla_engine_observe(G_MOZILLA_NETWORK_IO_SERVICE_CONTRACTID, NULL, G_MOZILLA_PROFILE_CHANGE_NET_RESTORE, NULL);

        ULOG_INFO_F("CON_IC_STATUS_CONNECTED: self:%p, global:%p, transferitem: %p\n", self, webglobal, transferitem);

        // fix for bug 55796
        if (transferitem) {
            // Continue the download after we have established the connection again
            g_web_transfer_item_continue (G_WEB_TRANSFER_ITEM (transferitem));

            // init progress timeout.
            init_progress_timeout (G_MOZILLA_TRANSFER_ITEM (transferitem));

            // setting it to NULL
            transferitem = NULL;

        } else if (self) {
            if (self->waiting_connection && self->will_open_location) {
                g_mozilla_connectivity_set_proxy();
                ULOG_INFO_F("Engine: %p: LoadUrl: '%s'\n", self->engine, self->will_open_location);
                self->waiting_connection = FALSE;
                if (self->will_open_location && !g_str_has_prefix(self->will_open_location, G_MOZILLA_URL_PREFIX_LOCAL))
                    gtk_moz_embed_load_url(GTK_MOZ_EMBED(self->engine), self->will_open_location);
            } else if (self->progress_bar_state == 0 && !self->is_loading) {
                g_signal_emit_by_name(G_OBJECT(self), G_MOZILLA_FINISHED_LOADING, 0, self->last_net_status, self->page_size, TRUE);
                self->progress_bar_state = -1;
            }
              /*else if (self->is_download)
                g_signal_emit_by_name(G_OBJECT(self), G_MOZILLA_FINISHED_LOADING, 0, self->last_net_status, 15000, TRUE);*/
        }
        if (gsloop && g_main_loop_is_running(gsloop))
            g_main_loop_quit (gsloop);
        gsloop = NULL;
        return;
    case CON_IC_STATUS_DISCONNECTED:
        ULOG_INFO_F("CON_IC_STATUS_DISCONNECTED, arg:%p, global:%p\n", self, webglobal);
        g_mozilla_web_pause_all_transfer_item (G_MOZILLA_WEB(webglobal));

        // from g_mozilla_engine_stop_load body
        if (self->is_loading && self->will_open_location && !g_str_has_prefix(self->will_open_location, G_MOZILLA_URL_PREFIX_LOCAL)) {
            g_mozilla_connectivity_check_buggy();
            gtk_moz_embed_stop_load(GTK_MOZ_EMBED(self->engine));
            self->is_loading = FALSE;
            //load_finished_cb(GTK_MOZ_EMBED(self->engine), self);
        }

        connected = FALSE;
        break;
    // FIXME: Handle ERRORS here, please ...
    // case OSSO_IAP_ERROR:
    //    connected = FALSE;
    //    break;
    case OSSO_CON_IC_CONNECTING:
        ULOG_INFO_F("OSSO_CON_IC_CONNECTING, arg:%p, global:%p\n", self, webglobal);
        connected = FALSE;
        if (self->progress_bar_state != 0 && self->user_iteraction_happens && self->waiting_connection)
          g_signal_emit_by_name(G_OBJECT(self), G_MOZILLA_START_LOADING , 0, self->will_open_location);
        self->progress_bar_state = 0;
        return;
#endif
    }
    can_connect = FALSE;
    if (self) {
        if (self->waiting_connection) {
            g_signal_emit_by_name(G_OBJECT(self), G_MOZILLA_FINISHED_LOADING, 0, self->last_net_status, self->page_size, TRUE);
            self->progress_bar_state = -1;
            self->user_iteraction_happens = FALSE;
        }
    self->waiting_connection = FALSE;
    }
    can_connect = TRUE;
}

#ifdef USE_DBUS
/* Called whenever ICD status changes */
static DBusHandlerResult
connectivity_cb (DBusConnection *connection,
                 DBusMessage    *message,
                 gpointer        data)
{
    TRACE_LOG ("connectivity_cb:, %p", data);
    const char *iap_name = NULL;
    const char *iap_type = NULL;
    const char *iap_state = NULL;

    DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    //GMozillaWeb *priv = (GMozillaWeb *)data;
#ifdef USE_CONIC
    if (!message ||
        !dbus_message_is_signal (message,
                                 ICD_DBUS_INTERFACE,
                                 ICD_STATUS_CHANGED_SIG)) {
        return result;
    }
#endif
    if (!dbus_message_get_args (message,
                                NULL,
                                DBUS_TYPE_STRING, &iap_name,
                                DBUS_TYPE_STRING, &iap_type,
                                DBUS_TYPE_STRING, &iap_state,
                                DBUS_TYPE_INVALID) || !iap_name || !iap_state) {
        return result;
    }
    TRACE_LOG ("iap_name: %s", iap_name);
    TRACE_LOG ("iap_type: %s", iap_type);
    TRACE_LOG ("iap_state: %s", iap_state);

    //       Disconnected
    if (!strcmp (iap_state, G_MOZILLA_IAP_STATE_IDLE)) {
        connected = FALSE;
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, CON_IC_STATUS_DISCONNECTED);
#endif
        if (gsloop && g_main_loop_is_running(gsloop))
            g_main_loop_quit (gsloop);
        gsloop = NULL;
        is_connecting = FALSE;
    } else if (!strcmp (iap_state, G_MOZILLA_IAP_STATE_CONNECTED)) {
        g_mozilla_connectivity_set_proxy();
        connected = TRUE;
        is_connecting = FALSE;
        can_connect = FALSE;
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, CON_IC_STATUS_CONNECTED);
#endif

    } else if (!strcmp (iap_state, G_MOZILLA_IAP_STATE_CONNECTING)) {
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), G_WEBWIDGET_SIGNAL_CONNECTIVITY_STATUS, OSSO_CON_IC_CONNECTING);
#endif
    }

    return result;
}
#endif

static void
g_mozilla_connectivity_conf_handler_func (GConfClient * client, guint cnxn_id, GConfEntry * entry, GMozillaWeb *self)
{
   g_debug("Changed GConf proxy settings\n");
   g_timeout_add(1000, g_mozilla_connectivity_set_proxy, self);
/*
   if (self)
   {
      GConfValue*  val = gconf_entry_get_value (entry);
      if (val && val->type == GCONF_VALUE_BOOL)
      {
        g_debug(">>>>>>Func:%s::%d, val:%i\n", __PRETTY_FUNCTION__, __LINE__, gconf_value_get_bool (val));
      }

      if (val && val->type == GCONF_VALUE_INT)
        g_debug(">>>>>>Func:%s::%d, val:%i\n", __PRETTY_FUNCTION__, __LINE__, gconf_value_get_int (val));
      if (val && val->type == GCONF_VALUE_STRING)
        g_debug(">>>>>>Func:%s::%d, val:%s\n", __PRETTY_FUNCTION__, __LINE__, gconf_value_get_string (val));
  }
*/
}

void
g_mozilla_connectivity_set_global (GMozillaWeb *self)
{
    TRACE_LOG();
    webglobal = G_OBJECT(self);
#ifdef USE_CONIC
    connection = con_ic_connection_new();
    g_assert(connection != NULL);

    /* Make sure we are in the initial state */
    connection_count = -1;
    
    g_signal_connect(G_OBJECT(connection), G_MOZILLA_CONNECTION_EVENT,
                     G_CALLBACK(connection_cb),
                     GINT_TO_POINTER(USER_DATA_MAGIC));

    /* Receive notification about all connectivity events */
    g_object_set(G_OBJECT(connection),
                 G_MOZILLA_AUTOMATIC_CONNECTION_EVENTS, TRUE, NULL);
#endif
#ifdef USE_DBUS
    DBusConnection *sys_conn;
    char filter[256] = {0};
    sys_conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    g_return_if_fail(sys_conn);
    dbus_connection_add_filter (sys_conn, connectivity_cb, self, NULL);
#ifdef USE_CONIC
    g_snprintf(filter, sizeof (filter),
               "type='signal',interface='%s'",
               ICD_DBUS_INTERFACE);
#endif
    TRACE_LOG("filter: %s", filter);
    dbus_bus_add_match (sys_conn, filter, NULL);
#endif /* USE_DBUS */
    const char *val = GCONF_USE_HTTP_PROXY;
    GConfClient *client = gconf_client_get_default();
    gconf_client_add_dir(client, val, GCONF_CLIENT_PRELOAD_NONE, NULL);
    gconf_client_notify_add(client,
                            val,
                            (GConfClientNotifyFunc) g_mozilla_connectivity_conf_handler_func, 
                            self, 
                            NULL, NULL);
}

void
g_mozilla_connectivity_unset_global (GMozillaWeb *self)
{
    TRACE_LOG();
#ifdef USE_DBUS
    DBusConnection *sys_conn;
    char filter[256] = {0};
    sys_conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
#ifdef USE_CONIC
    g_snprintf(filter, sizeof (filter),
               "type='signal',interface='%s'",
               ICD_DBUS_INTERFACE);
#endif /* USE_CONIC */
    dbus_bus_remove_match(sys_conn, filter, NULL);
    dbus_connection_remove_filter(sys_conn, connectivity_cb, self);
#endif /* USE_DBUS */
#ifdef USE_CONIC
    /* Stop receiving automatic events and free the connection object */
    g_object_set(G_OBJECT(connection),
                 G_MOZILLA_AUTOMATIC_CONNECTION_EVENTS, FALSE, NULL);
    g_object_unref(connection);
    connection = NULL;
    /* Free all IAP tracking resources */
    g_slist_foreach(connections, g_mozilla_iap_free, NULL);
    g_slist_free(connections);
    connections = NULL;
    /* Return to initial state */
    connection_count = -1;
#endif /* USE_CONIC */
}

gboolean
g_mozilla_connectivity_connect (gpointer data, gboolean reconnect)
{
    TRACE_LOG();
    //Fixme
#ifdef __arm__
    if (connected && !reconnect)
#endif
        return FALSE;

    if (gsloop)
        return TRUE;

    if (!can_connect || is_connecting)
        return FALSE;

    if (data) {
        if (G_IS_MOZILLA_ENGINE (data)) {
            webglobal = G_MOZILLA_ENGINE(data)->global;
            if (!G_MOZILLA_ENGINE(data)->user_iteraction_happens)
                return TRUE;
            G_MOZILLA_ENGINE(data)->waiting_connection = TRUE;
            if (G_MOZILLA_ENGINE(data)->progress_bar_state != 0)
                g_signal_emit_by_name(G_OBJECT(data), G_MOZILLA_START_LOADING , 0, G_MOZILLA_ENGINE(data)->will_open_location);
            G_MOZILLA_ENGINE(data)->progress_bar_state = 0;

        // fix for bug 55796
        } else if (G_IS_MOZILLA_TRANSFER_ITEM (data)) {
            transferitem = data;
        }
    }
#ifdef USE_CONIC
    connected = FALSE;
    is_connecting = TRUE;
    con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
#endif

    if (gsloop) {
        g_main_loop_run(gsloop);
        g_main_loop_unref(gsloop);
    }

    return TRUE;
}

gboolean
g_mozilla_connectivity_is_connected()
{
#ifdef USE_CONIC
    return connected;
#else
    return TRUE;
#endif
}

#define SAFE_ENV_READ 100

static gboolean
g_mozilla_parse_env_proxy (const gchar *env_name, const gchar *moz_proto)
{
    g_return_val_if_fail(env_name && moz_proto, FALSE);
    TRACE_LOG(">>>>>>>>>>.Env: %s > %s", env_name, moz_proto);
    const gchar * env = getenv(env_name);
    if (!env || !strlen(env))
        return FALSE;
    gchar server[20];
    gchar port[11];
    gint i = 0, k = 0, env_len = strlen(env);
    gboolean port_parse = FALSE;
    gchar c = 0;
    for (i = 0; i < env_len || i > SAFE_ENV_READ; i++) {
        c = env[i];
        if (env[i] == '/' && env[i+1] == '/') {
            k = 0; i++; port_parse = FALSE; continue;
        }
        if (c == ':') {
            port_parse = TRUE; server[k] = 0; k = 0; continue;
        }
        if (!port_parse && k < 20 && (c == '.' || g_ascii_isalnum(c)))
            server[k++] = c;
        else if (port_parse && k < 10 && g_ascii_isdigit(c))
            port[k++] = c;
        else break;
    }
    port[k<11 ? k : 10] = 0;
    gint port_int = atoi((const char*)&port);
    gchar *moz_port = g_strdup_printf(G_MOZILLA_NETWORK_PROXY_PARAM_PORT, moz_proto);
    gchar *moz_server = g_strdup_printf(G_MOZILLA_NETWORK_PROXY_PARAM, moz_proto);
    g_mozilla_engine_set_pref (G_TYPE_STRING, moz_server, &server);
    g_mozilla_engine_set_pref (G_TYPE_INT, moz_port, &port_int);
    g_free(moz_port);
    g_free(moz_server);
    return TRUE;
}

gboolean g_mozilla_connectivity_set_proxy (void)
{
    TRACE_LOG();
    gchar *proxytype = NULL, *autoconf = NULL;
    gchar *proxy_http = NULL, *proxy_https = NULL, *proxy_ftp = NULL;
    gint proxy_http_port, proxy_https_port, proxy_ftp_port;
    gchar *omit_proxy = NULL;
    gint gval = 0;
#ifdef USE_GCONF
    GConfClient *gconf_client;
    gconf_client = gconf_client_get_default();
    proxytype = gconf_client_get_string(gconf_client,
                                        GCONF_SYSTEM_PROXY_MODE,
                                        NULL);
#endif
    gval = 1;
    g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    if (getenv(G_MOZILLA_HTTP_PROXY)) {
        if (g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_HTTP)) {
            if (!g_mozilla_parse_env_proxy (G_MOZILLA_FTP_PROXY, G_MOZILLA_FTP))
                g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_FTP);
            if (!g_mozilla_parse_env_proxy (G_MOZILLA_HTTPS_PROXY, G_MOZILLA_SSL))
                g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_SSL);
            gval = 1;
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, G_MOZILLA_LOCAL_HOST_IP);
        }
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_NONE) == 0) {
        gval = 0;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_AUTO) == 0) {
        gval = 2;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        autoconf = gconf_client_get_string(gconf_client,
                                           GCONF_PROXY_AUTOCONFIG_URL,
                                           NULL);
#endif
        //g_print ("%s\n", autoconf);
        if (autoconf) {
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_AUTOCONFIG_URL_PROXY, autoconf);
            EAL_GFREE(autoconf);
        }
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_MANUAL) == 0) {
        gval = 1;
        g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        proxy_http = gconf_client_get_string(gconf_client,
                                             GCONF_HTTP_PROXY_HOST,
                                             NULL);
#endif
        //printf("HTTP PROXY: %s\n", proxy_http);
        if (proxy_http) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_HTTP_PROXY,
                                          g_str_has_prefix(proxy_http, G_MOZILLA_URL_PREFIX_HTTP) ? &proxy_http[7] : proxy_http);
            EAL_GFREE(proxy_http);
#ifdef USE_GCONF
            proxy_http_port = gconf_client_get_int(gconf_client,
                                                   GCONF_HTTP_PROXY_PORT,
                                                   NULL);
            //printf("HTTP PORT: %d\n", proxy_http_port);
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTP_PROXY_PORT, &proxy_http_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_HTTP_PROXY, G_MOZILLA_HTTP);

#ifdef USE_GCONF
        proxy_https = gconf_client_get_string(gconf_client,
                                              GCONF_SECURE_PROXY_HOST,
                                              NULL);
#endif
        //printf("HTTPS PROXY: %s\n", proxy_http);
        if (proxy_https) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_HTTPS_PROXY,
                                          g_str_has_prefix(proxy_https, G_MOZILLA_URL_PREFIX_HTTPS) ? &proxy_https[8] : proxy_https);
            EAL_GFREE(proxy_https);
#ifdef USE_GCONF
            proxy_https_port = gconf_client_get_int(gconf_client,
                                                GCONF_SECURE_PROXY_PORT,
                                                NULL);
            //printf("HTTPS PORT: %d\n", proxy_https_port);
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTPS_PROXY_PORT, &proxy_https_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_HTTPS_PROXY, G_MOZILLA_SSL);

#ifdef USE_GCONF
        proxy_ftp = gconf_client_get_string(gconf_client,
                                            GCONF_FTP_PROXY_HOST,
                                            NULL);
        //printf("FTP PROXY: %s\n", proxy_ftp);
#endif
        if (proxy_ftp) {
            g_mozilla_engine_set_pref(G_TYPE_STRING,
                                          G_MOZILLA_PREF_FTP_PROXY,
                                          g_str_has_prefix(proxy_ftp, G_MOZILLA_URL_PREFIX_FTP) ? &proxy_ftp[6] : proxy_ftp);
            EAL_GFREE(proxy_ftp);
#ifdef USE_GCONF
            proxy_ftp_port = gconf_client_get_int(gconf_client,
                                                  GCONF_FTP_PROXY_PORT,
                                                  NULL);
            //printf("FTP PORT: %d\n", proxy_ftp_port);
#endif
            g_mozilla_engine_set_pref (G_TYPE_INT, G_MOZILLA_PREF_FTP_PROXY_PORT, &proxy_ftp_port);
        }
        else
            g_mozilla_parse_env_proxy (G_MOZILLA_FTP_PROXY, G_MOZILLA_FTP);


#ifdef USE_GCONF

        omit_proxy = gconf_client_get_string(gconf_client,
                                             GCONF_PROXY_IGNORE_HOSTS,
                                             NULL);
#endif
        //printf("OMITED_HOSTS: %s\n", omit_proxy);
        if (omit_proxy)
            g_mozilla_engine_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, omit_proxy);
    }
    g_mozilla_engine_save_prefs ();
    EAL_IF_GFREE(proxytype);
    EAL_IF_GFREE(omit_proxy);
    TRACE_LOG();
    return FALSE;
}
