/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "MicrobEalCertificate.h"

#include "nsCOMPtr.h"
#include "nsIDOMWindow.h"
#include "nsPIDOMWindow.h"
#include "nsIDOMDocument.h"
#include "nsIDOM3Document.h"
#include "nsIDOMNode.h"
#include "nsStringAPI.h"
#include "nsIDOMEventTarget.h"
#include "nsIWindowWatcher.h"
#include "nsServiceManagerUtils.h"
#include "nsIDOMEvent.h"
#include "nsIDOMNSEvent.h"
#include "nsIPrefBranch.h"
#include "nsIPrefService.h"
#include "nsIDialogParamBlock.h"
#include "nsComponentManagerUtils.h"
#include "nsIDOMHistory.h"
#include "MicrobEalUtils.h"

#define COMMAND_LITERAL         "command"
#define SSL_ERROR_PAGE_URI      "about:neterror?e=nssBadCert"
#define EXCEPTION_DIALOG_BUTTON "exceptionDialogButton"
#define OUT_OF_HERE_BUTTON      "getMeOutOfHereButton"
#define EXCEPTION_DIALOG_URI    "chrome://pippki/content/exceptionDialog.xul"
#define EXCEPTION_DIALOG_FLAGS  "chrome,centerscreen,modal,resizable"
#define PREFS_BEHAVIOR          "browser.ssl_override_behavior"
#define PREFS_BEHAVIOR_DEFAULT  1

NS_IMPL_ISUPPORTS2(MicrobEalCertificate, nsIDOMEventListener, nsISSLCertErrorDialog)

//////////////////////////////////////////////////////////////////////////////////////////////////////////
nsresult
MicrobEalCertificate::Init()
{
  return NS_OK;
}


MicrobEalCertificate::MicrobEalCertificate() : mDialogOpen(PR_FALSE)
{
}

MicrobEalCertificate::~MicrobEalCertificate()
{
}

// Set up the event handler for "command"-events
nsresult MicrobEalCertificate::SetUp(nsISupports *aTarget)
{
  nsCOMPtr<nsPIDOMWindow> pidomWindow = do_QueryInterface(aTarget);
  NS_ENSURE_TRUE(pidomWindow, NS_ERROR_FAILURE);
  nsCOMPtr<nsIDOMEventTarget> target = do_QueryInterface(pidomWindow->GetChromeEventHandler());
  NS_ENSURE_TRUE(target, NS_ERROR_FAILURE);

  return target->AddEventListener(NS_LITERAL_STRING(COMMAND_LITERAL), this,  PR_FALSE);
}

// The handler for "command"-events
NS_IMETHODIMP MicrobEalCertificate::HandleEvent(nsIDOMEvent *aEvent)
{
  NS_ASSERTION(aEvent, "event must not be null");

  if (mDialogOpen)
    return NS_ERROR_FAILURE;

  // Is this a "command"-event? We should not be getting any other events
  // but check it anyway
  nsString type;
  if (aEvent)
    aEvent->GetType(type);

  if (!type.EqualsLiteral(COMMAND_LITERAL)) {
    // Something we didn't want
    return NS_ERROR_FAILURE;
  }

  nsCOMPtr<nsIDOMEventTarget> originalTarget;
  nsCOMPtr<nsIDOMNSEvent> nsEvent(do_QueryInterface(aEvent));
  NS_ENSURE_TRUE(nsEvent, NS_ERROR_FAILURE);

  // It the event trusted?
  PRBool trusted=PR_FALSE;
  nsEvent->GetIsTrusted(&trusted);
  if (!trusted) {
    // Not trusted, but that's ok, we just don't care then
    return NS_OK;
  }

  // Get the original target of the event
  nsEvent->GetOriginalTarget(getter_AddRefs(originalTarget));

  nsCOMPtr<nsIDOMDocument> document;
  nsCOMPtr<nsIDOMNode> documentNode(do_QueryInterface(originalTarget));
  NS_ENSURE_TRUE(documentNode, NS_ERROR_FAILURE);
  documentNode->GetOwnerDocument(getter_AddRefs(document));
  nsCOMPtr<nsIDOM3Document> doc(do_QueryInterface(document));
  NS_ENSURE_TRUE(doc, NS_ERROR_FAILURE);

  // Check if the URI matches the SSL error page URI
  nsString uri;
  doc->GetDocumentURI(uri);

  char sslErrorPageUri[] = SSL_ERROR_PAGE_URI;
  if (uri.EqualsLiteral(sslErrorPageUri)) {
    // It wasn't the SSL error page so we can exit this event handler now
    return NS_OK;
  }

  // We now know that we are on the SSL error page

  // Which element triggered the "command" event?
  nsString elementId;
  nsCOMPtr<nsIDOMElement> element(do_QueryInterface(originalTarget));
  NS_ENSURE_TRUE(element, NS_ERROR_FAILURE);
  element->GetAttribute(NS_LITERAL_STRING("id"), elementId);

  if (elementId.EqualsLiteral(OUT_OF_HERE_BUTTON))
    return OutOfHere();

  if (elementId.EqualsLiteral(EXCEPTION_DIALOG_BUTTON))
    return ExceptionDialog();

  // We don't know what caused this event, so we don't care

  return NS_OK;
}

NS_IMETHODIMP MicrobEalCertificate::ExceptionDialog()
{
  nsCOMPtr<nsIWindowWatcher> wwatch = do_GetService(NS_WINDOWWATCHER_CONTRACTID);
  nsCOMPtr<nsIDOMWindow> active;
  nsCOMPtr<nsIDOMWindowInternal> parent;
  if (!wwatch) {
    return NS_ERROR_FAILURE;
  }
  wwatch->GetActiveWindow(getter_AddRefs(active));
  parent = do_QueryInterface(active);
  NS_ENSURE_TRUE(parent, NS_ERROR_FAILURE);

  // Get the behaviour from the preferences
  nsCOMPtr<nsIPrefBranch> prefs = do_GetService(NS_PREFSERVICE_CONTRACTID);
  PRInt32 override=PREFS_BEHAVIOR_DEFAULT;
  if (prefs) {
    prefs->GetIntPref(PREFS_BEHAVIOR, &override);
  }

  nsCOMPtr<nsIDOMLocation> domLocation;
  parent->GetLocation(getter_AddRefs(domLocation));
  NS_ENSURE_TRUE(domLocation, NS_ERROR_FAILURE);

  // Get the current URI
  nsString uri;
  domLocation->ToString(uri);

  // Set up the dialog parameters
  nsCOMPtr<nsIDialogParamBlock> params(do_CreateInstance(NS_DIALOGPARAMBLOCK_CONTRACTID));
  NS_ENSURE_TRUE(params, NS_ERROR_FAILURE);

  // Set the return value to default value
  params->SetInt(0, 0);       // exceptionAdded = false

  // Pre-fill dialog if required the dialog
  switch (override) {
  case 2:         // Pre-fetch & pre-populate
    params->SetInt(1, 1);
    // Fall through

  case 1:         // Pre-populate
    params->SetNumberStrings(1);
    params->SetString(0, uri.get());    // location
    break;

  default:
    break;
  }

  mDialogOpen = PR_TRUE;

  // Open the exception dialog
  nsCOMPtr<nsIDOMWindow> newWindow;
  parent->OpenDialog(
    NS_ConvertASCIItoUTF16(EXCEPTION_DIALOG_URI),
    NS_LITERAL_STRING("_blank"),
    NS_LITERAL_STRING(EXCEPTION_DIALOG_FLAGS),
    params,
    getter_AddRefs(newWindow));

  // Did the user add the exception?
  PRInt32 exceptionAdded=0;
  params->GetInt(0, &exceptionAdded); // prefetchCert = true

  if (exceptionAdded)
    domLocation->Reload(PR_FALSE);
  else
    domLocation->ToString(uri);

  mDialogOpen = PR_FALSE;

  return NS_OK;
}

NS_IMETHODIMP MicrobEalCertificate::OutOfHere()
{
  // Get the active window, its document and its location
  nsCOMPtr<nsIWindowWatcher> wwatch = do_GetService(NS_WINDOWWATCHER_CONTRACTID);
  NS_ENSURE_TRUE(wwatch, NS_ERROR_FAILURE);

  nsCOMPtr<nsIDOMWindow> active;
  nsCOMPtr<nsIDOMWindowInternal> parent;
  nsCOMPtr<nsIDOMHistory> history;
  wwatch->GetActiveWindow(getter_AddRefs(active));
  NS_ENSURE_TRUE(active, NS_ERROR_FAILURE);
  parent = do_QueryInterface(active);
  NS_ENSURE_TRUE(parent, NS_ERROR_FAILURE);

  // Find out if we have history
  parent->GetHistory(getter_AddRefs(history));
  PRInt32 historyLength=0;
  if (history) {
    history->GetLength(&historyLength);
  }

  if (historyLength > 1) {
    parent->Back();
  } else {
    // No history
    GMozillaEngine* engine = GetGMozEngineForDOMWindow(active);

    if (engine) {
      // Send the signal to close the window and show start up view if
      // it is available
      g_signal_emit_by_name (G_OBJECT (engine), "close_window", TRUE);
    }
  }

  return NS_OK;
}

NS_IMETHODIMP
MicrobEalCertificate::ShowCertError(nsIInterfaceRequestor *ctx,
                                    nsISSLStatus *status,
                                    nsIX509Cert *cert,
                                    const nsAString & textErrorMessage,
                                    const nsAString & htmlErrorMessage,
                                    const nsACString & hostName,
                                    PRUint32 portNumber)
{
  return NS_OK;
}
