/**
    @file plugin.c

    Maemo-Blocks plugin for the startup screen

  This file is part of Maemo Blocks

  Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
  http://www.indt.org/maemo
 
  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2.1 of
  the License, or (at your option) any later version.
 
  This software is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with this software; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  02110-1301 USA
*/

#include <stdio.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <startup_plugin.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-banner.h>
#include <gdk/gdkkeysyms.h>

/** The Gconf path to the difficulty level settings */
#define DIFFICULTY_LEVEL	"/apps/indt/maemoblocks/difficulty_level"

/** This gconf value will be true while the games startup is running */
#define BLOCKS_RUNNING	"/apps/indt/maemoblocks/running"

/** The Gconf path to the noise level settings */
#define NOISE_LEVEL		    "/apps/indt/maemoblocks/noise_level"

/** The Gconf path to the noise height settings */
#define NOISE_HEIGHT		"/apps/indt/maemoblocks/noise_height"

/** The path to the help page */
#define MAEMOBLOCKS_HELP_PATH "Applications_blocks_startupview"

/** The numbers of the menuitems */
#define MENU_ITEM_NUMBER 1

/** The width of the speed combobox */
#define COMBOBOX_SPEED_WIDTH		70

/** The width of the other comboboxes */
#define COMBOBOX_WIDTH		70

/** The Code for the plugin callback which indicates the game starts */
#define MA_GAME_PLAYING_START 30

/** The Code for the plugin callback which indicates the game has started */
#define MA_GAME_PLAYING 31

/** The Code for the plugin callback which indicates the original settings should be reseted */
#define MA_GAME_RESET_OLD_SETTINGS 35
#define MA_GAME_ENDED 37
#define MA_GAME_RESTORE_LAST_SETTINGS 38

#define _(String)                 dgettext("osso-games", String)

GConfClient *gcc = NULL;
GtkWidget *difficult_box;
GtkWidget *noise_level_box;
GtkWidget *noise_height_box;
GtkWidget *difficult_box_caption;
GtkWidget *noise_level_box_caption;
GtkWidget *noise_height_box_caption;

GtkWidget *help_item;
GtkWidget *menu_items[MENU_ITEM_NUMBER];
gint selected_difficulty = -1;
gint selected_noise = -1;
gint selected_height = -1;

static GtkWidget **load_menu(guint * nitems);
/*static void plugin_game_help_cb(GtkMenuItem * menuitem, gpointer user_data);*/
static GtkWidget *load_plugin(void);
static void unload_plugin(void);
static void write_config(void);
static void plugin_show_infoprint(gchar * msg);
static void plugin_callback(GtkWidget *, gpointer data);
static void plugin_restore_original_settings(void);
static void plugin_restore_last_settings(void);
static void set_all_insensitive(void);
static void set_all_sensitive(void);

gboolean plugin_difficult_keypress_cb(GtkWidget * widget, GdkEventKey * event,
                                      gpointer user_data);
gboolean plugin_noise_keypress_cb(GtkWidget * widget, GdkEventKey * event,
                                  gpointer user_data);

static StartupPluginInfo plugin_info = {
    load_plugin,
    unload_plugin,
    write_config,
    load_menu,
    NULL,
    plugin_callback,
    NULL
};

GameStartupInfo gs;

STARTUP_INIT_PLUGIN(plugin_info, gs, FALSE, FALSE)
/**
  Sets all widgets to insensitive. This needs, when it starts to load the game
*/
  static void set_all_insensitive(void)
{
    gtk_widget_set_sensitive(difficult_box_caption, FALSE);
    gtk_widget_set_sensitive(noise_level_box_caption, FALSE);
    gtk_widget_set_sensitive(noise_height_box_caption, FALSE);
    if (GTK_WIDGET_VISIBLE(gs.ui->play_button))
    {
        gtk_widget_set_sensitive(gs.ui->play_button, FALSE);
    }
    if (GTK_WIDGET_VISIBLE(gs.ui->restart_button))
    {
        gtk_widget_set_sensitive(gs.ui->restart_button, FALSE);
    }

}

/**
  Sets all widget to sensitive
*/
static void
set_all_sensitive(void)
{
    gtk_widget_set_sensitive(difficult_box_caption, TRUE);
    gtk_widget_set_sensitive(noise_level_box_caption, TRUE);
    gtk_widget_set_sensitive(noise_height_box_caption, TRUE);
}

/**
  Callback function to communicate with the startup screen
  @param This selects the correct function
*/
static void
plugin_callback(GtkWidget * widget, gpointer data)
{
    switch ((int) data)
    {
        case MA_GAME_PLAYING_START:
        {
            set_all_insensitive();
            break;
        }
        case MA_GAME_PLAYING:
        {
            set_all_sensitive();
            break;
        }
        case MA_GAME_RESET_OLD_SETTINGS:
        {
            plugin_restore_original_settings();
            break;
        }
        case MA_GAME_ENDED:
        case MA_GAME_RESTORE_LAST_SETTINGS:
        {
            plugin_restore_last_settings();
            break;
        }
    }
}

/**
  Shows an infoprint
  @param msg The string of the infoprint
*/
static void
plugin_show_infoprint(gchar * msg)
{

    hildon_banner_show_information(GTK_WIDGET(gs.ui->hildon_app), NULL, msg);
}

/**
  Restores the last selected settings, which was selected by the user before continue the game
  The fresh changes against the original ones should not be updated using
  previous user selected ones.
*/
static void
plugin_restore_last_settings(void)
{
    gint original_difficulty =
        gconf_client_get_int(gcc, DIFFICULTY_LEVEL, NULL);
    gint original_noise = gconf_client_get_int(gcc, NOISE_LEVEL, NULL);
    gint original_height = gconf_client_get_int(gcc, NOISE_HEIGHT, NULL);
    gint current_difficulty =
        gtk_combo_box_get_active(GTK_COMBO_BOX(difficult_box));
    gint current_noise =
        gtk_combo_box_get_active(GTK_COMBO_BOX(noise_level_box));
    gint current_height =
        gtk_combo_box_get_active(GTK_COMBO_BOX(noise_height_box));

    if (original_difficulty != current_difficulty)
        selected_difficulty = current_difficulty;
    if (original_noise != current_noise)
        selected_noise = current_noise;
    if (original_height != current_height)
        selected_height = current_height;

    if (selected_difficulty != -1)
    {
        gtk_combo_box_set_active(GTK_COMBO_BOX(difficult_box),
                                 selected_difficulty);
        selected_difficulty = -1;
    }

    if (selected_noise != -1)
    {
        gtk_combo_box_set_active(GTK_COMBO_BOX(noise_level_box),
                                 selected_noise);
        selected_noise = -1;
    }

    if (selected_height != -1)
    {
        gtk_combo_box_set_active(GTK_COMBO_BOX(noise_height_box),
                                 selected_height);
        selected_height = -1;
    }

}

/**
  Restores the original settings, which is stored in the gconf, and it prints an infoprint
*/
static void
plugin_restore_original_settings(void)
{
    gint original_difficulty =
        gconf_client_get_int(gcc, DIFFICULTY_LEVEL, NULL);
    gint original_noise = gconf_client_get_int(gcc, NOISE_LEVEL, NULL);
    gint original_height = gconf_client_get_int(gcc, NOISE_HEIGHT, NULL);
    gint current_difficulty =
        gtk_combo_box_get_active(GTK_COMBO_BOX(difficult_box));
    gint current_noise =
        gtk_combo_box_get_active(GTK_COMBO_BOX(noise_level_box));
    gint current_height =
        gtk_combo_box_get_active(GTK_COMBO_BOX(noise_height_box));

    if (original_difficulty != current_difficulty)
    {
        selected_difficulty = current_difficulty;
        gtk_combo_box_set_active(GTK_COMBO_BOX(difficult_box),
                                 original_difficulty);
    }

    if (original_noise != current_noise)
    {
        selected_noise = current_noise;
        gtk_combo_box_set_active(GTK_COMBO_BOX(noise_level_box),
                                 original_noise);
    }

    if (original_height != current_height)
    {
        selected_height = current_height;
        gtk_combo_box_set_active(GTK_COMBO_BOX(noise_height_box),
                                 original_height);
    }

    if ((original_difficulty != current_difficulty) ||
            (original_noise != current_noise) ||
            (original_height != current_height))
    {
        plugin_show_infoprint(_("game_ib_changes_effect_next_game"));
    }
}

/**
  Loads the menu
  @param nitems The number of the menuitems
  @return The menu array
*/
static GtkWidget **
load_menu(guint * nitems)
{
	/* Rama - Bug#100799 - removed help from main menu */
    *nitems = MENU_ITEM_NUMBER-1;

    menu_items[0] = NULL;
    return menu_items;
}

/**
  Opens the help page
*/
/* Rama - Bug#100799 */
#if 0
static void
plugin_game_help_cb(GtkMenuItem * menuitem, gpointer user_data)
{
    StartupApp *app = NULL;
    app = gs.ui->app;
    hildon_help_show(app->osso, MAEMOBLOCKS_HELP_PATH, 0);
}
#endif

/**
  Callback which is called, when the user press a button, when the difficulty combobox is activated.
  If the pressed key was the down key, the nise combobox will get the focus
  @param widget The combobox widget
  @param event The called event
  @param user_data NULL, not needed here
  @return TRUE, if the event was handled
*/
gboolean
plugin_difficult_keypress_cb(GtkWidget * widget,
                             GdkEventKey * event, gpointer user_data)
{
    if (event->keyval == GDK_Down)
    {
        gtk_widget_grab_focus(noise_level_box_caption);
        return TRUE;
    }
    return FALSE;
}

/**
  Callback which is called, when the user press a button, when the noise combobox is activated.
  If the pressed key was the up key, the difficulty combobox will get the focus
  @param widget The combobox widget
  @param event The called event
  @param user_data NULL, not needed here
  @return TRUE, if the event was handled
*/
gboolean
plugin_noise_keypress_cb(GtkWidget * widget,
                         GdkEventKey * event, gpointer user_data)
{
    if (event->keyval == GDK_Up)
    {
        gtk_widget_grab_focus(difficult_box_caption);
        return TRUE;
    }
    return FALSE;
}

/**
  Loads the plugin, generates the startup screen
  @return The generated startup screen
*/
static GtkWidget *
load_plugin(void)
{
    int difficulty_level, noise_level, noise_height;
    GtkWidget *table;
    GtkSizeGroup *group =
        GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));
    GtkSizeGroup *combo_group =
        GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));
    int i;
    GtkAttachOptions xoptions = GTK_EXPAND, yoptions = GTK_EXPAND;
    g_type_init();
    gcc = gconf_client_get_default();
    gconf_client_set_bool(gcc, BLOCKS_RUNNING, TRUE, NULL);
    difficulty_level = gconf_client_get_int(gcc, DIFFICULTY_LEVEL, NULL);
    noise_level = gconf_client_get_int(gcc, NOISE_LEVEL, NULL);
    noise_height = gconf_client_get_int(gcc, NOISE_HEIGHT, NULL);
    table = gtk_table_new(2, 2, FALSE);
    g_assert(table);
    gtk_widget_show(table);

    difficult_box = gtk_combo_box_new_text();
    g_assert(difficult_box);
    for (i = 1; i <= 20; i++)
    {
        gchar *buffer = g_new0(gchar, 3);
        gtk_combo_box_append_text(GTK_COMBO_BOX(difficult_box),
                                  g_ascii_dtostr(buffer, 3, i));
    }

    gtk_size_group_add_widget(combo_group, difficult_box);
    gtk_combo_box_set_active(GTK_COMBO_BOX(difficult_box), difficulty_level);
    difficult_box_caption =
        hildon_caption_new(NULL, _("game_fi_blocks_settings_difficulty"),
                           difficult_box, NULL, HILDON_CAPTION_OPTIONAL);
    g_assert(difficult_box_caption);
    gtk_table_attach(GTK_TABLE(table), difficult_box_caption, 0, 1, 0, 1,
                     (GtkAttachOptions) xoptions,
                     (GtkAttachOptions) yoptions, 0, 0);


    noise_level_box = gtk_combo_box_new_text();
    g_assert(noise_level_box);
    for (i = 0; i <= 9; i++)
    {
        gchar *buffer = g_new0(gchar, 3);
        gtk_combo_box_append_text(GTK_COMBO_BOX(noise_level_box),
                                  g_ascii_dtostr(buffer, 3, i));
    }
    gtk_size_group_add_widget(combo_group, noise_level_box);
    gtk_combo_box_set_active(GTK_COMBO_BOX(noise_level_box), noise_level);
    noise_level_box_caption =
        hildon_caption_new(group, _("game_fi_blocks_settings_noise_level"),
                           noise_level_box, NULL, HILDON_CAPTION_OPTIONAL);
    g_assert(noise_level_box_caption);

    gtk_table_attach(GTK_TABLE(table), noise_level_box_caption, 1, 2, 0, 1,
                     (GtkAttachOptions) xoptions,
                     (GtkAttachOptions) yoptions, 0, 0);

    noise_height_box = gtk_combo_box_new_text();
    g_assert(noise_height_box);

    for (i = 0; i <= 10; i++)
    {
        gchar *buffer = g_new0(gchar, 3);
        gtk_combo_box_append_text(GTK_COMBO_BOX(noise_height_box),
                                  g_ascii_dtostr(buffer, 3, i));
    }

    gtk_size_group_add_widget(combo_group, noise_height_box);
    gtk_combo_box_set_active(GTK_COMBO_BOX(noise_height_box), noise_height);

    noise_height_box_caption =
        hildon_caption_new(group, _("game_fi_blocks_settings_noise_height"),
                           noise_height_box, NULL, HILDON_CAPTION_OPTIONAL);
    g_assert(noise_height_box_caption);

    gtk_table_attach(GTK_TABLE(table), noise_height_box_caption, 1, 2, 1, 2,
                     (GtkAttachOptions) xoptions,
                     (GtkAttachOptions) yoptions, 0, 0);
    g_signal_connect(G_OBJECT(noise_level_box_caption), "key-press-event",
                     G_CALLBACK(plugin_noise_keypress_cb), NULL);
    g_signal_connect(G_OBJECT(difficult_box_caption), "key-press-event",
                     G_CALLBACK(plugin_difficult_keypress_cb), NULL);


    gtk_widget_show_all(table);
    return table;
}

/**
  Called, when to startup screen will be closed
*/
static void
unload_plugin(void)
{
    remove("/tmp/.maemoblocks-save");
    gconf_client_set_bool(gcc, BLOCKS_RUNNING, FALSE, NULL);
}

/**
  Writes the configuration to the GConf
*/
static void
write_config(void)
{
    gconf_client_set_int(gcc, DIFFICULTY_LEVEL,
                         gtk_combo_box_get_active(GTK_COMBO_BOX
                                                  (difficult_box)), NULL);
    gconf_client_set_int(gcc, NOISE_LEVEL,
                         gtk_combo_box_get_active(GTK_COMBO_BOX
                                                  (noise_level_box)), NULL);
    gconf_client_set_int(gcc, NOISE_HEIGHT,
                         gtk_combo_box_get_active(GTK_COMBO_BOX
                                                  (noise_height_box)), NULL);
}
