/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "EmbedProgress.h"
#include "EmbedWindow.h"

#include <nsIChannel.h>
#include <nsIWebProgress.h>
#include <nsIDOMWindow.h>
#include "EmbedGtkTools.h"
#ifdef FIXED_BUG347731
#include "nsPresContext.h"
#include "nsIFrame.h"
#endif
#include "nsILoginManager.h"
#include "nsILoginInfo.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDOMDocument.h"
#include "nsIDOMHTMLCollection.h"
#include "nsIForm.h"
#include "nsIFormControl.h"
#include "nsIDocument.h"

#include "nsIURI.h"
#include "EmbedEventListener.h"

static PRBool sNastyProgress = PR_FALSE;

static gboolean
progress_emit_stop(void * data)
{
    g_return_val_if_fail(data, FALSE);
    EmbedPrivate *owner = (EmbedPrivate*)data;
    if (owner->mIsDestroyed) return FALSE;
    if (owner->mIsUploading) {
        g_signal_emit(GTK_OBJECT(owner->mOwningWidget),
                      moz_embed_signals[FILE_UPLOAD_FINISHED], 0);
        owner->mIsUploading = PR_FALSE;
        return FALSE;
    }

    if (!owner->mLoadFinished) {
      owner->mLoadFinished = PR_TRUE;
      g_signal_emit(GTK_OBJECT(owner->mOwningWidget),
                    moz_embed_signals[NET_STOP], 0);
    }
    return FALSE;
}

EmbedProgress::EmbedProgress(void)
{
  mOwner = nsnull;
  mStopSignalTimer = 0;
  mConnectingState = PR_TRUE;
  mPageDownloaded = PR_FALSE;
  mPagePainted = PR_FALSE;
  mShowProgressStopped = PR_TRUE;
  mLoginAttached = PR_FALSE;
  mStopLevel = 0;
  mGlobalUserField = nsnull;
  mGlobalPassField = nsnull;
}

EmbedProgress::~EmbedProgress()
{
  Shutdown();
}

nsresult
EmbedProgress::Shutdown()
{
  mOwner = nsnull;
  GLIB_SOURCE_IF_REMOVE(mStopSignalTimer);
  return NS_OK;
}

NS_IMPL_ISUPPORTS3(EmbedProgress,
		   nsIWebProgressListener,
		   nsIWebProgressListener2,
		   nsISupportsWeakReference)

nsresult
EmbedProgress::Init(EmbedPrivate *aOwner)
{
  mOwner = aOwner;
  mStopLevel = 0;
  gboolean enabled = PR_FALSE;
 	if (gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN,"gtkmozembed.nasty_progress", &enabled) && !enabled)
 		sNastyProgress = PR_FALSE;
 	return NS_OK;
}

static inline nsresult
EnsureAlive(EmbedPrivate *mOwner)
{
  NS_ENSURE_ARG(mOwner);
  if (mOwner->mIsDestroyed)
    return NS_ERROR_FAILURE;
  if (!mOwner->mProgress)
    return NS_ERROR_FAILURE;
  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnStateChange(nsIWebProgress *aWebProgress,
			     nsIRequest     *aRequest,
			     PRUint32        aStateFlags,
			     nsresult        aStatus)
{
  PRBool mIsLoadingDocument = PR_TRUE;
  aWebProgress->GetIsLoadingDocument(&mIsLoadingDocument);
  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);

  if (mOwner->mWindow->mHidden == 2) {
    mOwner->Show();
    gtk_widget_grab_focus(GTK_WIDGET(mOwner->mOwningWidget));
    mOwner->mWindow->mHidden = 0;
  }
  if (sNastyProgress &&
      aStateFlags & GTK_MOZ_EMBED_FLAG_NEGOTIATING &&
      aStateFlags & GTK_MOZ_EMBED_FLAG_IS_DOCUMENT) {
#ifdef FIXED_BUG347731
    if (aStatus == NS_FRAME_HAS_VIEW) {
      mPagePainted = PR_FALSE;
      mConnectingState = PR_FALSE;
      mShowProgressStopped = PR_FALSE;
    }
    if (aStatus == NS_FRAME_IMPL_RESERVED)
      mPagePainted = PR_TRUE;
#endif
  }

  // give the widget a chance to attach any listeners
  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  mOwner->ContentStateChange();

  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  if (mStopSignalTimer &&
      (
       (aStateFlags & GTK_MOZ_EMBED_FLAG_TRANSFERRING)
       || (aStateFlags & GTK_MOZ_EMBED_FLAG_REDIRECTING)
       || (aStateFlags & GTK_MOZ_EMBED_FLAG_NEGOTIATING)
      )
     ) {
      GLIB_SOURCE_IF_REMOVE(mStopSignalTimer);
      mStopLevel = 0;
      g_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
                    moz_embed_signals[NET_START], 0);
      NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
      mOwner->mLoadFinished = PR_FALSE;
      mPageDownloaded = PR_FALSE;
      mShowProgressStopped = PR_FALSE;
  }

  // if we've got the start flag, emit the signal
  if ((aStateFlags & GTK_MOZ_EMBED_FLAG_IS_NETWORK || aStateFlags & GTK_MOZ_EMBED_FLAG_NEGOTIATING) &&
      (aStateFlags & GTK_MOZ_EMBED_FLAG_START)) {
    // FIXME: workaround for broken progress values.
    mOwner->mOwningWidget->current_number_of_requests = 0;
    mOwner->mOwningWidget->total_number_of_requests = 0;

    if (mOwner->mLoadFinished) {
      mOwner->mLoadFinished = PR_FALSE;
      mStopLevel = 0;
      mConnectingState = PR_TRUE;
      mPageDownloaded = PR_FALSE;
      mPagePainted = PR_FALSE;
      mShowProgressStopped = PR_FALSE;

      nsCAutoString uriString;
      RequestToURIString(aRequest, uriString);
      if (!mOwner->mURI.Equals(uriString))
        mOwner->SetURI(uriString.get());

      g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                    moz_embed_signals[NET_START], 0);
      NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
    }
  }

  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  if (sNastyProgress && mPagePainted && mPageDownloaded && !mShowProgressStopped) {
    g_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
	 								moz_embed_signals[PROGRESS_ALL], 0,
 									(const gchar *) nsnull,
									100, 100);
    g_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
									moz_embed_signals[NET_STOP], 0);
    mShowProgressStopped = PR_TRUE;
  }

  // FIXME: workaround for broken progress values.
  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  if (mOwner->mOwningWidget) {
    if (aStateFlags & GTK_MOZ_EMBED_FLAG_IS_REQUEST) {
      if (aStateFlags & GTK_MOZ_EMBED_FLAG_START)
        mOwner->mOwningWidget->total_number_of_requests ++;
      else if (aStateFlags & GTK_MOZ_EMBED_FLAG_STOP)
        mOwner->mOwningWidget->current_number_of_requests++;
     }

    if (!sNastyProgress || (!mConnectingState && !mShowProgressStopped))
      g_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
		 								moz_embed_signals[PROGRESS_ALL], 0,
 	 									(const gchar *) NULL,
 										mOwner->mOwningWidget->current_number_of_requests,
										mOwner->mOwningWidget->total_number_of_requests);

    NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
    g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                  moz_embed_signals[NET_STATE_ALL], 0,
                  NULL,
                  (gint)aStateFlags, (gint)aStatus);
    NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  }
  // and for stop, too
  if (aStateFlags & GTK_MOZ_EMBED_FLAG_STOP) {
    if (aStateFlags & GTK_MOZ_EMBED_FLAG_IS_REQUEST)
      mStopLevel = 1;
    if (aStateFlags & GTK_MOZ_EMBED_FLAG_IS_DOCUMENT)
       mStopLevel = mStopLevel == 1 ? 2 : 0;
    if (aStateFlags & GTK_MOZ_EMBED_FLAG_IS_WINDOW) {
      mStopLevel = mStopLevel == 2 ? 3 : 0;
    }
  }

  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  if (aStateFlags & GTK_MOZ_EMBED_FLAG_STOP) {
    if (aStateFlags & GTK_MOZ_EMBED_FLAG_IS_NETWORK) {
      GLIB_SOURCE_IF_REMOVE(mStopSignalTimer);
      mShowProgressStopped = PR_TRUE;
      progress_emit_stop(mOwner);
      NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
      // let our owner know that the load finished
      mOwner->ContentFinishedLoading();

    } else if (mStopLevel == 3) {
      GLIB_SOURCE_IF_REMOVE(mStopSignalTimer);
      mStopLevel = 0;
      mShowProgressStopped = PR_TRUE;
      mStopSignalTimer = g_timeout_add(1000, progress_emit_stop, mOwner);
    }
  }

  NS_ENSURE_SUCCESS(EnsureAlive(mOwner), NS_ERROR_FAILURE);
  StoreUserNameAndLoginFields(aWebProgress, aRequest, aStateFlags, aStatus);

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnProgressChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				PRInt32         aCurSelfProgress,
				PRInt32         aMaxSelfProgress,
				PRInt32         aCurTotalProgress,
				PRInt32         aMaxTotalProgress)
{

  nsCAutoString uriString;
  RequestToURIString(aRequest, uriString);

  // is it the same as the current uri?
  if (aMaxTotalProgress != -1 && aCurTotalProgress == aMaxTotalProgress
      && aMaxSelfProgress != -1 && aCurSelfProgress == aMaxSelfProgress) {
    mPageDownloaded = PR_TRUE;
  }

  if (mOwner->mURI.Equals(uriString)) {
    if (!mPagePainted && mOwner->mLoadFinished && !mStopSignalTimer) {
      mOwner->mLoadFinished = PR_FALSE;
      mStopSignalTimer = g_timeout_add(1000, progress_emit_stop, mOwner);
      g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                    moz_embed_signals[NET_START], 0);
    }
    g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                  moz_embed_signals[PROGRESS], 0,
                  aCurTotalProgress, aMaxTotalProgress);
  }

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnLocationChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				nsIURI         *aLocation)
{
  nsCAutoString newURI;
  nsCAutoString prePath;
  NS_ENSURE_ARG_POINTER(aLocation);
  aLocation->GetSpec(newURI);
  aLocation->GetPrePath(prePath);

  // Make sure that this is the primary frame change and not
  // just a subframe.
  PRBool isSubFrameLoad = PR_FALSE;
  if (aWebProgress) {
    nsCOMPtr<nsIDOMWindow> domWindow;
    nsCOMPtr<nsIDOMWindow> topDomWindow;

    aWebProgress->GetDOMWindow(getter_AddRefs(domWindow));

    // get the root dom window
    if (domWindow)
      domWindow->GetTop(getter_AddRefs(topDomWindow));

    if (domWindow != topDomWindow)
      isSubFrameLoad = PR_TRUE;
  }

  if (!isSubFrameLoad) {
    mOwner->SetURI(newURI.get());
    mOwner->mPrePath.Assign(prePath.get());
    g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                  moz_embed_signals[LOCATION], 0);
  }

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnStatusChange(nsIWebProgress  *aWebProgress,
			      nsIRequest      *aRequest,
			      nsresult         aStatus,
			      const PRUnichar *aMessage)
{
  g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                moz_embed_signals[STATUS_CHANGE], 0,
                static_cast<void *>(aRequest),
                static_cast<int>(aStatus),
                static_cast<const void *>(aMessage));

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnSecurityChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				PRUint32         aState)
{
  g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                moz_embed_signals[SECURITY_CHANGE], 0,
                static_cast<void *>(aRequest),
                aState);
  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnRefreshAttempted(nsIWebProgress *aWebProgress,
                                  nsIURI *aRefreshURI,
                                  PRInt32 aMillis,
                                  PRBool aSameURI,
                                  PRBool *_retval NS_OUTPARAM)
{
  NS_ENSURE_ARG_POINTER(_retval);
  *_retval = TRUE;      // Don't block this refresh

  nsCAutoString uri;
  aRefreshURI->GetSpec(uri);

  g_signal_emit(G_OBJECT(mOwner->mOwningWidget),
                moz_embed_signals[META_REFRESH], 
                0,
                uri.get(),
                aSameURI,
                aMillis,
                &_retval);

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnProgressChange64(nsIWebProgress *aWebProgress,
                                  nsIRequest *aRequest,
                                  PRInt64 aCurSelfProgress,
                                  PRInt64 aMaxSelfProgress,
                                  PRInt64 aCurTotalProgress,
                                  PRInt64 aMaxTotalProgress)
{
  return NS_ERROR_NOT_IMPLEMENTED;
}


void
EmbedProgress::InsertLogin(const char* username, const char* password)
{
  if (username && mGlobalUserField) {
    mGlobalUserField->SetValue(NS_ConvertUTF8toUTF16(username));
  }

  if (password && mGlobalPassField) {
    mGlobalPassField->SetValue(NS_ConvertUTF8toUTF16(password));
  }
}

PRBool
EmbedProgress::GetPasswordRealm(nsIURI* aURI, nsACString& aRealm)
{
  NS_ENSURE_TRUE(aURI, PR_FALSE);
  // Note: this _is_ different from getting the uri's prePath!
  // We don't want to include a username or password that's part of the
  // URL in the host key... it will cause lookups to work incorrectly, and will
  // also cause usernames and passwords to be stored in cleartext.
  nsCAutoString buffer;
  aURI->GetScheme(buffer);
  aRealm.Append(buffer);
  aRealm.Append(NS_LITERAL_CSTRING("://"));
  aURI->GetHostPort(buffer);
  if (buffer.IsEmpty()) {
    // The scheme does not support hostnames, so don't attempt to save/restore
    // any signon data. (see bug 159484)
    return PR_FALSE;
  }
  aRealm.Append(buffer);
  return PR_TRUE;
}

nsresult
EmbedProgress::StoreUserNameAndLoginFields(nsIWebProgress *aWebProgress,
			                   nsIRequest     *aRequest,
			                   PRUint32        aStateFlags,
			                   nsresult        aStatus)
{
  // We're only interested in successful document loads.
  if (NS_FAILED(aStatus) ||
      !(aStateFlags & nsIWebProgressListener::STATE_IS_DOCUMENT) ||
      !(aStateFlags & nsIWebProgressListener::STATE_STOP)) {
    return NS_OK;
  }

  // Don't do anything if the global signon pref is disabled
  PRBool remember = PR_FALSE;
  gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, "signon.rememberSignons", &remember);
  if (!remember) {
    return NS_OK;
  }

  nsCOMPtr<nsIDOMWindow> domWin;
  nsresult rv = aWebProgress->GetDOMWindow(getter_AddRefs(domWin));
  NS_ENSURE_SUCCESS(rv, rv);
  nsCOMPtr<nsIDOMDocument> domDoc;
  domWin->GetDocument(getter_AddRefs(domDoc));
  NS_ASSERTION(domDoc, "DOM window should always have a document!");
  // For now, only prefill forms in HTML documents.
  nsCOMPtr<nsIDOMHTMLDocument> htmlDoc = do_QueryInterface(domDoc);
  if (!htmlDoc) {
    return NS_OK;
  }

  nsCOMPtr<nsIDOMHTMLCollection> forms;
  htmlDoc->GetForms(getter_AddRefs(forms));
  nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
  nsCAutoString realm;
  if (!GetPasswordRealm(doc->GetDocumentURI(), realm)) {
    return NS_OK;
  }

  nsCAutoString buffer;
  doc->GetDocumentURI()->GetSpec(buffer);

  nsCOMPtr<nsILoginManager> loginmgr(
        do_GetService("@mozilla.org/login-manager;1"));
    
  if (!loginmgr) {
    return NS_OK;
  }
    
  nsILoginInfo** logins;
  PRUint32 count = 0;
  
  nsString empty;

  rv = loginmgr->FindLogins(&count, NS_ConvertUTF8toUTF16(realm),
    empty, empty, &logins);
  NS_ENSURE_SUCCESS(rv, NS_OK);

  PRUint32 formCount;
  forms->GetLength(&formCount);

  // Check to see if we should formfill.  failure is non-fatal
  PRBool prefillForm = PR_TRUE;
  gtk_moz_embed_common_get_pref(G_TYPE_BOOLEAN, "signon.autofillForms", &prefillForm);

  // We can auto-prefill the username and password if there is only
  // one stored login that matches the username and password field names
  // on the form in question.  Note that we only need to worry about a
  // single login per form.
  for (PRUint32 i = 0; i < formCount; ++i) {
    nsCOMPtr<nsIDOMNode> formNode;
    forms->Item(i, getter_AddRefs(formNode));
    nsCOMPtr<nsIForm> form = do_QueryInterface(formNode);
    nsILoginInfo* firstMatch = nsnull;
    PRBool attachedToInput = PR_FALSE;
    PRBool prefilledUser = PR_FALSE;
    nsCOMPtr<nsIDOMHTMLInputElement> userField, passField;
    nsCOMPtr<nsIDOMHTMLInputElement> temp;
    nsAutoString fieldType;

    for (PRUint32 i = 0; i < count; i++) {
      nsCOMPtr<nsISupports> foundNode;
      nsILoginInfo* login = logins[i];
      nsString usernameField;
      login->GetUsernameField(usernameField);
      if (!usernameField.IsEmpty()) {
        foundNode = form->ResolveName(usernameField);
        temp = do_QueryInterface(foundNode);
      }
      nsAutoString oldUserValue;
      if (temp) {
        temp->GetType(fieldType);
        if (!fieldType.Equals(NS_LITERAL_STRING("text"))) {
          continue;
        }
        temp->GetValue(oldUserValue);
        userField = temp;
      }
      nsString passwordField;
      login->GetPasswordField(passwordField);
      if (!passwordField.IsEmpty()) {
        foundNode = form->ResolveName(passwordField);
        temp = do_QueryInterface(foundNode);
      } else if (userField) {
        // No password field name was supplied, try to locate one in the form,
        // but only if we have a username field.
        nsCOMPtr<nsIFormControl> fc(do_QueryInterface(foundNode));
        PRInt32 index = -1;
        index = form->IndexOfControl(fc);
        if (index >= 0) {
          PRUint32 count;
          count = form->GetElementCount();
          PRUint32 i;
          temp = nsnull;
          // Search forwards
          nsCOMPtr<nsIFormControl> passField;
          for (i = index + 1; i < count; ++i) {
            form->GetElementAt(i, getter_AddRefs(passField));
            if (passField && passField->GetType() == NS_FORM_INPUT_PASSWORD) {
              foundNode = passField;
              temp = do_QueryInterface(foundNode);
            }
          }
          if (!temp && index != 0) {
            // Search backwards
            i = index;
            do {
              form->GetElementAt(i, getter_AddRefs(passField));
              if (passField && passField->GetType() == NS_FORM_INPUT_PASSWORD) {
                foundNode = passField;
                temp = do_QueryInterface(foundNode);
              }
            } while (i-- != 0);
          }
        }
      }
      nsAutoString oldPassValue;
      if (temp) {
        temp->GetType(fieldType);
        if (!fieldType.Equals(NS_LITERAL_STRING("password")))
          continue;
        temp->GetValue(oldPassValue);
        passField = temp;
      } else {
        continue;
      }
      if (firstMatch && userField && !attachedToInput) {
        // We've found more than one possible signon for this form.
        // Listen for blur and autocomplete events on the username field so
        // that we can attempt to prefill the password after the user has
        // entered the username.
        mLoginAttached = PR_TRUE;
        attachedToInput = PR_TRUE;
      } else {
        firstMatch = login;
      }
    }

    if (mGlobalUserField && mGlobalPassField && !userField && !passField)
      continue; // Code is very bad need to be fixed.
    mGlobalUserField = userField;
    mGlobalPassField = passField;
  }


  NS_FREE_XPCOM_ISUPPORTS_POINTER_ARRAY(count, logins);

  return NS_OK;
}

/* static */
void
EmbedProgress::RequestToURIString(nsIRequest *aRequest, nsACString &aString)
{
  // is it a channel
  nsCOMPtr<nsIChannel> channel;
  channel = do_QueryInterface(aRequest);
  if (!channel)
    return;
  
  nsCOMPtr<nsIURI> uri;
  channel->GetURI(getter_AddRefs(uri));
  if (!uri)
    return;
  
  uri->GetSpec(aString);
}
