/* GStreamer
 *
 * Copyright (C) 2002 Ronald Bultje <rbultje@ronald.bitfreak.net>
 *               2006 Edgard Lima <edgard.lima@indt.org.br>
 *               2008 Nokia Corporation <multimedia@maemo.org>
 *
 * v4l2camsrc.c - system calls
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <unistd.h>
#ifdef __sun
/* Needed on older Solaris Nevada builds (72 at least) */
#include <stropts.h>
#include <sys/ioccom.h>
#endif

#include "v4l2camsrc_calls.h"
#include "gstv4l2camcolorbalance.h"
#include "gstv4l2camphotoiface.h"

GST_DEBUG_CATEGORY_EXTERN (v4l2camsrc_debug);
#define GST_CAT_DEFAULT v4l2camsrc_debug

/* lalala... */
#define GST_V4L2_SET_ACTIVE(element) (element)->buffer = GINT_TO_POINTER (-1)
#define GST_V4L2_SET_INACTIVE(element) (element)->buffer = NULL

/* On some systems MAP_FAILED seems to be missing */
#ifndef MAP_FAILED
#define MAP_FAILED ((caddr_t) -1)
#endif

#define RESIZER_MAX_DOWNSCALE_FACTOR 4
#define V4L2CAMSRC_POLL_TIMEOUT (3 * GST_SECOND)

// #define USE_MLOCK

/* Local functions */
static gboolean
gst_v4l2camsrc_get_nearest_size (GstV4l2CamSrc * v4l2camsrc, guint32 pixelformat,
    gint * width, gint * height);
static void gst_v4l2camsrc_buffer_pool_destroy (GstV4l2BufferPool * pool,
    GstV4l2CamSrc * v4l2camsrc);
static gboolean
gst_v4l2camsrc_detect_max_zoom (GstV4l2CamSrc *v4l2camsrc, gfloat *max_zoom);

static GstBuffer *
gst_v4l2camsrc_buffer_new (GstV4l2BufferPool * pool, GstV4l2CamSrc * v4l2camsrc,
    guint index, GstCaps * caps)
{
  GstBuffer *ret = NULL;
  GstBuffer *buf = NULL;
  GstFlowReturn flow_ret;
  struct v4l2_buffer *vbuffer;

  vbuffer = &pool->vbuffers[index];
  vbuffer->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  vbuffer->memory = V4L2_MEMORY_USERPTR;
  vbuffer->length = v4l2camsrc->frame_byte_size;
  vbuffer->index = index;

  GST_LOG ("creating buffer %u, in pool %p (size %d)", index, pool,
      vbuffer->length);

  if (ioctl (pool->video_fd, VIDIOC_QUERYBUF, vbuffer) < 0)
    goto querybuf_failed;

  if (gst_pad_is_linked (GST_BASE_SRC_PAD (v4l2camsrc))) {
    GST_LOG ("using pad_alloc");

    flow_ret = gst_pad_alloc_buffer_and_set_caps (GST_BASE_SRC_PAD (v4l2camsrc),
        0LL, v4l2camsrc->frame_byte_size, caps, &buf);
    if (flow_ret != GST_FLOW_OK)
      goto pad_alloc_failed;
  }
  else {
    void *data;

    GST_LOG ("using posix_memalign");
    if (posix_memalign (&data, getpagesize(), vbuffer->length) != 0) {
      goto memalign_failed;
    }
    buf = gst_buffer_new ();
    GST_BUFFER_MALLOCDATA (buf) = (guint8 *) data;
    GST_BUFFER_DATA (buf) = (guint8 *) data;
    GST_BUFFER_SIZE (buf) = vbuffer->length;
  }

#ifdef USE_MLOCK
  GST_DEBUG ("mlocking buffer data");
  if (mlock ((void *) GST_BUFFER_DATA (buf), v4l2camsrc->frame_byte_size) == -1)
    goto mlock_failed;
#endif

  /* mlocking succeeded, now we can set the pointer to vbuffer. The existence
   * of this pointer will be used later to determine if the munlock() is
   * needed */
  vbuffer->m.userptr = (unsigned int) GST_BUFFER_DATA (buf);

  ret = buf;

#if 1
  GST_LOG ("  index:     %u", vbuffer->index);
  GST_LOG ("  type:      %d", vbuffer->type);
  GST_LOG ("  bytesused: %u", vbuffer->bytesused);
  GST_LOG ("  flags:     %08x", vbuffer->flags);
  GST_LOG ("  field:     %d", vbuffer->field);
  GST_LOG ("  memory:    %d", vbuffer->memory);
  if (vbuffer->memory == V4L2_MEMORY_MMAP)
    GST_LOG ("  MMAP offset:  %u", vbuffer->m.offset);
  else if (vbuffer->memory == V4L2_MEMORY_USERPTR)
    GST_LOG ("  user address: %u", vbuffer->m.userptr);
  GST_LOG ("  length:    %u", vbuffer->length);
  GST_LOG ("  input:     %u", vbuffer->input);
#endif

  return ret;

  /* ERRORS */
#ifdef USE_MLOCK
mlock_failed:
  {
    GST_WARNING ("Failed to mlock memory: %s", g_strerror (errno));
    gst_buffer_unref (buf);
    return NULL;
  }
#endif
querybuf_failed:
  {
    gint errnosave = errno;

    GST_WARNING ("Failed QUERYBUF: %s", g_strerror (errnosave));
    errno = errnosave;
    return NULL;
  }
memalign_failed:
  {
    GST_WARNING ("Failed to posix_memalign a buffer");
    return NULL;
  }
pad_alloc_failed:
  {
    GST_WARNING ("Failed to pad_alloc_buffer: %s",
        gst_flow_get_name (flow_ret));
    return NULL;
  }
}

#define GST_TYPE_V4L2_BUFFER_POOL (gst_v4l2camsrc_buffer_pool_get_type())
#define GST_IS_V4L2_BUFFER_POOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GST_TYPE_V4L2_BUFFER_POOL))
#define GST_V4L2_BUFFER_POOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GST_TYPE_V4L2_BUFFER_POOL, GstV4l2BufferPool))

static GstMiniObjectClass *buffer_pool_parent_class = NULL;

static void
gst_v4l2camsrc_buffer_pool_finalize (GstV4l2BufferPool * pool)
{
  g_mutex_free (pool->lock);
  pool->lock = NULL;

  if (pool->video_fd >= 0)
    close (pool->video_fd);

  if (pool->buffers) {
    g_free (pool->buffers);
    pool->buffers = NULL;
  }
  if (pool->vbuffers) {
    g_free (pool->vbuffers);
    pool->vbuffers = NULL;
  }
  if (pool->queued) {
    g_free (pool->queued);
    pool->queued = NULL;
  }
  GST_MINI_OBJECT_CLASS (buffer_pool_parent_class)->finalize (GST_MINI_OBJECT
      (pool));
}

/*
 */
static void
gst_v4l2camsrc_buffer_pool_init (GstV4l2BufferPool * pool, gpointer g_class)
{
  pool->lock = g_mutex_new ();
  pool->running = FALSE;
  pool->num_live_buffers = 0;
}

/*
 */
static void
gst_v4l2camsrc_buffer_pool_class_init (gpointer g_class, gpointer class_data)
{
  GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS (g_class);

  buffer_pool_parent_class = g_type_class_peek_parent (g_class);

  mini_object_class->finalize = (GstMiniObjectFinalizeFunction)
      gst_v4l2camsrc_buffer_pool_finalize;
}

/*
 */
static GType
gst_v4l2camsrc_buffer_pool_get_type (void)
{
  static GType _gst_v4l2camsrc_buffer_pool_type;

  if (G_UNLIKELY (_gst_v4l2camsrc_buffer_pool_type == 0)) {
    static const GTypeInfo v4l2camsrc_buffer_pool_info = {
      sizeof (GstBufferClass),
      NULL,
      NULL,
      gst_v4l2camsrc_buffer_pool_class_init,
      NULL,
      NULL,
      sizeof (GstV4l2BufferPool),
      0,
      (GInstanceInitFunc) gst_v4l2camsrc_buffer_pool_init,
      NULL
    };
    _gst_v4l2camsrc_buffer_pool_type = g_type_register_static (GST_TYPE_MINI_OBJECT,
        "GstV4l2BufferPool", &v4l2camsrc_buffer_pool_info, 0);
  }
  return _gst_v4l2camsrc_buffer_pool_type;
}

/*
 */
static GstV4l2BufferPool *
gst_v4l2camsrc_buffer_pool_new (GstV4l2CamSrc * v4l2camsrc, gint fd, GstCaps * caps)
{
  GstV4l2BufferPool *pool;
  gint n;

  pool = (GstV4l2BufferPool *) gst_mini_object_new (GST_TYPE_V4L2_BUFFER_POOL);

  pool->video_fd = dup (fd);
  if (pool->video_fd < 0)
    goto dup_failed;


  pool->buffer_count = v4l2camsrc->num_buffers;
  pool->buffers = g_new0 (GstBuffer *, pool->buffer_count);
  pool->vbuffers = g_new0 (struct v4l2_buffer, pool->buffer_count);
  pool->queued = g_new0 (guint, pool->buffer_count);

  for (n = 0; n < pool->buffer_count; n++) {
    pool->buffers[n] = gst_v4l2camsrc_buffer_new (pool, v4l2camsrc, n, caps);
    if (!pool->buffers[n])
      goto buffer_new_failed;
  }

  return pool;

  /* ERRORS */
dup_failed:
  {
    gint errnosave = errno;

    gst_v4l2camsrc_buffer_pool_destroy (pool, v4l2camsrc);

    errno = errnosave;

    return NULL;
  }
buffer_new_failed:
  {
    gint errnosave = errno;

    GST_LOG ("creating a new buffer failed");
    gst_mini_object_unref (GST_MINI_OBJECT (pool));

    errno = errnosave;

    return NULL;
  }
}

/*
 */
static gboolean
gst_v4l2camsrc_buffer_pool_activate (GstV4l2BufferPool * pool, GstV4l2CamSrc * v4l2camsrc)
{
  gint n;
  struct v4l2_buffer *vbuffer;

  g_mutex_lock (pool->lock);

  for (n = 0; n < pool->buffer_count; n++) {

    vbuffer = &pool->vbuffers[n];

    GST_LOG ("enqueue pool buffer %d", n);

    if (ioctl (pool->video_fd, VIDIOC_QBUF, vbuffer) < 0)
      goto queue_failed;

    pool->queued[n] = 1;
  }
  pool->running = TRUE;

  g_mutex_unlock (pool->lock);

  return TRUE;

  /* ERRORS */
queue_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
        ("Could not enqueue buffers in device '%s'.",
            v4l2camsrc->videodev),
        ("enqueing buffer %d/%d failed: %s",
            n, v4l2camsrc->num_buffers, g_strerror (errno)));
    g_mutex_unlock (pool->lock);
    return FALSE;
  }
}

/* requeue buffers that are writable again */
static gboolean
gst_v4l2_buffer_pool_update (GstV4l2BufferPool * pool, GstV4l2CamSrc * v4l2camsrc)
{
  gint n;
  struct v4l2_buffer *vbuffer;

  g_mutex_lock (pool->lock);

  for (n = 0; n < pool->buffer_count; n++) {
    vbuffer = &pool->vbuffers[n];

    if (!pool->queued[n] && gst_buffer_is_writable (pool->buffers[n])) {
      if (ioctl (pool->video_fd, VIDIOC_QBUF, vbuffer) < 0) {
        GST_WARNING_OBJECT (v4l2camsrc, "could not requeue buffer %p at index %d",
          pool->buffers[n], n);
      } else {
        GST_LOG ("reviving buffer %p, %d", pool->buffers[n], n);
        pool->num_live_buffers--;
        pool->queued[n] = 1;
      }
    }
  }

  g_mutex_unlock (pool->lock);

  return TRUE;
}

/*
 */
static void
gst_v4l2camsrc_buffer_pool_destroy (GstV4l2BufferPool * pool,
    GstV4l2CamSrc * v4l2camsrc)
{
  gint n;

  g_mutex_lock (pool->lock);
  pool->running = FALSE;
  g_mutex_unlock (pool->lock);

  GST_DEBUG ("destroy pool");

  /* after this point, no more buffers will be queued or dequeued; no buffer
   * from pool->buffers that is NULL will be set to a buffer, and no buffer that
   * is not NULL will be pushed out. */

  /* miniobjects have no dispose, so they can't break ref-cycles, as buffers ref
   * the pool, we need to unref the buffer to properly finalize te pool */
  for (n = 0; n < pool->buffer_count; n++) {
    GstBuffer *buf;
    struct v4l2_buffer *vbuf;

    g_mutex_lock (pool->lock);
    buf = GST_BUFFER (pool->buffers[n]);
    vbuf = &pool->vbuffers[n];
    g_mutex_unlock (pool->lock);

    if (buf)
#ifdef USE_MLOCK
      if (vbuf->m.userptr) {
        GST_DEBUG ("munlocking buffer data");
        munlock ((void *)vbuf->m.userptr, v4l2camsrc->frame_byte_size);
      }
#endif
      /* we own the ref if the buffer is in pool->buffers; drop it. */
      gst_buffer_unref (buf);
  }

  gst_mini_object_unref (GST_MINI_OBJECT (pool));
}

/******************************************************
 * gst_v4l2camsrc_get_capabilities():
 *   get the device's capturing capabilities
 * return value: TRUE on success, FALSE on error
 ******************************************************/
static gboolean
gst_v4l2camsrc_get_capabilities (GstV4l2CamSrc *v4l2camsrc)
{
  GST_DEBUG_OBJECT (v4l2camsrc, "getting capabilities");

  if (!GST_V4L2_IS_OPEN (v4l2camsrc))
    return FALSE;

  if (ioctl (v4l2camsrc->video_fd, VIDIOC_QUERYCAP, &v4l2camsrc->vcap) < 0)
    goto cap_failed;

  GST_LOG_OBJECT (v4l2camsrc, "driver:      '%s'", v4l2camsrc->vcap.driver);
  GST_LOG_OBJECT (v4l2camsrc, "card:        '%s'", v4l2camsrc->vcap.card);
  GST_LOG_OBJECT (v4l2camsrc, "bus_info:    '%s'", v4l2camsrc->vcap.bus_info);
  GST_LOG_OBJECT (v4l2camsrc, "version:     %08x", v4l2camsrc->vcap.version);
  GST_LOG_OBJECT (v4l2camsrc, "capabilites: %08x", v4l2camsrc->vcap.capabilities);

  return TRUE;

  /* ERRORS */
cap_failed:
{
  GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
                     ("Error getting capabilities for device '%s': "
                         "It isn't a v4l2 driver. Check if it is a v4l1 driver.",
                         v4l2camsrc->videodev), GST_ERROR_SYSTEM);
  return FALSE;
}
}


/******************************************************
 * gst_v4l2camsrc_empty_lists() and gst_v4l2camsrc_fill_lists():
 *   fill/empty the lists of enumerations
 * return value: TRUE on success, FALSE on error
 ******************************************************/
static gboolean
gst_v4l2camsrc_fill_lists (GstV4l2CamSrc *v4l2camsrc)
{
  gint n;

  GST_DEBUG_OBJECT (v4l2camsrc, "getting enumerations");
  GST_V4L2_CHECK_OPEN (v4l2camsrc);

  GST_DEBUG_OBJECT (v4l2camsrc, "  controls+menus");
  /* and lastly, controls+menus (if appropriate) */
  for (n = V4L2_CID_BASE;; n++) {
    struct v4l2_queryctrl control = { 0, };
    GstV4l2ColorBalanceChannel *v4l2channel;

    GstColorBalanceChannel *channel;

    /* when we reached the last official CID, continue with private CIDs */
    if (n == V4L2_CID_LASTP1) {
      GST_DEBUG_OBJECT (v4l2camsrc, "checking private CIDs");
      n = V4L2_CID_PRIVATE_BASE;
      /* FIXME: We are still not handling private controls. We need a new GstInterface
      to export those controls */
      break;
    }

    control.id = n;
    if (ioctl (v4l2camsrc->video_fd, VIDIOC_QUERYCTRL, &control) < 0) {
      if (errno == EINVAL) {
        if (n < V4L2_CID_PRIVATE_BASE)
          /* continue so that we also check private controls */
          continue;
        else
          break;
      } else {
        GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
                           ("Failed getting controls attributes on device '%s.'",
                            v4l2camsrc->videodev),
                            ("Failed querying control %d on device '%s'. (%d - %s)",
                             n, v4l2camsrc->videodev, errno, strerror (errno)));
        return FALSE;
      }
    }
    if (control.flags & V4L2_CTRL_FLAG_DISABLED)
      continue;

    switch (n) {
      case V4L2_CID_BRIGHTNESS:
      case V4L2_CID_CONTRAST:
      case V4L2_CID_SATURATION:
      case V4L2_CID_HUE:
      case V4L2_CID_BLACK_LEVEL:
      case V4L2_CID_AUTO_WHITE_BALANCE:
      case V4L2_CID_DO_WHITE_BALANCE:
      case V4L2_CID_RED_BALANCE:
      case V4L2_CID_BLUE_BALANCE:
      case V4L2_CID_GAMMA:
      case V4L2_CID_EXPOSURE:
      case V4L2_CID_AUTOGAIN:
      case V4L2_CID_GAIN:
        /* we only handle these for now (why?) */
        break;
      case V4L2_CID_HFLIP:
      case V4L2_CID_VFLIP:
      case V4L2_CID_HCENTER:
      case V4L2_CID_VCENTER:
#ifdef V4L2_CID_PAN_RESET
      case V4L2_CID_PAN_RESET:
#endif
#ifdef V4L2_CID_TILT_RESET
      case V4L2_CID_TILT_RESET:
#endif
        /* not handled here, handled by VideoOrientation interface */
        control.id++;
        break;
      case V4L2_CID_AUDIO_VOLUME:
      case V4L2_CID_AUDIO_BALANCE:
      case V4L2_CID_AUDIO_BASS:
      case V4L2_CID_AUDIO_TREBLE:
      case V4L2_CID_AUDIO_MUTE:
      case V4L2_CID_AUDIO_LOUDNESS:
        /* FIXME: We should implement GstMixer interface */
        /* fall through */
      default:
        GST_DEBUG_OBJECT (v4l2camsrc,
                          "ControlID %s (%x) unhandled, FIXME", control.name, n);
        control.id++;
        break;
    }
    if (n != control.id)
      continue;

    GST_DEBUG_OBJECT (v4l2camsrc, "Adding ControlID %s (%x)", control.name, n);
    v4l2channel = g_object_new (GST_TYPE_V4L2_COLOR_BALANCE_CHANNEL, NULL);
    channel = GST_COLOR_BALANCE_CHANNEL (v4l2channel);
    channel->label = g_strdup ((const gchar *) control.name);
    v4l2channel->id = n;

    switch (control.type) {
      case V4L2_CTRL_TYPE_INTEGER:
        channel->min_value = control.minimum;
        channel->max_value = control.maximum;
        break;
      case V4L2_CTRL_TYPE_BOOLEAN:
        channel->min_value = FALSE;
        channel->max_value = TRUE;
        break;
      default:
        /* FIXME we should find out how to handle V4L2_CTRL_TYPE_BUTTON.
        BUTTON controls like V4L2_CID_DO_WHITE_BALANCE can just be set (1) or
        unset (0), but can't be queried */
        GST_DEBUG_OBJECT (v4l2camsrc,
                          "Control with non supported type %s (%x), type=%d",
                          control.name, n, control.type);
        channel->min_value = channel->max_value = 0;
        break;
    }

    v4l2camsrc->colors = g_list_append (v4l2camsrc->colors, (gpointer) channel);
  }

  GST_DEBUG_OBJECT (v4l2camsrc, "done");
  return TRUE;
}

/******************************************************
 * gst_v4l2camsrc_empty_lists():
 *   Clean up device information resources.
 ******************************************************/
void
gst_v4l2camsrc_empty_lists (GstV4l2CamSrc *v4l2camsrc)
{
  GST_DEBUG_OBJECT (v4l2camsrc, "deleting enumerations");

  g_list_foreach (v4l2camsrc->colors, (GFunc) g_object_unref, NULL);
  g_list_free (v4l2camsrc->colors);
  v4l2camsrc->colors = NULL;
}

/******************************************************
 * gst_v4l2camsrc_open():
 *   open the video device (v4l2camsrc->videodev)
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_open (GstV4l2CamSrc *v4l2camsrc)
{
  struct stat st;
  GstPollFD pollfd = GST_POLL_FD_INIT;

  GST_DEBUG_OBJECT (v4l2camsrc, "Trying to open device %s",
      v4l2camsrc->videodev);

  GST_V4L2_CHECK_NOT_OPEN (v4l2camsrc);
  GST_V4L2_CHECK_NOT_ACTIVE (v4l2camsrc);

  /* be sure we have a device */
  if (!v4l2camsrc->videodev)
    v4l2camsrc->videodev = g_strdup ("/dev/video");

  /* check if it is a device */
  if (stat (v4l2camsrc->videodev, &st) == -1)
    goto stat_failed;

  if (!S_ISCHR (st.st_mode))
    goto no_device;

  /* open the device */
  v4l2camsrc->video_fd =
      open (v4l2camsrc->videodev, O_RDWR /* | O_NONBLOCK */ );

  if (!GST_V4L2_IS_OPEN (v4l2camsrc))
    goto not_open;

  /* A hack to speed up re-opening the device */
  if (!v4l2camsrc->colors) {
    /* get capabilities, error will be posted */
    if (!gst_v4l2camsrc_get_capabilities (v4l2camsrc))
      goto error;

    /* do we need to be a capture device? */
    if (GST_IS_V4L2CAMSRC (v4l2camsrc) &&
        !(v4l2camsrc->vcap.capabilities & V4L2_CAP_VIDEO_CAPTURE))
      goto not_capture;

    /* create enumerations, posts errors. */
    if (!gst_v4l2camsrc_fill_lists (v4l2camsrc))
      goto error;
  }
  else {
    GST_INFO_OBJECT (v4l2camsrc, "No need to fill device lists");
  }

  GST_INFO_OBJECT (v4l2camsrc,
      "Opened device '%s' (%s) successfully",
      v4l2camsrc->vcap.card, v4l2camsrc->videodev);

  pollfd.fd = v4l2camsrc->video_fd;
  gst_poll_add_fd (v4l2camsrc->poll, &pollfd);
  gst_poll_fd_ctl_read (v4l2camsrc->poll, &pollfd, TRUE);

  return TRUE;

  /* ERRORS */
stat_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, NOT_FOUND,
        ("Cannot identify device '%s'.", v4l2camsrc->videodev),
        GST_ERROR_SYSTEM);
    goto error;
  }
no_device:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, NOT_FOUND,
        ("This isn't a device '%s'.", v4l2camsrc->videodev),
        GST_ERROR_SYSTEM);
    goto error;
  }
not_open:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, OPEN_READ_WRITE,
        ("Could not open device '%s' for reading and writing.",
            v4l2camsrc->videodev), GST_ERROR_SYSTEM);
    goto error;
  }
not_capture:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, NOT_FOUND,
        ("Device '%s' is not a capture device.",
            v4l2camsrc->videodev),
        ("Capabilities: 0x%x", v4l2camsrc->vcap.capabilities));
    goto error;
  }
error:
  {
    if (GST_V4L2_IS_OPEN (v4l2camsrc)) {
      /* close device */
      close (v4l2camsrc->video_fd);
      v4l2camsrc->video_fd = -1;
    }
    /* empty lists */
    gst_v4l2camsrc_empty_lists (v4l2camsrc);

    return FALSE;
  }
}

/******************************************************
 * gst_v4l2camsrc_close():
 *   close the video device (v4l2camsrc->video_fd)
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_close (GstV4l2CamSrc *v4l2camsrc)
{
  GstPollFD pollfd = GST_POLL_FD_INIT;

  GST_DEBUG_OBJECT (v4l2camsrc, "Trying to close %s",
      v4l2camsrc->videodev);

  GST_V4L2_CHECK_OPEN (v4l2camsrc);
  GST_V4L2_CHECK_NOT_ACTIVE (v4l2camsrc);

  /* close device */
  close (v4l2camsrc->video_fd);
  pollfd.fd = v4l2camsrc->video_fd;
  gst_poll_remove_fd (v4l2camsrc->poll, &pollfd);
  v4l2camsrc->video_fd = -1;

  return TRUE;
}

/******************************************************
 * gst_v4l2camsrc_get_attribute():
 *   try to get the value of one specific attribute
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_get_attribute (GstV4l2CamSrc *v4l2camsrc,
                              int attribute_num, int *value)
{
  struct v4l2_control control;

  GST_DEBUG_OBJECT (v4l2camsrc, "getting value of attribute %d",
                    attribute_num);

  if (!GST_V4L2_IS_OPEN (v4l2camsrc))
    return FALSE;

  control.id = attribute_num;

  if (ioctl (v4l2camsrc->video_fd, VIDIOC_G_CTRL, &control) < 0)
    goto ctrl_failed1;

  *value = control.value;

  return TRUE;

ctrl_failed1:
{
  struct v4l2_ext_controls controls;
  struct v4l2_ext_control control;

  controls.ctrl_class = V4L2_CTRL_CLASS_USER;
  controls.count = 1;
  controls.controls = &control;

  control.id = attribute_num;

  if (ioctl (v4l2camsrc->video_fd, VIDIOC_G_EXT_CTRLS, &controls) < 0)
    goto ctrl_failed2;

  *value = control.value;

  return TRUE;
 
}

  /* ERRORS */
ctrl_failed2:
{
  GST_ELEMENT_WARNING (v4l2camsrc, RESOURCE, SETTINGS,
                       ("Failed to get value for control %d on device '%s'.",
                        attribute_num, v4l2camsrc->videodev), GST_ERROR_SYSTEM);
  return FALSE;
}
}

/******************************************************
 * gst_v4l2camsrc_set_attribute():
 *   try to set the value of one specific attribute
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_set_attribute (GstV4l2CamSrc *v4l2camsrc,
                              int attribute_num, const int value)
{
  struct v4l2_control control;

  GST_DEBUG_OBJECT (v4l2camsrc, "setting value of attribute %d to %d",
                    attribute_num, value);

  if (!GST_V4L2_IS_OPEN (v4l2camsrc))
    return FALSE;

  control.id = attribute_num;
  control.value = value;
  if (ioctl (v4l2camsrc->video_fd, VIDIOC_S_CTRL, &control) < 0)
    goto ctrl_failed1;

  return TRUE;

ctrl_failed1:
{
  struct v4l2_ext_controls controls;
  struct v4l2_ext_control control;

  controls.ctrl_class = V4L2_CTRL_CLASS_USER;
  controls.count = 1;
  controls.controls = &control;

  control.id = attribute_num;
  control.value = value;

  if (ioctl (v4l2camsrc->video_fd, VIDIOC_S_EXT_CTRLS, &controls) < 0)
    goto ctrl_failed2;

  return TRUE;
}

  /* ERRORS */
ctrl_failed2:
{
  GST_ELEMENT_WARNING (v4l2camsrc, RESOURCE, SETTINGS,
                       ("Failed to set value %d for control %d on device '%s'.",
                        value, attribute_num, v4l2camsrc->videodev), GST_ERROR_SYSTEM);
  return FALSE;
}
}


/* complete made up ranking, the values themselves are meaningless */
#define YUV_BASE_RANK     1000
#define BAYER_BASE_RANK     15

static gint
gst_v4l2camsrc_format_get_rank (guint32 fourcc)
{
  switch (fourcc) {

    case V4L2_PIX_FMT_YUYV:    /* YUY2, 16 bits per pixel */
      return YUV_BASE_RANK + 10;
    case V4L2_PIX_FMT_UYVY:    /* UYVY, 16 bits per pixel */
      return YUV_BASE_RANK + 9;

#ifdef V4L2_PIX_FMT_SGRBG10
    case V4L2_PIX_FMT_SGRBG10:
#endif
    case V4L2_PIX_FMT_SBGGR8:
      return BAYER_BASE_RANK;

    default:
      break;
  }

  return 0;
}

static gint
gst_v4l2camsrc_format_cmp_func (gconstpointer a, gconstpointer b)
{
  guint32 pf1 = ((struct v4l2_fmtdesc *) a)->pixelformat;
  guint32 pf2 = ((struct v4l2_fmtdesc *) b)->pixelformat;

  if (pf1 == pf2)
    return 0;

  return gst_v4l2camsrc_format_get_rank (pf2) - gst_v4l2camsrc_format_get_rank (pf1);
}

/******************************************************
 * gst_v4l2camsrc_fill_format_list():
 *   create list of supported capture formats
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_fill_format_list (GstV4l2CamSrc * v4l2camsrc)
{
  gint n;
  struct v4l2_fmtdesc *format;

  GST_DEBUG_OBJECT (v4l2camsrc, "getting src format enumerations");

  /* format enumeration */
  for (n = 0;; n++) {
    format = g_new0 (struct v4l2_fmtdesc, 1);

    format->index = n;
    format->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

    if (ioctl (v4l2camsrc->video_fd, VIDIOC_ENUM_FMT, format) < 0) {
      if (errno == EINVAL) {
        g_free (format);
        break;                  /* end of enumeration */
      } else {
        goto failed;
      }
    }

    GST_LOG_OBJECT (v4l2camsrc, "index:       %u", format->index);
    GST_LOG_OBJECT (v4l2camsrc, "type:        %d", format->type);
    GST_LOG_OBJECT (v4l2camsrc, "flags:       %08x", format->flags);
    GST_LOG_OBJECT (v4l2camsrc, "description: '%s'", format->description);
    GST_LOG_OBJECT (v4l2camsrc, "pixelformat: %" GST_FOURCC_FORMAT,
        GST_FOURCC_ARGS (format->pixelformat));

    /* sort formats according to our preference;  we do this, because caps
     * are probed in the order the formats are in the list, and the order of
     * formats in the final probed caps matters for things like fixation */
    v4l2camsrc->formats = g_slist_insert_sorted (v4l2camsrc->formats, format,
        (GCompareFunc) gst_v4l2camsrc_format_cmp_func);
  }

  GST_DEBUG_OBJECT (v4l2camsrc, "got %d format(s)", n);

  return TRUE;

  /* ERRORS */
failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
        ("Failed to enumerate possible video formats device '%s' can work with",
        v4l2camsrc->videodev),
        ("Failed to get number %d in pixelformat enumeration for %s. (%d - %s)",
        n, v4l2camsrc->videodev, errno, g_strerror (errno)));
    g_free (format);
    return FALSE;
  }
}

/******************************************************
 * gst_v4l2camsrc_clear_format_list():
 *   free list of supported capture formats
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_clear_format_list (GstV4l2CamSrc * v4l2camsrc)
{
  g_slist_foreach (v4l2camsrc->formats, (GFunc) g_free, NULL);
  g_slist_free (v4l2camsrc->formats);
  v4l2camsrc->formats = NULL;

  return TRUE;
}

/* The frame interval enumeration code first appeared in Linux 2.6.19. */
#ifdef VIDIOC_ENUM_FRAMEINTERVALS 
static GstStructure *
gst_v4l2camsrc_probe_caps_for_format_and_size (GstV4l2CamSrc * v4l2camsrc,
    guint32 pixelformat,
    guint32 width, guint32 height, const GstStructure * template)
{
  gint fd = v4l2camsrc->video_fd;
  struct v4l2_frmivalenum ival;
  guint32 num, denom;
  GstStructure *s;
  GValue rates = { 0, };

  memset (&ival, 0, sizeof (struct v4l2_frmivalenum));
  ival.index = 0;
  ival.pixel_format = pixelformat;
  ival.width = width;
  ival.height = height;

  GST_LOG_OBJECT (v4l2camsrc, "get frame interval for %ux%u, %" GST_FOURCC_FORMAT,
      width, height, GST_FOURCC_ARGS (pixelformat));

  /* keep in mind that v4l2 gives us frame intervals (durations); we invert the
   * fraction to get framerate */
  if (ioctl (fd, VIDIOC_ENUM_FRAMEINTERVALS, &ival) < 0)
    goto enum_frameintervals_failed;

  if (ival.type == V4L2_FRMIVAL_TYPE_DISCRETE) {
    GValue rate = { 0, };

    g_value_init (&rates, GST_TYPE_LIST);
    g_value_init (&rate, GST_TYPE_FRACTION);

    do {
      num = ival.discrete.numerator;
      denom = ival.discrete.denominator;

      if (num > G_MAXINT || denom > G_MAXINT) {
        /* let us hope we don't get here... */
        num >>= 1;
        denom >>= 1;
      }

      GST_LOG_OBJECT (v4l2camsrc, "adding discrete framerate: %d/%d", denom, num);

      /* swap to get the framerate */
      gst_value_set_fraction (&rate, denom, num);
      gst_value_list_append_value (&rates, &rate);

      ival.index++;
    } while (ioctl (fd, VIDIOC_ENUM_FRAMEINTERVALS, &ival) >= 0);
  } else if (ival.type == V4L2_FRMIVAL_TYPE_STEPWISE) {
    GValue min = { 0, };
    GValue step = { 0, };
    GValue max = { 0, };
    gboolean added = FALSE;
    guint32 minnum, mindenom;
    guint32 maxnum, maxdenom;

    g_value_init (&rates, GST_TYPE_LIST);

    g_value_init (&min, GST_TYPE_FRACTION);
    g_value_init (&step, GST_TYPE_FRACTION);
    g_value_init (&max, GST_TYPE_FRACTION);

    /* get the min */
    minnum = ival.stepwise.min.numerator;
    mindenom = ival.stepwise.min.denominator;
    if (minnum > G_MAXINT || mindenom > G_MAXINT) {
      minnum >>= 1;
      mindenom >>= 1;
    }
    GST_LOG_OBJECT (v4l2camsrc, "stepwise min frame interval: %d/%d", minnum,
        mindenom);
    gst_value_set_fraction (&min, minnum, mindenom);

    /* get the max */
    maxnum = ival.stepwise.max.numerator;
    maxdenom = ival.stepwise.max.denominator;
    if (maxnum > G_MAXINT || maxdenom > G_MAXINT) {
      maxnum >>= 1;
      maxdenom >>= 1;
    }

    GST_LOG_OBJECT (v4l2camsrc, "stepwise max frame interval: %d/%d", maxnum,
        maxdenom);
    gst_value_set_fraction (&max, maxnum, maxdenom);

    /* get the step */
    num = ival.stepwise.step.numerator;
    denom = ival.stepwise.step.denominator;
    if (num > G_MAXINT || denom > G_MAXINT) {
      num >>= 1;
      denom >>= 1;
    }

    if (num == 0 || denom == 0) {
      /* in this case we have a wrong fraction or no step, set the step to max
       * so that we only add the min value in the loop below */
      num = maxnum;
      denom = maxdenom;
    }

    /* since we only have gst_value_fraction_subtract and not add, negate the
     * numerator */
    GST_LOG_OBJECT (v4l2camsrc, "stepwise step frame interval: %d/%d", num, denom);
    gst_value_set_fraction (&step, -num, denom);

    while (gst_value_compare (&min, &max) <= 0) {
      GValue rate = { 0, };

      num = gst_value_get_fraction_numerator (&min);
      denom = gst_value_get_fraction_denominator (&min);
      GST_LOG_OBJECT (v4l2camsrc, "adding stepwise framerate: %d/%d", denom, num);

      /* invert to get the framerate */
      g_value_init (&rate, GST_TYPE_FRACTION);
      gst_value_set_fraction (&rate, denom, num);
      gst_value_list_append_value (&rates, &rate);
      added = TRUE;

      /* we're actually adding because step was negated above. This is because
       * there is no _add function... */
      if (!gst_value_fraction_subtract (&min, &min, &step)) {
        GST_WARNING_OBJECT (v4l2camsrc, "could not step fraction!");
        break;
      }
    }
    if (!added) {
      /* no range was added, leave the default range from the template */
      GST_WARNING_OBJECT (v4l2camsrc, "no range added, leaving default");
      g_value_unset (&rates);
    }
  } else if (ival.type == V4L2_FRMIVAL_TYPE_CONTINUOUS) {
    guint32 maxnum, maxdenom;

    g_value_init (&rates, GST_TYPE_FRACTION_RANGE);

    num = ival.stepwise.min.numerator;
    denom = ival.stepwise.min.denominator;
    if (num > G_MAXINT || denom > G_MAXINT) {
      num >>= 1;
      denom >>= 1;
    }

    maxnum = ival.stepwise.max.numerator;
    maxdenom = ival.stepwise.max.denominator;
    if (maxnum > G_MAXINT || maxdenom > G_MAXINT) {
      maxnum >>= 1;
      maxdenom >>= 1;
    }

    GST_LOG_OBJECT (v4l2camsrc, "continuous frame interval %d/%d to %d/%d",
        maxdenom, maxnum, denom, num);

    gst_value_set_fraction_range_full (&rates, maxdenom, maxnum, denom, num);
  } else {
    goto unknown_type;
  }

return_data:
  s = gst_structure_copy (template);
  /* https://projects.maemo.org/bugzilla/show_bug.cgi?id=105590 */
  gst_structure_set (s,
      "width", G_TYPE_INT, (gint) width & ~0x0F,
      "height", G_TYPE_INT, (gint) height & ~0x0F,
      NULL);

  if (G_IS_VALUE (&rates)) {
    /* only change the framerate on the template when we have a valid probed new
     * value */
    gst_structure_set_value (s, "framerate", &rates);
    g_value_unset (&rates);
  }
  return s;

  /* ERRORS */
enum_frameintervals_failed:
  {
    GST_DEBUG_OBJECT (v4l2camsrc,
        "Unable to enumerate intervals for %" GST_FOURCC_FORMAT "@%ux%u",
        GST_FOURCC_ARGS (pixelformat), width, height);
    goto return_data;
  }
unknown_type:
  {
    /* I don't see how this is actually an error, we ignore the format then */
    GST_WARNING_OBJECT (v4l2camsrc,
        "Unknown frame interval type at %" GST_FOURCC_FORMAT "@%ux%u: %u",
        GST_FOURCC_ARGS (pixelformat), width, height, ival.type);
    return NULL;
  }
}
#endif /* defined VIDIOC_ENUM_FRAMEINTERVALS */

gint compare_resolutions (gconstpointer a, gconstpointer b)
{
  GstStructure *as = (GstStructure *) a;
  GstStructure *bs = (GstStructure *) b;
  gint aw, bw, ah, bh;

  gst_structure_get_int (as, "width", &aw);
  gst_structure_get_int (bs, "width", &bw);
  gst_structure_get_int (as, "height", &ah);
  gst_structure_get_int (bs, "height", &bh);

  /* FIXME: What is the best way to compare resolutions if their aspect
   * ratio isn't the same? */
  return ((aw-bw) + (ah-bh));
}

GstCaps *
gst_v4l2camsrc_probe_caps_for_format (GstV4l2CamSrc * v4l2camsrc, guint32 pixelformat,
    const GstStructure * template)
{
  GstCaps *ret = gst_caps_new_empty ();
  gint min_w, max_w, min_h, max_h;
  GstStructure *tmp;

  min_w = min_h = 1;
  max_w = max_h = GST_V4L2_MAX_SIZE;
  if (!gst_v4l2camsrc_get_nearest_size (v4l2camsrc, pixelformat, &min_w, &min_h)) {
    GST_WARNING_OBJECT (v4l2camsrc,
        "Could not probe minimum capture size for pixelformat %"
         GST_FOURCC_FORMAT, GST_FOURCC_ARGS (pixelformat));
  }
  if (!gst_v4l2camsrc_get_nearest_size (v4l2camsrc, pixelformat, &max_w, &max_h)) {
    GST_WARNING_OBJECT (v4l2camsrc,
        "Could not probe maximum capture size for pixelformat %"
        GST_FOURCC_FORMAT, GST_FOURCC_ARGS (pixelformat));
  }

#ifdef VIDIOC_ENUM_FRAMESIZES
  {
    gint fd = v4l2camsrc->video_fd;
    struct v4l2_frmsizeenum size;
    GList *results = NULL;
    gint w, h;

    memset (&size, 0, sizeof (struct v4l2_frmsizeenum));
    size.index = 0;
    size.pixel_format = pixelformat;

    if (ioctl (fd, VIDIOC_ENUM_FRAMESIZES, &size) < 0)
      goto enum_framesizes_failed;

    if (size.type == V4L2_FRMSIZE_TYPE_DISCRETE) {
      do {
        w = MIN (size.discrete.width, G_MAXINT);
        h = MIN (size.discrete.height, G_MAXINT);

        tmp = gst_v4l2camsrc_probe_caps_for_format_and_size (v4l2camsrc,
            pixelformat, w, h, template);

        if (tmp)
          results = g_list_prepend (results, tmp);

        size.index++;
      } while (ioctl (fd, VIDIOC_ENUM_FRAMESIZES, &size) >= 0);
    } else {
      goto unknown_type;
    }

    /* Sort the resolutions from smallest to largest */
    results = g_list_sort (results, compare_resolutions);

    /* If we have resizer we can use ranges instead of discrete
       resolutions reported by driver. FIXME: how to detect resizer?
       It would be nice if the resizer was transparent to v4l2camsrc.
    */
    while (results != NULL) {
      tmp = GST_STRUCTURE (results->data);
      gst_structure_get_int (tmp, "width", &max_w);
      gst_structure_get_int (tmp, "height", &max_h);

      min_w = max_w / RESIZER_MAX_DOWNSCALE_FACTOR;
      min_h = max_h / RESIZER_MAX_DOWNSCALE_FACTOR;

      GST_DEBUG_OBJECT (v4l2camsrc,
          "Setting up range %dx%d - %dx%d", min_w, min_h, max_w, max_h);

      gst_structure_set (tmp,
                         "width", GST_TYPE_INT_RANGE, min_w, max_w,
                         "height", GST_TYPE_INT_RANGE, min_h,max_h,
                         NULL);

      gst_caps_append_structure (ret, tmp);

      results = g_list_delete_link (results, results);
    }

    if (gst_caps_is_empty (ret))
      goto enum_framesizes_no_results;

    return ret;

    /* ERRORS */
  enum_framesizes_failed:
    {
      /* I don't see how this is actually an error */
      GST_DEBUG_OBJECT (v4l2camsrc,
          "Failed to enumerate frame sizes for pixelformat %" GST_FOURCC_FORMAT
          " (%s)", GST_FOURCC_ARGS (pixelformat), g_strerror (errno));
      goto default_frame_sizes;
    }
  enum_framesizes_no_results:
    {
      /* it's possible that VIDIOC_ENUM_FRAMESIZES is defined but the driver in
       * question doesn't actually support it yet */
      GST_DEBUG_OBJECT (v4l2camsrc, "No results for pixelformat %" GST_FOURCC_FORMAT
          " enumerating frame sizes, trying fallback",
          GST_FOURCC_ARGS (pixelformat));
      goto default_frame_sizes;
    }
  unknown_type:
    {
      GST_WARNING_OBJECT (v4l2camsrc,
          "Unknown frame sizeenum type for pixelformat %" GST_FOURCC_FORMAT
          ": %u", GST_FOURCC_ARGS (pixelformat), size.type);
      goto default_frame_sizes;
    }
  }
#endif /* defined VIDIOC_ENUM_FRAMESIZES */

default_frame_sizes:

  tmp = gst_structure_copy (template);
  gst_structure_set (tmp,
                     "width", GST_TYPE_INT_RANGE, min_w, max_w,
                     "height", GST_TYPE_INT_RANGE, min_h, max_h, NULL);
  gst_caps_append_structure (ret, tmp);

  return ret;
}

/******************************************************
 * gst_v4l2camsrc_grab_frame ():
 *   grab a frame for capturing
 * return value: GST_FLOW_OK, GST_FLOW_WRONG_STATE or GST_FLOW_ERROR
 ******************************************************/
GstFlowReturn
gst_v4l2camsrc_grab_frame (GstV4l2CamSrc * v4l2camsrc, GstBuffer ** buf)
{
#define NUM_TRIALS 50
  struct v4l2_buffer buffer;
  gint32 trials = NUM_TRIALS;
  GstBuffer *pool_buffer;
  gboolean need_copy;
  gint index;
  gint ret;

  /* requeue buffers that are writable again */
  g_mutex_lock (v4l2camsrc->driver->device_mutex);
  gst_v4l2_buffer_pool_update (v4l2camsrc->pool, v4l2camsrc);
  g_mutex_unlock (v4l2camsrc->driver->device_mutex);

  memset (&buffer, 0x00, sizeof (buffer));
  buffer.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  buffer.memory = V4L2_MEMORY_USERPTR;

  for (;;) {
    ret = gst_poll_wait (v4l2camsrc->poll, V4L2CAMSRC_POLL_TIMEOUT);
    if (G_UNLIKELY (ret < 0)) {
      if (errno == EBUSY)
        goto stopped;
      if (errno != EAGAIN && errno != EINTR)
        goto select_error;
    }

    if (G_UNLIKELY (ret == 0)) {
      goto timeout;
    }

    g_mutex_lock (v4l2camsrc->driver->device_mutex);
    ret = ioctl (v4l2camsrc->video_fd, VIDIOC_DQBUF, &buffer);
    g_mutex_unlock (v4l2camsrc->driver->device_mutex);

    if (ret >= 0)
      break;

    GST_WARNING_OBJECT (v4l2camsrc,
        "problem grabbing frame %d (ix=%d), trials=%d, pool-ct=%d, buf.flags=%d",
        buffer.sequence, buffer.index, trials,
        GST_MINI_OBJECT_REFCOUNT (v4l2camsrc->pool), buffer.flags);

    /* if the sync() got interrupted, we can retry */
    switch (errno) {
      case EAGAIN:
        GST_WARNING_OBJECT (v4l2camsrc,
            "Non-blocking I/O has been selected using O_NONBLOCK and"
            " no buffer was in the outgoing queue. device %s",
            v4l2camsrc->videodev);
        break;
      case EINVAL:
        goto einval;
      case ENOMEM:
        goto enomem;
      case EIO:
        GST_INFO_OBJECT (v4l2camsrc,
            "VIDIOC_DQBUF failed due to an internal error."
            " Can also indicate temporary problems like signal loss."
            " Note the driver might dequeue an (empty) buffer despite"
            " returning an error, or even stop capturing."
            " device %s", v4l2camsrc->videodev);
        /* have we de-queued a buffer ? */
        if (!(buffer.flags & (V4L2_BUF_FLAG_QUEUED | V4L2_BUF_FLAG_DONE))) {
          /* this fails
             if ((buffer.index >= 0) && (buffer.index < v4l2camsrc->breq.count)) {
             GST_DEBUG_OBJECT (v4l2camsrc, "reenqueing buffer (ix=%ld)", buffer.index);
             gst_v4l2camsrc_queue_frame (v4l2camsrc, buffer.index);
             }
             else {
           */
          GST_DEBUG_OBJECT (v4l2camsrc, "reenqueing buffer");
          /* FIXME: this is not a good idea, as drivers usualy return the buffer
           * with index-number set to 0, thus the re-enque will fail unless it
           * was incidentialy 0.
           * We could try to re-enque all buffers without handling the ioctl
           * return.
           */
          /*
             if (ioctl (v4l2camsrc->video_fd, VIDIOC_QBUF, &buffer) < 0) {
             goto qbuf_failed;
             }
           */
          /*} */
        }
        break;
      case EINTR:
        GST_WARNING_OBJECT (v4l2camsrc,
            "could not sync on a buffer on device %s",
            v4l2camsrc->videodev);
        break;
      default:
        GST_WARNING_OBJECT (v4l2camsrc,
            "Grabbing frame got interrupted on %s. No expected reason.",
            v4l2camsrc->videodev);
        break;
    }

    /* check nr. of attempts to capture */
    if (--trials == -1) {
      goto too_many_trials;
    } else {
      memset (&buffer, 0x00, sizeof (buffer));
      buffer.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      buffer.memory = V4L2_MEMORY_USERPTR;
    }
  }

  g_mutex_lock (v4l2camsrc->pool->lock);

  index = buffer.index;

  /* get our GstBuffer with that index from the pool, if the buffer is
   * in use (NULL) we have a serious problem (should only be possible in mmap
   * case). */
  pool_buffer = GST_BUFFER (v4l2camsrc->pool->buffers[index]);

  if (G_UNLIKELY (pool_buffer == NULL))
    goto no_buffer;

  GST_LOG_OBJECT (v4l2camsrc, "grabbed buffer %p at index %d (refct = %d)",
    pool_buffer, index, GST_MINI_OBJECT_REFCOUNT (pool_buffer));

  /* ref the buffer and requeue, when if becomes writable again */
  v4l2camsrc->pool->buffers[index] = gst_buffer_ref (pool_buffer);
  v4l2camsrc->pool->num_live_buffers++;
  v4l2camsrc->pool->queued[index] = 0;

  /* if we are handing out the last buffer in the pool, we need to make a
   * copy and bring the buffer back in the pool. */
  need_copy = v4l2camsrc->always_copy
      || (v4l2camsrc->pool->num_live_buffers == v4l2camsrc->pool->buffer_count);

  g_mutex_unlock (v4l2camsrc->pool->lock);

  /* this can change at every frame, esp. with jpeg */
  GST_BUFFER_SIZE (pool_buffer) = buffer.bytesused;

  GST_BUFFER_OFFSET (pool_buffer) = v4l2camsrc->offset++;
  GST_BUFFER_OFFSET_END (pool_buffer) = v4l2camsrc->offset;

  /* timestamps, LOCK to get clock and base time. */
  {
    GstClock *clock;
    GstClockTime timestamp;

    GST_OBJECT_LOCK (v4l2camsrc);
    if ((clock = GST_ELEMENT_CLOCK (v4l2camsrc))) {
      /* we have a clock, get base time and ref clock */
      timestamp = GST_ELEMENT (v4l2camsrc)->base_time;
      gst_object_ref (clock);
    } else {
      /* no clock, can't set timestamps */
      timestamp = GST_CLOCK_TIME_NONE;
    }
    GST_OBJECT_UNLOCK (v4l2camsrc);

    if (clock) {
      GstClockTime latency;

      /* the time now is the time of the clock minus the base time */
      timestamp = gst_clock_get_time (clock) - timestamp;
      gst_object_unref (clock);

      latency =
          gst_util_uint64_scale_int (GST_SECOND, v4l2camsrc->fps_d,
          v4l2camsrc->fps_n);

      if (timestamp > latency)
        timestamp -= latency;
      else
        timestamp = 0;
    }

    /* FIXME: use the timestamp from the buffer itself! */
    GST_BUFFER_TIMESTAMP (pool_buffer) = timestamp;
  }

  if (G_UNLIKELY (need_copy)) {
    GST_DEBUG_OBJECT (v4l2camsrc, "copying buffer");
    *buf = gst_buffer_copy (pool_buffer);
    GST_BUFFER_FLAG_UNSET (*buf, GST_BUFFER_FLAG_READONLY);
    /* this will requeue */
    gst_buffer_unref (pool_buffer);
  } else {
    *buf = pool_buffer;
  }

  GST_LOG_OBJECT (v4l2camsrc, "grabbed frame %d (ix=%d), flags %08x, pool-ct=%d",
      buffer.sequence, buffer.index, buffer.flags,
      v4l2camsrc->pool->num_live_buffers);

  return GST_FLOW_OK;

  /* ERRORS */
select_error:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ, (NULL),
        ("select error %d: %s (%d)", ret, g_strerror (errno), errno));
    return GST_FLOW_ERROR;
  }
stopped:
  {
    GST_DEBUG ("stop called");
    return GST_FLOW_WRONG_STATE;
  }
einval:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, FAILED,
        ("Failed trying to get video frames from device '%s'.",
            v4l2camsrc->videodev),
        ("The buffer type is not supported, or the index is out of bounds,"
                " or no buffers have been allocated yet, or the userptr"
                " or length are invalid. device %s",
            v4l2camsrc->videodev));
    return GST_FLOW_ERROR;
  }
enomem:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, FAILED,
        ("Failed trying to get video frames from device '%s'. Not enough memory.",
         v4l2camsrc->videodev), ("insufficient memory to enqueue a user "
             "pointer buffer. device %s.", v4l2camsrc->videodev));
    return GST_FLOW_ERROR;
  }
too_many_trials:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, FAILED,
        ("Failed trying to get video frames from device '%s'.",
            v4l2camsrc->videodev),
        ("Failed after %d tries. device %s. system error: %s",
            NUM_TRIALS, v4l2camsrc->videodev, g_strerror (errno)));
    return GST_FLOW_ERROR;
  }
no_buffer:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, FAILED,
        ("Failed trying to get video frames from device '%s'.",
            v4l2camsrc->videodev),
        ("No free buffers found in the pool at index %d.", index));
    g_mutex_unlock (v4l2camsrc->pool->lock);
    return GST_FLOW_ERROR;  }
/*
qbuf_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, WRITE,
        ("Could not exchange data with device '%s'.",
            v4l2camsrc->videodev),
        ("Error queueing buffer on device %s. system error: %s",
            v4l2camsrc->videodev, g_strerror (errno)));
    return GST_FLOW_ERROR;
  }
*/
timeout:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, FAILED,
        ("Timeout when trying to get video frames from device '%s'.",
         v4l2camsrc->videodev), NULL);
    return GST_FLOW_ERROR;
  }
}

static gboolean
fractions_are_equal (gint num1, gint den1, gint num2, gint den2)
{
  GValue fraction1 = { 0, }, fraction2 = {
  0,};

  g_value_init (&fraction1, GST_TYPE_FRACTION);
  g_value_init (&fraction2, GST_TYPE_FRACTION);
  gst_value_set_fraction (&fraction1, num1, den1);
  gst_value_set_fraction (&fraction2, num2, den2);
  /* we know we don't have to unset the values in this case */
  return (gst_value_compare (&fraction1, &fraction2) == GST_VALUE_EQUAL);
}

/******************************************************
 * gst_v4l2camsrc_set_capture():
 *   set capture parameters
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_set_capture (GstV4l2CamSrc * v4l2camsrc, guint32 pixelformat,
    guint32 width, guint32 height, guint fps_n, guint fps_d)
{
  gint fd = v4l2camsrc->video_fd;
  struct v4l2_format format;
  struct v4l2_streamparm stream;

  GST_DEBUG_OBJECT (v4l2camsrc, "Setting capture format to %dx%d, format "
      "%" GST_FOURCC_FORMAT, width, height, GST_FOURCC_ARGS (pixelformat));

  GST_V4L2_CHECK_OPEN (v4l2camsrc);
  GST_V4L2_CHECK_NOT_ACTIVE (v4l2camsrc);

  memset (&format, 0x00, sizeof (struct v4l2_format));
  format.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

  if (ioctl (fd, VIDIOC_G_FMT, &format) < 0)
    goto get_fmt_failed;

  format.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  format.fmt.pix.width = width;
  format.fmt.pix.height = height;
  format.fmt.pix.pixelformat = pixelformat;
  /* request whole frames; change when gstreamer supports interlaced video
   * (INTERLACED mode returns frames where the fields have already been
   *  combined, there are other modes for requesting fields individually) */
  format.fmt.pix.field = V4L2_FIELD_INTERLACED;

  if (ioctl (fd, VIDIOC_S_FMT, &format) < 0) {
    if (errno != EINVAL)
      goto set_fmt_failed;

    /* try again with progressive video */
    format.fmt.pix.width = width;
    format.fmt.pix.height = height;
    format.fmt.pix.pixelformat = pixelformat;
    format.fmt.pix.field = V4L2_FIELD_NONE;
    if (ioctl (fd, VIDIOC_S_FMT, &format) < 0)
      goto set_fmt_failed;
  }

  if (format.fmt.pix.width != width || format.fmt.pix.height != height)
    goto invalid_dimensions;

  if (format.fmt.pix.pixelformat != pixelformat)
    goto invalid_pixelformat;

  /* Is there a reason we require the caller to always specify a framerate? */
  GST_LOG_OBJECT (v4l2camsrc, "Desired framerate: %u/%u", fps_n, fps_d);

  memset (&stream, 0x00, sizeof (struct v4l2_streamparm));
  stream.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (ioctl (fd, VIDIOC_G_PARM, &stream) < 0) {
    GST_ELEMENT_WARNING (v4l2camsrc, RESOURCE, SETTINGS,
        ("Could not get parameters on device '%s'",
            v4l2camsrc->videodev), GST_ERROR_SYSTEM);
    goto done;
  }

  /* Note: V4L2 provides the frame interval, we have the frame rate */
  if (fractions_are_equal (stream.parm.capture.timeperframe.numerator,
          stream.parm.capture.timeperframe.denominator, fps_d, fps_n)) {
    GST_LOG_OBJECT (v4l2camsrc, "Desired framerate already set, nothing to do");
    goto done;
  }

  /* We want to change the frame rate, so check whether we can. Some cheap USB
   * cameras don't have the capability */
  if ((stream.parm.capture.capability & V4L2_CAP_TIMEPERFRAME) == 0) {
    GST_DEBUG_OBJECT (v4l2camsrc, "Not setting framerate (not supported)");
    goto done;
  }

  GST_LOG_OBJECT (v4l2camsrc, "Setting framerate to %u/%u", fps_n, fps_d);

  /* Note: V4L2 wants the frame interval, we have the frame rate */
  stream.parm.capture.timeperframe.numerator = fps_d;
  stream.parm.capture.timeperframe.denominator = fps_n;

  /* some cheap USB cam's won't accept any change */
  if (ioctl (fd, VIDIOC_S_PARM, &stream) < 0) {
    GST_ELEMENT_WARNING (v4l2camsrc, RESOURCE, SETTINGS,
        ("Video input device did not accept new frame rate setting."),
        GST_ERROR_SYSTEM);
    goto done;
  }

  GST_INFO_OBJECT (v4l2camsrc, "Set frame interval to %u/%u", 
                   stream.parm.capture.timeperframe.numerator,
                   stream.parm.capture.timeperframe.denominator);
done:
  v4l2camsrc->fps_n = fps_n;
  v4l2camsrc->fps_d = fps_d;

  if (ioctl (v4l2camsrc->video_fd, VIDIOC_CROPCAP, &v4l2camsrc->vcrop) < 0) {
    GST_DEBUG_OBJECT (v4l2camsrc, "crop not supported");
    v4l2camsrc->max_zoom_factor = 1.0;
    goto no_crop;
  }

  v4l2camsrc->crop_supported = TRUE;

  GST_DEBUG_OBJECT (v4l2camsrc, "got cropping bounds: x:%d, y:%d, w:%d, h:%d",
                    v4l2camsrc->vcrop.bounds.left,
                    v4l2camsrc->vcrop.bounds.top,
                    v4l2camsrc->vcrop.bounds.width,
                    v4l2camsrc->vcrop.bounds.height);

  GST_DEBUG_OBJECT (v4l2camsrc, "cropping defrect: x:%d, y:%d, w:%d, h:%d",
                    v4l2camsrc->vcrop.defrect.left,
                    v4l2camsrc->vcrop.defrect.top,
                    v4l2camsrc->vcrop.defrect.width,
                    v4l2camsrc->vcrop.defrect.height);

  if (gst_v4l2camsrc_detect_max_zoom (v4l2camsrc,
        &v4l2camsrc->max_zoom_factor) == FALSE)
  {
    v4l2camsrc->max_zoom_factor = 1.0;
  }

  v4l2camsrc->photoconf.zoom =
      CLAMP (v4l2camsrc->photoconf.zoom, 1.0, v4l2camsrc->max_zoom_factor);

  gst_v4l2camsrc_update_cropping (v4l2camsrc, width, height,
                                  v4l2camsrc->photoconf.zoom);

no_crop:

  return TRUE;

  /* ERRORS */
get_fmt_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
        ("Device '%s' does not support video capture",
            v4l2camsrc->videodev),
        ("Call to G_FMT failed: (%s)", g_strerror (errno)));
    return FALSE;
  }
set_fmt_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
        ("Device '%s' cannot capture at %dx%d",
            v4l2camsrc->videodev, width, height),
        ("Call to S_FMT failed for %" GST_FOURCC_FORMAT " @ %dx%d: %s",
            GST_FOURCC_ARGS (pixelformat), width, height, g_strerror (errno)));
    return FALSE;
  }
invalid_dimensions:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
        ("Device '%s' cannot capture at %dx%d",
            v4l2camsrc->videodev, width, height),
        ("Tried to capture at %dx%d, but device returned size %dx%d",
            width, height, format.fmt.pix.width, format.fmt.pix.height));
    return FALSE;
  }
invalid_pixelformat:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, SETTINGS,
        ("Device '%s' cannot capture in the specified format",
            v4l2camsrc->videodev),
        ("Tried to capture in %" GST_FOURCC_FORMAT
            ", but device returned format" " %" GST_FOURCC_FORMAT,
            GST_FOURCC_ARGS (pixelformat),
            GST_FOURCC_ARGS (format.fmt.pix.pixelformat)));
    return FALSE;
  }
}

/******************************************************
 * gst_v4l2camsrc_capture_init():
 *   initialize the capture system
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_capture_init (GstV4l2CamSrc * v4l2camsrc, GstCaps * caps)
{
  gint fd = v4l2camsrc->video_fd;
  struct v4l2_requestbuffers breq;

  memset (&breq, 0, sizeof (struct v4l2_requestbuffers));

  GST_DEBUG_OBJECT (v4l2camsrc, "initializing the capture system");

  GST_V4L2_CHECK_OPEN (v4l2camsrc);
  GST_V4L2_CHECK_NOT_ACTIVE (v4l2camsrc);

  if (!(v4l2camsrc->vcap.capabilities & V4L2_CAP_STREAMING))
    goto no_capture_method;

  GST_DEBUG_OBJECT (v4l2camsrc, "STREAMING, requesting %d CAPTURE buffers",
      v4l2camsrc->num_buffers);

  breq.count = v4l2camsrc->num_buffers;
  breq.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  breq.memory = V4L2_MEMORY_USERPTR;

  if (ioctl (fd, VIDIOC_REQBUFS, &breq) < 0)
    goto reqbufs_failed;

  GST_LOG_OBJECT (v4l2camsrc, " count:  %u", breq.count);
  GST_LOG_OBJECT (v4l2camsrc, " type:   %d", breq.type);
  GST_LOG_OBJECT (v4l2camsrc, " memory: %d", breq.memory);

  if (breq.count < GST_V4L2_MIN_BUFFERS)
    goto no_buffers;

  if (v4l2camsrc->num_buffers != breq.count) {
    GST_WARNING_OBJECT (v4l2camsrc, "using %u buffers instead", breq.count);
    v4l2camsrc->num_buffers = breq.count;
    g_object_notify (G_OBJECT (v4l2camsrc), "queue-size");
  }

  /* Map the buffers */
  GST_LOG_OBJECT (v4l2camsrc, "initiating buffer pool");

  if (!(v4l2camsrc->pool = gst_v4l2camsrc_buffer_pool_new (v4l2camsrc, fd, caps)))
    goto buffer_pool_new_failed;

  GST_INFO_OBJECT (v4l2camsrc, "capturing buffers");

  GST_V4L2_SET_ACTIVE (v4l2camsrc);

  return TRUE;

  /* ERRORS */
reqbufs_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
        ("Could not get buffers from device '%s'.",
            v4l2camsrc->videodev),
        ("error requesting %d buffers: %s",
            v4l2camsrc->num_buffers, g_strerror (errno)));
    return FALSE;
  }
no_buffers:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
        ("Could not get enough buffers from device '%s'.",
            v4l2camsrc->videodev),
        ("we received %d from device '%s', we want at least %d",
            breq.count, v4l2camsrc->videodev, GST_V4L2_MIN_BUFFERS));
    return FALSE;
  }
buffer_pool_new_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
        ("Could not map buffers from device '%s'",
            v4l2camsrc->videodev),
        ("Failed to create buffer pool: %s", g_strerror (errno)));
    return FALSE;
  }
no_capture_method:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, READ,
        ("The driver of device '%s' does not support streaming. ",
         v4l2camsrc->videodev), (NULL));
    return FALSE;
  }
}


/******************************************************
 * gst_v4l2camsrc_capture_start():
 *   start streaming capture
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_capture_start (GstV4l2CamSrc * v4l2camsrc)
{
  gint type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  gint fd = v4l2camsrc->video_fd;

  GST_DEBUG_OBJECT (v4l2camsrc, "starting the capturing");
  //GST_V4L2_CHECK_OPEN (v4l2camsrc);
  GST_V4L2_CHECK_ACTIVE (v4l2camsrc);

  v4l2camsrc->quit = FALSE;

  if (!gst_v4l2camsrc_buffer_pool_activate (v4l2camsrc->pool, v4l2camsrc))
    goto pool_activate_failed;

  /* Are we going to capture one high-quality frame only? */
  GST_V4L2CAM_STATE_LOCK (v4l2camsrc);
  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE) {
    guint32 flash_intensity;

    v4l2camsrc->driver->get_flash_intensity (v4l2camsrc->driver,
                                             &flash_intensity);

    /* For still capture privacy light must be on during the capture phase */
    v4l2camsrc->driver->set_privacy_light (v4l2camsrc->driver, TRUE);

    switch (v4l2camsrc->photoconf.flash_mode) {
      case GST_PHOTOGRAPHY_FLASH_MODE_OFF:
        break;

      case GST_PHOTOGRAPHY_FLASH_MODE_AUTO:
        /* FIXME: What is the threshold value? */
        if (flash_intensity <= 10)
          break;

      case GST_PHOTOGRAPHY_FLASH_MODE_ON:
      case GST_PHOTOGRAPHY_FLASH_MODE_FILL_IN:
      case GST_PHOTOGRAPHY_FLASH_MODE_RED_EYE:
      default:
        v4l2camsrc->driver->set_flash_intensity (v4l2camsrc->driver,
            flash_intensity);
        /* FIXME: What is a good timeout value? */
        v4l2camsrc->driver->set_flash_duration (v4l2camsrc->driver, 10000000);
        v4l2camsrc->driver->set_flash (v4l2camsrc->driver, TRUE);
    }
  }
  else if (v4l2camsrc->capture_mode == GST_V4L2CAMSRC_CAPTURE_MODE_VIDEO) {
    /* For video recording privacy light must be on all the time */
    v4l2camsrc->driver->set_privacy_light (v4l2camsrc->driver, TRUE);
  }
  GST_V4L2CAM_STATE_UNLOCK (v4l2camsrc);

  if (ioctl (fd, VIDIOC_STREAMON, &type) < 0)
    goto streamon_failed;

  v4l2camsrc->is_capturing = TRUE;

  return TRUE;

  /* ERRORS */
pool_activate_failed:
  {
    /* already errored */
    return FALSE;
  }
streamon_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, OPEN_READ,
        ("Error starting streaming capture from device '%s'.",
            v4l2camsrc->videodev), GST_ERROR_SYSTEM);
    return FALSE;
  }
}

/******************************************************
 * gst_v4l2camsrc_capture_stop():
 *   stop streaming capture
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_capture_stop (GstV4l2CamSrc * v4l2camsrc)
{
  gint type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

  GST_DEBUG_OBJECT (v4l2camsrc, "stopping capturing");

  if (!GST_V4L2_IS_OPEN (v4l2camsrc)) {
    goto done;
  }
  if (!GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    goto done;
  }

  /* we actually need to sync on all queued buffers but not
   * on the non-queued ones */
  if (ioctl (v4l2camsrc->video_fd, VIDIOC_STREAMOFF, &type) < 0)
    goto streamoff_failed;

  GST_DEBUG_OBJECT (v4l2camsrc, "STREAMOFF called");

  /* if we were in photo capturing phase, make sure that privacy light
   * and flash led is turned off */
  GST_V4L2CAM_STATE_LOCK (v4l2camsrc);
  if (v4l2camsrc->photo_capture_phase == GST_V4L2PHOTO_CAPTURE_DONE) {
    if (v4l2camsrc->photoconf.flash_mode != GST_PHOTOGRAPHY_FLASH_MODE_OFF) {
      v4l2camsrc->driver->set_flash (v4l2camsrc->driver, FALSE);
    }
  }
  GST_V4L2CAM_STATE_UNLOCK (v4l2camsrc);

  v4l2camsrc->driver->set_privacy_light (v4l2camsrc->driver, FALSE);

done:

  /* make an optional pending wait stop */
  v4l2camsrc->quit = TRUE;
  v4l2camsrc->is_capturing = FALSE;

  return TRUE;

  /* ERRORS */
streamoff_failed:
  {
    GST_ELEMENT_ERROR (v4l2camsrc, RESOURCE, CLOSE,
        ("Error stopping streaming capture from device '%s'.",
            v4l2camsrc->videodev), GST_ERROR_SYSTEM);
    return FALSE;
  }
}

/******************************************************
 * gst_v4l2camsrc_capture_deinit():
 *   deinitialize the capture system
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_capture_deinit (GstV4l2CamSrc * v4l2camsrc)
{
  GST_DEBUG_OBJECT (v4l2camsrc, "deinitting capture system");

  if (!GST_V4L2_IS_OPEN (v4l2camsrc)) {
    return TRUE;
  }
  if (!GST_V4L2_IS_ACTIVE (v4l2camsrc)) {
    return TRUE;
  }

  if (v4l2camsrc->pool) {
    gst_v4l2camsrc_buffer_pool_destroy (v4l2camsrc->pool, v4l2camsrc);
    v4l2camsrc->pool = NULL;
  }

  GST_V4L2_SET_INACTIVE (v4l2camsrc);

  return TRUE;
}

/*
 */
static gboolean
gst_v4l2camsrc_get_nearest_size (GstV4l2CamSrc * v4l2camsrc, guint32 pixelformat,
    gint * width, gint * height)
{
  struct v4l2_format fmt;
  int fd;
  int r;

  g_return_val_if_fail (width != NULL, FALSE);
  g_return_val_if_fail (height != NULL, FALSE);

  GST_LOG_OBJECT (v4l2camsrc,
      "getting nearest size to %dx%d with format %" GST_FOURCC_FORMAT,
      *width, *height, GST_FOURCC_ARGS (pixelformat));

  fd = v4l2camsrc->video_fd;

  /* get size delimiters */
  memset (&fmt, 0, sizeof (fmt));
  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  fmt.fmt.pix.width = *width;
  fmt.fmt.pix.height = *height;
  fmt.fmt.pix.pixelformat = pixelformat;
  fmt.fmt.pix.field = V4L2_FIELD_INTERLACED;

  r = ioctl (fd, VIDIOC_TRY_FMT, &fmt);
  if (r < 0 && errno == EINVAL) {
    /* try again with progressive video */
    fmt.fmt.pix.width = *width;
    fmt.fmt.pix.height = *height;
    fmt.fmt.pix.pixelformat = pixelformat;
    fmt.fmt.pix.field = V4L2_FIELD_NONE;
    r = ioctl (fd, VIDIOC_TRY_FMT, &fmt);
  }

  if (r < 0) {
    /* The driver might not implement TRY_FMT, in which case we will try
       S_FMT to probe */
    if (errno != ENOTTY)
      return FALSE;

    /* Only try S_FMT if we're not actively capturing yet, which we shouldn't
       be, because we're still probing */
    if (GST_V4L2_IS_ACTIVE (v4l2camsrc))
      return FALSE;

    GST_LOG_OBJECT (v4l2camsrc,
        "Failed to probe size limit with VIDIOC_TRY_FMT, trying VIDIOC_S_FMT");

    fmt.fmt.pix.width = *width;
    fmt.fmt.pix.height = *height;

    r = ioctl (fd, VIDIOC_S_FMT, &fmt);
    if (r < 0 && errno == EINVAL) {
      /* try again with progressive video */
      fmt.fmt.pix.width = *width;
      fmt.fmt.pix.height = *height;
      fmt.fmt.pix.pixelformat = pixelformat;
      fmt.fmt.pix.field = V4L2_FIELD_NONE;
      r = ioctl (fd, VIDIOC_S_FMT, &fmt);
    }

    if (r < 0)
      return FALSE;
  }

  GST_LOG_OBJECT (v4l2camsrc,
      "got nearest size %dx%d", fmt.fmt.pix.width, fmt.fmt.pix.height);

  *width = fmt.fmt.pix.width;
  *height = fmt.fmt.pix.height;

  return TRUE;
}


/******************************************************
 * gst_v4l2camsrc_set_crop():
 *   set cropping bounds
 * return value: TRUE on success, FALSE on error
 ******************************************************/
static gboolean
gst_v4l2camsrc_set_crop (GstV4l2CamSrc *v4l2camsrc,
                         gint x, gint y, gint w, gint h)
{
  gboolean ret = FALSE;
  struct v4l2_crop scrop;
  struct v4l2_crop gcrop;
  gint cx, cy, cw, ch;
  int fd;

  fd = v4l2camsrc->video_fd;
  cx = v4l2camsrc->vcrop.bounds.left;
  cy = v4l2camsrc->vcrop.bounds.top;
  cw = v4l2camsrc->vcrop.bounds.width;
  ch = v4l2camsrc->vcrop.bounds.height;

  memset (&scrop, 0, sizeof (scrop));
  scrop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

  /* Make sure that cropping area stays inside the crop boundaries */
  scrop.c.left = x > cx ? x : cx;
  scrop.c.top = y > cy ? y : cy;
  scrop.c.width = (scrop.c.left+w) < (cx+cw) ? w : cx+cw-scrop.c.left;
  scrop.c.height = (scrop.c.top+h) < (cy+ch) ? h : cy+ch-scrop.c.top;

  GST_DEBUG_OBJECT (v4l2camsrc,
                    "Crop request: left = %d, top = %d, w = %d, h = %d",
                     scrop.c.left, scrop.c.top, scrop.c.width, scrop.c.height);

  g_mutex_lock (v4l2camsrc->driver->device_mutex);

  if (-1 == ioctl (fd, VIDIOC_S_CROP, &scrop) && errno != EINVAL) {
    goto s_crop_failed;
  }

  if (-1 == ioctl (fd, VIDIOC_G_CROP, &gcrop)) {
    goto g_crop_failed;
  }

  GST_DEBUG_OBJECT (v4l2camsrc,
                   "Crop selected: left = %d, top = %d, w = %d, h = %d",
                   gcrop.c.left, gcrop.c.top, gcrop.c.width, gcrop.c.height);

#if 0
  if (gcrop.c.left != scrop.c.left || gcrop.c.top != scrop.c.top ||
      gcrop.c.width != scrop.c.width || gcrop.c.height != scrop.c.height)
  {
    goto crop_not_supported;
  }
#endif

  ret = TRUE;

done:
  g_mutex_unlock (v4l2camsrc->driver->device_mutex);

  return ret;

/* ERRORS */
s_crop_failed:
  GST_DEBUG_OBJECT (v4l2camsrc, "VIDIOC_S_CROP not supported");
  goto done;

g_crop_failed:
  GST_DEBUG_OBJECT (v4l2camsrc, "VIDIOC_G_CROP not supported");
  goto done;

#if 0
crop_not_supported:
  GST_DEBUG_OBJECT (v4l2camsrc, "Given crop value not accepted");
  goto done;
#endif
}


/******************************************************
 * gst_v4l2camsrc_set_cropping():
 *   update cropping area according to  width, height and zoom factors.
 * return value: TRUE on success, FALSE on error
 ******************************************************/
gboolean
gst_v4l2camsrc_update_cropping (GstV4l2CamSrc *v4l2camsrc, gint width,
                                gint height, gfloat zoom)
{
  gfloat sensor_ar, reso_ar;
  gint crop_x, crop_y, crop_w, crop_h;

  g_return_val_if_fail (width != 0, FALSE);
  g_return_val_if_fail (height != 0, FALSE);
  g_return_val_if_fail (zoom != 0, FALSE);

  if (zoom < 1.0 || zoom > v4l2camsrc->max_zoom_factor) {
    GST_DEBUG_OBJECT (v4l2camsrc, "invalid zoom = %.2f", zoom);
    return FALSE;
  }
  if (!v4l2camsrc->crop_supported) {
    GST_DEBUG_OBJECT (v4l2camsrc, "crop not supported");
    return FALSE;
  }

  sensor_ar = v4l2camsrc->vcrop.defrect.width /
              v4l2camsrc->vcrop.defrect.height;
  reso_ar = width / height;

  if (sensor_ar > reso_ar) {
    crop_w = (width * v4l2camsrc->vcrop.defrect.height / height) / zoom;
    crop_h = MAX (v4l2camsrc->vcrop.defrect.height, height) / zoom;
  }
  else {
    crop_w = MAX (v4l2camsrc->vcrop.defrect.width, width) / zoom;
    crop_h = (height * v4l2camsrc->vcrop.defrect.width / width) / zoom;
  }

  crop_x = ABS ((v4l2camsrc->vcrop.defrect.width - crop_w)  / 2);
  crop_y = ABS ((v4l2camsrc->vcrop.defrect.height - crop_h)  / 2);

  GST_LOG_OBJECT (v4l2camsrc, "set cropping: x: %d, y: %d, w: %d, h: %d",
                  crop_x, crop_y, crop_w, crop_h);

  return gst_v4l2camsrc_set_crop (v4l2camsrc, crop_x, crop_y, crop_w, crop_h);
}


/******************************************************
 * gst_v4l2camsrc_detect_max_zoom();
 *   find out the maximum zoom factor for the camera hardware.
 * return value: TRUE on success, FALSE on error
 ******************************************************/
static gboolean
gst_v4l2camsrc_detect_max_zoom (GstV4l2CamSrc *v4l2camsrc, gfloat *max_zoom)
{
  struct v4l2_cropcap vcrop;
  struct v4l2_crop scrop;
  struct v4l2_crop gcrop;
  gboolean found = FALSE;
  gboolean ret = FALSE;

  int fd = v4l2camsrc->video_fd;
  float zoomfactor, tmp;
  guint ow, oh;

  GST_DEBUG_OBJECT (v4l2camsrc, "Finding maximum zoom level");

  if (ioctl (fd, VIDIOC_CROPCAP, &vcrop) < 0) {
    goto cropcap_failed;
  }

  zoomfactor = 2.0;
  ow = oh = G_MAXINT;

  memset (&scrop, 0, sizeof (scrop));
  scrop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

  g_mutex_lock (v4l2camsrc->driver->device_mutex);

  do {
    scrop.c.width  = vcrop.defrect.width / zoomfactor;
    scrop.c.height = vcrop.defrect.height / zoomfactor;
    scrop.c.left   = (vcrop.defrect.width - scrop.c.width) / 2;
    scrop.c.top    = (vcrop.defrect.height - scrop.c.height) / 2;

    GST_LOG_OBJECT (v4l2camsrc, "Checking for zoom factor %f", zoomfactor);

    GST_LOG_OBJECT (v4l2camsrc,
                    "Setting CROP: left = %d, top = %d, w = %d, h = %d",
                    scrop.c.left, scrop.c.top, scrop.c.width, scrop.c.height);

    if (-1 == ioctl (fd, VIDIOC_S_CROP, &scrop) && errno != EINVAL) {
      goto s_crop_failed;
    }

    if (-1 == ioctl (fd, VIDIOC_G_CROP, &gcrop)) {
      goto g_crop_failed;
    }

    GST_LOG_OBJECT (v4l2camsrc,
                    "Got crop: left = %d, top = %d, w = %d, h = %d",
                    gcrop.c.left, gcrop.c.top, gcrop.c.width, gcrop.c.height);

    tmp = (float) vcrop.defrect.height / gcrop.c.height;
    GST_LOG_OBJECT (v4l2camsrc, "Found zoom (height): %f", tmp);

    tmp = (float) vcrop.defrect.width / gcrop.c.width;
    GST_LOG_OBJECT (v4l2camsrc, "Found zoom (width): %f", tmp);

    if (gcrop.c.width < ow && gcrop.c.height < oh) {
      ow = gcrop.c.width;
      oh = gcrop.c.height;
      zoomfactor += 0.5;
    }
    else {
      found = TRUE;
    }
  }
  while (!found);

  *max_zoom = zoomfactor - 0.5;
  GST_DEBUG_OBJECT (v4l2camsrc, "Found maximum zoom: %f", *max_zoom);

  ret = TRUE;

done:
  g_mutex_unlock (v4l2camsrc->driver->device_mutex);

  return ret;

/* ERRORS */
cropcap_failed:
  GST_DEBUG_OBJECT (v4l2camsrc, "VIDIOC_CROPCAP not supported");
  goto done;

s_crop_failed:
  GST_DEBUG_OBJECT (v4l2camsrc, "VIDIOC_S_CROP not supported");
  goto done;

g_crop_failed:
  GST_DEBUG_OBJECT (v4l2camsrc, "VIDIOC_G_CROP not supported");
  goto done;
}
