/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <bfd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include "compress.h"
#include "list.h"
#include "options.h"
#include "trace_file.h"

static int process_data ()
{
	t_trace_file *trace_data;
	int ret;

	trace_data = calloc(1, sizeof(t_trace_file));
	if (trace_data == NULL) {
		fprintf(stderr, "error: could not allocate trace_data\n");
		return -ENOMEM;
	}
	/* open trace file for reading */
	if (!(trace_data->trace_file = fopen(arguments.trace_file, "r"))) {
		fprintf(stderr, "error: could not open trace file %s\n",
			arguments.trace_file);
		ret = -ENOENT;
		goto error;
	}

	/* initialize BFD library */
	bfd_init();

	/* initialize trace list */
	if ((ret = list_init(&trace_data->list)))
		goto error;

	/* initialize map list */
	if ((ret = maplist_init(&trace_data->maplist)))
		goto error;

	/* read data from input file */
	if ((ret = read_trace_file(trace_data)))
		goto error;

#if 0
	/* FIXME: implement verbose option properly */
	if ((arguments.verbose) &&
		(ret = list_print(trace_data)))
		goto error;
#endif

	/* Header file must be printed when the name resolution is not
	 * enabled. */
	if (!arguments.resolve)
		print_header_file(trace_data);

	/* filter freed allocations and/or leaks info */
	if ((arguments.leak || arguments.freed) &&
		(ret = list_filter(trace_data)))
		goto error;

	if (arguments.compress)
		compact(trace_data);

	if (arguments.blocks)
		blocks_sort(trace_data);
	else if (arguments.size)
		size_sort(trace_data);

	list_print(trace_data);

	/* close trace file */
	fclose(trace_data->trace_file);

	/* free map list allocations */
	if ((ret = maplist_free(&trace_data->maplist)))
		goto error;

	/* free list memory allocations */
	if ((ret = list_free(&trace_data->list)))
		goto error;

error:
	free(trace_data);

	return ret;
		
}

/* main program */
int main (int argc, char *argv[])
{
	int ret;

	/* process program options from parameters */
	if ((ret = process_options (argc, argv)))
		return ret;

	/* process data according to options specified */
	if ((ret = process_data ()))
		return ret;

	return 0;
}
