/*
 * This file is part of functracer-postproc.
 *
 * Copyright (C) 2008 by Nokia Corporation
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Based on backtrace code from libleaks.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "list.h"
#include "maps.h"
#include "options.h"
#include "trace_file.h"

static int size_compare(t_trace_file *trace_data, struct t_line* item1,
		      struct t_line* item2)
{
	if (item1->total_size < item2->total_size)
		return -1;
	if (item1->total_size > item2->total_size)
		return 1;
	return 0;
}

static int blocks_compare(t_trace_file *trace_data, struct t_line* item1,
		      struct t_line* item2)
{
	if (item1->blocks < item2->blocks)
		return -1;
	if (item1->blocks > item2->blocks)
		return 1;
	return 0;
}

static int backtrace_compare(t_trace_file *trace_data, struct t_line* item1,
		      struct t_line* item2)
{
	long offset1 = item1->offset;
	long offset2 = item2->offset;
	char line[LINE_MAX];
	t_address a, b;

	while (1) {
		a = read_backtrace_lines(trace_data, offset1, line,
					 sizeof(line));
		offset1 = ftell(trace_data->trace_file);
		memset(line, 0, sizeof(line));
		b = read_backtrace_lines(trace_data, offset2, line,
					 sizeof(line));
		offset2 = ftell(trace_data->trace_file);

		/* Some sopthisticated operations */
		if (a < b)
			return -1;
		if (a > b)
			return 1;
		if ( !a )
			break;
	}
	
	/* Well, they are equal - so return 0 */
	return 0;
}

void compact(t_trace_file *trace_data)
{
	t_line *current;
	t_list *list = &trace_data->list;
	int total_size, blocks;

	if (list_is_empty(list))
		return;

	list_sort(trace_data, backtrace_compare);

	total_size = blocks = 0;
	current = list->first->next;
	while (current && current->next) {
        	/* Set flag if that is the last or next item has another stack
		 * trace */
		if (current->type == TYPE_ALLOC)
			total_size += current->size;

		if (backtrace_compare(trace_data, current, current->next) == 0) {
			current->another_stack = 0;
			if (current->type != TYPE_FREE)
				blocks++;
		} else {
			current->total_size = total_size;
			current->blocks = blocks + 1;
			total_size = blocks = 0;
		}

		current = current->next;
	}

	/* Fill data for last item */
	if (current->type == TYPE_ALLOC)
		total_size += current->size;
	current->total_size = total_size;
	current->blocks = blocks + 1;
}

void size_sort(t_trace_file *trace_data)
{
	list_sort(trace_data, size_compare);
}

void blocks_sort(t_trace_file *trace_data)
{
	list_sort(trace_data, blocks_compare);
}
