/* vim: set ts=2 sw=2 cino= et: */
/*
 * This file is part of eds-backend-telepathy
 *
 * Copyright (C) 2008-2009 Nokia Corporation
 *   @author Rob Bradford
 *   @author Travis Reitter <travis.reitter@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <telepathy-glib/dbus.h>
#include <libmcclient/mc-profile.h>
#include <libedata-book/e-data-book-view.h>
#include <gio/gio.h>
#include <telepathy-glib/util.h>

#include "e-book-backend-tp.h"
#include "e-book-backend-tp-cl.h"
#include "e-book-backend-tp-contact.h"
#include "e-book-backend-tp-db.h"
#include "e-book-backend-tp-log.h"

G_DEFINE_TYPE (EBookBackendTp, e_book_backend_tp, E_TYPE_BOOK_BACKEND_SYNC);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), E_TYPE_BOOK_BACKEND_TP, EBookBackendTpPrivate))

struct _EBookBackendTpPrivate {
  EBookBackendTpCl *tpcl;
  GHashTable *handle_to_contact;
  GHashTable *name_to_contact;
  GHashTable *uid_to_contact;
  EBookBackendTpDb *tpdb;
  gboolean load_started; /* initial populate from database */
  gboolean members_ready; /* members ready to report to views */
  gulong members_ready_signal_id;
  GList *views;
  McAccount *account;
  McProfile *profile;
  const gchar *vcard_field;
  const gchar *profile_name;
  gboolean load_error; /* we cannot report errors back when asynchronously
                        * loading an account, so we have to use this hack */

  GHashTable *contacts_to_delete; /* contacts scheduled for deletion */
  GHashTable *contacts_to_update; /* contacts scheduled for update */
  GHashTable *contacts_to_add; /* contacts scheduled for addition */

  GHashTable *contacts_to_update_in_db;
  guint db_update_timer;

  /* When we receive a signal like aliases-changed we delay the update to
   * an idle callback, so:
   * 1. we avoid to starve the main loop
   * 2. if more signals arrive in sequence we generate the vcards only once
   */
  GHashTable *contacts_remotely_changed; /* the contacts that changed */
  guint contacts_remotely_changed_update_id; /* source id of the callback */
};

enum
{
  READY_SIGNAL = 0,
  MEMBERS_READY_SIGNAL = 1,
  LAST_SIGNAL
};

static guint32 signals[LAST_SIGNAL] = { 0 };

typedef enum {
    CONTACT_SORT_ORDER_FIRST_LAST,
    CONTACT_SORT_ORDER_LAST_FIRST,
    CONTACT_SORT_ORDER_NICKNAME
} ContactSortOrder;

/* Key used to store the sort order on a book view using g_object_set_data */
#define BOOK_VIEW_SORT_ORDER_DATA_KEY "tp-backend-contact-sort-order"

static gchar *
e_book_backend_tp_generate_uid (EBookBackendTp *backend, const gchar *name)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  gchar *tmp = NULL;
  gint count = 0;
  const gchar *account_name;

  account_name = priv->account->name;

  tmp = g_strdup_printf ("%s-%s", account_name, name);

  while (g_hash_table_lookup (priv->uid_to_contact, tmp))
  {
    count++;
    g_free (tmp);
    tmp = g_strdup_printf ("%s-%s-%d", account_name, name, count);
  }

  return tmp;
}

/* Update callbacks. Fired by signal emissions when something gets updated in
 * the EBookBackendTpCl
 */

static void
notify_remotely_updated_contacts (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = NULL;
  GHashTableIter iter;
  gpointer contact_pointer;
  EBookBackendTpContact *contact = NULL;
  GList *l = NULL;
  EContact *ec;

  priv = GET_PRIVATE (backend);

  if (!priv->views)
    return;

  g_hash_table_iter_init (&iter, priv->contacts_remotely_changed);
  while (g_hash_table_iter_next (&iter, NULL, &contact_pointer))
  {
    contact = contact_pointer;
    ec = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
        priv->profile_name);

    /* For each view we known about tell them about the contact */
    for (l = priv->views; l != NULL; l = l->next)
    {
      DEBUG ("notifying contact: %s", contact->name);
      e_data_book_view_notify_update ((EDataBookView *)l->data, ec);
    }

    g_object_unref (ec);
  }

  /* For all the views send completion of the set of contacts */
  for (l = priv->views; l != NULL; l = l->next)
  {
    e_data_book_view_notify_complete ((EDataBookView *)l->data, 
        GNOME_Evolution_Addressbook_Success);
  }
}

static void
notify_updated_contact (EBookBackendTp *backend, 
    EBookBackendTpContact *contact)
{
  EBookBackendTpPrivate *priv = NULL;
  EContact *ec;
  GList *l = NULL;

  priv = GET_PRIVATE (backend);

  if (!priv->views)
    return;

  ec = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
      priv->profile_name);

  /* For each view we known about tell them about the contact */
  for (l = priv->views; l != NULL; l = l->next)
  {
    DEBUG ("notifying contact: %s", contact->name);
    e_data_book_view_notify_update ((EDataBookView *)l->data, ec);
  }

  g_object_unref (ec);

  /* For all the views send completion of the set of contacts */
  for (l = priv->views; l != NULL; l = l->next)
  {
    e_data_book_view_notify_complete ((EDataBookView *)l->data, 
        GNOME_Evolution_Addressbook_Success);
  }
}

static void
flush_db_updates (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);

  GList *tmp_list, *l;
  GArray *contacts;

  GError *error = NULL;

  DEBUG ("flushing pending contacts to db");

  /* 
   * Because we can be called outside the db_update_timeout_cb and still need
   * to ensure a consistent state
   */
  if (priv->db_update_timer)
  {
    g_source_remove (priv->db_update_timer);
    priv->db_update_timer = 0;
  }

  /*
   * It's possible that a flush of the contacts info is requested after the
   * DB has been deleted, so we avoid to fail or emit criticals if this
   * happen.
   */
  if (!priv->tpdb)
  {
    DEBUG ("skipping flush as the database was deleted");

    g_hash_table_remove_all (priv->contacts_to_update_in_db);
    return;
  }      

  tmp_list = g_hash_table_get_values (priv->contacts_to_update_in_db);
  contacts = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));

  for (l = tmp_list; l; l = l->next)
  {
    g_array_append_val (contacts, l->data);
  }

  if (!e_book_backend_tp_db_update_contacts (priv->tpdb, contacts, &error))
  {
    g_critical ("Error whilst flushing pending contacts to db: %s",
        error ? error->message : "unknown error");
    g_clear_error (&error);
  }

  g_hash_table_remove_all (priv->contacts_to_update_in_db);

  g_array_free (contacts, TRUE);
  g_list_free (tmp_list);
}

static gboolean
db_update_timeout_cb (gpointer userdata)
{
  EBookBackendTp *backend = (EBookBackendTp *)userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);

  DEBUG ("db update timeout fired");

  /* Clear timer */
  priv->db_update_timer = 0;

  flush_db_updates (backend);

  /* False so this timeout doesn't occur again */
  return FALSE;
}

static void
request_db_update_remotely_updated_contacts (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GHashTableIter iter;
  gpointer contact_pointer;
  EBookBackendTpContact *contact;

  g_hash_table_iter_init (&iter, priv->contacts_remotely_changed);
  while (g_hash_table_iter_next (&iter, NULL, &contact_pointer))
  {
    contact = contact_pointer;

    if (g_hash_table_lookup (priv->contacts_to_update_in_db, contact->uid))
    {
      DEBUG ("Update requested for already scheduled contact: %s",
          contact->uid);
    } else {
      g_hash_table_insert (priv->contacts_to_update_in_db,
          g_strdup (contact->uid),
          e_book_backend_tp_contact_ref (contact));
      DEBUG ("Update scheduled for contact: %s",
          contact->uid);
    }
  }

  if (priv->db_update_timer > 0)
  {
    DEBUG ("Update timer already setup");
  } else {
    priv->db_update_timer = g_timeout_add_seconds (5, db_update_timeout_cb,
        backend);
  }
}

static gboolean
update_contacts_idle_cb (gpointer userdata)
{
  EBookBackendTp *backend = userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);

  DEBUG ("update remotely changed contacts");

  notify_remotely_updated_contacts (backend);
  request_db_update_remotely_updated_contacts (backend);

  g_hash_table_remove_all (priv->contacts_remotely_changed);
  priv->contacts_remotely_changed_update_id = 0;

  return FALSE;
}

static void
update_contacts (EBookBackendTp *backend, GArray *contacts)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact;
  guint i = 0;

  for (i = 0; i < contacts->len; i++)
  {
    contact = g_array_index (contacts, EBookBackendTpContact *, i);

    if (!g_hash_table_lookup (priv->contacts_remotely_changed, contact->uid))
    {
      DEBUG ("Update scheduled for contact: %s",
          contact->uid);
      g_hash_table_insert (priv->contacts_remotely_changed,
          g_strdup (contact->uid), e_book_backend_tp_contact_ref (contact));
    }
  }

  if (!priv->contacts_remotely_changed_update_id)
    priv->contacts_remotely_changed_update_id = g_idle_add_full (
        G_PRIORITY_LOW, update_contacts_idle_cb, backend, NULL);
}

static void
delete_contacts (EBookBackendTp *backend, GArray *contacts)
{
  EBookBackendTpPrivate *priv = NULL;
  EBookBackendTpContact *contact = NULL;
  guint i = 0;
  gchar *tmp = NULL;
  GArray *uids_to_delete = NULL;
  GList *l;

  priv = GET_PRIVATE (backend);

  g_return_if_fail (priv->tpdb);

  uids_to_delete = g_array_sized_new (TRUE, TRUE, sizeof (gchar *), 
      contacts->len);

  for (i = 0; i < contacts->len; i++)
  {
    contact = g_array_index (contacts, EBookBackendTpContact *, i);

    MESSAGE ("removing contact %s", contact->name);

    if (contact->handle > 0)
    {
      DEBUG ("removing from handle to contact mapping");
      g_hash_table_remove (priv->handle_to_contact, 
          GINT_TO_POINTER (contact->handle));
    }

    DEBUG ("ensure there are no pending changes for the contact");
    g_hash_table_remove (priv->contacts_remotely_changed, contact->uid);

    tmp = g_strdup (contact->uid);
    g_array_append_val (uids_to_delete, tmp);

    DEBUG ("removing from name to contact mapping");
    g_hash_table_remove (priv->name_to_contact, contact->name);
    DEBUG ("removing from uid to contact mapping");
    g_hash_table_remove (priv->uid_to_contact, contact->uid);
  }

  DEBUG ("removing contacts from database");
  e_book_backend_tp_db_remove_contacts (priv->tpdb, uids_to_delete, NULL);

  for (i = 0; i < uids_to_delete->len; i++)
  {
    tmp = g_array_index (uids_to_delete, gchar *, i);
    DEBUG ("notifying %s", tmp);

    /* It's possible that we already sent a notification for this if the
     * contact was removed by us and not by another client, but deletion
     * of contacts is rare enough that there is no real need to optimise
     * this. */
    for (l = priv->views; l != NULL; l = l->next)
    {
      e_data_book_view_notify_remove ((EDataBookView *)l->data, 
          tmp);
    }

    g_free (tmp);
  }

  /* For all the views send completion of the set of contacts */
  for (l = priv->views; l != NULL; l = l->next)
  {
    e_data_book_view_notify_complete ((EDataBookView *)l->data, 
        GNOME_Evolution_Addressbook_Success);
  }

  g_array_free (uids_to_delete, TRUE);
}

static void
tp_cl_aliases_changed_cb (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  guint i = 0;
  EBookBackendTpContact *contact_in = NULL;
  EBookBackendTpContact *contact = NULL;
  GArray *contacts_to_update = NULL;

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);

    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GINT_TO_POINTER (contact_in->handle));

    if (contact)
    {
      DEBUG ("updating alias for uid %s, handle %d and name %s from %s to %s",
          contact->uid, contact->handle, contact->name,
          contact->alias, contact_in->alias);
      g_free (contact->alias);
      contact->alias = g_strdup (contact_in->alias);

      if (contacts_to_update == NULL)
      {
        contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
      }

      g_array_append_val (contacts_to_update, contact);
    }
  }

  if (contacts_to_update)
  {
    update_contacts (backend, contacts_to_update);
    g_array_free (contacts_to_update, TRUE);
  }
}

static void
tp_cl_presence_changed_cb (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  guint i = 0;
  EBookBackendTpContact *contact_in = NULL;
  EBookBackendTpContact *contact = NULL;
  GArray *contacts_to_update = NULL;

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);

    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GINT_TO_POINTER (contact_in->handle));

    if (contact)
    {
      DEBUG ("updating status for uid %s, handle %d and name %s "
          "from %s;%s to %s;%s",
          contact->uid, contact->handle, contact->name,
          contact->status, contact->status_message, 
          contact_in->status, contact_in->status_message);

      g_free (contact->status);
      g_free (contact->status_message);

      contact->generic_status = contact_in->generic_status;
      contact->status = g_strdup (contact_in->status);
      contact->status_message = g_strdup (contact_in->status_message);

      if (contacts_to_update == NULL)
      {
        contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
      }

      g_array_append_val (contacts_to_update, contact);
    }
  }

  if (contacts_to_update)
  {
    update_contacts (backend, contacts_to_update);
    g_array_free (contacts_to_update, TRUE);
  }
}

static void
tp_cl_flags_changed (EBookBackendTpCl *tpcl, GArray *contacts, 
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact = NULL;
  EBookBackendTpContact *contact_in = NULL;
  GArray *contacts_to_update = NULL;
  guint i = 0;

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);

    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GINT_TO_POINTER (contact_in->handle));

    if (contact)
    {
      DEBUG ("updating flags for uid %s, handle %d and name %s",
          contact->uid, contact->handle, contact->name);

      contact->flags = contact_in->flags;

      if (contacts_to_update == NULL)
      {
        contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
      }

      g_array_append_val (contacts_to_update, contact);
    }
  }

  if (contacts_to_update)
  {
    update_contacts (backend, contacts_to_update);
    g_array_free (contacts_to_update, TRUE);
  }
}

static void
tp_cl_contacts_added (EBookBackendTpCl *tpcl, GArray *contacts, 
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = NULL;
  EBookBackendTpContact *contact = NULL;
  EBookBackendTpContact *contact_in = NULL;
  guint i = 0;
  GArray *contacts_to_update = NULL;
  GArray *contacts_to_add = NULL;
  GError *error = NULL;

  priv = GET_PRIVATE (userdata);

  g_return_if_fail (priv->tpdb);

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);
    contact = g_hash_table_lookup (priv->name_to_contact, 
        contact_in->name);

    if (contact)
    {
      DEBUG ("contact %s already known.", contact_in->name);

      /* Let's update the flags, handle and alias at the least */
      contact->flags = contact_in->flags;
      contact->handle = contact_in->handle;

      g_free (contact->alias);
      contact->alias = g_strdup (contact_in->alias);

      /* Clear the schedule add flag */
      contact->pending_flags &= ~SCHEDULE_ADD;
    } else {
      MESSAGE ("new contact found %s", contact_in->name);
      contact = e_book_backend_tp_contact_dup (contact_in);
      contact->uid = e_book_backend_tp_generate_uid (backend, contact->name);

      g_hash_table_insert (priv->name_to_contact, 
          g_strdup (contact->name),
          e_book_backend_tp_contact_ref (contact));

      /* Get rid of the initial reference. If we didn't do this here we'd need
       * to make sure that we ref'ed everything going into the update_contacts
       * array and then unref after.
       */
      e_book_backend_tp_contact_unref (contact);

      if (!contacts_to_add)
        contacts_to_add = g_array_new (TRUE, TRUE, sizeof (EBookBackendTp *));

      g_array_append_val (contacts_to_add, contact);
    }

    if (!contacts_to_update)
      contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTp *));

    g_array_append_val (contacts_to_update, contact);

    g_hash_table_insert (priv->handle_to_contact, 
        GINT_TO_POINTER (contact->handle),
        e_book_backend_tp_contact_ref (contact));
    g_hash_table_insert (priv->uid_to_contact, 
        g_strdup (contact->uid),
        e_book_backend_tp_contact_ref (contact));
  }

  if (contacts_to_add)
  {
    if (!e_book_backend_tp_db_add_contacts (priv->tpdb, contacts_to_update, &error))
    {
      g_critical ("Error when trying to save new contacts to database: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    }
    g_array_free (contacts_to_add, TRUE);
  }

  if (contacts_to_update)
  {
    update_contacts (backend, contacts_to_update);
    g_array_free (contacts_to_update, TRUE);
  }
}

static void
tp_cl_contacts_removed (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = NULL;
  EBookBackendTpContact *contact = NULL;
  EBookBackendTpContact *contact_in = NULL;
  GArray *contacts_to_remove = NULL;
  guint i = 0;

  priv = GET_PRIVATE (backend);

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);

    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GINT_TO_POINTER (contact_in->handle));

    if (contact)
    {
      MESSAGE ("told about removal of %s", contact->name);
      if (!contacts_to_remove)
        contacts_to_remove = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));

      /* clear the schedule delete flag */
      contact->pending_flags &= ~SCHEDULE_DELETE;
      g_array_append_val (contacts_to_remove, contact);
    } else {
      DEBUG ("Told about the removal of unknown contact (%s)",
          contact_in->name);
    }
  }

  if (contacts_to_remove)
  {
    delete_contacts (backend, contacts_to_remove);
    g_array_free (contacts_to_remove, TRUE);
  }
}

static void
tp_cl_avatar_tokens_changed_cb (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  guint i = 0;
  EBookBackendTpContact *contact_in;
  EBookBackendTpContact *contact = NULL;
  GArray *contacts_to_request;
  GError *error = NULL;
  gchar *avatar_path = NULL;
  GArray *contacts_to_update = NULL;

  contacts_to_update = g_array_new (TRUE, TRUE,
      sizeof (EBookBackendTpContact *));
  contacts_to_request = g_array_new (TRUE, TRUE, 
      sizeof (EBookBackendTpContact *));

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);
    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GUINT_TO_POINTER (contact_in->handle));

    if (!contact)
    {
      DEBUG ("Told about changed token for unknown contact handle: %d",
          contact_in->handle);
      continue;
    }

    /* If the contact_in->avatar_token is NULL (for instance because the
     * contact is offline) just use the one from the DB, else update the
     * avatar token */
    if (contact_in->avatar_token &&
        tp_strdiff (contact->avatar_token, contact_in->avatar_token))
    {
      g_free (contact->avatar_token);
      contact->avatar_token = g_strdup (contact_in->avatar_token);
    } 

    if (contact->avatar_token && contact->avatar_token[0] != '\0')
      avatar_path = g_build_filename (g_get_home_dir (), ".osso-abook",
          "avatars", contact->avatar_token, NULL);
    else
      avatar_path = NULL;

    if (avatar_path && !g_file_test (avatar_path, G_FILE_TEST_EXISTS))
    {
      g_array_append_val (contacts_to_request, contact);
    } else {
      g_array_append_val (contacts_to_update, contact);
    }

    g_free (avatar_path);
  }

  /* We need to notify about contacts which have just changed avatars to an
   * existing image file (the contacts changed to avatars which need to retrieve
   * data will be updated once the data is stored) */
  if (contacts_to_update->len > 0)
  {
    update_contacts (backend, contacts_to_update);
  }

  if (!e_book_backend_tp_cl_request_avatar_data (priv->tpcl, contacts_to_request, &error))
  {
    WARNING ("Error whilst requesting avatar data for changed tokens");
    g_clear_error (&error);
  }

  g_array_free (contacts_to_update, TRUE);
  g_array_free (contacts_to_request, TRUE);
}

typedef struct
{
  EBookBackendTp *backend;
  EBookBackendTpContact *contact;
} AvatarDataSavedClosure;

static void
avatar_data_saved_cb (GObject *source, GAsyncResult *res, 
    gpointer userdata)
{
  AvatarDataSavedClosure *closure = (AvatarDataSavedClosure *)userdata;
  EBookBackendTp *backend = closure->backend;
  EBookBackendTpContact *contact = closure->contact;
  GError *error = NULL;
  GArray *contacts;

  if (!g_file_replace_contents_finish (G_FILE (source), res, NULL, &error))
  {
    WARNING ("Error whilst writing to avatar file: %s",
        error ? error->message : "unknown error");
    g_clear_error (&error);
    goto done;
  }

  contacts = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
  g_array_append_val (contacts, contact);
  update_contacts (backend, contacts);
  g_array_free (contacts, TRUE);

done:
  g_object_unref (backend);
  e_book_backend_tp_contact_unref (contact);
  g_free (closure);
}

static void
tp_cl_avatar_data_changed_cb (EBookBackendTpCl *tpcl, 
    EBookBackendTpContact *contact_in, gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact;
  gchar *avatar_path;
  GFile *avatar_file;
  AvatarDataSavedClosure *closure;

  contact = g_hash_table_lookup (priv->handle_to_contact, 
      GUINT_TO_POINTER (contact_in->handle));

  if (!contact)
  {
    DEBUG ("Given avatar data for unknown contact with handle: %d",
        contact_in->handle);
    return;
  }

  g_free (contact->avatar_mime);
  contact->avatar_mime = g_strdup (contact_in->avatar_mime);

  g_free (contact->avatar_data);
  contact->avatar_data = g_memdup (contact_in->avatar_data, contact_in->avatar_len);

  contact->avatar_len = contact_in->avatar_len;

  avatar_path = g_build_filename (g_get_home_dir (), ".osso-abook", "avatars",
      contact->avatar_token, NULL);
  avatar_file = g_file_new_for_path (avatar_path);

  closure = g_new0 (AvatarDataSavedClosure, 1);
  closure->backend = g_object_ref (backend);
  closure->contact = e_book_backend_tp_contact_ref (contact);

  g_file_replace_contents_async (avatar_file,
      contact->avatar_data,
      contact->avatar_len,
      NULL,
      FALSE,
      G_FILE_CREATE_NONE,
      NULL,
      avatar_data_saved_cb,
      closure);
  g_object_unref (avatar_file);
  g_free (avatar_path);
}

static void
tp_cl_capabilities_changed_cb (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact;
  EBookBackendTpContact *contact_in;
  guint i = 0;
  GArray *contacts_to_update = NULL;

  DEBUG ("capabalities_changed_cb");

  contacts_to_update = g_array_sized_new (TRUE, TRUE, 
      sizeof (EBookBackendTpContact *), contacts->len);

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);
    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GUINT_TO_POINTER (contact_in->handle));

    if (!contact)
    {
      DEBUG ("Unknown contact with handle %d", 
          contact_in->handle);
      continue;
    }

    contact->capabilities = contact_in->capabilities;

    g_array_append_val (contacts_to_update, contact);
  }
  
  if (contacts_to_update->len > 0 )
  {
    update_contacts (backend, contacts_to_update);
  }

  g_array_free (contacts_to_update, TRUE);
}

static void 
tp_cl_contact_info_changed_cb (EBookBackendTpCl *tpcl, GArray *contacts,
    gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact;
  EBookBackendTpContact *contact_in;
  guint i = 0;
  GArray *contacts_to_update = NULL;
  
  DEBUG ("tp_cl_contact_info_changed_cb");

  contacts_to_update = g_array_sized_new (TRUE, TRUE, 
      sizeof (EBookBackendTpContact *), contacts->len);
  
  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);
    contact = g_hash_table_lookup (priv->handle_to_contact, 
        GUINT_TO_POINTER (contact_in->handle));

    if (!contact)
    {
      DEBUG ("Unknown contact with handle %d", 
          contact_in->handle);
      continue;
    }
    
    g_free (contact->contact_info);
    
    contact->contact_info = g_strdup (contact_in->contact_info);
    g_array_append_val (contacts_to_update, contact);
  }
  
  if (contacts_to_update->len > 0 )
  {
    update_contacts (backend, contacts_to_update);
  }

  g_array_free (contacts_to_update, TRUE);
}

/* The following code pretty much responsible for merging the state of
 * telepathy with our original imported data from the database and then later
 * reconciliating the other way. Hold on to your hats it's going to get pretty
 * hairy
*/

typedef struct
{
  EBookBackendTp *backend;
  GArray *contacts_to_add;
  GArray *contacts_to_update;
} GetMembersClosure;

static gboolean
run_update_contact (EBookBackendTp *backend, EBookBackendTpContact *contact)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  gboolean changed = FALSE;
  GError *error= NULL;

  g_object_ref (backend);
  e_book_backend_tp_contact_ref (contact);

  if (contact->pending_flags & SCHEDULE_UPDATE_FLAGS)
  {
    if (!e_book_backend_tp_cl_run_update_flags (priv->tpcl, contact, &error))
    {
      WARNING ("Error whilst updating contact flags: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    }

    /* Clear the flag. We don't want this to happen again */
    contact->pending_flags &= ~SCHEDULE_UPDATE_FLAGS;
    changed = TRUE;
  }

  if (contact->pending_flags & SCHEDULE_UNBLOCK)
  {
    if (e_book_backend_tp_cl_run_unblock_contact (priv->tpcl, contact, &error))
    {
      /* Clear the flag. We don't want this to happen again */
      contact->pending_flags &= ~SCHEDULE_UNBLOCK;
    }
    else
    {
      WARNING ("Error whilst unblocking contact: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);

      /* We couldn't unblock the contact now (maybe we are offline?), so we
       * will try the next time we connect. In the meantime we pretend to
       * not be in the deny list anymore so the UI can show the contact */
      contact->flags &= ~DENY;
    }

    changed = TRUE;
  }

  if (contact->pending_flags & SCHEDULE_UPDATE_MASTER_UID)
  {
    /* Clear the flag. We don't want this to happen again */
    contact->pending_flags &= ~SCHEDULE_UPDATE_MASTER_UID;
    changed = TRUE;
  }

  g_object_unref (backend);
  e_book_backend_tp_contact_unref (contact);

  return changed;
}

/* Given a contact list returns the flags for all the associated contact
 * lists, for instance ALL_FLAGS_FROM_CL(CL_SUBSCRIBE) returns
 * SUBSCRIBE | SUBSCRIBE_LOCAL_PENDING | SUBSCRIBE_REMOTE_PENDING */
#define ALL_FLAGS_FROM_CL(cl) ((1 << (cl)) | \
                               (1 << ((cl) + 1)) | \
                               (1 << ((cl) + 2)))

static gboolean
run_add_contact (EBookBackendTp *backend, EBookBackendTpContact *contact,
    GError **error_in)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  gboolean success;
  GError *error = NULL;

  e_book_backend_tp_return_val_with_error_if_fail (contact, FALSE, error_in);

  if (contact->flags & CONTACT_INVALID)
    /* We already know it's invalid, no need to test again */
    return TRUE;

  g_object_ref (backend);
  e_book_backend_tp_contact_ref (contact);

  success = e_book_backend_tp_cl_run_add_contact (priv->tpcl, contact, &error);

  if (!success)
  {
    if (error && error->domain == TP_ERRORS &&
        error->code == TP_ERROR_INVALID_HANDLE)
    {
      /* The contact has an invalid ID, in this case we want to keep the
       * contact in our address book with some UI element showing it's
       * invalid.
       * Just showing an error banner is impossible as if we are offline
       * the error banner will appear only when we go online */
      g_clear_error (&error);

      contact->flags |= CONTACT_INVALID;

      /* Remove flags that would do nothing on invalid contacts */
      contact->pending_flags &= ~SCHEDULE_UPDATE_FLAGS;
      /* Remove flags for list membership */
      contact->pending_flags &= ~(
            ALL_FLAGS_FROM_CL (CL_SUBSCRIBE) |
            ALL_FLAGS_FROM_CL (CL_PUBLISH) |
            ALL_FLAGS_FROM_CL (CL_ALLOW) |
            ALL_FLAGS_FROM_CL (CL_DENY) |
            ALL_FLAGS_FROM_CL (CL_STORED)
          );

      success = TRUE;
    }
    else
    {
      g_propagate_error (error_in, error);
    }
  }

  g_object_unref (backend);
  e_book_backend_tp_contact_unref (contact);

  return success;
}

/*
 * Phase 3:
 *
 * Here we apply pending changes to the roster that have been queued in the
 * database. We don't need the closure here. Everything we care about is in
 * the private structure.
 */
static gboolean
_sync_phase_3_idle_cb (gpointer userdata)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (userdata);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GList *l = NULL;
  GList *contacts_to_update = NULL;
  GList *contacts_to_add = NULL;
  GList *contacts_to_delete = NULL;
  EBookBackendTpContact *contact = NULL;
  GError *error = NULL;
  GArray *contacts_to_update_in_db = NULL;
  EBookBackendTpClStatus status;

  g_return_val_if_fail (priv->tpdb, FALSE);

  /* Check if we're still online */
  status = e_book_backend_tp_cl_get_status (priv->tpcl);

  if (status != E_BOOK_BACKEND_TP_CL_ONLINE)
  {
    g_object_unref (backend);
    return FALSE;
  }

  contacts_to_update = g_hash_table_get_values (priv->contacts_to_update);

  for (l = contacts_to_update; l; l = l->next)
  {
    contact = l->data;

    if (run_update_contact (backend, contact))
    {
      if (!contacts_to_update_in_db)
      {
        contacts_to_update_in_db = g_array_new (TRUE, TRUE,
            sizeof (EBookBackendTpContact *));
      }

      g_array_append_val (contacts_to_update_in_db, contact);
    }
  }

  g_list_free (contacts_to_update);

  contacts_to_add = g_hash_table_get_values (priv->contacts_to_add);

  for (l = contacts_to_add; l; l = l->next)
  {
    contact = (EBookBackendTpContact *)l->data;

    if (!run_add_contact (backend, contact, &error))
    {
      WARNING ("Unable to create contact: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    } else {
      /* If successful clear that we should add this */
      contact->pending_flags &= ~SCHEDULE_ADD;
      g_hash_table_remove (priv->contacts_to_add, contact->uid);

      if (!contacts_to_update_in_db)
      {
        contacts_to_update_in_db = g_array_new (TRUE, TRUE, 
            sizeof (EBookBackendTpContact *));
      }

      /* Get the database updated wrt. saving these flags */
      g_array_append_val (contacts_to_update_in_db, contact);
    }
  }

  g_list_free (contacts_to_add);

  contacts_to_delete = g_hash_table_get_values (priv->contacts_to_delete);

  for (l = contacts_to_delete; l; l = l->next)
  {
    contact = (EBookBackendTpContact *)l->data;
    MESSAGE ("Deleting contact: %s", contact->uid);
    if (!e_book_backend_tp_cl_run_remove_contact (priv->tpcl, contact, &error))
    {
      WARNING ("Unable to delete contact: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    } else {
      contact->pending_flags &= ~SCHEDULE_DELETE;
      g_hash_table_remove (priv->contacts_to_delete, contact->uid);

      if (!contacts_to_update_in_db)
      {
        contacts_to_update_in_db = g_array_new (TRUE, TRUE, 
            sizeof (EBookBackendTpContact *));
      }

      /* Get the database updated wrt. saving these flags */
      g_array_append_val (contacts_to_update_in_db, contact);
    }
  }

  g_list_free (contacts_to_delete);

  if (contacts_to_update_in_db)
  {
    if (!e_book_backend_tp_db_update_contacts (priv->tpdb, contacts_to_update_in_db, &error))
    {
      g_critical ("Error whilst updating contacts in database: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    }

    g_array_free (contacts_to_update_in_db, TRUE);
  }

  g_object_unref (backend);

  return FALSE;
}

/*
 * Phase 2:
 *
 * Here we apply some of the changes that we have scheduled to apply to the
 * database. Including adding, updating and removing contacts that we find out
 * that have changed during the synchronisation process. These could be
 * considered part of phase 1 but we carry them out here inside an idle to
 * try and avoid starving out the main loop.
 */

static gboolean
_sync_phase_2_idle_cb (gpointer userdata)
{
  GetMembersClosure *closure = (GetMembersClosure *)userdata;
  EBookBackendTp *backend = closure->backend;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  guint i;
  GList *contacts, *l;
  EBookBackendTpContact *contact;
  GArray *contacts_to_delete = NULL;

  g_return_val_if_fail (priv->tpdb, FALSE);

  /* Add new contacts to the database */
  if (closure->contacts_to_add)
  {
    e_book_backend_tp_db_add_contacts (priv->tpdb, closure->contacts_to_add, NULL);

    for (i = 0; i < closure->contacts_to_add->len; i++)
    {
      contact = g_array_index (closure->contacts_to_add,
          EBookBackendTpContact *, i);
      e_book_backend_tp_contact_unref (contact);
    }

    g_array_free (closure->contacts_to_add, TRUE);
  }

  /* Update refreshed contacts in database */
  if (closure->contacts_to_update)
  {
    e_book_backend_tp_db_update_contacts (priv->tpdb, closure->contacts_to_update, NULL);

    for (i = 0; i < closure->contacts_to_update->len; i++)
    {
      contact = g_array_index (closure->contacts_to_update,
          EBookBackendTpContact *, i);
      e_book_backend_tp_contact_unref (contact);
    }

    g_array_free (closure->contacts_to_update, TRUE);
  }

  /* 
   * We must iterate through all contacts looking for the unseen flag and use
   * that to know which ones to remove
   */
  contacts = g_hash_table_get_values (priv->name_to_contact);

  for (l = contacts; l != NULL; l = l->next)
  {
    contact = (EBookBackendTpContact *)l->data;

    if (contact->flags & CONTACT_UNSEEN)
    {
      if (!contacts_to_delete)
        contacts_to_delete = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));

      DEBUG ("found unseen contact with uid %s and name %s",
          contact->uid, contact->name);

      g_array_append_val (contacts_to_delete, contact);
    }
  }

  if (contacts_to_delete)
  {
    delete_contacts (backend, contacts_to_delete);
    g_array_free (contacts_to_delete, TRUE);
  }

  g_list_free (contacts);

  g_idle_add (_sync_phase_3_idle_cb, g_object_ref (backend));

  g_object_unref (closure->backend);
  g_free (closure);
  return FALSE;
}

/*
 * Phase 1:
 *
 * The addition of new contacts and updates to contacts is done here. The
 * process is more complicated when you consider that we may have local
 * changes to the contacts.
 *
 * The local changes we could have to existing contacts is the change in
 * membership. e.g. something could be blocked offline or the alias can be
 * changed.
 *
 * The majority of the work for this phase is done in the tp_cl_get_members_cb
 * callback. This will fired when the data comes back from the request made in
 * the _sync_phase_1 function which is called when we have finished doing our
 * initial database population AND when we are online.
 */
static void
tp_cl_get_members_cb (EBookBackendTpCl *tpcl, GArray *contacts, 
    const GError *error, gpointer userdata)
{
  EBookBackendTp *backend = (EBookBackendTp *)userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact_in = NULL;
  EBookBackendTpContact *contact = NULL;
  guint i = 0;
  GetMembersClosure *closure;

  g_return_if_fail (error || contacts);

  if (error)
  {
    WARNING ("error retrieving members of contact list: %s", error->message);
    g_object_unref (backend);
    return;
  }

  /* Note that we cannot just return here even if the roster is empty
   * or we will skip some needed steps, for instance we will not mark
   * for deletion unseen contacts. */
  
  DEBUG ("get_members called with %d contacts", contacts->len);

  closure = g_new0 (GetMembersClosure, 1);
  closure->contacts_to_add = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
  closure->contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
  closure->backend = g_object_ref (backend);

  for (i = 0; i < contacts->len; i++)
  {
    contact_in = g_array_index (contacts, EBookBackendTpContact *, i);

    /* See if we have an existing contact with this 'name' */
    contact = g_hash_table_lookup (priv->name_to_contact, contact_in->name);

    /* we've already got this contact */
    if (contact != NULL)
    {
      gboolean changed = FALSE; /* whether we need to update contact in db */

      /* Update our fields with the latest ones from the contact list */
      /* TODO: Add more fields here */

      if (contact->alias == NULL || (contact_in->alias &&
          !g_str_equal (contact->alias, contact_in->alias)))
      {
        g_free (contact->alias);
        contact->alias = g_strdup (contact_in->alias);
        changed = TRUE;
      }

      contact->generic_status = contact_in->generic_status;

      if (contact->status == NULL || (contact_in->status &&
          !g_str_equal (contact->status, contact_in->status)))
      {
        g_free (contact->status);
        contact->status = g_strdup (contact_in->status);
      }

      if (contact->status_message == NULL || (contact_in->status_message &&
          !g_str_equal (contact->status_message, contact_in->status_message)))
      {
        g_free (contact->status_message);
        contact->status_message = g_strdup (contact_in->status_message);
      }

      if (contact->contact_info == NULL || (contact_in->contact_info &&
          !g_str_equal (contact->contact_info, contact_in->contact_info)))
      {
        g_free (contact->contact_info);
        contact->contact_info = g_strdup (contact_in->contact_info);
      }

      /* Update our idea of the handle */
      contact->handle = contact_in->handle;

      if (changed)
      {
        /* Add to the array of contacts to update in the database */
        e_book_backend_tp_contact_ref (contact);
        g_array_append_val (closure->contacts_to_update, contact);
      }

      /* Remove the UNSEEN flag because it's been seen now */
      contact->flags &= ~CONTACT_UNSEEN;

      DEBUG ("Refreshing contact with handle %d and name %s",
          contact->handle, contact->name);
    } else {
      /* Woohoo a new contact. We duplicate the contact here we do this so we
       * can maintain a completely separate set of the contacts to that used
       * in the backend. I think this is the best thing to do.
       */

      contact = e_book_backend_tp_contact_dup (contact_in);

      /* Generate a UID for it */
      contact->uid = e_book_backend_tp_generate_uid (backend, contact->name);

      /* Save in the uid hash table */
      g_hash_table_insert (priv->uid_to_contact, g_strdup (contact->uid), 
          e_book_backend_tp_contact_ref (contact));

      /* Save in the name hash table */
      g_hash_table_insert (priv->name_to_contact, g_strdup (contact->name), 
          e_book_backend_tp_contact_ref (contact));
      
      /* Save for adding to the database (leave ownership of the contact) */
      g_array_append_val (closure->contacts_to_add, contact);

      DEBUG ("New contact with handle %d and name %s", 
          contact->handle, contact->name);
    }

    /* Add to the handle lookup table */
    if (g_hash_table_lookup (priv->handle_to_contact, 
          GINT_TO_POINTER (contact_in->handle)) == NULL)
    {
      g_hash_table_insert (priv->handle_to_contact, 
          GINT_TO_POINTER (contact_in->handle),
          e_book_backend_tp_contact_ref (contact));
    } else {
      WARNING ("duplicate contact for handle: %d found", contact_in->handle);
    }
  }

  if (!priv->views)
  {
    DEBUG ("no known views; will notify about members later");
  }

  g_idle_add (_sync_phase_2_idle_cb, closure);

  g_object_unref (backend);
}

static void
_sync_phase_1 (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GError *error = NULL;

  DEBUG ("online and getting members");

  if (!e_book_backend_tp_cl_get_members (priv->tpcl, tp_cl_get_members_cb,
          g_object_ref (backend), &error))
  {
    WARNING ("Error when asking for members: %s", 
        error ? error->message : "unknown error");
    g_clear_error (&error);
  }
}

/*
 * Phase 0
 *
 * Cheekily calling this phase 0 because it's important in the synchronisation
 * prcoess but isn't actually part of it itself since we might not actually
 * get online ever in any given session.
 *
 * This code is responsible for doing the initial population from the
 * database. And then once that is completed waiting for a status change so
 * that we go online.
 *
 * This is process is started when the backend in loaded.
 */
static void
tp_cl_status_changed_cb (EBookBackendTpCl *tpcl, EBookBackendTpClStatus status,
    gpointer userdata)
{
  EBookBackendTp *backend = (EBookBackendTp *)userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);

  DEBUG ("Status changed to %s",
      status == E_BOOK_BACKEND_TP_CL_ONLINE ? "online" : "offline");

  /* Ahaha. We are online now so we can start the synchronisation process */
  if (status == E_BOOK_BACKEND_TP_CL_ONLINE)
  {
    _sync_phase_1 (backend);
  }
  else
  {
    /* Empty the handle table as handles are useful only as long as the
     * account is online */
    g_hash_table_remove_all (priv->handle_to_contact);
  }
}

static void
tp_ready_cb (EBookBackendTp *backend, gpointer userdata)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpClStatus status;

  MESSAGE ("database import complete, we're ready");

  /* Now that we are ready lets listen for the status changed signal */
  g_signal_connect (priv->tpcl, "status-changed", 
      (GCallback)tp_cl_status_changed_cb, backend);

  g_signal_connect (priv->tpcl, "aliases-changed",
      (GCallback)tp_cl_aliases_changed_cb, backend);

  g_signal_connect (priv->tpcl, "presences-changed",
      (GCallback)tp_cl_presence_changed_cb, backend);

  g_signal_connect (priv->tpcl, "flags-changed",
      (GCallback)tp_cl_flags_changed, backend);

  g_signal_connect (priv->tpcl, "contacts-added",
      (GCallback)tp_cl_contacts_added, backend);

  g_signal_connect (priv->tpcl, "contacts-removed",
      (GCallback)tp_cl_contacts_removed, backend);

  g_signal_connect (priv->tpcl, "avatar-tokens-changed",
      (GCallback)tp_cl_avatar_tokens_changed_cb, backend);

  g_signal_connect (priv->tpcl, "avatar-data-changed",
      (GCallback)tp_cl_avatar_data_changed_cb, backend);

  g_signal_connect (priv->tpcl, "capabilities-changed",
      (GCallback)tp_cl_capabilities_changed_cb, backend);

  g_signal_connect (priv->tpcl, "contact-info-changed",
      (GCallback)tp_cl_contact_info_changed_cb, backend);

  status = e_book_backend_tp_cl_get_status (priv->tpcl);

  /* Signal that we're ready to pass along the cached contacts; additional
   * contacts and their changes will follow */
  priv->members_ready = TRUE;
  g_signal_emit_by_name (backend, "members-ready");

  /* Of course if we are already online we should grab the contacts anyway */
  if (status == E_BOOK_BACKEND_TP_CL_ONLINE)
  {
    DEBUG ("online");
    _sync_phase_1 (backend);
  } else {
    DEBUG ("not online");
  }
}

/* TODO: May need to split this into a set of idle callback functions not to
 * starve the mainloop */
static gboolean
_sync_phase_0_idle_cb (gpointer userdata)
{
  EBookBackendTp *backend = (EBookBackendTp *)userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GArray *contacts = NULL;
  EBookBackendTpContact *contact = NULL;
  guint i = 0;

  g_return_val_if_fail (priv->tpdb, FALSE);

  /* We need to know when the database import is complete. So we can 
   * consider moving to phase 1 but only when ready*/
  g_signal_connect (backend, "ready", (GCallback)tp_ready_cb, userdata);

  /* TODO: GError foo */
  contacts = e_book_backend_tp_db_fetch_contacts (priv->tpdb, NULL);

  if (contacts != NULL)
  {
    DEBUG ("retrieved %d contacts from database", contacts->len);

    /* 
     * Import the contacts from the database into the initial set of hash
     * tables
     */
    for (i = 0; i < contacts->len; i++)
    {
      contact = g_array_index (contacts, EBookBackendTpContact *, i);
      g_hash_table_insert (priv->uid_to_contact, 
          g_strdup (contact->uid), 
          e_book_backend_tp_contact_ref (contact));
      g_hash_table_insert (priv->name_to_contact, 
          g_strdup (contact->name), 
          e_book_backend_tp_contact_ref (contact));

      if (contact->pending_flags & SCHEDULE_DELETE)
      {
        g_hash_table_insert (priv->contacts_to_delete,
            g_strdup (contact->uid),
            e_book_backend_tp_contact_ref (contact));
      }

      if (contact->pending_flags & SCHEDULE_ADD)
      {
        g_hash_table_insert (priv->contacts_to_add,
            g_strdup (contact->uid),
            e_book_backend_tp_contact_ref (contact));
      }

      if (contact->pending_flags & SCHEDULE_UPDATE_FLAGS
          || contact->pending_flags & SCHEDULE_UNBLOCK)
      {
        g_hash_table_insert (priv->contacts_to_update,
            g_strdup (contact->uid),
            e_book_backend_tp_contact_ref (contact));
      }

      if (!(contact->pending_flags & SCHEDULE_ADD) &&
          !(contact->flags & CONTACT_INVALID))
      {
        /* Marking as 'UNSEEN' so we know what to remove */
        contact->flags |= CONTACT_UNSEEN;
      }

      /* We don't need the reference ourselves anymore */
      e_book_backend_tp_contact_unref (contact);
    }

    g_array_free (contacts, TRUE);
  }

  /* Fire the signal so that any 'pending' book views can do their thing. */
  g_signal_emit_by_name (backend, "ready");

  /* The reference was added by account_compat_ready_cb */
  g_object_unref (backend);

  return FALSE;
}

/* backend implementations */

static void
account_compat_ready_cb (McAccount *account, const GError *error_in,
    gpointer userdata)
{
  EBookBackendTp *backend = userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend); 
  const gchar *profile_name = NULL;
  GError *error = NULL;

  g_return_if_fail (priv->tpdb);

  if (error_in)
  {
    WARNING ("Unable to retrieve the account properties");
    goto error;
  }

  profile_name = mc_account_compat_get_profile (priv->account);
  priv->profile = mc_profile_lookup (profile_name);

  if (!priv->profile)
  {
    WARNING ("Unable to get mission control profile");
    goto error;
  }

  if (!(mc_profile_get_capabilities (priv->profile) &
        MC_PROFILE_CAPABILITY_SUPPORTS_ROSTER))
  {
    g_critical (G_STRLOC ": the account %s doesn't support rosters",
        priv->account->name);
    goto error;
  }

  priv->vcard_field = mc_profile_get_vcard_field (priv->profile);
  priv->profile_name = mc_profile_get_unique_name (priv->profile);

  if (!priv->vcard_field)
  {
    WARNING ("No vcard field available in profile");
    goto error;
  }
  
  if (!e_book_backend_tp_db_open (priv->tpdb, priv->account->name, &error))
  {
    g_critical ("Error when opening database: %s",
        error ? error->message : "unknown error");
    g_clear_error (&error);
    goto error;
  }

  if (!e_book_backend_tp_cl_load (priv->tpcl, priv->account, &error))
  {
    g_critical ("Error when loading the contact list: %s",
        error ? error->message : "unknown error");
    g_clear_error (&error);
    goto error;
  }

  /* This idle will populate from the database and when it has done so fire
   * the 'ready' signal */
  g_idle_add (_sync_phase_0_idle_cb, backend);

  return;

error:
  /* Even implementing the async interface EDS wants use to do the load in a
   * sync way. This is not possible using Telepathy so we do the real load
   * asynchronously but we lose the ability to report proper errors if
   * something goes wrong. priv->load_error is used to prevent backend
   * methods to be called is the load didn't succeed. */
  priv->load_error = TRUE;
  g_object_unref (backend);
}

typedef struct 
{
  EBookBackendTp *backend;
  gchar *account_name;
} LoadSourceClosure;

static gboolean
load_source_idle_cb (gpointer userdata)
{
  LoadSourceClosure *closure = userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (closure->backend);
  DBusGConnection *bus;
  TpDBusDaemon *dbus_daemon;
  gchar *object_path;

  bus = tp_get_bus ();
  dbus_daemon = tp_dbus_daemon_new (bus);
  object_path = g_strconcat (MC_ACCOUNT_DBUS_OBJECT_BASE,
      closure->account_name, NULL);
  priv->account = mc_account_new (dbus_daemon, object_path);
  g_free (object_path);
  g_object_unref (dbus_daemon);

  if (priv->account)
    mc_account_compat_call_when_ready (priv->account, account_compat_ready_cb,
        g_object_ref (closure->backend));
  else
    WARNING ("Unable to open mission control account: %s",
        closure->account_name);

  g_object_unref (closure->backend);
  free (closure->account_name); /* allocated by sscanf, don't use g_free */
  g_free (closure);

  return FALSE;
}

static GNOME_Evolution_Addressbook_CallStatus
e_book_backend_tp_load_source (EBookBackend *backend, ESource *source, 
    gboolean only_if_exists)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  gchar *account_name = NULL;
  gchar *uri = NULL;
  GNOME_Evolution_Addressbook_CallStatus status =
    GNOME_Evolution_Addressbook_Success;

  /* e_book_backend_tp_load_source can be called more than once so we have to
   * avoid problems with it. */
  if (priv->load_started)
    return GNOME_Evolution_Addressbook_Success;

  priv->load_started = TRUE;

  uri = e_source_get_uri (source);

  if (sscanf (uri, "tp://%as", &account_name) == 1)
  {
    LoadSourceClosure *closure;

    closure = g_new0 (LoadSourceClosure, 1);
    closure->backend = g_object_ref (backend);
    closure->account_name = account_name; /* Leave ownership */

    e_book_backend_set_is_loaded (E_BOOK_BACKEND (backend), TRUE);
    e_book_backend_set_is_writable (E_BOOK_BACKEND (backend), TRUE);

    g_idle_add (load_source_idle_cb, closure);
  } else {
    WARNING ("Error when parsing uri %s", uri);
    status = GNOME_Evolution_Addressbook_OtherError;
  }

  g_free (uri);

  return status;
}

static char *
e_book_backend_tp_get_static_capabilities (EBookBackend *backend)
{
  return g_strdup ("local,do-initial-query,contact-lists");
}

/* Stream the contacts over to the client.in the view. */

typedef struct {
    /* g_utf8_collate_key can remove spaces so we cannot just concatenate
     * the first name with the last name */
    gchar *tag1;
    gchar *tag2;
    EContact *econtact;
} ContactSortData;

static ContactSortData *
contact_sort_data_new (EContact *ec, ContactSortOrder sort_order,
    const gchar *vcard_field)
{
  ContactSortData *data;
  const gchar *first;
  const gchar *last;
  const gchar *tag1 = NULL;
  const gchar *tag2 = NULL;
  gchar *tmp;
  gboolean free_tag1 = FALSE;

  data = g_new0 (ContactSortData, 1);
  data->econtact = g_object_ref (ec);

  if (sort_order == CONTACT_SORT_ORDER_LAST_FIRST ||
      sort_order == CONTACT_SORT_ORDER_FIRST_LAST) {
    first = e_contact_get_const (ec, E_CONTACT_GIVEN_NAME);
    last = e_contact_get_const (ec, E_CONTACT_FAMILY_NAME);

    if (first && last) {
        /* We have both first and last name */
        if (sort_order == CONTACT_SORT_ORDER_FIRST_LAST) {
            tag1 = first;
            tag2 = last;
        } else {
            tag1 = last;
            tag2 = first;
        }

        goto done;
    }

    if (first) {
        /* Only first name, just use it */
        tag1 = first;
        goto done;
    }

    if (last) {
        /* Only last name, just use it */
        tag1 = last;
        goto done;
    }

    /* No name fields, fallback to the nickname */
  }

  tag1 = e_contact_get_const (ec, E_CONTACT_NICKNAME);
  if (!tag1) {
      /* If the nickname (i.e. Telepathy alias) is not set then just use the
       * JID (or any other contact ID for other protocols) as a fallback */
      EVCardAttribute *attr;

      attr = e_vcard_get_attribute (E_VCARD (ec), vcard_field);
      tag1 = e_vcard_attribute_get_value (attr);
      free_tag1 = TRUE;
  }

done:
  tmp = g_utf8_casefold (tag1, -1);
  data->tag1 = g_utf8_collate_key (tmp, -1);
  g_free (tmp);

  if (tag2) {
    tmp = g_utf8_casefold (tag2, -1);
    data->tag2 = g_utf8_collate_key (tmp, -1);
    g_free (tmp);
  }

  /* tag1 can come from e_vcard_attribute_get_value that allocates a new
   * string instead of e_contact_get_const */
  if (free_tag1)
    g_free ((gchar *)tag1);

  return data;
}

static void
contact_sort_data_free (ContactSortData *data)
{
  if (!data)
    return;

  g_free (data->tag1);
  g_free (data->tag2);
  g_object_unref (data->econtact);
  g_free (data);
}

static gint
contact_sort_data_compare (const ContactSortData **pa, const ContactSortData **pb)
{
  const ContactSortData *a = *pa;
  const ContactSortData *b = *pb;
  gint cmp;

  cmp = g_strcmp0 (a->tag1, b->tag1);
  if (cmp == 0)
    cmp = g_strcmp0 (a->tag2, b->tag2);

  return cmp;
}

/* TODO: Should we perhaps split this up OR spin it into a separate thread to
 * allow freeze-thaw.
 */
static void
notify_all_contacts_updated_for_view (EBookBackendTp *backend,
    EDataBookView *book_view)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GPtrArray *all_contacts;
  ContactSortOrder sort_order;
  GHashTableIter iter;
  gpointer contact_pointer;
  guint i;

  DEBUG ("sending contacts");

  all_contacts = g_ptr_array_sized_new (g_hash_table_size (priv->uid_to_contact));

  /* If for some reason the sort order was not set then g_object_get_data will
   * return NULL, that will be casted to CONTACT_SORT_ORDER_FIRST_LAST. This
   * is fine as it's a good default. */
  sort_order = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (book_view),
        BOOK_VIEW_SORT_ORDER_DATA_KEY));

  g_hash_table_iter_init (&iter, priv->uid_to_contact);
  while (g_hash_table_iter_next (&iter, NULL, &contact_pointer)) {
    EBookBackendTpContact *contact;
    EContact *ec;

    contact = contact_pointer;

    if (contact->pending_flags & SCHEDULE_DELETE)
      /* The contact was deleted by the user but not deleted yet from the
       * server, for instance because we are offline. */
      continue;

    ec = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
        priv->profile_name);

    g_ptr_array_add (all_contacts,
        contact_sort_data_new (ec, sort_order, priv->vcard_field));

    g_object_unref (ec);
  }

  g_ptr_array_sort (all_contacts, (GCompareFunc) contact_sort_data_compare);

  for (i  = 0; i < all_contacts->len; i++) {
    ContactSortData *data = g_ptr_array_index (all_contacts, i);
    e_data_book_view_notify_update (book_view, data->econtact);
    contact_sort_data_free (data);
  }

  g_ptr_array_free (all_contacts, TRUE);

  e_data_book_view_notify_complete (book_view,
    GNOME_Evolution_Addressbook_Success);
}

static void
notify_all_contacts_updated (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GList *l;

  DEBUG ("notifying book views about members");

  for (l = priv->views; l != NULL; l = l->next)
    notify_all_contacts_updated_for_view (backend, l->data);
}

static void
book_view_tp_members_ready_cb (EBookBackendTp *backend, gpointer userdata)
{
  DEBUG ("members ready to send through book view (callback)");

  notify_all_contacts_updated (backend);
}

typedef struct 
{
  EBookBackendTp *backend;
  EDataBookView *book_view;
} BookViewClosure;

static gboolean
start_book_view_idle_cb (gpointer userdata)
{
  BookViewClosure *closure = (BookViewClosure *)userdata;
  EBookBackendTp *backend = (EBookBackendTp *)closure->backend;
  EBookBackendTpPrivate *priv = GET_PRIVATE (closure->backend);

  DEBUG ("book view idle callback");

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the book view cannot"
        "be started");

    e_data_book_view_notify_complete (closure->book_view,
        GNOME_Evolution_Addressbook_OtherError);

    goto done;
  }

  /* Store the list of views that we have since we need this to notify of
   * changes, etc.
   */
  priv->views = g_list_append (priv->views, closure->book_view);
  g_object_ref (closure->book_view);

  /* If we have finished import and have the members, send them along; otherwise
   * setup a callback to be fired when we are ready.
   */
  if (priv->members_ready)
  {
    DEBUG ("members ready to send through book view (immediate)");
    notify_all_contacts_updated_for_view (backend, closure->book_view);
  } else if (!priv->members_ready_signal_id) {
    priv->members_ready_signal_id = g_signal_connect (backend,
        "members-ready", (GCallback)book_view_tp_members_ready_cb, NULL);
  }

done:
  g_object_unref (closure->backend);
  g_object_unref (closure->book_view);
  g_free (closure);

  return FALSE;
}

/* 
 * Create a closure, populate with all the objects we need and then spin off
 * into the mainloop via an idle.
 */
static void
e_book_backend_tp_start_book_view (EBookBackend *backend, 
    EDataBookView *book_view)
{
  BookViewClosure *closure;

  closure = g_new0 (BookViewClosure, 1);
  closure->backend = (EBookBackendTp *)backend;
  closure->book_view = book_view;

  g_object_ref (closure->backend);
  g_object_ref (closure->book_view);

  g_idle_add (start_book_view_idle_cb, closure);
}

/* idle to avoid thread pain */
static gboolean
stop_book_view_idle_cb (gpointer userdata)
{
  BookViewClosure *closure = (BookViewClosure *)userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (closure->backend);

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the book view cannot "
        "be stopped");
    goto done;
  }

  flush_db_updates (closure->backend);

  priv->views = g_list_remove (priv->views, closure->book_view);
  g_object_unref (closure->book_view);

done:
  g_object_unref (closure->book_view);
  g_object_unref (closure->backend);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_stop_book_view (EBookBackend *backend,
    EDataBookView *book_view)
{
  BookViewClosure *closure = NULL;

  closure = g_new0 (BookViewClosure, 1);
  closure->book_view = g_object_ref (book_view);
  closure->backend = g_object_ref (backend);

  g_idle_add (stop_book_view_idle_cb, closure);
}

/* object class implementations */
static void
e_book_backend_tp_dispose (GObject *object)
{
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (object);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);

  if (priv->db_update_timer)
    /* There are still pending changes */
    flush_db_updates (backend);

  g_signal_handlers_disconnect_matched (priv->tpcl, G_SIGNAL_MATCH_DATA, 0, 0,
      NULL, NULL, object);

  if (priv->account)
  {
    g_object_unref (priv->account);
    priv->account = NULL;
  }

  if (priv->profile)
  {
    g_object_unref (priv->profile);
    priv->profile = NULL;
  }

  g_object_unref (priv->tpcl);

  if (priv->tpdb)
  {
    e_book_backend_tp_db_close (priv->tpdb, NULL);
    g_object_unref (priv->tpdb);
    priv->tpdb = NULL;
  }

  g_hash_table_unref (priv->uid_to_contact);
  g_hash_table_unref (priv->name_to_contact);
  g_hash_table_unref (priv->handle_to_contact);

  g_hash_table_unref (priv->contacts_to_delete);
  g_hash_table_unref (priv->contacts_to_update);
  g_hash_table_unref (priv->contacts_to_add);

  g_hash_table_unref (priv->contacts_to_update_in_db);

  g_hash_table_unref (priv->contacts_remotely_changed);
  if (priv->contacts_remotely_changed_update_id)
    g_source_remove (priv->contacts_remotely_changed_update_id);

  G_OBJECT_CLASS (e_book_backend_tp_parent_class)->dispose (object);
}

static void
e_book_backend_tp_finalize (GObject *object)
{
}

static void
e_book_backend_tp_init (EBookBackendTp *backend)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  gchar *avatar_dir;

  priv->tpcl = e_book_backend_tp_cl_new ();
  priv->tpdb = e_book_backend_tp_db_new ();

  priv->uid_to_contact = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);
  priv->name_to_contact = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);
  priv->handle_to_contact = g_hash_table_new_full (g_direct_hash, g_direct_equal,
      NULL, (GDestroyNotify) e_book_backend_tp_contact_unref);
  priv->contacts_to_delete = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);
  priv->contacts_to_update = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);
  priv->contacts_to_add = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);

  priv->contacts_to_update_in_db = g_hash_table_new_full (g_str_hash, g_str_equal,
      g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);

  priv->contacts_remotely_changed = g_hash_table_new_full (g_str_hash,
      g_str_equal, g_free, (GDestroyNotify) e_book_backend_tp_contact_unref);

  /* Create the avatar directory */
  avatar_dir = g_build_filename (g_get_home_dir (), ".osso-abook", "avatars",
      NULL);
  if (g_mkdir_with_parents (avatar_dir, 0755) < 0)
  {
    WARNING ("Error creating avatar directory: %s",
        g_strerror (errno));
  }
  g_free (avatar_dir);
}

static GNOME_Evolution_Addressbook_CallStatus
e_book_backend_tp_cancel_operation (EBookBackend *backend, EDataBook *book)
{
  return GNOME_Evolution_Addressbook_CouldNotCancel;
}

static void 
e_book_backend_tp_set_mode (EBookBackend *backend, 
    GNOME_Evolution_Addressbook_BookMode mode)
{

}

typedef struct
{
  EBookBackend *backend;
  EContact *contact;
  EDataBook *book;
  guint32 opid;
} ModifyContactClosure;

static gboolean
modify_contact_idle_cb (gpointer userdata)
{
  ModifyContactClosure *closure = (ModifyContactClosure *)userdata;
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (closure->backend);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GNOME_Evolution_Addressbook_CallStatus status;
  const gchar *uid = NULL;
  EBookBackendTpContact *contact = NULL;
  EBookBackendTpClStatus tpcl_status;
  GError *error = NULL;
  EContact *updated_econtact;

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the contact cannot "
        "be modified");
    status = GNOME_Evolution_Addressbook_OtherError;
    goto done;
  }

  if (!priv->tpdb)
  {
    status = BookRemoved;
    goto done;
  }

  if (!e_book_backend_tp_db_check_available_disk_space ())
  {
    status = GNOME_Evolution_Addressbook_NoSpace;
    goto done;
  }

  status = GNOME_Evolution_Addressbook_Success;

  uid = e_contact_get_const (closure->contact, E_CONTACT_UID);

  if (uid)
  {
    contact = g_hash_table_lookup (priv->uid_to_contact, uid);

    if (contact)
    {
      MESSAGE ("Modifying contact: %s", contact->name);
      if (e_log_will_print (e_book_backend_tp_log_domain_id, G_LOG_LEVEL_DEBUG))
      {
        gchar *tmp;
        tmp = e_vcard_to_string (E_VCARD (closure->contact), EVC_FORMAT_VCARD_30);
        DEBUG ("%s", tmp);
        g_free (tmp);
      }

      /* update our contact in place, noting any changes */
      e_book_backend_tp_contact_update_from_econtact (contact, 
          closure->contact, priv->vcard_field);

      DEBUG ("pending flags: %x %x", contact->pending_flags, SCHEDULE_UPDATE_MASTER_UID);

      if (contact->pending_flags & SCHEDULE_UPDATE_FLAGS 
          || contact->pending_flags & SCHEDULE_UNBLOCK
          || contact->pending_flags & SCHEDULE_UPDATE_MASTER_UID)
      {
        tpcl_status = e_book_backend_tp_cl_get_status (priv->tpcl);

        if ( tpcl_status == E_BOOK_BACKEND_TP_CL_ONLINE)
        {
          if (contact->pending_flags & SCHEDULE_UPDATE_FLAGS)
          {
            if (!e_book_backend_tp_cl_run_update_flags (priv->tpcl, contact, &error))
            {
              WARNING ("Error whilst trying to update flags: %s",
                  error ? error->message : "unknown error");
              g_clear_error (&error);
            } else {
              contact->pending_flags &= ~SCHEDULE_UPDATE_FLAGS;
            }
          }
        }

        g_hash_table_insert (priv->contacts_to_update,
            g_strdup (contact->uid),
            e_book_backend_tp_contact_ref (contact));

        if (!e_book_backend_tp_db_update_contact (priv->tpdb, contact, &error))
        {
          WARNING ("Error whilst updating database contact: %s",
              error ? error->message : "unknown error");
          g_clear_error (&error);
        } else {
          contact->pending_flags &= ~SCHEDULE_UPDATE_MASTER_UID;
        }

        notify_updated_contact (backend, contact);
      }
    } else {
      WARNING ("Unknown uid (%s) on submitted vcard", uid);
      status = GNOME_Evolution_Addressbook_OtherError;
    }
  } else {
    WARNING ("No uid found on submitted vcard");
    status = GNOME_Evolution_Addressbook_OtherError;
  }

done:
  if (status == GNOME_Evolution_Addressbook_Success)
  {
    updated_econtact = e_book_backend_tp_contact_to_econtact (contact, 
        priv->vcard_field, priv->profile_name);
    e_data_book_respond_modify (closure->book, closure->opid, status, 
        updated_econtact);
  } else {
    g_object_unref (closure->contact);
    e_data_book_respond_modify (closure->book, closure->opid, status, 
        NULL);
  }

  g_object_unref (closure->contact);
  g_object_unref (closure->book);
  g_object_unref (closure->backend);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_modify_contact (EBookBackend *backend, EDataBook *book,
    guint32 opid, const gchar *vcard)
{
  ModifyContactClosure *closure;

  g_critical ("Modifying contacts is not supported for IM contacts and "
      "can lead to race conditions. Just add the same contacts again to "
      "update it");

  closure = g_new0 (ModifyContactClosure, 1);

  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->contact = e_contact_new_from_vcard (vcard);
  closure->opid = opid;

  g_idle_add (modify_contact_idle_cb, closure);
}

typedef struct 
{
  EBookBackend *backend;
  EContact *econtact;
  EDataBook *book;
  guint32 opid;
} CreateContactClosure;

static EBookBackendTpContact *
run_create_contact (EBookBackendTp *backend, EContact *econtact, GError **error_out)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact = NULL;
  GError *error = NULL;
  EBookBackendTpContact *existing_contact = NULL;

  g_return_val_if_fail (priv->tpdb, FALSE);

  g_object_ref (backend);

  /* create our contact, populate with a UID and also with details from the
   * vcard */
  contact = e_book_backend_tp_contact_new ();

  if (!e_book_backend_tp_contact_update_from_econtact (contact, econtact,
        priv->vcard_field))
  {
    g_object_unref (backend);
    e_book_backend_tp_contact_unref (contact);
    return NULL;
  }

  contact->uid = e_book_backend_tp_generate_uid (backend, contact->name);

  contact->pending_flags |= SCHEDULE_ADD;
  contact->pending_flags |= SUBSCRIBE | PUBLISH | STORED;

  if (e_book_backend_tp_cl_get_status (priv->tpcl) == E_BOOK_BACKEND_TP_CL_ONLINE)
  {
    if (!run_add_contact (backend, contact, &error))
    {
      WARNING ("Error whilst creating contact: %s",
          error ? error->message : "unknown error");
      g_propagate_error (error_out, error);

      g_object_unref (backend);
      e_book_backend_tp_contact_unref (contact);
      return NULL;
    }

    /* Don't remove the SCHEDULE_ADD flag:
     * We probably have to store some master UID.
     */
  }

  /* Lets check to see if we already have a contact with this name before. We
   * can cheat and just return our existing contact back through EDS.
   *
   * We do this here. After perhaps trying to add in order to potentially
   * resolve the normalised name (which gets written back into the contact
   * field.)
   *
   * However even if we aren't online we want to look for other contacts that
   * also have been added with the same name.
   */
  existing_contact = g_hash_table_lookup (priv->name_to_contact, contact->name);

  if (existing_contact)
  {
    /* So we already had a contact with this name.
     *
     * The e_book_add_contact() function also is used to atomically add master
     * contact UIDs: Calling e_book_get_contact() and e_book_commit_contact()
     * sequencially bears the risk of loosing updates.
     *
     * Therefore we now check if this duplicate contact introduces new master
     * contact UIDS and if that's the case we update the database.
     */
    if (!e_book_backend_tp_contact_update_master_uids (existing_contact, contact->master_uids))
    {
      DEBUG ("Trying to add a contact with a duplicate name");
      contact->pending_flags &= ~SCHEDULE_ADD;
    }

    e_book_backend_tp_contact_unref (contact);
    contact = e_book_backend_tp_contact_ref (existing_contact);

    /* The add function is also called to unblock blocked contacts, so if
     * the contact is in the deny list we unblock it */
    if (contact->flags & DENY)
    {
      contact->pending_flags &= ~DENY;
      contact->pending_flags |= SCHEDULE_UNBLOCK;
    }

    if (run_update_contact (backend, contact))
    {
      if (!e_book_backend_tp_db_update_contact (priv->tpdb, contact, &error))
      {
        g_critical ("Error whilst updating contact in database: %s",
            error ? error->message : "unknown error");
        g_clear_error (&error);
      }

      notify_updated_contact (backend, contact);
    }
  } else {
    /* Add to our main tables */
    g_hash_table_insert (priv->uid_to_contact,
        g_strdup (contact->uid),
        e_book_backend_tp_contact_ref (contact));
    g_hash_table_insert (priv->name_to_contact,
        g_strdup (contact->name),
        e_book_backend_tp_contact_ref (contact));

    if (contact->pending_flags & SCHEDULE_ADD)
    {
      g_hash_table_insert (priv->contacts_to_add,
          g_strdup (contact->uid), e_book_backend_tp_contact_ref (contact));

      if (!e_book_backend_tp_db_add_contact (priv->tpdb, contact, &error))
      {
        g_critical ("Error adding contact to database: %s",
            error ? error->message : "unknown error");
        g_propagate_error (error_out, error);

        e_book_backend_tp_contact_unref (contact);
        g_object_unref (backend);
    
        return NULL;
      }
    }
  }

  g_object_unref (backend);
  return contact;
}

static gboolean
create_contact_idle_cb (gpointer userdata)
{
  CreateContactClosure *closure = (CreateContactClosure *)userdata;
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (closure->backend);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact = NULL;
  EContact *econtact = NULL;
  GError *error = NULL;
  GNOME_Evolution_Addressbook_CallStatus status;

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the contact cannot "
        "be created");
    status = GNOME_Evolution_Addressbook_OtherError;
    goto done;
  }

  if (!e_book_backend_tp_db_check_available_disk_space ())
  {
    status = GNOME_Evolution_Addressbook_NoSpace;
    goto done;
  }

  contact = run_create_contact (backend, closure->econtact, &error);
  if (contact)
  {
    status = GNOME_Evolution_Addressbook_Success;
    econtact = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
        priv->profile_name);
  } else {
    status = GNOME_Evolution_Addressbook_OtherError;
    econtact = NULL;
    g_clear_error (&error);
  }

done:
  e_data_book_respond_create (closure->book, closure->opid, status, econtact);

  if (contact)
    e_book_backend_tp_contact_unref (contact);
  if (econtact)
    g_object_unref (econtact);

  g_object_unref (closure->book);
  g_object_unref (closure->econtact);
  g_object_unref (closure->backend);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_create_contact (EBookBackend *backend, EDataBook *book,
    guint32 opid, const gchar *vcard)
{
  CreateContactClosure *closure;

  closure = g_new0 (CreateContactClosure, 1);

  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->econtact = e_contact_new_from_vcard (vcard);
  closure->opid = opid;

  g_idle_add (create_contact_idle_cb, closure);
}

typedef struct 
{
  EBookBackend *backend;
  GSList *econtacts; /* GSList of EContact* */
  EDataBook *book;
  guint32 opid;
} CreateContactsClosure;

static gboolean
create_contacts_idle_cb (gpointer userdata)
{
  CreateContactsClosure *closure = (CreateContactsClosure *)userdata;
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (closure->backend);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GSList *econtact_in;
  GList *econtacts = NULL;
  GNOME_Evolution_Addressbook_CallStatus status = GNOME_Evolution_Addressbook_Success;
  GError *error = NULL;

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the contacts cannot "
        "be created");
    status = GNOME_Evolution_Addressbook_OtherError;
    goto done;
  }

  if (!e_book_backend_tp_db_check_available_disk_space ())
  {
    status = GNOME_Evolution_Addressbook_NoSpace;
    goto done;
  }

  econtact_in = closure->econtacts;
  while (econtact_in)
  {
    EBookBackendTpContact *contact;

    contact = run_create_contact (backend, econtact_in->data, &error);
    if (contact)
    {
      EContact *econtact = e_book_backend_tp_contact_to_econtact (contact,
          priv->vcard_field, priv->profile_name);
      econtacts = g_list_prepend (econtacts, econtact);
      e_book_backend_tp_contact_unref (contact);
    } else {
      g_list_foreach (econtacts, (GFunc)g_object_unref, NULL);
      g_list_free (econtacts);
      econtacts = NULL;
      status = GNOME_Evolution_Addressbook_OtherError;
      g_clear_error (&error);
      break;
    }

    econtact_in = g_slist_next (econtact_in);
  }

done:
  e_data_book_respond_create_contacts (closure->book, closure->opid, status,
      econtacts);

  g_object_unref (closure->book);

  g_slist_foreach (closure->econtacts, (GFunc)g_object_unref, NULL);
  g_slist_free (closure->econtacts);
  g_list_foreach (econtacts, (GFunc)g_object_unref, NULL);
  g_list_free (econtacts);
  g_object_unref (closure->backend);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_create_contacts (EBookBackend *backend, EDataBook *book,
    guint32 opid, const gchar **vcards)
{
  CreateContactsClosure *closure;
  const gchar **iter;

  closure = g_new0 (CreateContactsClosure, 1);

  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->opid = opid;

  for (iter = vcards; *iter; iter++)
  {
    closure->econtacts = g_slist_prepend (closure->econtacts,
        e_contact_new_from_vcard (*iter));
  }

  g_idle_add (create_contacts_idle_cb, closure);
}

static EBookBackendTpContact*
run_remove_contact (EBookBackendTp         *backend,
                    EBookBackendTpClStatus  status,
                    const char             *uid,
                    gboolean               *ret_really_remove)
{
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  EBookBackendTpContact *contact = NULL;
  GError *error = NULL;
  char **uid_list;
  int i;
  gboolean really_remove = TRUE;

  g_object_ref (backend);

  /* The frontend appends master contact uids with a ; delimiter to notify us of
   * master contacts to remove from this contact before deletion */
  uid_list = g_strsplit (uid, ";", -1);
  contact = g_hash_table_lookup (priv->uid_to_contact, uid_list[0]);

  if (!contact)
  {
    WARNING ("Unknown UID asked to be deleted: %s", uid);
    goto cleanup;
  }

  /* Check special cases... */
  if (uid_list[1] && strcmp (uid_list[1], "*") == 0)
  {
    /* We want to remove all master uids, but not removing contact from roster */
    e_book_backend_tp_contact_remove_all_master_uids (contact);
    really_remove = FALSE;
  } else if (uid_list[1]) {
    /* We want to remove a list of master uids, and remove contact from roster
     * only if there are no more master uid. */
    for (i = 1; uid_list[i]; ++i)
      e_book_backend_tp_contact_remove_master_uid (contact, uid_list[i]);

    if (contact->master_uids->len > 0)
      really_remove = FALSE;
  }

  if (!really_remove)
  {
    /* We don't really want to remove this contact,
     * just an update of master uids */
    if (!(contact->pending_flags & SCHEDULE_UPDATE_MASTER_UID))
      goto cleanup;
    if (!e_book_backend_tp_db_update_contact (priv->tpdb, contact, &error))
    {
      WARNING ("Error whilst updating database contact: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    } else {
      contact->pending_flags &= ~SCHEDULE_UPDATE_MASTER_UID;
    }

    notify_updated_contact (backend, contact);
    goto cleanup;
  }

  contact->pending_flags = 0;

  if (contact->flags & CONTACT_INVALID)
  {
    /* Invalid contacts are not known to Telepathy, so there is no point in
     * calling e_book_backend_tp_cl_run_remove_contact on them and we just
     * remove them directly. */
    GArray *contacts_to_remove;

    contacts_to_remove = g_array_sized_new (TRUE, TRUE,
        sizeof (EBookBackendTpContact *), 1);
    g_array_append_val (contacts_to_remove, contact);

    delete_contacts (backend, contacts_to_remove);

    g_array_free (contacts_to_remove, TRUE);
  } else if (status == E_BOOK_BACKEND_TP_CL_ONLINE &&
      e_book_backend_tp_cl_run_remove_contact (priv->tpcl, contact, &error))
  {
    /* We've successfully asked for deletion. The actual removal from the
     * database, etc, will happen in the MembersChanged signal */
  } else {
    if (error)
    {
      WARNING ("Error whilst requesting contact deletion: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    }

    /* Mark for schedule removal */
    contact->pending_flags |= SCHEDULE_DELETE;
    g_hash_table_insert
      (priv->contacts_to_delete, g_strdup (contact->uid),
       e_book_backend_tp_contact_ref (contact));
  }

cleanup:
  g_strfreev (uid_list);
  g_object_unref (backend);

  if (ret_really_remove)
    *ret_really_remove = really_remove;

  return contact;
}

typedef struct
{
  EBookBackend *backend;
  EDataBook *book;
  guint32 opid;
  GList *id_list;
} RemoveMembersClosure;

static gboolean
remove_contacts_idle_cb (gpointer userdata)
{
  RemoveMembersClosure *closure = userdata;
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (closure->backend);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GNOME_Evolution_Addressbook_CallStatus status;
  EBookBackendTpContact *contact = NULL;
  EBookBackendTpClStatus tpcl_status;
  GArray *contacts_to_update = NULL;
  GList *ids_removed = NULL;
  GError *error = NULL;
  GList *l = NULL;

  if (priv->load_error)
  {
    g_critical ("the book was not loaded correctly so the contacts cannot "
        "be removed");
    status = GNOME_Evolution_Addressbook_OtherError;
    goto done;
  }

  if (!priv->tpdb)
  {
    status = BookRemoved;
    goto done;
  }

  if (!e_book_backend_tp_db_check_available_disk_space ())
  {
    status = GNOME_Evolution_Addressbook_NoSpace;
    goto done;
  }

  status = GNOME_Evolution_Addressbook_Success;
  tpcl_status = e_book_backend_tp_cl_get_status (priv->tpcl);

  /* Removing contacts is really easy. We basically just want to zero the
   * flags and then the members changed stuff deals with it fine. */

  for (l = closure->id_list; l; l = l->next)
  {
    gboolean really_remove = TRUE;

    contact = run_remove_contact (backend, tpcl_status, l->data, &really_remove);

    if (!contact)
      continue;

    if (really_remove)
    {
      ids_removed = g_list_prepend (ids_removed, g_strdup (contact->uid));
      if (!contacts_to_update)
        contacts_to_update = g_array_new (TRUE, TRUE, sizeof (EBookBackendTpContact *));
      g_array_append_val (contacts_to_update, contact);
    }
  }

done:
  e_data_book_respond_remove_contacts (closure->book, closure->opid, status,
      ids_removed);

  /* Now update the database */
  if (contacts_to_update)
  {
    if (!e_book_backend_tp_db_update_contacts (priv->tpdb, contacts_to_update,
          &error))
    {
      g_critical ("error whilst updating database: %s",
          error ? error->message : "unknown error");
      g_clear_error (&error);
    }

    g_array_free (contacts_to_update, TRUE);
  }

  while (ids_removed)
  {
    g_free (ids_removed->data);
    ids_removed = g_list_delete_link (ids_removed, ids_removed);
  }

  g_object_unref (closure->backend);
  g_object_unref (closure->book);
  g_list_foreach (closure->id_list, (GFunc) g_free, NULL);
  g_list_free (closure->id_list);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_remove_contacts (EBookBackend *backend, EDataBook *book,
    guint32 opid, GList *id_list)
{
  RemoveMembersClosure *closure = NULL;
  GList *l = NULL;

  closure = g_new0 (RemoveMembersClosure, 1);
  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->opid = opid;

  for (l = id_list; l; l = l->next)
    closure->id_list = g_list_append (closure->id_list, g_strdup (l->data));

  g_idle_add (remove_contacts_idle_cb, closure);
}

static void
unsupported_method (const gchar *method_name)
{
  WARNING ("unsupported method: %s", method_name);
}

static void
e_book_backend_tp_authenticate_user (EBookBackend *backend, EDataBook *book,
    guint32 opid, const char *user, const char *passwd, const char *auth_method)
{
  unsupported_method ("authenticate_user");
}

static void
e_book_backend_tp_get_changes (EBookBackend *backend, EDataBook *book, guint32 opid, const char *change_id)
{
  unsupported_method ("get_changes");
}

typedef struct
{
  EBookBackend *backend;
  EDataBook *book;
  guint32 opid;
  char *uid;
} GetContactClosure;

static gboolean
get_contact_idle_cb (gpointer userdata)
{
  GetContactClosure *closure = userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (closure->backend);
  GNOME_Evolution_Addressbook_CallStatus status =
      GNOME_Evolution_Addressbook_OtherError;
  EBookBackendTpContact *contact;
  EContact *ec;
  gchar *vcard = NULL;

  if (closure->uid == NULL || closure->uid == '\0') {
    WARNING ("Empty contact id");
    goto done;
  }

  contact = g_hash_table_lookup (priv->uid_to_contact, closure->uid);

  if (!contact) {
    status = GNOME_Evolution_Addressbook_ContactNotFound;
    goto done;
  }

  ec = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
      priv->profile_name);
  vcard = e_vcard_to_string (E_VCARD (ec), EVC_FORMAT_VCARD_30);
  g_object_unref (ec);

  if (vcard == NULL || vcard[0] == '\0') {
    WARNING ("Could not generate vcard from record.");
    g_free (vcard);
    vcard = NULL;
    goto done;
  }

  status = GNOME_Evolution_Addressbook_Success;

done:
  e_data_book_respond_get_contact (closure->book, closure->opid, status,
      vcard);

  g_object_unref (closure->backend);
  g_object_unref (closure->book);
  g_free (closure->uid);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_get_contact (EBookBackend *backend, EDataBook *book, guint32 opid, const char *id)
{
  GetContactClosure *closure = NULL;

  closure = g_new0 (GetContactClosure, 1);
  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->opid = opid;
  closure->uid = g_strdup (id);

  g_idle_add (get_contact_idle_cb, closure);
}

typedef struct
{
  EBookBackend *backend;
  EDataBook *book;
  guint32 opid;
  gchar *query;
} GetContactListClosure;

static gboolean
get_contact_list_idle_cb (gpointer userdata)
{
  GetContactListClosure *closure = userdata;
  EBookBackendTpPrivate *priv = GET_PRIVATE (closure->backend);
  GNOME_Evolution_Addressbook_CallStatus status =
    GNOME_Evolution_Addressbook_OtherError;
  EBookBackendSExp *sexp = NULL;
  gboolean get_all = FALSE;
  GHashTableIter iter;
  gpointer contact_pointer;
  GList *contact_list = NULL;

  if (closure->query == NULL || closure->query == '\0') {
    WARNING ("Empty query");
    goto done;
  }

  sexp = e_book_backend_sexp_new (closure->query);
  if (sexp == NULL) {
    WARNING ("Could not create sexp");
    goto done;
  }

  DEBUG ("query: %s", closure->query);
  if (!g_ascii_strcasecmp (closure->query,
        "(contains \"x-evolution-any-field\" \"\")")) {
    get_all = TRUE;
  }

  g_hash_table_iter_init (&iter, priv->uid_to_contact);
  /* We cannot pass directly an EBookBackendTpContact * as it would break
   * strict aliasing */
  while (g_hash_table_iter_next (&iter, NULL, &contact_pointer)) {
    EBookBackendTpContact *contact = contact_pointer;
    EContact *ec;
    gchar *vcard;

    ec = e_book_backend_tp_contact_to_econtact (contact, priv->vcard_field,
        priv->profile_name);
    vcard = e_vcard_to_string (E_VCARD (ec), EVC_FORMAT_VCARD_30);
    g_object_unref (ec);

    if (vcard == NULL || vcard[0] == '\0') {
      WARNING ("Could not generate vcard from record.");
      g_free (vcard);
      continue;
    }

    if (get_all || e_book_backend_sexp_match_vcard (sexp, vcard)) {
      contact_list = g_list_prepend (contact_list, vcard);
    }
  }

  status = GNOME_Evolution_Addressbook_Success;

done:
  e_data_book_respond_get_contact_list (closure->book, closure->opid, status,
      contact_list);

  /* elements are released by libedata-book */
  g_list_free (contact_list);

  if (sexp)
    g_object_unref (sexp);

  g_object_unref (closure->backend);
  g_object_unref (closure->book);
  g_free (closure->query);
  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_get_contact_list (EBookBackend *backend, EDataBook *book, guint32 opid, const char *query)
{
  GetContactListClosure *closure = NULL;

  closure = g_new0 (GetContactListClosure, 1);
  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->opid = opid;
  closure->query = g_strdup (query);

  g_idle_add (get_contact_list_idle_cb, closure);
}

static void
e_book_backend_tp_get_required_fields (EBookBackend *backend, EDataBook *book, guint32 opid)
{
  unsupported_method ("get_required_fields");
}

static void
e_book_backend_tp_get_supported_auth_methods (EBookBackend *backend, EDataBook *book, guint32 opid)
{
  unsupported_method ("get_supported_auth_methods");
}

static void
e_book_backend_tp_get_supported_fields (EBookBackend *backend, EDataBook *book, guint32 opid)
{
  unsupported_method ("get_supported_fields");
}

static void
e_book_backend_tp_modify_contacts (EBookBackend *backend, EDataBook *book, guint32 opid, const char **vcards)
{
  unsupported_method ("modify_contacts");
}

typedef struct
{
  EBookBackend *backend;
  EDataBook *book;
  guint32 opid;
} RemoveClosure;

static gboolean
remove_idle_cb (gpointer userdata)
{
  RemoveClosure *closure = (RemoveClosure *)userdata;
  EBookBackendTp *backend = E_BOOK_BACKEND_TP (closure->backend);
  EBookBackendTpPrivate *priv = GET_PRIVATE (backend);
  GNOME_Evolution_Addressbook_CallStatus status;
  GError *error = NULL;

  if (!priv->tpdb)
  {
    status = BookRemoved;
    goto done;
  }

  if (e_book_backend_tp_db_delete (priv->tpdb, &error)) {
    status = GNOME_Evolution_Addressbook_Success;
  } else {
    WARNING ("Error whilst deleting the db: %s",
        error ? error->message : "unknown error");
    g_clear_error (&error);
    status = GNOME_Evolution_Addressbook_OtherError;
  }

  g_object_unref (priv->tpdb);
  priv->tpdb = NULL;

done:
  e_data_book_respond_remove (closure->book, closure->opid, status); 

  g_object_unref (closure->book);
  g_object_unref (closure->backend);

  g_free (closure);

  return FALSE;
}

static void
e_book_backend_tp_remove (EBookBackend *backend, EDataBook *book, guint32 opid)
{
  RemoveClosure *closure;

  closure = g_new0 (RemoveClosure, 1);

  closure->backend = g_object_ref (backend);
  closure->book = g_object_ref (book);
  closure->opid = opid;

  g_idle_add (remove_idle_cb, closure);
}

static void
e_book_backend_tp_set_view_sort_order (EBookBackend *backend, EDataBookView *book_view, const gchar *query_term)
{
  ContactSortOrder sort_order;

  if (g_strcmp0 (query_term, "first-last") == 0)
    sort_order = CONTACT_SORT_ORDER_FIRST_LAST;
  else if (g_strcmp0 (query_term, "last-first") == 0)
    sort_order = CONTACT_SORT_ORDER_LAST_FIRST;
  else if (g_strcmp0 (query_term, "nick") == 0)
    sort_order = CONTACT_SORT_ORDER_NICKNAME;
  else {
    WARNING ("unsupported sort order '%s'", query_term);
    return;
  }

  g_object_set_data (G_OBJECT (book_view), BOOK_VIEW_SORT_ORDER_DATA_KEY,
      GINT_TO_POINTER (sort_order));
}

static void
e_book_backend_tp_sync (EBookBackend *backend)
{
  unsupported_method ("sync");
}

static void
e_book_backend_tp_class_init (EBookBackendTpClass *klass)
{
  GObjectClass *object_class = NULL;
  EBookBackendClass *backend_class = NULL;

  object_class = G_OBJECT_CLASS (klass);
  backend_class = E_BOOK_BACKEND_CLASS (klass);

  object_class->dispose = e_book_backend_tp_dispose;
  object_class->finalize = e_book_backend_tp_finalize;

  backend_class->load_source = e_book_backend_tp_load_source;
  backend_class->get_static_capabilities = e_book_backend_tp_get_static_capabilities;
  backend_class->start_book_view = e_book_backend_tp_start_book_view;
  backend_class->stop_book_view = e_book_backend_tp_stop_book_view;
  backend_class->cancel_operation = e_book_backend_tp_cancel_operation;
  backend_class->set_mode = e_book_backend_tp_set_mode;
  backend_class->modify_contact = e_book_backend_tp_modify_contact;
  backend_class->create_contact = e_book_backend_tp_create_contact;
  backend_class->create_contacts = e_book_backend_tp_create_contacts;
  backend_class->remove_contacts = e_book_backend_tp_remove_contacts;

  /* Unsupported methods. These are included to prevent a crash due to eds
   * assuming the methods are non-NULL (and dereferencing them) */
  backend_class->authenticate_user = e_book_backend_tp_authenticate_user;
  backend_class->get_changes = e_book_backend_tp_get_changes;
  backend_class->get_contact = e_book_backend_tp_get_contact;
  backend_class->get_contact_list = e_book_backend_tp_get_contact_list;
  backend_class->get_required_fields = e_book_backend_tp_get_required_fields;
  backend_class->get_supported_auth_methods = e_book_backend_tp_get_supported_auth_methods;
  backend_class->get_supported_fields = e_book_backend_tp_get_supported_fields;
  backend_class->modify_contacts = e_book_backend_tp_modify_contacts;
  backend_class->remove = e_book_backend_tp_remove;
  backend_class->set_view_sort_order = e_book_backend_tp_set_view_sort_order;
  backend_class->sync = e_book_backend_tp_sync;

  /* There should be exactly one async OR sync implementation of each function,
   * so we don't create stubs for the sync functions here. */

  signals[READY_SIGNAL] = g_signal_new ("ready",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_FIRST,
      G_STRUCT_OFFSET (EBookBackendTpClass, ready),
      NULL, NULL,
      g_cclosure_marshal_VOID__VOID,
      G_TYPE_NONE,
      0);

  signals[MEMBERS_READY_SIGNAL] = g_signal_new ("members-ready",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_FIRST,
      G_STRUCT_OFFSET (EBookBackendTpClass, members_ready),
      NULL, NULL,
      g_cclosure_marshal_VOID__VOID,
      G_TYPE_NONE,
      0);

  g_type_class_add_private (klass, sizeof (EBookBackendTpPrivate));
}

EBookBackend *
e_book_backend_tp_new (void)
{
  return (EBookBackend *)g_object_new (E_TYPE_BOOK_BACKEND_TP, NULL);
}
