/*
 * UPnP Browser for Maemo
 *
 * resource_selection.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-defines.h>
#endif

#include <gtk/gtk.h>
#include <cybergarage/util/cstring.h>
#include <string.h>

#include "resource_selection.h"

ResourceSelectionDialog* resource_selection_dialog_new()
{
	ResourceSelectionDialog* dialog = g_new(ResourceSelectionDialog, 1);

	/* Create the view */
	dialog->dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog->dialog), "Select Resource");
	gtk_window_set_type_hint(GTK_WINDOW(dialog->dialog), 
				 GDK_WINDOW_TYPE_HINT_DIALOG);
	
	dialog->vbox = GTK_DIALOG(dialog->dialog)->vbox;
	gtk_widget_show(dialog->vbox);
	
	/* Create the tree view */
	resource_selection_dialog_create_view_model(dialog);

	/* Action area */
	dialog->dialog_action_area =
			GTK_DIALOG(dialog->dialog)->action_area;
	gtk_widget_show(dialog->dialog_action_area);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog->dialog_action_area),
				  GTK_BUTTONBOX_END);

	/* OK button */
	dialog->ok_button = gtk_button_new_with_label("OK");
	gtk_widget_show(dialog->ok_button);
	gtk_dialog_add_action_widget(GTK_DIALOG(dialog->dialog), 
				     dialog->ok_button, GTK_RESPONSE_OK);

	/* Cancel button */
	dialog->cancel_button = gtk_button_new_with_label("Cancel");
	gtk_widget_show(dialog->cancel_button);
	gtk_dialog_add_action_widget(GTK_DIALOG(dialog->dialog),
				     dialog->cancel_button, GTK_RESPONSE_CANCEL);

	gtk_widget_set_size_request(GTK_WIDGET(dialog->dialog), 600, 300);
	
	return dialog;
}

void resource_selection_dialog_create_view_model(ResourceSelectionDialog* dialog)
{
	GtkCellRenderer *renderer = NULL;
	GtkTreeViewColumn *column = NULL;
	
	/* Create list model */
	dialog->model = gtk_list_store_new(RESOURCEMODEL_COLUMNS,
					   G_TYPE_STRING);
	
	dialog->view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(dialog->model));
	gtk_widget_show(dialog->view);
	
	/* Create a scrolled window */
	dialog->scrolled_window = gtk_scrolled_window_new(
		gtk_tree_view_get_hadjustment(GTK_TREE_VIEW(dialog->view)),
		gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(dialog->view)));
	gtk_widget_show(dialog->scrolled_window);

	/* Set the scrollbar showing policy */
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->scrolled_window),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);

	/* Add the tree view to the scrolled window */
	gtk_container_add(GTK_CONTAINER(dialog->scrolled_window),
			  dialog->view);

	/* Add the scrolled window to the dialog */
	gtk_container_add(GTK_CONTAINER(dialog->vbox),
			  dialog->scrolled_window);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes("Resource", renderer,
						"text", 
						RESOURCEMODEL_COLUMN_NAME,
						NULL);

	gtk_tree_view_append_column(GTK_TREE_VIEW(dialog->view), column);
}

void resource_selection_dialog_delete(ResourceSelectionDialog* dialog)
{
	g_return_if_fail(dialog != NULL);
	
	gtk_object_destroy(GTK_OBJECT(dialog->dialog));
	
	g_free(dialog);
}

/**
 * Parse multiple URI strings that are passed in resString as <URI><URI>...
 */
void resource_selection_dialog_update(ResourceSelectionDialog* dialog,
				      gchar* resString)
{
	int i = 0;
	int len = 0;
	int head = -1;
	int tail = -1;
	gchar* tmp = NULL;
	GtkTreeIter iter;
	
	g_return_if_fail(dialog != NULL);
	g_return_if_fail(resString != NULL);

	len = strlen(resString);
	tmp = g_new0(gchar, len);
	
	while(i < len)
	{
		if (resString[i] == '<')
		{
			head = i + 1;
		}
		else if (resString[i] == '>')
		{
			tail = i - 1;			
		}
		
		if (head != -1 && tail != -1 && tail > head && tail < len)
		{
			memcpy(tmp, resString + head, tail - head + 1);
			tmp[tail+1] = '\0';

			gtk_list_store_append(dialog->model, &iter);
			gtk_list_store_set(dialog->model, &iter,
					   RESOURCEMODEL_COLUMN_NAME,
						   tmp,
					   -1);
			head = tail = -1;
		}
		
		i++;
	}
}

gboolean resource_selection_dialog_get_selection(ResourceSelectionDialog* dialog,
						 gchar** uri)
{	
	GtkTreeModel *model = NULL;
	GtkTreeSelection *selection = NULL;
	GtkTreeIter iter;

	g_return_val_if_fail(dialog != NULL, FALSE);
	
	/* Get the selection from the folder view */
	selection = gtk_tree_view_get_selection(
				GTK_TREE_VIEW(dialog->view));

	if (!gtk_tree_selection_get_selected(selection,
					     &model,
					     &iter))
	{
		return FALSE;
	}

	/* Get the node's URI and class */
	gtk_tree_model_get(model, &iter, RESOURCEMODEL_COLUMN_NAME, uri, -1);
	
	return TRUE;
}

gboolean resource_selection_dialog_run(gchar* resString, gchar** selection)
{
	gboolean result = FALSE;
	ResourceSelectionDialog* dialog = NULL;

	if (resString == NULL || selection == NULL)
	{
		return FALSE;
	}
	
	dialog = resource_selection_dialog_new();
	if (dialog == NULL)
	{
		return FALSE;
	}

	resource_selection_dialog_update(dialog, resString);
		
	if (gtk_dialog_run(GTK_DIALOG(dialog->dialog)) == GTK_RESPONSE_OK)
	{
		resource_selection_dialog_get_selection(dialog, selection);
		result = TRUE;
	}
	else
	{
		result = FALSE;
	}

	resource_selection_dialog_delete(dialog);
	
	return result;
}
