/*
 * UPnP Browser for Maemo (player part)
 *
 * player.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#include <glib.h>
#include <libosso.h>
#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#include "player.h"

gboolean player_play_uri(osso_context_t *osso, const gchar *arg_uri, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	gboolean result = FALSE;
	gchar *uri = NULL;
	gchar *query_begin = NULL;
	
	g_return_val_if_fail(osso != NULL, FALSE);
	
	message = dbus_message_new_method_call(
            PLAYER_DBUS_ENGINE_SERVICE_NAME,
            PLAYER_DBUS_ENGINE_OBJECT_NAME,
            PLAYER_DBUS_ENGINE_INTERFACE_NAME,
            "play_media" );

        g_return_val_if_fail(message != NULL, FALSE);
	
	/* It seems that mediaengine can't handle URIs which have query part,
	   so just strip the query - makes things to work e.g. with WMC */
	uri = g_strdup(arg_uri);
	query_begin = strstr(uri, "?");
	if (query_begin != NULL)
	{
		query_begin[0] = 0;
	}
	
	result = dbus_message_append_args(
			message,
			DBUS_TYPE_STRING, &uri,
			DBUS_TYPE_INVALID );
	
	if (query_begin != NULL)
	{
		query_begin[0] = '?';
	}

	if (result != TRUE)
	{
		g_free(uri);
		return FALSE;
	}

        dbus_error_init( &dbus_error );

	reply = dbus_connection_send_with_reply_and_block(
	      osso_get_dbus_connection(osso), 
	      message, PLAYER_PLAY_TIMEOUT,
              &dbus_error);

	g_free(uri);
	uri = NULL;
	
	dbus_message_unref(message);
	if (reply != NULL) dbus_message_unref(reply);
	
	return TRUE;
}

gboolean player_stop(osso_context_t *osso, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	dbus_uint32_t client_id = 0;

	g_return_val_if_fail(osso != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"stop" );

	g_return_val_if_fail(message != NULL, FALSE);
	
	dbus_message_set_no_reply( message, TRUE );
		
	result = dbus_connection_send(osso_get_dbus_connection(osso), 
				      message, &client_id);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_connection_flush(osso_get_dbus_connection(osso));
	
	dbus_message_unref(message);

	return TRUE;
}

gboolean player_pause(osso_context_t *osso, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	
	g_return_val_if_fail(osso != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"pause" );
	
	g_return_val_if_fail(message != NULL, FALSE);
	
        reply = dbus_connection_send_with_reply_and_block(
			osso_get_dbus_connection(osso), 
			message, -1, &dbus_error);
	
        dbus_message_unref( message );
	if (reply != NULL) dbus_message_unref( reply );
	
	return TRUE;
}

gboolean player_set_mute(osso_context_t *osso, gboolean muted, GError **err)
{
	DBusError dbus_error;
	dbus_uint32_t client_id = 0;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	
	g_return_val_if_fail(osso != NULL, FALSE);
	
	dbus_error_init( &dbus_error );
	
	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"mute" );

	g_return_val_if_fail(message != NULL, FALSE);
	
	result = dbus_message_append_args(
			message,
			DBUS_TYPE_BOOLEAN, &muted,
			DBUS_TYPE_INVALID);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_message_set_no_reply(message, TRUE);

	result = dbus_connection_send(
		osso_get_dbus_connection(osso), 
		message, &client_id);
	
	g_return_val_if_fail(result != FALSE, FALSE);
	
	dbus_message_unref(message);
	
	return TRUE;
}

gboolean player_set_volume(osso_context_t *osso, gint value, GError **err)
{
	DBusError dbus_error;
	dbus_uint32_t client_id = 0;
	DBusMessage *message = NULL;
	gboolean result = FALSE;
	gdouble volume_level = 0.0;
	
	g_return_val_if_fail(osso != NULL, FALSE);
	
	dbus_error_init( &dbus_error );

	message = dbus_message_new_method_call(
		PLAYER_DBUS_ENGINE_SERVICE_NAME,
		PLAYER_DBUS_ENGINE_OBJECT_NAME,
		PLAYER_DBUS_ENGINE_INTERFACE_NAME,
		"set_volume" );

	dbus_message_set_no_reply( message, TRUE );

	volume_level = (gdouble)(value / PLAYER_VOLUME_MODIFIER);
        result = dbus_message_append_args(
			message,
			DBUS_TYPE_DOUBLE, value / PLAYER_VOLUME_MODIFIER,
			DBUS_TYPE_DOUBLE, &volume_level,
			DBUS_TYPE_INVALID);
        
	result = dbus_connection_send(
            osso_get_dbus_connection(osso), message, &client_id);
        
	dbus_connection_flush(osso_get_dbus_connection(osso));

        dbus_message_unref( message );
	
	return TRUE;
}

gboolean player_get_position(osso_context_t *osso, gint *pos, gint *length, GError **err)
{
	DBusError dbus_error;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	DBusMessageIter iter;
	dbus_int32_t int32;
	
	g_return_val_if_fail(osso != NULL, FALSE);
	
	message = dbus_message_new_method_call(
            PLAYER_DBUS_ENGINE_SERVICE_NAME,
            PLAYER_DBUS_ENGINE_OBJECT_NAME,
            PLAYER_DBUS_ENGINE_INTERFACE_NAME,
            "get_position");

        g_return_val_if_fail(message != NULL, FALSE);

	dbus_error_init(&dbus_error);

	reply = dbus_connection_send_with_reply_and_block(
	      osso_get_dbus_connection(osso), 
	      message, PLAYER_PLAY_TIMEOUT,
              &dbus_error);
	
	g_return_val_if_fail(reply != NULL, FALSE);

	if (reply != NULL)
	{
		dbus_message_iter_init(reply, &iter);
		dbus_message_iter_get_basic(&iter, &int32);
		*pos = (gint) int32;
		
		if (length != NULL && dbus_message_iter_next(&iter) == TRUE)
		{
			dbus_message_iter_get_basic(&iter, &int32);
			*length = (gint) int32;
		}
	}
	
	dbus_message_unref(message);
	if (reply != NULL) dbus_message_unref(reply);

	return TRUE;
}
