/* 
 * CyberLinkC AV API for Rendering Control Service
 *
 * crcscontrolpoint.h
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license,
 * see file COPYING.
 */

#ifndef CG_UPNP_AV_CRCSCONTROLPOINT_H
#define CG_UPNP_AV_CRCSCONTROLPOINT_H

#include <cybergarage/renderingcontrol/crenderingcontrol.h>
#include <cybergarage/renderingcontrol/crcsactionarguments.h>
#include <cybergarage/upnp/cdevice.h>

/*****************************************************************************
 * Rendering Control Service action invocations
 *****************************************************************************/

/**
 * Invoke an action on the upnp device
 *
 * @param upnpDevice The UPnP device
 * @param actionName The name of the action to invoke
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param ... A list of action-specific arguments
 *
 * @return An integer containing the UPnP status code
 *
 * The list of action-specific arguments is a comma-separated list, 
 * that is organised as pairs of char* ARG_NAME and char** ARG_VALUE.
 *
 * ARG_NAME contains the name of the argument, while ARG_VALUE contains
 * the value of the argument. You don't need to put anything inside OUT
 * arguments, as they are not sent along with the action; their contents
 * are set when the action post returns with OUT values that were received
 * from the peer. Finally, put a single NULL to end the argument list.
 */
int cg_upnp_av_rcs_control_invokeaction(CgUpnpDevice* upnpDevice,
					char* actionName,
					char** errorString,
					...);

/****************************************************************************
 * Presets
 ****************************************************************************/

/**
 * Get a list of rendering control presets from the device
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param currentPresetNameList List of RCS presets (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_listpresets(upnpDevice, \
					   errorString, \
					   instanceID, \
					   currentPresetNameList) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_LISTPRESETS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTPRESETNAMELIST, \
			currentPresetNameList, \
			NULL)


/**
 * Select a valid RCS preset from a list provided by \ref cg_upnp_av_rcs_control_listpresets
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param presetName The name of the RCS preset to select (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_selectpreset(upnpDevice, \
					    errorString, \
					    instanceID, \
					    presetName) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SELECTPRESET, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_PRESETNAME, \
			presetName, \
			NULL)

/****************************************************************************
 * Video
 ****************************************************************************/

/**
 * Get brightness
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param brightness The current brightness level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getbrightness(upnpDevice, \
					     errorString, \
					     instanceID, \
					     brightness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETBRIGHTNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTBRIGHNESS, \
			brightness, \
			NULL)

/**
 * Set brightness
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param brightness The desired brightness level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setbrightness(upnpDevice, \
					     errorString, \
					     instanceID, \
					     brightness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETBRIGHTNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDBRIGHTNESS, \
			brightness, \
			NULL)

/**
 * Get contrast
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param contrast The current contrast level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getcontrast(upnpDevice, \
					   errorString, \
					   instanceID, \
					   contrast) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETCONTRAST, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTCONTRAST, \
			contrast, \
			NULL)

/**
 * Set contrast
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param contrast The desired contrast level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setcontrast(upnpDevice, \
					   errorString, \
					   instanceID, \
					   contrast) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETCONTRAST, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDCONTRAST, \
			contrast, \
			NULL)

/**
 * Get sharpness
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param sharpness The current sharpness level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getsharpness(upnpDevice, \
					    errorString, \
					    instanceID, \
					    sharpness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETSHARPNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTSHARPNESS, \
			sharpness, \
			NULL)

/**
 * Set sharpness
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param sharpness The desired sharpness level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setsharpness(upnpDevice, \
					    errorString, \
					    instanceID, \
					    sharpness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETSHARPNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDSHARPNESS, \
			sharpness, \
			NULL)

/**
 * Get red video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The current red video gain (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getredvideogain(upnpDevice, \
					       errorString, \
					       instanceID, \
					       gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETREDVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTREDVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Set red video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The desired red video gain (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setredvideogain(upnpDevice, \
					       errorString, \
					       instanceID, \
					       gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETREDVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDREDVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Get green video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The current green video gain (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getgreenvideogain(upnpDevice, \
						 errorString, \
						 instanceID, \
						 gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETGREENVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTGREENVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Set green video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The desired green video gain (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setgreenvideogain(upnpDevice, \
						 errorString, \
						 instanceID, \
						 gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETGREENVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDGREENVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Get blue video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The current blue video gain (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getbluevideogain(upnpDevice, \
						errorString, \
						instanceID, \
						gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETBLUEVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTBLUEVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Set blue video gain
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param gain The desired blue video gain (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setbluevideogain(upnpDevice, \
						errorString, \
						instanceID, \
						gain) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETBLUEVIDEOGAIN, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDBLUEVIDEOGAIN, \
			gain, \
			NULL)

/**
 * Get red video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The current red video black level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getredvideoblacklevel(upnpDevice, \
						     errorString, \
						     instanceID, \
						     blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETREDVIDEOBLACKLEVEL, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTREDVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Set red video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The desired red video black level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setredvideoblacklevel(upnpDevice, \
						     errorString, \
						     instanceID, \
						     blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETREDVIDEOBLACKLEVEL, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDREDVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Get green video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The current green video black level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getgreenvideoblacklevel(upnpDevice, \
						       errorString, \
						       instanceID, \
						       blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETGREENVIDEOBLACKLEVEL, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTGREENVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Set green video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The desired green video black level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setgreenvideoblacklevel(upnpDevice, \
						       errorString, \
						       instanceID, \
						       blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETGREENVIDEOBLACKLEVEL, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDGREENVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Get blue video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The current blue video black level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getbluevideoblacklevel(upnpDevice, \
						      errorString, \
						      instanceID, \
						      blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETBLUEVIDEOBLACKLEVEL, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTBLUEVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Set blue video black level
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param blackLevel The desired blue video black level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setbluevideoblacklevel(upnpDevice, \
						      errorString, \
						      instanceID, \
						      blackLevel) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETBLUEVIDEOBLACKLEVEL, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDBLUEVIDEOBLACKLEVEL, \
			blackLevel, \
			NULL)

/**
 * Get color temperature
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param colorTemp The current color temperature (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getcolortemperature(upnpDevice, \
						   errorString, \
						   instanceID, \
						   colorTemp) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETCOLORTEMPERATURE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTCOLORTEMPERATURE, \
			colorTemp, \
			NULL)

/**
 * Set color temperature
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param colorTemp The desired color temperature (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setcolortemperature(upnpDevice, \
						   errorString, \
						   instanceID, \
						   colorTemp) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETCOLORTEMPERATURE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDCOLORTEMPERATURE, \
			colorTemp, \
			NULL)

/**
 * Get horizontal keystone
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param keystone The current horizontal keystone value (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_gethorizontalkeystone(upnpDevice, \
						     errorString, \
						     instanceID, \
						     keystone) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETHORIZONTALKEYSTONE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTHORIZONTALKEYSTONE, \
			keystone, \
			NULL)

/**
 * Set horizontal keystone
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param keystone The desired horizontal keystone value (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_sethorizontalkeystone(upnpDevice, \
						     errorString, \
						     instanceID, \
						     keystone) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETHORIZONTALKEYSTONE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDHORIZONTALKEYSTONE, \
			keystone, \
			NULL)

/**
 * Get vertical keystone
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param keystone The current vertical keystone value (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getverticalkeystone(upnpDevice, \
						   errorString, \
						   instanceID, \
						   keystone) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETVERTICALKEYSTONE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CURRENTVERTICALKEYSTONE, \
			keystone, \
			NULL)

/**
 * Set vertical keystone
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param keystone The desired vertical keystone value (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setverticalkeystone(upnpDevice, \
						   errorString, \
						   instanceID, \
						   keystone) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETVERTICALKEYSTONE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_DESIREDVERTICALKEYSTONE, \
			keystone, \
			NULL)

/****************************************************************************
 * Audio
 ****************************************************************************/

/**
 * Get mute state
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param mute The current mute state (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getmute(upnpDevice, \
				       errorString, \
				       instanceID, \
				       channel, \
				       mute) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETMUTE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_CURRENTMUTE, \
			mute, \
			NULL)

/**
 * Set mute state
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param mute The desired mute state (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setmute(upnpDevice, \
				       errorString, \
				       instanceID, \
				       channel, \
				       mute) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETMUTE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_DESIREDMUTE, \
			mute, \
			NULL)

/**
 * Get volume
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param volume The current volume level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getvolume(upnpDevice, \
					 errorString, \
					 instanceID, \
					 channel, \
					 volume) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETVOLUME, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_CURRENTVOLUME, \
			volume, \
			NULL)

/**
 * Set volume
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param volume The desired volume level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setvolume(upnpDevice, \
					 errorString, \
					 instanceID, \
					 channel, \
					 volume) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETVOLUME, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_DESIREDVOLUME, \
			volume, \
			NULL)

/**
 * Get volume in deciBels (dB)
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param volume The current volume level (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getvolumedb(upnpDevice, \
					   errorString, \
					   instanceID, \
					   channel, \
					   volume) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETVOLUMEDB, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_CURRENTVOLUMEDB, \
			volume, \
			NULL)

/**
 * Set volume in deciBels (dB)
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param volume The desired volume level (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setvolumedb(upnpDevice, \
					   errorString, \
					   instanceID, \
					   channel, \
					   volume) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETVOLUMEDB, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_DESIREDVOLUMEDB, \
			volume, \
			NULL)


/**
 * Get the valid value range for get/setvolumedb
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param min The minimum value (OUT)
 * @param max The maximum value (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getvolumedbrange(upnpDevice, \
						errorString, \
						instanceID, \
						channel, \
						min, \
						max) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETVOLUMEDBRANGE, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_MINVALUE, \
			min, \
			CG_UPNP_RCS_ARG_MAXVALUE, \
			max, \
			NULL)

/**
 * Get loudness state
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param loudness The current loudness state (OUT)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_getloudness(upnpDevice, \
					   errorString, \
					   instanceID, \
					   channel, \
					   loudness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_GETLOUDNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_CURRENTLOUDNESS, \
			loudness, \
			NULL)

/**
 * Set loudness state
 *
 * @param upnpDevice The UPnP device
 * @param errorString A newly-created error string if the action fails. Pass
 *                    NULL if you don't want to use it.
 * @param instanceID The ID of the current RCS instance (IN)
 * @param channel Channel name (IN)
 * @param loudness The desired loudness state (IN)
 *
 * @return An integer containing the UPnP status code
 */
#define cg_upnp_av_rcs_control_setloudness(upnpDevice, \
					   errorString, \
					   instanceID, \
					   channel, \
					   loudness) \
	\
	cg_upnp_av_rcs_control_invokeaction(upnpDevice, \
			CG_UPNP_RCS_ACTION_SETLOUDNESS, \
			errorString, \
			CG_UPNP_RCS_ARG_INSTANCEID, \
			instanceID, \
			CG_UPNP_RCS_ARG_CHANNEL, \
			channel, \
			CG_UPNP_RCS_ARG_DESIREDLOUDNESS, \
			loudness, \
			NULL)

#endif /* CG_UPNP_AV_CRCSCONTROLPOINT_H */
