/*
  copyright   : GPL
  title       : fakeroot
  description : create a "fake" root shell, by wrapping 
                functions like chown, stat, etc. Useful for debian
                packaging mechanism
  author      : joost witteveen, joostje@debian.org

 Network-enabled version by timo.savola@movial.fi
*/

/*
  upon startup, the fakeroot script (/usr/bin/fakeroot) 
  forks faked (this program), and the shell or user program that
  will run with the libtricks.so.0.0 wrapper.
  
  These tree running programs have the following tasks:
    
    fakeroot scripot
       starts the other two processes, waits for the user process to
       die, and then send a SIGTERM signal to faked, causing
       Faked to clear the ipc message queues.

    faked
       the ``main'' daemon, creates ipc message queues, and later
       receives ipc messages from the user program, maintains
       fake inode<->ownership database (actually just a 
       lot of struct stat entries). Will clear ipc message ques
       upon receipt of a SIGTERM. Will show debug output upon
       receipt of a SIGUSR1 (if started with -d debug option)

    user program
       Any shell or other programme, run with 
       LD_PRELOAD=libtricks.so.0.0, and FAKEROOTKEY=ipc-key,
       thus the executed commands will communicate with
       faked. libtricks will wrap all file ownership etc modification
       calls, and send the info to faked. Also the stat() function
       is wrapped, it will first ask the database kept by faked
       and report the `fake' data if available.

  The following functions are currently wrapped:
     getuid(), geteuid(), getgid(), getegid(),
     mknod()
     chown(), fchown() lchown()
     chmod(), fchmod() 
     mkdir(),
     lstat(), fstat(), stat() (actually, __xlstat, ...)
     unlink(), remove(), rmdir(), rename()
    
  comments:
    I need to wrap unlink because of the following:
        install -o admin foo bar
	rm bar
        touch bar         //bar now may have the same inode:dev as old bar,
	                  //but unless the rm was caught,
			  //fakeroot still has the old entry.
        ls -al bar
    Same goes for all other ways to remove inodes form the filesystem,
    like rename(existing_file, any_file).

    The communication between client (user progamme) and faked happens 
    with inode/dev information, not filenames. This is 
    needed, as the client is the only one who knows what cwd is,
    so it's much easier to stat in the client. Otherwise, the daemon
    needs to keep a list of client pids vs cwd, and I'd have to wrap
    fork e.d., as they inherit their parent's cwd. Very compilcated.
    
    */			       
/* ipc documentation bugs: msgsnd(2): MSGMAX=4056, not 4080 
   (def in ./linux/msg.h, couldn't find other def in /usr/include/ 
   */

#include "config.h"
#include "message.h"
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif


void process_chown(struct fake_msg *buf);
void process_chmod(struct fake_msg *buf);
void process_mknod(struct fake_msg *buf);
void process_stat(struct fake_msg *buf);
void process_unlink(struct fake_msg *buf);

typedef void (*process_func)(struct fake_msg *);

process_func func_arr[]={process_chown,
			 process_chmod,
			 process_mknod,
			 process_stat,
			 process_unlink,
			 };

int highest_funcid = sizeof(func_arr)/sizeof(func_arr[0]);

unsigned int debug = 0, unknown_is_real = 0;


static int comm_sd = -1;

static void fail(const char *msg)
{
  if (errno > 0)
    fprintf(stderr, "fakeroot daemon: %s (%s)\n", msg, strerror(errno));
  else
    fprintf(stderr, "fakeroot daemon: %s\n", msg);

  exit(1);
}

static void cpyfakefake(struct fakestat *dest,
			const struct fakestat *source)
{
  dest->mode = source->mode;
  dest->ino  = source->ino ;
  dest->uid  = source->uid ;
  dest->gid  = source->gid ;
  dest->dev  = source->dev ;
  dest->rdev = source->rdev;
  /* DON'T copy the nlink count! The system always knows this one better! */
}

static void send_fakem(const struct fake_msg *buf)
{
  struct fake_msg fm;

  fm.id = htonl(buf->id);
  fm.st.uid = htonl(buf->st.uid);
  fm.st.gid = htonl(buf->st.gid);
  fm.st.ino = htonll(buf->st.ino);
  fm.st.dev = htonll(buf->st.dev);
  fm.st.rdev = htonll(buf->st.rdev);
  fm.st.mode = htonl(buf->st.mode);
  fm.st.nlink = htonl(buf->st.nlink);

  while (1) {
    ssize_t len;

    len = write(comm_sd, &fm, sizeof (fm));
    if (len > 0)
      break;

    if (errno == EINTR)
      continue;

    fail("write");
  }
}

static int get_fakem(struct fake_msg *buf)
{
  while (1) {
    ssize_t len;

    len = read(comm_sd, buf, sizeof (struct fake_msg));
    if (len > 0)
      break;

    if (len == 0)
      return -1;

    if (errno == EINTR)
      continue;

    fail("read");
  }

  buf->id = ntohl(buf->id);
  buf->st.uid = ntohl(buf->st.uid);
  buf->st.gid = ntohl(buf->st.gid);
  buf->st.ino = ntohll(buf->st.ino);
  buf->st.dev = ntohll(buf->st.dev);
  buf->st.rdev = ntohll(buf->st.rdev);
  buf->st.mode = ntohl(buf->st.mode);
  buf->st.nlink = ntohl(buf->st.nlink);

  return 0;
}




#define fakestat_equal(a, b)  ((a)->dev == (b)->dev && (a)->ino == (b)->ino)

struct data_node_s;
typedef struct data_node_s {
  struct data_node_s *next;
  struct fakestat     buf;
} data_node_t;

#define data_node_get(n)   ((struct fakestat *) &(n)->buf)
#define data_node_next(n)  ((n)->next)

static data_node_t *data_head = NULL;

static data_node_t *data_find(const struct fakestat *key)
{
  data_node_t *n;

  for (n = data_head; n; n = n->next)
    if (fakestat_equal(&n->buf, key))
      break;

  return n;
}

static void data_insert(const struct fakestat *buf)
{
  data_node_t *n, *last = NULL;

  for (n = data_head; n; last = n, n = n->next)
    if (fakestat_equal(&n->buf, buf))
      break;

  if (n == NULL) {
    n = calloc(1, sizeof (data_node_t));

    if (last)
      last->next = n;
    else
      data_head = n;
  }

  memcpy(&n->buf, buf, sizeof (struct fakestat));
}

static data_node_t *data_erase(data_node_t *pos)
{
  data_node_t *n, *prev = NULL, *next;

  for (n = data_head; n; prev = n, n = n->next)
    if (n == pos)
      break;

  next = n->next;

  if (n == data_head)
    data_head = next;
  else
    prev->next = next;

  free(n);

  return next;
}

static unsigned int data_size(void)
{
  unsigned int size = 0;
  data_node_t *n;

  for (n = data_head; n; n = n->next)
    size++;

  return size;
}

#define data_begin()  (data_head)
#define data_end()    (NULL)


static struct {
  unsigned int capacity;
  unsigned int size;
  int *array;
} sd_list = {
  0, 0, NULL
};

static void sd_list_add(int sd)
{
  if (sd_list.capacity == sd_list.size) {
    int *mem;

    sd_list.capacity += 16;

    if (sd_list.array == NULL) {

      sd_list.array = malloc(sd_list.capacity * sizeof (int));
      if (!sd_list.array)
	fail("malloc");

    } else {

      sd_list.array = realloc(sd_list.array, sd_list.capacity * sizeof (int));
      if (!sd_list.array)
	fail("realloc");

    }
  }

  sd_list.array[sd_list.size] = sd;
  sd_list.size++;
}

static void sd_list_remove(unsigned int i)
{
  for (i++; i < sd_list.size; i++)
    sd_list.array[i - 1] = sd_list.array[i];

  sd_list.size--;
}

#define sd_list_size()    (sd_list.size)
#define sd_list_index(i)  (sd_list.array[(i)])


/*********************************/
/*                               */
/* data base maintainance        */
/*                               */
/*********************************/
void debug_stat(const struct fakestat *st){
  fprintf(stderr,"dev:ino=(%lx:%li), mode=0%lo, own=(%li,%li), nlink=%li, rdev=%li\n",
	  st->dev,
	  st->ino,
	  st->mode,
	  st->uid,
	  st->gid,
	  st->nlink,
	  st->rdev);
}

void insert_or_overwrite(struct fakestat *st){
  data_node_t *i;
  
  i = data_find(st);
  if (i == data_end()) {
    if(debug){
      fprintf(stderr,"FAKEROOT: insert_or_overwrite unknown stat: ");
      debug_stat(st);
    }
    data_insert(st);
  }
  else
    memcpy(data_node_get(i), st, sizeof (struct fakestat));
}


/*******************************************/
/*                                         */
/* process requests from wrapper functions */
/*                                         */
/*******************************************/


void process_chown(struct fake_msg *buf){
  struct fakestat *stptr;
  struct fakestat st;
  data_node_t *i;
  
  if(debug){
    fprintf(stderr,"FAKEROOT: chown ");
    debug_stat(&buf->st);
  }
  i = data_find(&buf->st);
  if (i != data_end()) {
    stptr = data_node_get(i);
    /* From chown(2): If  the owner or group is specified as -1, 
       then that ID is not changed. 
       Cannot put that test in libtricks, as at that point it isn't
       known what the fake user/group is (so cannot specify `unchanged')
       
       I typecast to (uint32_t), as st.uid may be bigger than uid_t.
       In that case, the msb in st.uid should be discarded.
       I don't typecaset to (uid_t), as the size of uid_t may vary
       depending on what libc (headers) were used to compile. So,
       different clients might actually use different uid_t's
       concurrently. Yes, this does seem farfeched, but was
       actually the case with the libc5/6 transition.
    */
    if ((uint32_t)buf->st.uid != (uint32_t)-1) 
      stptr->uid=buf->st.uid;
    if ((uint32_t)buf->st.gid != (uint32_t)-1)
      stptr->gid=buf->st.gid;
  }
  else{
    st=buf->st;
    /* See comment above.  We pretend that unknown files are owned
       by root.root, so we have to maintain that pretense when the
       caller asks to leave an id unchanged. */
    if ((uint32_t)st.uid == (uint32_t)-1)
       st.uid = 0;
    if ((uint32_t)st.gid == (uint32_t)-1)
       st.gid = 0;
    insert_or_overwrite(&st);
  }
}

void process_chmod(struct fake_msg *buf){
  struct fakestat *st;
  data_node_t *i;
  
  if(debug)
    fprintf(stderr,"FAKEROOT: chmod, mode=%lo\n",
	    buf->st.mode);
  
  i = data_find(&buf->st);
  if (i != data_end()) {
    st = data_node_get(i);
    st->mode = (buf->st.mode&~S_IFMT) | (st->mode&S_IFMT);
  }
  else{
    st=&buf->st;
    st->uid=0;
    st->gid=0;
  }
  insert_or_overwrite(st);
}
void process_mknod(struct fake_msg *buf){
  struct fakestat *st;
  data_node_t *i;
  
  if(debug)
    fprintf(stderr,"FAKEROOT: chmod, mode=%lo\n",
	    buf->st.mode);
  
  i = data_find(&buf->st);
  if (i != data_end()) {
    st = data_node_get(i);
    st->mode = buf->st.mode;
    st->rdev = buf->st.rdev;
  }
  else{
    st=&buf->st;
    st->uid=0;
    st->gid=0;
  }
  insert_or_overwrite(st);
}

void process_stat(struct fake_msg *buf){
  data_node_t *i;

  i = data_find(&buf->st);
  if(debug){
    fprintf(stderr,"FAKEROOT: process stat oldstate=");
    debug_stat(&buf->st);
  }
  if (i == data_end()) {
    if (debug)
      fprintf(stderr,"FAKEROOT: (previously unknown)\n");
    if (!unknown_is_real) {
      buf->st.uid=0;
      buf->st.gid=0;
    }
  }
  else{
    cpyfakefake(&buf->st, data_node_get(i));
    if(debug){
      fprintf(stderr,"FAKEROOT: (previously known): fake=");
      debug_stat(&buf->st);      
    }

  }
  send_fakem(buf);
}
//void process_fstat(struct fake_msg *buf){
//  process_stat(buf);
//}

void process_unlink(struct fake_msg *buf){

  if((buf->st.nlink==1)||
     (S_ISDIR(buf->st.mode)&&(buf->st.nlink==2))){
    data_node_t *i;
    i = data_find(&buf->st);
    if (i != data_end()) {
      if(debug){
	fprintf(stderr,"FAKEROOT: unlink known file, old stat=");
	debug_stat(data_node_get(i));
      }
      data_erase(i);
    }
    if (data_find(&buf->st) != data_end()) {
      fprintf(stderr,"FAKEROOT************************************************* cannot remove stat (a \"cannot happen\")\n");
    }
  }
}

void debugdata(int dummy){
  data_node_t *i;

  fprintf(stderr," FAKED keeps data of %i inodes:\n", data_size());
  for (i = data_begin(); i != data_end(); i = data_node_next(i))
    debug_stat(data_node_get(i));
}


void process_msg(struct fake_msg *buf){

  func_id f;
  f= buf->id;
  if (f <= highest_funcid)
    func_arr[f]((struct fake_msg*)buf);
  
}

void get_msg(const int listen_sd)
{
  fd_set readfds;
  struct fake_msg buf;

  while (1) {
    int maxfd, count, i;

    FD_ZERO(&readfds);

    FD_SET(listen_sd, &readfds);
    maxfd = listen_sd;

    for (i = 0; i < sd_list_size(); i++) {
      const int sd = sd_list_index(i);

      FD_SET(sd, &readfds);
      maxfd = MAX(sd, maxfd);
    }

    count = select(maxfd + 1, &readfds, NULL, NULL, NULL);
    if (count < 0) {
      if (errno == EINTR)
	continue;
      
      fail("select");
    }

    for (i = 0; i < sd_list_size(); ) {
      const int sd = sd_list_index(i);

      if (FD_ISSET(sd, &readfds)) {
	if (debug)
	  fprintf(stderr, "fakeroot: message from fd=%d\n", sd);

	comm_sd = sd;

	if (get_fakem(&buf) < 0) {
	  if (debug)
	    fprintf(stderr, "fakeroot: closing fd=%d\n", sd);

	  close(sd);

	  sd_list_remove(i);
	  continue;
	}

	process_msg(&buf);
      }

      i++;
    }

    if (FD_ISSET(listen_sd, &readfds)) {
      struct sockaddr_in addr;
      socklen_t len = sizeof (addr);
      const int sd = accept(listen_sd, (struct sockaddr *) &addr, &len);
      if (sd < 0)
	fail("accept");

      if (debug) {
	char host[256];
	if (getnameinfo((struct sockaddr *) &addr, len, host, sizeof (host),
			NULL, 0, 0) == 0)
	  fprintf(stderr, "fakeroot: connection from %s, fd=%d\n", host, sd);
      }

      comm_sd = sd;

      if (get_fakem(&buf) < 0) {
	if (debug)
	  fprintf(stderr, "fakeroot: closing fd=%d\n", sd);

	close(sd);
	continue;
      }

      process_msg(&buf);

      sd_list_add(sd);
    }
  }
}

/***********/
/*         */
/* misc    */
/*         */
/***********/



void cleanup(int g){
  if(debug)
    fprintf(stderr, "fakeroot:"
	    " signal=%i\n",  g);
  if(g!=-1)
    exit(0);
}

/*************/
/*           */
/*   main    */
/*           */
/*************/


static long int read_intarg(char **argv){
  if(!*argv){
    fprintf(stderr,"%s needs numeric argument\n",*(argv-1));
    exit(1);
  } else{
    return atoi(*argv);
  }
}

int main(int argc, char **argv){
  struct sigaction sa,sa_debug;
  int i;
  unsigned int foreground = 0;
  int pid;
  int sd, yes = 1;
  unsigned int port = 0;
  struct sockaddr_in addr;
  socklen_t addr_len;

  if(getenv(FAKEROOTKEY_ENV)){
    //(I'm not sure -- maybe this can work?)
    fprintf(stderr,"Please, don't run fakeroot from within fakeroot!\n");
    exit(1);
  }
  while(*(++argv)){
    if(!strcmp(*argv,"--port"))
      port=read_intarg(++argv);
    else if(!strcmp(*argv,"--foreground"))
      foreground = 1;
    else if(!strcmp(*argv,"--debug"))
      debug = 1;
    else if(!strcmp(*argv,"--unknown-is-real"))
      unknown_is_real = 1;
    else if(!strcmp(*argv,"--version")) {
      fprintf(stderr,"fakeroot-net version " VERSION "\n");
      exit(0);
    } else {
      fprintf(stderr,"faked, daemon for fake root enfironment\n");
      fprintf(stderr,"Best used from the shell script `fakeroot'\n");
      fprintf(stderr,"options for fakeroot: --port, --foreground, --debug, --unknown-is-real, --version\n");
      exit(1);
    }
  }
  
  sd = socket(PF_INET, SOCK_STREAM, 0);
  if (sd < 0)
    fail("socket");

  if (setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof (yes)) < 0)
    fail("setsockopt(SO_REUSEADDR)");

  if (port > 0) {
    memset((char *) &addr, 0, sizeof (addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port = htons(port);

    if (bind(sd, (struct sockaddr *) &addr, sizeof (addr)) < 0)
      fail("bind");
  }

  if (listen(sd, SOMAXCONN) < 0)
    fail("listen");

  addr_len = sizeof (addr);
  if (getsockname(sd, (struct sockaddr *) &addr, &addr_len) < 0)
    fail("getsockname");

  port = ntohs(addr.sin_port);

  sa.sa_handler=cleanup;
  sigemptyset(&sa.sa_mask);
  sa.sa_flags=0;
  //  sa.sa_restorer=0;


  sa_debug.sa_handler=debugdata;
  sigemptyset(&sa_debug.sa_mask);
  sa_debug.sa_flags=0;
  //  sa_debug.sa_restorer=0;
  
  for(i=1; i< NSIG; i++){
    switch (i){
    case SIGKILL:
    case SIGTSTP:
    case SIGCONT:
      break;
    case SIGUSR2:
      sigaction(i,&sa_debug,NULL);
      break;
    default:
      sigaction(i,&sa,NULL);
      break;
    }
  }
  


  if(!foreground){
    /* literally copied from the linux klogd code, go to background */
    if ((pid=fork()) == 0){
      int fl;
      int num_fds = getdtablesize();
      
      fflush(stdout);

      /* This is the child closing its file descriptors. */
      for (fl= 0; fl <= num_fds; ++fl)
	if (fl != sd)
	  close(fl);
      setsid();
    } else{
      printf("%i:%i\n",port,pid);

      exit(0);
    }
  } else{
    printf("%i:%i\n",port,getpid());
    fflush(stdout);
  }
  get_msg(sd);    /* we shouldn't return from this function */

  cleanup(-1);  /* if we do return, try to clean up and exit with a nonzero
		   return status */
  return 1;
}
