/*
  copyright : GPL
    (author: joost witteveen, joostje@debian.org)

 Network-enabled version by timo.savola@movial.fi


 This file contains the code (wrapper functions) that gets linked with 
 the programes run from inside fakeroot. These programes then communicate
 with the fakeroot daemon, that keeps information about the "fake" 
 ownerships etc. of the files etc.
    
 */

#include "communicate.h"
#include <dlfcn.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pthread.h>
#include <endian.h>


#ifndef _UTSNAME_LENGTH
/* for LINUX libc5 */
#  define _UTSNAME_LENGTH _SYS_NMLN
#endif


static int comm_sd = -1;
static pthread_mutex_t comm_sd_mutex = PTHREAD_MUTEX_INITIALIZER;


static void fail(const char *msg)
{
  if (errno > 0)
    fprintf(stderr, "libfakeroot: %s: %s\n", msg, strerror(errno));
  else
    fprintf(stderr, "libfakeroot: %s\n", msg);

  exit(1);
}


const char *env_var_set(const char *env){
  const char *s;
  
  s=getenv(env);
  
  if(s && *s)
    return s;
  else
    return NULL;
}

void cpyfakemstat(struct fake_msg *f,
                 const struct stat     *st){
  
  f->st.mode =st->st_mode;
  f->st.ino  =st->st_ino ;
  f->st.uid  =st->st_uid ;
  f->st.gid  =st->st_gid ;
  f->st.dev  =st->st_dev ;
  f->st.rdev =st->st_rdev;

  /* DO copy the nlink count. Although the system knows this
     one better, we need it for unlink().
     This actually opens up a race condition, if another command
     makes a hardlink on a file, while we try to unlink it. This
     may cause the record to be deleted, while the link continues
     to live on the disk. But the chance is small, and unlikely
     to occur in practical fakeroot conditions. */

  f->st.nlink=st->st_nlink;
}
void cpystatfakem(struct     stat *st,
                 const struct fake_msg *f){
  st->st_mode =f->st.mode;
  st->st_ino  =f->st.ino ;
  st->st_uid  =f->st.uid ;
  st->st_gid  =f->st.gid ;
  st->st_dev  =f->st.dev ;
  st->st_rdev =f->st.rdev;
  /* DON'T copy the nlink count! The system always knows
     this one better! */
  /*  st->st_nlink=f->st.nlink;*/
}
void cpyfakefake(struct fakestat *dest,
                 const struct fakestat *source){
  dest->mode =source->mode;
  dest->ino  =source->ino ;
  dest->uid  =source->uid ;
  dest->gid  =source->gid ;
  dest->dev  =source->dev ;
  dest->rdev =source->rdev;
  /* DON'T copy the nlink count! The system always knows
     this one better! */
  /*  dest->nlink=source->nlink;*/
}


#ifdef _LARGEFILE_SOURCE

void stat64from32(struct stat64 *s64, const struct stat *s32)
{
  /* I've added st_size and st_blocks here. 
     Don't know why they were missing -- joost*/
   s64->st_dev = s32->st_dev;
   s64->st_ino = s32->st_ino;
   s64->st_mode = s32->st_mode;
   s64->st_nlink = s32->st_nlink;
   s64->st_uid = s32->st_uid;
   s64->st_gid = s32->st_gid;
   s64->st_rdev = s32->st_rdev;
   s64->st_size = s32->st_size;
   s64->st_blksize = s32->st_blksize;
   s64->st_blocks = s32->st_blocks;
   s64->st_atime = s32->st_atime;
   s64->st_mtime = s32->st_mtime;
   s64->st_ctime = s32->st_ctime;
}

/* This assumes that the 64 bit structure is actually filled in and does not
   down case the sizes from the 32 bit one.. */
void stat32from64(struct stat *s32, const struct stat64 *s64)
{
   s32->st_dev = s64->st_dev;
   s32->st_ino = s64->st_ino;
   s32->st_mode = s64->st_mode;
   s32->st_nlink = s64->st_nlink;
   s32->st_uid = s64->st_uid;
   s32->st_gid = s64->st_gid;
   s32->st_rdev = s64->st_rdev;
   s32->st_size = (long)s64->st_size;
   s32->st_blksize = s64->st_blksize;
   s32->st_blocks = (long)s64->st_blocks;
   s32->st_atime = s64->st_atime;
   s32->st_mtime = s64->st_mtime;
   s32->st_ctime = s64->st_ctime;
}

#endif


static struct sockaddr *get_addr(void)
{
  static struct sockaddr_in addr = { 0 };

  if (!addr.sin_port) {
    char *str;
    int port;

    str = (char *) env_var_set(FAKEROOTKEY_ENV);
    if (!str) {
      errno = 0;
      fail("FAKEROOTKEY not defined in environment");
    }

    port = atoi(str);
    if (port <= 0 || port >= 65536) {
      errno = 0;
      fail("invalid port number in FAKEROOTKEY");
    }

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
    addr.sin_port = htons(port);
  }

  return (struct sockaddr *) &addr;
}

static void open_comm_sd(void)
{
  char *str;
  int base, val, yes = 1;

  if (comm_sd >= 0) {
    return;
  }

  base = getdtablesize() - 100;
  if (base < 3)
    base = 3;

  if ((str = getenv(FD_BASE_ENV))) {
    int fd = atoi(str);
    if (fd >= 3)
      base = fd;
  }

  comm_sd = socket(PF_INET, SOCK_STREAM, 0);
  if (comm_sd < 0)
    fail("socket");

  if (comm_sd < base) {
    int fd;

    do {
      fd = fcntl(comm_sd, F_DUPFD, base);
      if (fd < 0) {
	if (errno == EINTR)
	  continue;
	fail("fcntl(F_DUPFD)");
      }
    } while (0);

    close(comm_sd);
    comm_sd = fd;
  }

  if (setsockopt(comm_sd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof (yes)) < 0)
    fail("setsockopt(SO_REUSEADDR)");

  if (connect(comm_sd, get_addr(), sizeof (struct sockaddr_in)) < 0)
    fail("connect");
}

void close_comm_sd(void)
{
  /* We must synchronize since the vfork calls us in the parent process. */
  pthread_mutex_lock(&comm_sd_mutex);

  if (comm_sd >= 0) {
    close(comm_sd);
    comm_sd = -1;
  }

  pthread_mutex_unlock(&comm_sd_mutex);
}

static void __send_fakem(const struct fake_msg *buf)
{
  struct fake_msg fm;

  fm.id = htonl(buf->id);
  fm.st.uid = htonl(buf->st.uid);
  fm.st.gid = htonl(buf->st.gid);
  fm.st.ino = htonll(buf->st.ino);
  fm.st.dev = htonll(buf->st.dev);
  fm.st.rdev = htonll(buf->st.rdev);
  fm.st.mode = htonl(buf->st.mode);
  fm.st.nlink = htonl(buf->st.nlink);

  while (1) {
    ssize_t len;

    len = write(comm_sd, &fm, sizeof (fm));
    if (len > 0)
      break;

    if (len == 0) {
      errno = 0;
      fail("write: socket is closed");
    }

    if (errno == EINTR)
      continue;

    fail("write");
  }
}

void send_fakem(const struct fake_msg *buf)
{
  pthread_mutex_lock(&comm_sd_mutex);

  open_comm_sd();
  __send_fakem(buf);

  pthread_mutex_unlock(&comm_sd_mutex);
}

static void __get_fakem(struct fake_msg *buf)
{
  while (1) {
    ssize_t len;

    len = read(comm_sd, buf, sizeof (struct fake_msg));
    if (len > 0)
      break;

    if (len == 0) {
      errno = 0;
      fail("read: socket is closed");
    }

    if (errno == EINTR)
      continue;

    fail("read");
  }

  buf->id = ntohl(buf->id);
  buf->st.uid = ntohl(buf->st.uid);
  buf->st.gid = ntohl(buf->st.gid);
  buf->st.ino = ntohll(buf->st.ino);
  buf->st.dev = ntohll(buf->st.dev);
  buf->st.rdev = ntohll(buf->st.rdev);
  buf->st.mode = ntohl(buf->st.mode);
  buf->st.nlink = ntohl(buf->st.nlink);
}

void send_get_fakem(struct fake_msg *buf)
{
  pthread_mutex_lock(&comm_sd_mutex);

  open_comm_sd();
  __send_fakem(buf);
  __get_fakem(buf);

  pthread_mutex_unlock(&comm_sd_mutex);
}

void send_stat(const struct stat *st,
	       func_id f){
  struct fake_msg buf;
  
    cpyfakemstat(&buf,st);
    buf.id=f;
    send_fakem(&buf);
}

void send_get_stat(struct stat *st){
  struct fake_msg buf;

    cpyfakemstat(&buf,st);
    
    buf.id=stat_func;
    send_get_fakem(&buf);
    cpystatfakem(st,&buf);
}
