/*
 * Copyright (C) 2004 Nokia
 * by Lauri Leukkunen <lauri.leukkunen@nokia.com>
 * Based on miext/damage/damage.c
 * */

#include <config.h>

#include    "scrnintstr.h"
#include    "windowstr.h"
#include    "font.h"
#include    "dixfontstr.h"
#include    "fontstruct.h"
#include    "mi.h"
#include    "regionstr.h"
#include    "globals.h"
#include    "gcstruct.h"

#include "omap.h"

#define OMAP_GC_OP_PROLOGUE(pGC, pDrawable) \
    GCFuncs *oldFuncs = pGC->funcs; \
omapGCPriv(pGC); \
unwrap(pGCPriv, pGC, funcs);  \
unwrap(pGCPriv, pGC, ops)

#define OMAP_GC_OP_EPILOGUE(pGC, pDrawable) \
    wrap(pGCPriv, pGC, funcs, oldFuncs); \
wrap(pGCPriv, pGC, ops, &omapGCOps)

#define OMAP_GC_FUNC_PROLOGUE(pGC) \
    omapGCPriv(pGC); \
unwrap(pGCPriv, pGC, funcs); \
if (pGCPriv->ops) unwrap(pGCPriv, pGC, ops)

#define OMAP_GC_FUNC_EPILOGUE(pGC) \
    wrap(pGCPriv, pGC, funcs, &omapGCFuncs);  \
if (pGCPriv->ops) wrap(pGCPriv, pGC, ops, &omapGCOps)




static void omapValidateGC(GCPtr, unsigned long, DrawablePtr);
static void omapChangeGC(GCPtr, unsigned long);
static void omapCopyGC(GCPtr, unsigned long, GCPtr);
static void omapDestroyGC(GCPtr);
static void omapChangeClip(GCPtr, int, pointer, int);
static void omapDestroyClip(GCPtr);
static void omapCopyClip(GCPtr, GCPtr);



GCFuncs omapGCFuncs = {
    omapValidateGC,
    omapChangeGC,
    omapCopyGC,
    omapDestroyGC,
    omapChangeClip,
    omapDestroyClip,
    omapCopyClip
};

GCOps omapGCOps;

Bool omapCreateGC(GCPtr pGC)
{
    ScreenPtr pScreen = pGC->pScreen;
    KdScreenPriv(pScreen);
    omapScreenInfo(pScreenPriv);
    omapGCPriv(pGC);
    Bool ret;

    unwrap(omaps, pScreen, CreateGC);

    if ((ret = (*pScreen->CreateGC)(pGC))) {
        pGCPriv->ops = NULL;
        pGCPriv->funcs = pGC->funcs;
        pGC->funcs = &omapGCFuncs;
    }
    wrap(omaps, pScreen, CreateGC, omapCreateGC);
    return ret;
}

static void omapValidateGC(GCPtr         pGC,
        unsigned long changes,
        DrawablePtr   pDrawable)
{
    OMAP_GC_FUNC_PROLOGUE (pGC);
    (*pGC->funcs->ValidateGC)(pGC, changes, pDrawable);
    pGCPriv->ops = pGC->ops;  /* just so it's not NULL */
    OMAP_GC_FUNC_EPILOGUE (pGC);
}

static void omapDestroyGC(GCPtr pGC)
{
    OMAP_GC_FUNC_PROLOGUE (pGC);
    (*pGC->funcs->DestroyGC)(pGC);
    OMAP_GC_FUNC_EPILOGUE (pGC);
}

static void omapChangeGC (GCPtr           pGC,
        unsigned long   mask)
{
    OMAP_GC_FUNC_PROLOGUE (pGC);
    (*pGC->funcs->ChangeGC) (pGC, mask);
    pGCPriv->ops = pGC->ops;  /* just so it's not NULL */
    OMAP_GC_FUNC_EPILOGUE (pGC);
}

static void omapCopyGC (GCPtr         pGCSrc,
        unsigned long mask,
        GCPtr         pGCDst)
{
    OMAP_GC_FUNC_PROLOGUE (pGCDst);
    (*pGCDst->funcs->CopyGC) (pGCSrc, mask, pGCDst);
    pGCPriv->ops = pGCDst->ops;  /* just so it's not NULL */
    OMAP_GC_FUNC_EPILOGUE (pGCDst);
}

static void omapChangeClip (GCPtr     pGC,
        int       type,
        pointer   pvalue,
        int       nrects)
{
    OMAP_GC_FUNC_PROLOGUE (pGC);
    (*pGC->funcs->ChangeClip) (pGC, type, pvalue, nrects);
    OMAP_GC_FUNC_EPILOGUE (pGC);
}

static void omapCopyClip(GCPtr pgcDst, GCPtr pgcSrc)
{
    OMAP_GC_FUNC_PROLOGUE (pgcDst);
    (* pgcDst->funcs->CopyClip)(pgcDst, pgcSrc);
    OMAP_GC_FUNC_EPILOGUE (pgcDst);
}

static void omapDestroyClip(GCPtr pGC)
{
    OMAP_GC_FUNC_PROLOGUE (pGC);
    (* pGC->funcs->DestroyClip)(pGC);
    OMAP_GC_FUNC_EPILOGUE (pGC);
}

/******** GCOps ******/

static void omapFillSpans(DrawablePtr pDrawable,
        GC	    *pGC,
        int	    npt,
        DDXPointPtr ppt,
        int	    *pwidth,
        int	    fSorted)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (npt)
    {
        int	    nptTmp = npt;
        DDXPointPtr pptTmp = ppt;
        int	    *pwidthTmp = pwidth;
        BoxRec	    box;

        box.x1 = pptTmp->x;
        box.x2 = box.x1 + *pwidthTmp;
        box.y2 = box.y1 = pptTmp->y;

        while(--nptTmp) 
        {
            pptTmp++;
            pwidthTmp++;
            if(box.x1 > pptTmp->x) box.x1 = pptTmp->x;
            if(box.x2 < (pptTmp->x + *pwidthTmp))
                box.x2 = pptTmp->x + *pwidthTmp;
            if(box.y1 > pptTmp->y) box.y1 = pptTmp->y;
            else if(box.y2 < pptTmp->y) box.y2 = pptTmp->y;
        }

        box.y2++;

        if(!pGC->miTranslate) {
            TRANSLATE_BOX(box, pDrawable);
        }
        TRIM_BOX(box, pGC); 

        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }

    (*pGC->ops->FillSpans)(pDrawable, pGC, npt, ppt, pwidth, fSorted);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapSetSpans(DrawablePtr  pDrawable,
        GCPtr	    pGC,
        char	    *pcharsrc,
        DDXPointPtr  ppt,
        int	    *pwidth,
        int	    npt,
        int	    fSorted)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (npt)
    {
        DDXPointPtr pptTmp = ppt;
        int	    *pwidthTmp = pwidth;
        int	    nptTmp = npt;
        BoxRec	    box;

        box.x1 = pptTmp->x;
        box.x2 = box.x1 + *pwidthTmp;
        box.y2 = box.y1 = pptTmp->y;

        while(--nptTmp) 
        {
            pptTmp++;
            pwidthTmp++;
            if(box.x1 > pptTmp->x) box.x1 = pptTmp->x;
            if(box.x2 < (pptTmp->x + *pwidthTmp))
                box.x2 = pptTmp->x + *pwidthTmp;
            if(box.y1 > pptTmp->y) box.y1 = pptTmp->y;
            else if(box.y2 < pptTmp->y) box.y2 = pptTmp->y;
        }

        box.y2++;

        if(!pGC->miTranslate) {
            TRANSLATE_BOX(box, pDrawable);
        }
        TRIM_BOX(box, pGC); 

        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->SetSpans)(pDrawable, pGC, pcharsrc, ppt, pwidth, npt, fSorted);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPutImage(DrawablePtr  pDrawable,
        GCPtr	    pGC,
        int	    depth,
        int	    x,
        int	    y,
        int	    w,
        int	    h,
        int	    leftPad,
        int	    format,
        char	    *pImage)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    BoxRec box;

    box.x1 = x + pDrawable->x;
    box.x2 = box.x1 + w;
    box.y1 = y + pDrawable->y;
    box.y2 = box.y1 + h;

    TRIM_BOX(box, pGC);
    if(BOX_NOT_EMPTY(box))
        od = omapDamageBox (pDrawable, &box);
    (*pGC->ops->PutImage)(pDrawable, pGC, depth, x, y, w, h,
                          leftPad, format, pImage);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}


void omapPrepareCopyClip(DrawablePtr	pSrcDrawable,
        DrawablePtr	pDstDrawable,
        GCPtr		pGC,
        int		xIn, 
        int		yIn,
        int		widthSrc, 
        int		heightSrc,
        int		xOut, 
        int		yOut)
{
    RegionPtr	prgnSrcClip = NULL; /* may be a new region, or just a copy */
    Bool	freeSrcClip = FALSE;
    RegionPtr	prgnExposed = NULL;
    RegionRec	rgnDst;
    int		dx;
    int		dy;
    int		numRects;
    BoxRec	box;
    Bool	fastSrc = FALSE;    /* for fast clipping with pixmap source */
    Bool	fastDst = FALSE;    /* for fast clipping with one rect dest */
    Bool	fastExpose = FALSE; /* for fast exposures with pixmap source */

    /* Short cut for unmapped windows */

    if (pDstDrawable->type == DRAWABLE_WINDOW && 
            !((WindowPtr)pDstDrawable)->realized)
    {
        return;
    }

    if (pSrcDrawable->pScreen->SourceValidate)
    {
        (*pSrcDrawable->pScreen->SourceValidate) (pSrcDrawable, xIn, yIn, widthSrc, heightSrc);
    }

    /* Compute source clip region */
    if (pSrcDrawable->type == DRAWABLE_PIXMAP)
    {
        if ((pSrcDrawable == pDstDrawable) && (pGC->clientClipType == CT_NONE))
            prgnSrcClip = fbGetCompositeClip(pGC);
        else
            fastSrc = TRUE;
    }
    else
    {
        if (pGC->subWindowMode == IncludeInferiors)
        {
            /*
             * XFree86 DDX empties the border clip when the
             * VT is inactive, make sure the region isn't empty
             */
            if (!((WindowPtr) pSrcDrawable)->parent &&
                    REGION_NOTEMPTY (pSrcDrawable->pScreen,
                        &((WindowPtr) pSrcDrawable)->borderClip))
            {
                /*
                 * special case bitblt from root window in
                 * IncludeInferiors mode; just like from a pixmap
                 */
                fastSrc = TRUE;
            }
            else if ((pSrcDrawable == pDstDrawable) &&
                    (pGC->clientClipType == CT_NONE))
            {
                prgnSrcClip = fbGetCompositeClip(pGC);
            }
            else
            {
                prgnSrcClip = NotClippedByChildren((WindowPtr)pSrcDrawable);
                freeSrcClip = TRUE;
            }
        }
        else
        {
            prgnSrcClip = &((WindowPtr)pSrcDrawable)->clipList;
        }
    }

    xIn += pSrcDrawable->x;
    yIn += pSrcDrawable->y;

    xOut += pDstDrawable->x;
    yOut += pDstDrawable->y;

    box.x1 = xIn;
    box.y1 = yIn;
    box.x2 = xIn + widthSrc;
    box.y2 = yIn + heightSrc;

    dx = xIn - xOut;
    dy = yIn - yOut;

    /* Don't create a source region if we are doing a fast clip */
    if (fastSrc)
    {
        RegionPtr cclip;

        fastExpose = TRUE;
        /*
         * clip the source; if regions extend beyond the source size,
         * make sure exposure events get sent
         */
        if (box.x1 < pSrcDrawable->x)
        {
            box.x1 = pSrcDrawable->x;
            fastExpose = FALSE;
        }
        if (box.y1 < pSrcDrawable->y)
        {
            box.y1 = pSrcDrawable->y;
            fastExpose = FALSE;
        }
        if (box.x2 > pSrcDrawable->x + (int) pSrcDrawable->width)
        {
            box.x2 = pSrcDrawable->x + (int) pSrcDrawable->width;
            fastExpose = FALSE;
        }
        if (box.y2 > pSrcDrawable->y + (int) pSrcDrawable->height)
        {
            box.y2 = pSrcDrawable->y + (int) pSrcDrawable->height;
            fastExpose = FALSE;
        }

        /* Translate and clip the dst to the destination composite clip */
        box.x1 -= dx;
        box.x2 -= dx;
        box.y1 -= dy;
        box.y2 -= dy;

        /* If the destination composite clip is one rectangle we can
           do the clip directly.  Otherwise we have to create a full
           blown region and call intersect */

        cclip = fbGetCompositeClip(pGC);
        if (REGION_NUM_RECTS(cclip) == 1)
        {
            BoxPtr pBox = REGION_RECTS(cclip);

            if (box.x1 < pBox->x1) box.x1 = pBox->x1;
            if (box.x2 > pBox->x2) box.x2 = pBox->x2;
            if (box.y1 < pBox->y1) box.y1 = pBox->y1;
            if (box.y2 > pBox->y2) box.y2 = pBox->y2;
            fastDst = TRUE;
        }
    }

    /* Check to see if the region is empty */
    if (box.x1 >= box.x2 || box.y1 >= box.y2)
    {
        REGION_INIT(pGC->pScreen, &rgnDst, NullBox, 0);
    }
    else
    {
        REGION_INIT(pGC->pScreen, &rgnDst, &box, 1);
    }

    /* Clip against complex source if needed */
    if (!fastSrc)
    {
        REGION_INTERSECT(pGC->pScreen, &rgnDst, &rgnDst, prgnSrcClip);
        REGION_TRANSLATE(pGC->pScreen, &rgnDst, -dx, -dy);
    }

    /* Clip against complex dest if needed */
    if (!fastDst)
    {
        REGION_INTERSECT(pGC->pScreen, &rgnDst, &rgnDst,
                fbGetCompositeClip(pGC));
    }
    REGION_UNINIT(pGC->pScreen, &rgnDst);
    if (freeSrcClip)
        REGION_DESTROY(pGC->pScreen, prgnSrcClip);
}




static RegionPtr omapCopyArea(DrawablePtr   pSrc,
        DrawablePtr  pDst,
        GC	    *pGC,
        int	    srcx,
        int	    srcy,
        int	    width,
        int	    height,
        int	    dstx,
        int	    dsty)
{
    RegionPtr ret;
    OMAP_GC_OP_PROLOGUE(pGC, pDst);
    OmapDamageRegionPtr odr = NULL;
    OmapDamageBoxPtr odl;
    BoxRec box;
    int rir;
    int odc = 0;
    //omapPrepareCopyClip(pSrc, pDst, pGC, srcx, srcy, width, height, dstx, dsty);
    ret = (*pGC->ops->CopyArea)(pSrc, pDst,
            pGC, srcx, srcy, width, height, dstx, dsty);

    box.x1 = dstx + pDst->x;
    box.x2 = box.x1 + width;
    box.y1 = dsty + pDst->y;
    box.y2 = box.y1 + height;

    /* Find out if the drawing is entirely within the clip */
    rir = RECT_IN_REGION (pDst->pScreen, pGC->pCompositeClip, &box);
    if (rir != rgnOUT) {
        RegionRec region;

        REGION_INIT (pDst->pScreen, &region, &box, 1);

        if (rir == rgnPART) {
            REGION_INTERSECT (pDst->pScreen, &region,
                    &region, pGC->pCompositeClip);
        }

        odr = omapDamageRegion(pDst, &region, FALSE);

        if (odr) {
            odc = odr->count;
            odl = odr->box;
            while (odc--) {
                omapSendDamage(odl++);
            }
            xfree(odr->box);
            xfree(odr);
        }
        REGION_UNINIT (pDst->pScreen, &region);

    } 

    /* ret = (*pGC->ops->CopyArea)(pSrc, pDst,
       pGC, srcx, srcy, width, height, dstx, dsty);
       */
    OMAP_GC_OP_EPILOGUE(pGC, pDst);
    return ret;
}

    static RegionPtr
omapCopyPlane(DrawablePtr	pSrc,
        DrawablePtr	pDst,
        GCPtr		pGC,
        int		srcx,
        int		srcy,
        int		width,
        int		height,
        int		dstx,
        int		dsty,
        unsigned long	bitPlane)
{
    RegionPtr ret;
    OMAP_GC_OP_PROLOGUE(pGC, pDst);
    OmapDamageBoxPtr od = NULL;
    BoxRec box;

    box.x1 = dstx + pDst->x;
    box.x2 = box.x1 + width;
    box.y1 = dsty + pDst->y;
    box.y2 = box.y1 + height;

    TRIM_BOX(box, pGC);

    if(BOX_NOT_EMPTY(box))
        od = omapDamageBox (pDst, &box);

    ret = (*pGC->ops->CopyPlane)(pSrc, pDst,
            pGC, srcx, srcy, width, height, dstx, dsty, bitPlane);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDst);
    return ret;
}

static void omapPolyPoint(DrawablePtr pDrawable,
        GCPtr	    pGC,
        int	    mode,
        int	    npt,
        xPoint	    *ppt)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;

    if (npt)
    {
        BoxRec	box;
        int	nptTmp = npt;
        xPoint	*pptTmp = ppt;

        box.x2 = box.x1 = pptTmp->x;
        box.y2 = box.y1 = pptTmp->y;

        /* this could be slow if the points were spread out */

        while(--nptTmp) 
        {
            pptTmp++;
            if(box.x1 > pptTmp->x) box.x1 = pptTmp->x;
            else if(box.x2 < pptTmp->x) box.x2 = pptTmp->x;
            if(box.y1 > pptTmp->y) box.y1 = pptTmp->y;
            else if(box.y2 < pptTmp->y) box.y2 = pptTmp->y;
        }

        box.x2++;
        box.y2++;

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->PolyPoint)(pDrawable, pGC, mode, npt, ppt);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPolylines(DrawablePtr pDrawable,
        GCPtr	    pGC,
        int	    mode,
        int	    npt,
        DDXPointPtr ppt)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (npt)
    {
        int	    nptTmp = npt;
        DDXPointPtr pptTmp = ppt;
        BoxRec	    box;
        int	    extra = pGC->lineWidth >> 1;

        box.x2 = box.x1 = pptTmp->x;
        box.y2 = box.y1 = pptTmp->y;

        if(nptTmp > 1) 
        {
            if(pGC->joinStyle == JoinMiter)
                extra = 6 * pGC->lineWidth;
            else if(pGC->capStyle == CapProjecting)
                extra = pGC->lineWidth;
        }

        if(mode == CoordModePrevious) 
        {
            int x = box.x1;
            int y = box.y1;
            while(--nptTmp) 
            {
                pptTmp++;
                x += pptTmp->x;
                y += pptTmp->y;
                if(box.x1 > x) box.x1 = x;
                else if(box.x2 < x) box.x2 = x;
                if(box.y1 > y) box.y1 = y;
                else if(box.y2 < y) box.y2 = y;
            }
        }
        else 
        {
            while(--nptTmp) 
            {
                pptTmp++;
                if(box.x1 > pptTmp->x) box.x1 = pptTmp->x;
                else if(box.x2 < pptTmp->x) box.x2 = pptTmp->x;
                if(box.y1 > pptTmp->y) box.y1 = pptTmp->y;
                else if(box.y2 < pptTmp->y) box.y2 = pptTmp->y;
            }
        }

        box.x2++;
        box.y2++;

        if(extra) 
        {
            box.x1 -= extra;
            box.x2 += extra;
            box.y1 -= extra;
            box.y2 += extra;
        }

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->Polylines)(pDrawable, pGC, mode, npt, ppt);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPolySegment(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		nSeg,
        xSegment	*pSeg)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (nSeg)
    {
        BoxRec	    box;
        int	    extra = pGC->lineWidth;
        int	    nsegTmp = nSeg;
        xSegment    *pSegTmp = pSeg;

        if(pGC->capStyle != CapProjecting)
            extra >>= 1;

        if(pSegTmp->x2 > pSegTmp->x1) {
            box.x1 = pSegTmp->x1;
            box.x2 = pSegTmp->x2;
        } else {
            box.x2 = pSegTmp->x1;
            box.x1 = pSegTmp->x2;
        }

        if(pSegTmp->y2 > pSegTmp->y1) {
            box.y1 = pSegTmp->y1;
            box.y2 = pSegTmp->y2;
        } else {
            box.y2 = pSegTmp->y1;
            box.y1 = pSegTmp->y2;
        }

        while(--nsegTmp) 
        {
            pSegTmp++;
            if(pSegTmp->x2 > pSegTmp->x1) 
            {
                if(pSegTmp->x1 < box.x1) box.x1 = pSegTmp->x1;
                if(pSegTmp->x2 > box.x2) box.x2 = pSegTmp->x2;
            }
            else 
            {
                if(pSegTmp->x2 < box.x1) box.x1 = pSegTmp->x2;
                if(pSegTmp->x1 > box.x2) box.x2 = pSegTmp->x1;
            }
            if(pSegTmp->y2 > pSegTmp->y1) 
            {
                if(pSegTmp->y1 < box.y1) box.y1 = pSegTmp->y1;
                if(pSegTmp->y2 > box.y2) box.y2 = pSegTmp->y2;
            }
            else
            {
                if(pSegTmp->y2 < box.y1) box.y1 = pSegTmp->y2;
                if(pSegTmp->y1 > box.y2) box.y2 = pSegTmp->y1;
            }
        }

        box.x2++;
        box.y2++;

        if(extra) 
        {
            box.x1 -= extra;
            box.x2 += extra;
            box.y1 -= extra;
            box.y2 += extra;
        }

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->PolySegment)(pDrawable, pGC, nSeg, pSeg);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPolyRectangle(DrawablePtr  pDrawable,
        GCPtr        pGC,
        int	         nRects,
        xRectangle  *pRects)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od[nRects*4];
    int od_c;
    memset(od, '\0', sizeof(OmapDamageBoxPtr) * nRects * 4);
    //DBGOUT("howdy!\n");
    if (nRects)
    {
        BoxRec	    box;
        int	    offset1, offset2, offset3;
        int	    nRectsTmp = nRects;
        xRectangle  *pRectsTmp = pRects;

        offset2 = pGC->lineWidth;
        if(!offset2) offset2 = 1;
        offset1 = offset2 >> 1;
        offset3 = offset2 - offset1;

        od_c = nRects * 4;

        while(nRectsTmp--)
        {
            box.x1 = pRectsTmp->x - offset1;
            box.y1 = pRectsTmp->y - offset1;
            box.x2 = box.x1 + pRectsTmp->width + offset2;
            box.y2 = box.y1 + offset2;
            TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
            if(BOX_NOT_EMPTY(box))
                od[--od_c] = omapDamageBox (pDrawable, &box);

            box.x1 = pRectsTmp->x - offset1;
            box.y1 = pRectsTmp->y + offset3;
            box.x2 = box.x1 + offset2;
            box.y2 = box.y1 + pRectsTmp->height - offset2;
            TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
            if(BOX_NOT_EMPTY(box))
                od[--od_c] = omapDamageBox (pDrawable, &box);

            box.x1 = pRectsTmp->x + pRectsTmp->width - offset1;
            box.y1 = pRectsTmp->y + offset3;
            box.x2 = box.x1 + offset2;
            box.y2 = box.y1 + pRectsTmp->height - offset2;
            TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
            if(BOX_NOT_EMPTY(box))
                od[--od_c] = omapDamageBox (pDrawable, &box);

            box.x1 = pRectsTmp->x - offset1;
            box.y1 = pRectsTmp->y + pRectsTmp->height - offset1;
            box.x2 = box.x1 + pRectsTmp->width + offset2;
            box.y2 = box.y1 + offset2;
            TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
            if(BOX_NOT_EMPTY(box))
                od[--od_c] = omapDamageBox (pDrawable, &box);

            pRectsTmp++;
        }
    }
    (*pGC->ops->PolyRectangle)(pDrawable, pGC, nRects, pRects);
    od_c = nRects*4;
    while (od_c--) {
        if (od[od_c]) {
            omapSendDamage(od[od_c]);
            xfree(od[od_c]);
        }
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPolyArc(DrawablePtr   pDrawable,
        GCPtr	    pGC,
        int	    nArcs,
        xArc	    *pArcs)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (nArcs)
    {
        int	extra = pGC->lineWidth >> 1;
        BoxRec	box;
        int	nArcsTmp = nArcs;
        xArc	*pArcsTmp = pArcs;

        box.x1 = pArcsTmp->x;
        box.x2 = box.x1 + pArcsTmp->width;
        box.y1 = pArcsTmp->y;
        box.y2 = box.y1 + pArcsTmp->height;

        while(--nArcsTmp) 
        {
            pArcsTmp++;
            if(box.x1 > pArcsTmp->x)
                box.x1 = pArcsTmp->x;
            if(box.x2 < (pArcsTmp->x + pArcsTmp->width))
                box.x2 = pArcsTmp->x + pArcsTmp->width;
            if(box.y1 > pArcsTmp->y) 
                box.y1 = pArcsTmp->y;
            if(box.y2 < (pArcsTmp->y + pArcsTmp->height))
                box.y2 = pArcsTmp->y + pArcsTmp->height;
        }

        if(extra) 
        {
            box.x1 -= extra;
            box.x2 += extra;
            box.y1 -= extra;
            box.y2 += extra;
        }

        box.x2++;
        box.y2++;

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->PolyArc)(pDrawable, pGC, nArcs, pArcs);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapFillPolygon(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		shape,
        int		mode,
        int		npt,
        DDXPointPtr	ppt)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;

    if (npt > 2)
    {
        DDXPointPtr pptTmp = ppt;
        int	    nptTmp = npt;
        BoxRec	    box;

        box.x2 = box.x1 = pptTmp->x;
        box.y2 = box.y1 = pptTmp->y;

        if(mode != CoordModeOrigin) 
        {
            int x = box.x1;
            int y = box.y1;
            while(--nptTmp) 
            {
                pptTmp++;
                x += pptTmp->x;
                y += pptTmp->y;
                if(box.x1 > x) box.x1 = x;
                else if(box.x2 < x) box.x2 = x;
                if(box.y1 > y) box.y1 = y;
                else if(box.y2 < y) box.y2 = y;
            }
        }
        else 
        {
            while(--nptTmp) 
            {
                pptTmp++;
                if(box.x1 > pptTmp->x) box.x1 = pptTmp->x;
                else if(box.x2 < pptTmp->x) box.x2 = pptTmp->x;
                if(box.y1 > pptTmp->y) box.y1 = pptTmp->y;
                else if(box.y2 < pptTmp->y) box.y2 = pptTmp->y;
            }
        }

        box.x2++;
        box.y2++;

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }

    (*pGC->ops->FillPolygon)(pDrawable, pGC, shape, mode, npt, ppt);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}


static void omapPolyFillRect(DrawablePtr	pDrawable,
        GCPtr	pGC,
        int		nRects,
        xRectangle	*pRects)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (nRects)
    {
        BoxRec	    box;
        xRectangle  *pRectsTmp = pRects;
        int	    nRectsTmp = nRects;

        box.x1 = pRectsTmp->x;
        box.x2 = box.x1 + pRectsTmp->width;
        box.y1 = pRectsTmp->y;
        box.y2 = box.y1 + pRectsTmp->height;

        while(--nRectsTmp) 
        {
            pRectsTmp++;
            if(box.x1 > pRectsTmp->x) box.x1 = pRectsTmp->x;
            if(box.x2 < (pRectsTmp->x + pRectsTmp->width))
                box.x2 = pRectsTmp->x + pRectsTmp->width;
            if(box.y1 > pRectsTmp->y) box.y1 = pRectsTmp->y;
            if(box.y2 < (pRectsTmp->y + pRectsTmp->height))
                box.y2 = pRectsTmp->y + pRectsTmp->height;
        }

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->PolyFillRect)(pDrawable, pGC, nRects, pRects);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}


static void omapPolyFillArc(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		nArcs,
        xArc		*pArcs)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    if (nArcs)
    {
        BoxRec	box;
        int	nArcsTmp = nArcs;
        xArc	*pArcsTmp = pArcs;

        box.x1 = pArcsTmp->x;
        box.x2 = box.x1 + pArcsTmp->width;
        box.y1 = pArcsTmp->y;
        box.y2 = box.y1 + pArcsTmp->height;

        while(--nArcsTmp) 
        {
            pArcsTmp++;
            if(box.x1 > pArcsTmp->x)
                box.x1 = pArcsTmp->x;
            if(box.x2 < (pArcsTmp->x + pArcsTmp->width))
                box.x2 = pArcsTmp->x + pArcsTmp->width;
            if(box.y1 > pArcsTmp->y)
                box.y1 = pArcsTmp->y;
            if(box.y2 < (pArcsTmp->y + pArcsTmp->height))
                box.y2 = pArcsTmp->y + pArcsTmp->height;
        }

        TRIM_AND_TRANSLATE_BOX(box, pDrawable, pGC);
        if(BOX_NOT_EMPTY(box))
            od = omapDamageBox (pDrawable, &box);
    }
    (*pGC->ops->PolyFillArc)(pDrawable, pGC, nArcs, pArcs);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

/*
 * general Poly/Image text function.  Extract glyph information,
 * compute bounding box and remove cursor if it is overlapped.
 */

static OmapDamageBoxPtr omapDamageChars (DrawablePtr	pDrawable,
        FontPtr	font,
        int		x,
        int		y,
        unsigned int	n,
        CharInfoPtr	*charinfo,
        Bool		imageblt)
{
    ExtentInfoRec   extents;
    BoxRec	    box;

    QueryGlyphExtents(font, charinfo, n, &extents);
    if (imageblt)
    {
        if (extents.overallWidth > extents.overallRight)
            extents.overallRight = extents.overallWidth;
        if (extents.overallWidth < extents.overallLeft)
            extents.overallLeft = extents.overallWidth;
        if (extents.overallLeft > 0)
            extents.overallLeft = 0;
        if (extents.fontAscent > extents.overallAscent)
            extents.overallAscent = extents.fontAscent;
        if (extents.fontDescent > extents.overallDescent)
            extents.overallDescent = extents.fontDescent;
    }
    box.x1 = x + extents.overallLeft;
    box.y1 = y - extents.overallAscent;
    box.x2 = x + extents.overallRight;
    box.y2 = y + extents.overallDescent;
    return omapDamageBox (pDrawable, &box);
}

/*
 * values for textType:
 */
#define TT_POLY8   0
#define TT_IMAGE8  1
#define TT_POLY16  2
#define TT_IMAGE16 3


static int omapText (DrawablePtr	    pDrawable,
        GCPtr	    pGC,
        int		    x,
        int		    y,
        unsigned long   count,
        char	    *chars,
        FontEncoding    fontEncoding,
        Bool	    textType)
{
    CharInfoPtr	    *charinfo;
    CharInfoPtr	    *info;
    unsigned long   i;
    unsigned int    n;
    int		    w;
    Bool	    imageblt;

    imageblt = (textType == TT_IMAGE8) || (textType == TT_IMAGE16);

    charinfo = (CharInfoPtr *) ALLOCATE_LOCAL(count * sizeof(CharInfoPtr));
    if (!charinfo)
        return x;

    GetGlyphs(pGC->font, count, (unsigned char *)chars,
            fontEncoding, &i, charinfo);
    n = (unsigned int)i;
    w = 0;
    if (!imageblt)
        for (info = charinfo; i--; info++)
            w += (*info)->metrics.characterWidth;

    if (n != 0) {
        OmapDamageBoxPtr od = omapDamageChars (pDrawable, pGC->font, x + pDrawable->x, y + pDrawable->y, n,
                charinfo, imageblt);
        if (imageblt)
            (*pGC->ops->ImageGlyphBlt)(pDrawable, pGC, x, y, n, charinfo,
                                       FONTGLYPHS(pGC->font));
        else
            (*pGC->ops->PolyGlyphBlt)(pDrawable, pGC, x, y, n, charinfo,
                                      FONTGLYPHS(pGC->font));
        if (od) {
            omapSendDamage(od);
            xfree(od);
        }
    }
    DEALLOCATE_LOCAL(charinfo);
    return x + w;
}

static int omapPolyText8(DrawablePtr pDrawable,
        GCPtr	    pGC,
        int	    x,
        int	    y,
        int	    count,
        char	    *chars)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);

    omapText (pDrawable, pGC, x, y, (unsigned long) count, chars,
            Linear8Bit, TT_POLY8);

    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
    return x;
}

static int omapPolyText16(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		x,
        int		y,
        int		count,
        unsigned short	*chars)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);

    omapText (pDrawable, pGC, x, y, (unsigned long) count, (char *) chars,
            FONTLASTROW(pGC->font) == 0 ? Linear16Bit : TwoD16Bit,
            TT_POLY16);
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
    return x;
}

static void omapImageText8(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		x,
        int		y,
        int		count,
        char		*chars)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);

    omapText (pDrawable, pGC, x, y, (unsigned long) count, chars,
            Linear8Bit, TT_IMAGE8);

    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapImageText16(DrawablePtr	pDrawable,
        GCPtr		pGC,
        int		x,
        int		y,
        int		count,
        unsigned short *chars)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);

    omapText (pDrawable, pGC, x, y, (unsigned long) count, (char *) chars,
            FONTLASTROW(pGC->font) == 0 ? Linear16Bit : TwoD16Bit,
            TT_IMAGE16);

    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}


static void omapImageGlyphBlt(DrawablePtr	    pDrawable,
        GCPtr	    pGC,
        int		    x,
        int		    y,
        unsigned int    nglyph,
        CharInfoPtr	    *ppci,
        pointer	    pglyphBase)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = omapDamageChars (pDrawable, pGC->font, x + pDrawable->x, y + pDrawable->y,
            nglyph, ppci, TRUE);
    (*pGC->ops->ImageGlyphBlt)(pDrawable, pGC, x, y, nglyph,
                               ppci, pglyphBase);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPolyGlyphBlt(DrawablePtr	pDrawable,
        GCPtr	pGC,
        int		x,
        int		y,
        unsigned int	nglyph,
        CharInfoPtr	*ppci,
        pointer	pglyphBase)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = omapDamageChars (pDrawable, pGC->font, x + pDrawable->x, y + pDrawable->y,
            nglyph, ppci, FALSE);
    (*pGC->ops->PolyGlyphBlt)(pDrawable, pGC, x, y, nglyph,
                              ppci, pglyphBase);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}

static void omapPushPixels(GCPtr pGC,
        PixmapPtr	pBitMap,
        DrawablePtr	pDrawable,
        int		dx,
        int		dy,
        int		xOrg,
        int		yOrg)
{
    OMAP_GC_OP_PROLOGUE(pGC, pDrawable);
    OmapDamageBoxPtr od = NULL;
    BoxRec box;

    box.x1 = xOrg;
    box.y1 = yOrg;

    if(!pGC->miTranslate) {
        box.x1 += pDrawable->x;          
        box.y1 += pDrawable->y;          
    }

    box.x2 = box.x1 + dx;
    box.y2 = box.y1 + dy;

    TRIM_BOX(box, pGC);
    if(BOX_NOT_EMPTY(box))
        od = omapDamageBox (pDrawable, &box);

    (*pGC->ops->PushPixels)(pGC, pBitMap, pDrawable, dx, dy, xOrg, yOrg);
    if (od) {
        omapSendDamage(od);
        xfree(od);
    }
    OMAP_GC_OP_EPILOGUE(pGC, pDrawable);
}



GCOps omapGCOps = {
    omapFillSpans, 
    omapSetSpans,
    omapPutImage, 
    omapCopyArea,
    omapCopyPlane, 
    omapPolyPoint,
    omapPolylines, 
    omapPolySegment,
    omapPolyRectangle, 
    omapPolyArc,
    omapFillPolygon, 
    omapPolyFillRect,
    omapPolyFillArc, 
    omapPolyText8,
    omapPolyText16, 
    omapImageText8,
    omapImageText16, 
    omapImageGlyphBlt,
    omapPolyGlyphBlt, 
    omapPushPixels,
#ifdef NEED_LINEHELPER
    NULL,
#endif
    {NULL}              /* devPrivate */
};
