/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of telepathy-feed
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Onne Gorter <onne.gorter@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <glib.h>

#include <libgalago/galago.h>

#include <dbus/dbus-glib-bindings.h>

#include <libtelepathy/tp-connmgr.h>
#include <libtelepathy/tp-conn.h>
#include <libtelepathy/tp-conn-gen.h>
#include <libtelepathy/tp-conn-iface-presence-gen.h>
#include <libtelepathy/tp-conn-iface-capabilities-gen.h>
#include <libtelepathy/tp-chan.h>
#include <libtelepathy/tp-chan-iface-group-gen.h>

#include "galago.h"
#include "shared.h"
#include "util.h"

#define STREAMEDMEDIA_CHANNEL \
  "org.freedesktop.Telepathy.Channel.Type.StreamedMedia"
#define TEXT_CHANNEL \
  "org.freedesktop.Telepathy.Channel.Type.Text"

typedef enum {
  GALAGO_CAPS_CHANGED = 1 << 0,
  GALAGO_CAPS_CHAT    = 1 << 1,
  GALAGO_CAPS_AUDIO   = 1 << 2,
  GALAGO_CAPS_VIDEO   = 1 << 3
} GalagoCaps;

/* TODO:
 * - listen for connection crashing
 */

/**
 * Get a GalagoService from a service name.
 */
static GalagoService*
get_service (const char *service_name)
{
  static GHashTable *hash = NULL;
  GalagoService *service;

  g_assert (service_name != NULL);

  if (hash == NULL) {
    hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
    g_assert (hash != NULL);
  }

  service = g_hash_table_lookup (hash, service_name);
  if (service == NULL) {
    service = galago_create_service (service_name, NULL, 0) ;
    g_hash_table_insert (hash, g_strdup (service_name), service);
    p_debug ("Created service %p for %s", service, service_name);
  }

  return service;
}

static GalagoCaps
telepathy_to_galago_caps (const char *channel, guint generic, guint specific)
{
  GalagoCaps caps = 0;
  
  if (0 == strcmp (channel, STREAMEDMEDIA_CHANNEL)) {
    if (specific & TP_CHANNEL_MEDIA_CAPABILITY_AUDIO)
      caps |= GALAGO_CAPS_AUDIO;
    if (specific & TP_CHANNEL_MEDIA_CAPABILITY_VIDEO)
      caps |= GALAGO_CAPS_VIDEO;
  }
  if (0 == strcmp (channel, TEXT_CHANNEL)) {
    if (generic)
      caps |= GALAGO_CAPS_CHAT;
  }
  return caps;
}

/**
 * Map Telepathy status names to the Galago status enum. Note that this is wrong
 * and we should be using Presence:GetStatus() to get the status IDs from there.
 */
static GalagoStatusType
name_to_status (const char *status)
{
  g_assert (status != NULL);

  switch (status[0]) {
  case 'o': /* offline */
    return GALAGO_STATUS_OFFLINE;

  case 'h': /* hidden */
    return GALAGO_STATUS_HIDDEN;

  case 'x': /* xa */
    return GALAGO_STATUS_EXTENDED_AWAY;

  case 'b': /* busy, brb */
  case 'd': /* dnd */
    return GALAGO_STATUS_AWAY;

  case 'a': /* available, away */
    switch (status[1]) {
    case 'v': /* available */
      return GALAGO_STATUS_AVAILABLE;

    case 'w': /* away */
      return GALAGO_STATUS_AWAY;

    default:
      break;
    }

  default:
    break;
  }

  return GALAGO_STATUS_UNSET;
}

static char*
get_service_username_from_tp_handle (TpConn* conn,
                                     guint   handle)
{
  GError *error = NULL;
  char *retval = NULL;

  if (!tf_tp_conn_inspect_handle (DBUS_G_PROXY (conn), 
                                  TP_CONN_HANDLE_TYPE_CONTACT, 
                                  handle, &retval, &error)) {
    g_warning ("Cannot get name for handle %d: %s", handle, error->message);
    g_error_free (error);
    return NULL;
  }
  return retval;
}

static char*
get_service_protocol (TpConn *conn)
{
  GError *error = NULL;
  char *retval = NULL;

  if (!tp_conn_get_protocol (DBUS_G_PROXY (conn), &retval, &error)) {
      g_warning ("Cannot get protocol: %s\n", error->message);
      g_error_free (error);
      return NULL;
    }
  return retval;
}

static void
create_accounts (FeedContext *context,
                 GArray      *handles)
{
  gboolean ret;
  char **names;
  GalagoService *service;
  GError *error = NULL;
  char *protocol;
  int i;

  ret = tp_conn_inspect_handles (DBUS_G_PROXY (context->conn),
                                 TP_CONN_HANDLE_TYPE_CONTACT,
                                 handles, &names, &error);
  if (ret == FALSE) {
    g_warning ("Cannot get names: %s", error->message);
    g_error_free (error);

    return;
  }

  /* Get this protocol here so we only get it once */
  protocol = get_service_protocol (context->conn);
  if (protocol == NULL) {
    g_free (names);
    return;
  }

  service = get_service (protocol);
  g_free (protocol);
  
  /* Add all the accounts */
  for (i = 0; i < handles->len; i++) {
    GalagoAccount *account;
    guint handle = g_array_index (handles, guint, i);

    account = g_hash_table_lookup (context->account_hash,
                                   GINT_TO_POINTER (handle));
    if (account == NULL) {
      GalagoPerson *person;
      char *username_bound;
      
      person = galago_create_person (NULL);
      username_bound = g_strdup_printf ("%s\n%s", names[i],
                                        context->account_username);
      account = galago_service_create_account (service, person, username_bound);
      g_free (username_bound);
      g_free (names[i]);

      g_hash_table_insert (context->account_hash, GINT_TO_POINTER (handle),
                           account);
    }
  }

  g_free (names);
}

static GalagoAccount*
handle_to_account (FeedContext *context, 
                   guint        handle)
{
  GalagoAccount *account;

  g_assert (context);

  account = g_hash_table_lookup (context->account_hash, 
                                 GINT_TO_POINTER (handle));
  if (!account) {
    GalagoService *service;
    GalagoPerson *person;
    char *username = NULL, *protocol = NULL;
    char *username_bound;
    
    protocol = get_service_protocol (context->conn);
    if (!protocol)
      return NULL;

    service = get_service (protocol);
    g_free (protocol);

    username = get_service_username_from_tp_handle (context->conn, handle);
    if (!username) {
      return NULL;
    }

    person = galago_create_person (NULL);
    /* Galago will happily return existing accounts here, because the same 
     * username/id on two different rosters is the same account for galago 
     * but not for the rest of the world so we do our own ref counting for it 
     * in order that we don't destroy accounts that still have a presence
     */
    username_bound = g_strdup_printf ("%s\n%s", username, context->account_username);
    account = galago_service_create_account (service, person, username_bound);
    g_free (username_bound);
    g_free (username);
    g_assert (GALAGO_IS_ACCOUNT (account));

    g_hash_table_insert (context->account_hash, GINT_TO_POINTER (handle), 
                         account);
  }

  return account;
}

static void
populate_presence (gpointer key,
                   gpointer value,
                   gpointer user_data)
{
  char *id = key;
  GHashTable *params = value;
  FeedContext *context;
  GalagoPresence *presence;
  GalagoStatus *status;
  GValue *message;
  const char *s;
  guint type;
  gpointer *data = user_data;

  presence = data[0];
  context = data[1];

  /* TODO at some point: pass idle time */
  if (context->statuses == NULL) {
    type = name_to_status (id);
  } else {
    GValueArray *values;

    values = g_hash_table_lookup (context->statuses, id);
    type = g_value_get_uint (g_value_array_get_nth (values, 0));
  }

  status = galago_status_new (type, id, id, FALSE);

  if (params) {
    message = g_hash_table_lookup (params, "message");
    s = message ? g_value_get_string (message) : NULL;
    if (s && s[0] != '\0') {
      galago_object_set_attr_string (GALAGO_OBJECT (status), "message", s);
    }
  }

  galago_presence_add_status (presence, status);
}

static void
presence_handle (gpointer key, 
                 gpointer value, 
                 gpointer user_data)
{
  GalagoPresence *presence;
  GalagoAccount *account;
  GHashTable *presences;
  GValueArray *values;
  FeedContext *context = user_data;
  guint handle;
  gpointer data[2];
  GList *l;
  
  handle = GPOINTER_TO_UINT (key);

  if (handle == context->self_handle) {
    p_debug ("Ignoring our own presence");
  }

  values = value;
  presences = g_value_get_boxed (g_value_array_get_nth (values, 1));

/* Debugging: Have nice names instead of tp-handles */
#ifdef NICE_DEBUGGING
  if (debug_flags & DEBUG_PRESENCE) {
    char *id, *status;

    id = get_service_username_from_tp_handle (context->conn, handle);
    if (!id) {
      id = g_strdup_printf ("unknown handle %d", handle);
    }

    status = get_hashtable_as_string (presences);
    p_debug ("Handling presence for %s: %s (context %p)", id, status, context);
    g_free (id);
    g_free (status);
  }
#else
  p_debug ("Handling presence for %d (context %p)", handle, context);
#endif

  account = handle_to_account (context, handle);
  if (!account) {
    g_warning (G_STRLOC ": could not create galago account from tp-handle %d",
               handle);
    return;
  }
  presence = galago_account_get_presence (account, FALSE);
  if (presence == NULL) {
    presence = galago_account_create_presence (account);
  }
  
  data[0] = presence;
  data[1] = context;

  for (l = galago_presence_get_statuses (presence); l; l = l->next) {
    galago_presence_remove_status (presence,
                                   galago_status_get_id (GALAGO_STATUS (l->data)));
  }
  g_hash_table_foreach (presences, populate_presence, data);
}

static void
on_presence_update (DBusGProxy *proxy, 
                    GHashTable *hash, 
                    gpointer    user_data)
{
  g_return_if_fail (DBUS_IS_G_PROXY (proxy));

  p_debug ("Got presence update (context %p)", user_data);
  g_hash_table_foreach (hash, presence_handle, user_data);
  p_debug ("End of update.");
}

static void
handle_member_presence (FeedContext *context,
                        GArray      *members,
                        GArray      *pending)
{
  int i;
  GHashTable *hash = NULL;
  GError *error = NULL;
  GArray *all;

  /* Prime all the accounts in one go so that handle_to_account will
     not require any DBus calls */
  all = g_array_sized_new (FALSE, FALSE, sizeof (guint), 
                           members->len + pending->len);
  g_array_append_vals (all, members->data, members->len);
  g_array_append_vals (all, pending->data, pending->len);

  create_accounts (context, all);
  g_array_free (all, TRUE);

  p_debug ("Members - %d, pending - %d (%p)", 
           members->len, pending->len, context);

  for (i = 0; i < pending->len; i++) {
    GalagoAccount *account;
    GalagoPresence *presence;
    GalagoStatus *status;
    guint handle = g_array_index (pending, guint, i);

/* Debugging: Have nice names instead of tp-handles */
#ifdef NICE_DEBUGGING
    if (debug_flags & DEBUG_PRESENCE) {
      char *id;

      id = get_service_username_from_tp_handle (context->conn, handle);
      if (!id) {
        id = g_strdup_printf ("unknown handle %d", handle);
      }

      p_debug ("Handling pending for %s (%p)", id, context);
      g_free (id);
    }
#else
    p_debug ("Handling pending for %d (%p)", handle, context);
#endif
    account = handle_to_account (context, handle);
    if (!account) {
      g_warning (G_STRLOC ": could not create galago account from tp-handle %d", handle);
      return;
    }
    presence = galago_account_get_presence (account, FALSE);
    if (presence == NULL) {
      presence = galago_account_create_presence (account);
    }

    status = galago_presence_get_status (presence, "pending");
    if (status == NULL) {
      status = galago_status_new (GALAGO_STATUS_PENDING,
                                  "pending", "pending", 
                                  FALSE);
      galago_presence_add_status (presence, status);
    }
  }

  p_debug ("Getting Presence for %d contact list members", members->len);
  if (!tp_conn_iface_presence_get_presence (context->presence_proxy, 
                                            members, &hash, &error)) {
    g_warning ("Could not get presences for contact list members: %s", error->message);
    g_error_free (error);
    return;
  }
  g_hash_table_foreach (hash, presence_handle, context);
  p_debug ("Presences stored.");
}

static inline void
update_capabilities (FeedContext *context)
{
  int handle;
  char *caps_string;
  
  if (NULL == context->caps_proxy)
    return;
  
  for (handle = 0; handle < context->galago_caps->len; handle++) {
    GalagoCaps caps;
    GalagoAccount *account;

    caps = g_array_index (context->galago_caps, GalagoCaps, handle);
  
    if (FALSE == (caps & GALAGO_CAPS_CHANGED))
      continue;
    
    caps ^= GALAGO_CAPS_CHANGED;
    g_array_index (context->galago_caps, GalagoCaps, handle) = caps;
    account = handle_to_account (context, handle);
    if (!account) {
      g_warning (G_STRLOC ": could not create galago account from tp-handle %d", handle);
      return;
    }

#ifdef NICE_DEBUGGING
  if (debug_flags & DEBUG_CAPS) {
    char *id;

    id = get_service_username_from_tp_handle (context->conn, handle);
    if (!id) {
      id = g_strdup_printf ("unknown handle %d", handle);
    }

    c_debug ("Capabilities for GalagoAccount %s (handle: %d) are now: %s%s%s(%X)",
             id,
             handle,
             caps & GALAGO_CAPS_CHAT ? "chat " : "",
             caps & GALAGO_CAPS_AUDIO ? "audio " : "",
             caps & GALAGO_CAPS_VIDEO ? "video " : "",
             caps >> 1);
    g_free (id);
  }
#else
    c_debug ("Capabilities for handle %d are now: %s%s%s (%X)",
             handle,
             caps & GALAGO_CAPS_CHAT ? "chat " : "",
             caps & GALAGO_CAPS_AUDIO ? "audio " : "",
             caps & GALAGO_CAPS_VIDEO ? "video" : "",
             caps >> 1);
#endif

    /* Before we send the stuff over dbus, remove the CHANGED field */
    caps = caps >> 1;
    caps_string = g_strdup_printf ("%d", caps);
    galago_account_set_display_name (account, caps_string);
    g_free (caps_string);
#if 0
    galago_object_set_attr_int (GALAGO_OBJECT (account),
                                "capabilities",
                                caps);
#endif
  }
}
  
static inline void
handle_member_capabilities (FeedContext *context, GArray *members)
{
  GError *error = NULL;
  GPtrArray *all_caps = NULL;
  int i;

  if (NULL == context->caps_proxy)
    return;
  
  if (!tp_conn_iface_capabilities_get_capabilities (context->caps_proxy,
                                                    members, &all_caps,
                                                    &error)) {
    g_warning ("Cannot get capabilities: %s", error->message);
    g_error_free (error);
    return;
  }
  for (i = 0; i < all_caps->len; i++) {
    guint handle;
    GValueArray *caps;
    const gchar *channel;
    guint generic_caps, specific_caps;
    GalagoCaps channel_caps, handle_caps;
    
    caps = g_ptr_array_index (all_caps, i);
    g_assert (caps);

    handle = g_value_get_uint (g_value_array_get_nth (caps, 0));
    if (handle == 0) {
      c_debug ("Ignoring channel capabilities");
      continue;
    }
    if (handle == context->self_handle) {
      c_debug ("Ignoring our own capabilities");
      continue;
    }
    if (context->galago_caps->len <= handle)
        g_array_set_size (context->galago_caps, handle + 1);
    channel = g_value_get_string (g_value_array_get_nth (caps, 1));

    generic_caps = g_value_get_uint (g_value_array_get_nth (caps, 2));
    specific_caps = g_value_get_uint (g_value_array_get_nth (caps, 3));

    channel_caps = telepathy_to_galago_caps (channel, generic_caps, specific_caps);
    handle_caps = g_array_index (context->galago_caps, GalagoCaps, handle);
    handle_caps |= channel_caps;
    handle_caps |= GALAGO_CAPS_CHANGED;
    g_array_index (context->galago_caps, GalagoCaps, handle) = handle_caps;
    g_value_array_free (caps);
  }
  update_capabilities (context);
  g_ptr_array_free (all_caps, TRUE);
}

static void
on_members_changed (DBusGProxy *proxy,
                    char *message,
                    GArray *added,
                    GArray *removed,
                    GArray *local_pending,
                    GArray *remote_pending,
                    guint actor,
                    guint reason,
                    gpointer user_data)
{
  FeedContext *context;
  GArray *pending, *all;
  int i;

  context = user_data;

  g_return_if_fail (DBUS_IS_G_PROXY (proxy));
  g_return_if_fail (context);

  p_debug ("%d members to be removed", removed->len);
  for (i = 0; i < removed->len; i++) {
    guint id = g_array_index (removed, guint, i);
#ifdef NICE_DEBUGGING
                if (debug_flags & DEBUG_PRESENCE) {
                        char *name;
                        
                        name = get_service_username_from_tp_handle (context->conn, id);
                        if (!name) {
                                name = g_strdup_printf ("unknown handle %d", id);
                        }
                        p_debug ("Removing %s (%p)", name, context);
                        g_free (name);
                }
#else
                p_debug ("Removing %d (%p)", id, context);
#endif
    g_hash_table_remove (context->account_hash, GINT_TO_POINTER (id));
    g_array_index (context->galago_caps, GalagoCaps, id) = 0;
  }

  pending = g_array_new (FALSE, FALSE, sizeof (guint));
  g_array_append_vals (pending, local_pending->data, local_pending->len);
  g_array_append_vals (pending, remote_pending->data, remote_pending->len);

  handle_member_presence (context, added, pending);
  
  all = g_array_new (FALSE, FALSE, sizeof (guint));
  g_array_append_vals (all, added->data, added->len);
  g_array_append_vals (all, pending->data, pending->len);
  handle_member_capabilities (context, all);
  
  g_array_free (pending, TRUE);
  g_array_free (all, TRUE);
}

static void
on_capabilities_changed (DBusGProxy *proxy,
                         GPtrArray *all_caps,
                         FeedContext *context)
{
  int i;

  g_assert (context);
  
  c_debug ("Capabilities changed (context %p)\n", context);
  
  for (i = 0; i < all_caps->len; i++) {
    guint handle;
    GValueArray *caps;
    const gchar *channel;
    guint generic_caps, specific_caps;
    GalagoCaps channel_caps, handle_caps, old_channel_caps;
    
    caps = g_ptr_array_index (all_caps, i);
    g_assert (caps);

    handle = g_value_get_uint (g_value_array_get_nth (caps, 0));
    if (handle == 0) {
      c_debug ("Ignoring channel capability update");
      continue;
    }
    if (handle == context->self_handle) {
      c_debug ("Ignoring update of our own capabilities");
      continue;
    }

    channel = g_value_get_string (g_value_array_get_nth (caps, 1));
    
    generic_caps = g_value_get_uint (g_value_array_get_nth (caps, 2));
    specific_caps = g_value_get_uint (g_value_array_get_nth (caps, 4));
    old_channel_caps = telepathy_to_galago_caps (channel, generic_caps, specific_caps);

    generic_caps = g_value_get_uint (g_value_array_get_nth (caps, 3));
    specific_caps = g_value_get_uint (g_value_array_get_nth (caps, 5));
    channel_caps = telepathy_to_galago_caps (channel, generic_caps, specific_caps);
    
    handle_caps = g_array_index (context->galago_caps, GalagoCaps, handle);
    handle_caps ^= old_channel_caps;
    handle_caps |= channel_caps;
    handle_caps |= GALAGO_CAPS_CHANGED;
    g_array_index (context->galago_caps, GalagoCaps, handle) = handle_caps;
  }
  update_capabilities (context);
}

/* Disconnect the signals */
void
context_disconnected (FeedContext *context)
{
  d_debug ("Disconnecting context %p", context);

  g_return_if_fail (context->status == 2);
  context->status = (guint)-2;

  if (context->presence_proxy) {
    dbus_g_proxy_disconnect_signal (context->presence_proxy, "PresenceUpdate",
                                    G_CALLBACK (on_presence_update), context);
  }

  if (context->group_proxy) {
    dbus_g_proxy_disconnect_signal (context->group_proxy, "MembersChanged",
                                    G_CALLBACK (on_members_changed), context);
  }

  if (context->caps_proxy) {
    dbus_g_proxy_disconnect_signal (context->caps_proxy, "CapabilitiesChanged",
                                    G_CALLBACK (on_capabilities_changed),
                                    context);
  }

  /* Do not unref proxies returned by tp_[chan|conn]_get_interface */
  context->group_proxy = NULL;
  context->presence_proxy = NULL;
  context->caps_proxy = NULL;

  if (context->chan) {
    g_object_unref (context->chan);
    context->chan = NULL;
  }

  if (context->conn) {
    g_object_unref (context->conn);
    context->conn = NULL;
  }

  if (context->statuses) {
    g_hash_table_destroy (context->statuses);
    context->statuses = NULL;
  }
  
  if (context->galago_caps) {
    g_array_free (context->galago_caps, TRUE);
    context->galago_caps = NULL;
  }

  if (context->account_hash) {
    g_hash_table_destroy (context->account_hash);
    context->account_hash = NULL;
  }
}

void
context_connected (FeedContext *context)
{
  GError *error = NULL;
  GArray *members;
  GArray *local_pending;
  GArray *remote_pending;

  d_debug ("Connecting context %p", context);

  /* Our code assumes that a context is destroyed after tp-connection is closed
   * If we get here for a second time, we somehow missed the disconnect signal
   */
  g_return_if_fail (context->account_hash == NULL);

  if (!context->list_handle) {
    if (!tf_tp_conn_request_handle (DBUS_G_PROXY (context->conn),
                                    TP_CONN_HANDLE_TYPE_LIST,
                                    "subscribe", &context->list_handle,
                                    &error)) {
      g_warning ("Cannot get handle: %s", error->message);
      g_error_free (error);
      return;
    }
  }

  if (!context->self_handle) {
    if (!tp_conn_get_self_handle (DBUS_G_PROXY (context->conn),
                                  &context->self_handle, &error)) {
      g_warning ("Cannot get self handle: %s", error->message);
      g_error_free (error);
      return;
    }
  }
  if (!context->account_username) {
    context->account_username = get_service_username_from_tp_handle (context->conn, context->self_handle);
    if (!context->account_username)
      return;
  }

  if (!context->chan) {
    context->chan = tp_conn_new_channel (gconnection, context->conn,
                                         dbus_g_proxy_get_bus_name (DBUS_G_PROXY (context->conn)),
                                         TP_IFACE_CHANNEL_TYPE_CONTACT_LIST,
                                         TP_CONN_HANDLE_TYPE_LIST, 
                                         context->list_handle, FALSE);
    if (!context->chan) {
      g_warning ("Cannot open channel");
      return;
    }

    g_object_add_weak_pointer (G_OBJECT (context->chan),
                               (gpointer)&context->chan);
  }

  if (!context->group_proxy) {
    context->group_proxy = tp_chan_get_interface (context->chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
    if (context->group_proxy == NULL) {
      g_warning ("Cannot get subscribe group interface for connection");
      return;
    }

    g_object_add_weak_pointer (G_OBJECT (context->group_proxy),
                               (gpointer)&context->group_proxy);
  }

  if (!context->presence_proxy) {
    context->presence_proxy = tp_conn_get_interface (context->conn, TELEPATHY_CONN_IFACE_PRESENCE_QUARK);
    if (context->presence_proxy == NULL) {
      g_warning ("Cannot get presence interface for connection");
      return;
    }

    g_object_add_weak_pointer (G_OBJECT (context->presence_proxy),
                               (gpointer)&context->presence_proxy);
  }

  if (!context->caps_proxy) {
    context->caps_proxy = tp_conn_get_interface (context->conn, TELEPATHY_CONN_IFACE_CAPABILITIES_QUARK);
    if (context->caps_proxy == NULL) {
      g_warning ("Cannot get capabilities interface for connection");
    } else {
      g_object_add_weak_pointer (G_OBJECT (context->caps_proxy),
                                 (gpointer)&context->caps_proxy);
    }
  }
  
  if (!tp_chan_iface_group_get_all_members (context->group_proxy, &members, 
                                            &local_pending, &remote_pending, 
                                            &error)) {
    g_warning ("Cannot get subscribed members: %s", error->message);
    g_error_free (error);
    return;
  }

  /* If we have come this far, all calls should have succeeded
   * Now set up the signal handlers and let's go
   */
  context->account_hash = g_hash_table_new_full (NULL, NULL, NULL, g_object_unref);
  context->galago_caps = g_array_new (TRUE, TRUE, sizeof (GalagoCaps));

  dbus_g_proxy_connect_signal (context->presence_proxy, "PresenceUpdate",
                               G_CALLBACK (on_presence_update),
                               context, NULL);
  dbus_g_proxy_connect_signal (context->group_proxy, "MembersChanged",
                               G_CALLBACK (on_members_changed),
                               context, NULL);
  if (context->caps_proxy) {
    dbus_g_proxy_connect_signal (context->caps_proxy, "CapabilitiesChanged",
                                 G_CALLBACK (on_capabilities_changed),
                                 context, NULL);
  }

  if (!tp_conn_iface_presence_get_statuses (context->presence_proxy,
                                            &context->statuses, &error)) {
    g_warning ("Cannot get statuses: %s", error->message);
    g_warning ("Falling back on hardcoded statuses");
    g_error_free (error);
  }

  g_array_append_vals (local_pending, remote_pending->data,
                       remote_pending->len);
  handle_member_presence (context, members, local_pending);
  
  g_array_append_vals (members, local_pending->data,
                       local_pending->len);
  handle_member_capabilities (context, members);
  
  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);
}

