/*
 * This file is part of sp-response-time
 *
 * Copyright (C) 2005-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */ 

/**
   @file responsetime.c

   Implementation of configuration operations
   <p>
*/

#include <stdlib.h>
#include <signal.h>
#include <sys/time.h>

#include <X11/Xlib.h>

#include <pthread.h>

#include <popt.h>

#include "config.h"
#include "common.h"
#include "analyser.h"
#include "record.h"
#include "damage.h"

static pthread_t threads[3];
static Bool end_application = False;

static void signal_handler(int sig) {
	switch (sig) {
		case SIGUSR1:
			send_dummy_event();
			break;
		case SIGUSR2:
			reset_analyser();
			break;
		case SIGINT:
		case SIGTERM:
			terminate_analyser();
			end_application = True;
			break;
	}
}

int main(int argc, char **argv) {
	int rc1, rc2, rc3;
	poptContext pctx;
	int status;
	
	static AnalyserInitData analyser_init_data
		= {DEFAULT_WAIT_TIME, DEFAULT_IGNORE_AREA};
	
	char *output_path = NULL;
	
	// TODO: Should we support i18n.
	struct poptOption commandline_options[] = {
		{
			"ignore-area",	'i',	POPT_ARG_INT,	&analyser_init_data.ignore_area,	0,
			"All damages smaller than this area are ignored. Where X is area in square pixels.",
			"X"
		},
		{
			"wait-time",	'w',	POPT_ARG_INT,	&analyser_init_data.wait_time,	0,
			"Time to wait after last non ignored damage to be sure that update is done. Where T is time in milliseconds.",
			"T"
		},
		{
			"out",		'o',	POPT_ARG_STRING,&output_path,	0,
			"Output file. If not given stdout will be used.",
			"path/to/file"
		},
		POPT_AUTOHELP
		{ NULL, '\0', 0, NULL, 0, NULL, NULL }
	};

	(void) signal (SIGINT, signal_handler);
	(void) signal (SIGTERM, signal_handler);
	(void) signal (SIGUSR1, signal_handler);
	(void) signal (SIGUSR2, signal_handler);

	pctx = poptGetContext(NULL, argc, (const char**)argv, commandline_options, 0);

	for (int opt = 0; opt >= 0; opt = poptGetNextOpt(pctx));

	if ( output_path == NULL) {
		init_logging(USE_STDOUT, NULL);
	} else {
		init_logging(USE_FILE, output_path);
	}
	
	// TODO: We should do proper error checking when starting threads
	// and also later when ending threads.
	
	rc1 = pthread_create(&threads[0], NULL, run_analyser, (void*)&analyser_init_data);
	
	// TODO: implement more convinient way of waiting for analyser
	// to be initialized correctly. This is just quick hack.	
	sleep(2);
	
	rc2 = pthread_create(&threads[1], NULL, event_recorder, NULL);
	rc3 = pthread_create(&threads[2], NULL, damage_listener, NULL);
	
	while(end_application == False) {
		sleep(-1);
	}

	// TODO: maybe we should terminate recorder and damage lister properly
	// instead of killing the threads.
	// rc = pthread_join(threads[2], (void **)&status);
	// rc = pthread_join(threads[1], (void **)&status);
	rc3 = pthread_kill(threads[2], SIGTERM);
	rc2 = pthread_kill(threads[1], SIGTERM);
	rc1 = pthread_join(threads[0], (void **)&status);
	end_logging();
	
	return EXIT_SUCCESS;
}
