/*
 * This file is part of sp-response-time
 *
 * Copyright (C) 2005-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
   @file damage.c

   Implementation of configuration operations
   <p>
*/


#include <X11/Xlib.h>

#include <stdlib.h>
#include <pthread.h>
#include <sys/time.h>
#include <X11/extensions/Xdamage.h>

#include "common.h"
#include "damage.h"
#include "analyser.h"

static int DamageEventNum;		/* Damage Ext Event ID */
static int DamageWaitSecs = 5;		/* Max time to collect damamge */
static Rectangle InterestedDamageRect;	/* Damage rect to monitor */

static Bool initialise_damage(Display* dpy) {
	Damage damage;
	int unused;

	if (!XDamageQueryExtension(dpy, &DamageEventNum, &unused)) {
		log_error(0, "No DAMAGE extension found.");
		return False;
	}
	
	/*
	 * Set up our interested rect 
	 */
	InterestedDamageRect.x = 0;
	InterestedDamageRect.y = 0;
	InterestedDamageRect.width = DisplayWidth(dpy, DefaultScreen(dpy));
	InterestedDamageRect.height = DisplayHeight(dpy, DefaultScreen(dpy));

	/*
	 * XXX Return/global this ? 
	 */
	damage = XDamageCreate(dpy,
		DefaultRootWindow(dpy),
		XDamageReportBoundingBox);
		
	return True;
}


/**  
 * Get an X event with a timeout ( in secs ). The timeout is
 * updated for the number of secs left.
 */
static Bool get_xevent_timed(Display * dpy, XEvent * event_return, struct timeval *tv) {

	if (tv == NULL || (tv->tv_sec == 0 && tv->tv_usec == 0)) {
		XNextEvent(dpy, event_return);
		return True;
	}

	XFlush(dpy);

	if (XPending(dpy) == 0) {
		int fd = ConnectionNumber(dpy);
		fd_set readset;

		FD_ZERO(&readset);
		FD_SET(fd, &readset);

		if (select(fd+1, &readset, NULL, NULL, tv) == 0) {
			return False;
		} else {
			XNextEvent(dpy, event_return);
			return True;
		}
	} else {
		XNextEvent(dpy, event_return);
		return True;
	}
}

/** 
 * Waits for a damage 'response' to above click
 */
static Bool wait_response(Display * dpy) {
	XEvent e;
	struct timeval tv; 
	int    waitsecs, lastsecs;

	tv.tv_sec  = lastsecs = DamageWaitSecs;
	tv.tv_usec = 0;
	
	while (get_xevent_timed(dpy, &e, &tv)) {
		if (e.type == DamageEventNum + XDamageNotify) {
			XDamageNotifyEvent *dev = (XDamageNotifyEvent *) & e;
				
			if (dev->area.x >= InterestedDamageRect.x
				&& dev->area.width <= InterestedDamageRect.width
				&& dev->area.y >= InterestedDamageRect.y
				&& dev->area.height <= InterestedDamageRect.height) {
				handle_damage(dev->timestamp, dev);
			} else {
				waitsecs = lastsecs; /* Reset */
				tv.tv_sec = lastsecs;
			}
			
			XDamageSubtract(dpy, dev->damage, None, None);
		} else {
			waitsecs = lastsecs; /* Reset */
			tv.tv_sec = lastsecs;
			// TODO: do we need to know when these occur?
			// fprintf(stderr, "Got unwanted event type %d\n", e.type);
		}
	}
	
	return True;
}

void* damage_listener() {
	Display* dpy = NULL;

	if ((dpy = setup_display(getenv("DISPLAY"))) == NULL) {
		pthread_exit(NULL);
	}

	if ((initialise_damage(dpy)) == False) {
		pthread_exit(NULL);
	}
	
	while (wait_response(dpy));

	XCloseDisplay(dpy);

	pthread_exit(NULL);
}
