/*
 * This file is part of sp-response-time
 *
 * Copyright (C) 2005-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */ 

/**
   @file analyser.c

   Implementation of configuration operations
*/

#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <pthread.h>
#include <signal.h>

#include <X11/Xlib.h>
#include <sys/time.h>

#include "common.h"
#include "analyser.h"

static pthread_mutex_t event_mutex;
static pthread_mutex_t damage_mutex;

// These are timestamps got from X APIs.
static Time latest_event = 0;
static Time latest_damage = 0;
static Time latest_big_damage = 0;

static Bool unhandled_event = False;
static int ignore_area = 0;

static Bool end_thread = False;

static Time first_event_xserver_time = 0;

struct timeval first_event_time_of_day;

static Time translate_timeval_time(const struct timeval* time_value);
static void translate_xserver_time(const Time, struct timeval* time_value);

void terminate_analyser() {
	end_thread = True;
}

void reset_analyser() {
	// TODO: This should be rewritten because this is not correct way 
	// to do this, but at least this won't end up into dead lock.
	// Correct way would be to get both mutices.
	pthread_mutex_lock(&event_mutex);
	unhandled_event = False;
	pthread_mutex_unlock(&event_mutex);
	pthread_mutex_lock(&damage_mutex);
	latest_big_damage = 0;
	latest_damage = 0;
	pthread_mutex_unlock(&damage_mutex);
}

void send_dummy_event() {
	struct timeval tv;
	gettimeofday(&tv, NULL);
	handle_event(translate_timeval_time(&tv));
}

static long get_millisecond_diff(const struct timeval from_time, const struct timeval to_time) {
	long secs = to_time.tv_sec - from_time.tv_sec;
	long msecs = (to_time.tv_usec - from_time.tv_usec) / 1000;
	return secs * 1000 + msecs;
}

static void translate_xserver_time(const Time xserver_timestamp, struct timeval* time_value) {
	Time tmp = 0;
	tmp = xserver_timestamp - first_event_xserver_time;
	time_value->tv_sec = first_event_time_of_day.tv_sec + (time_t)(tmp / 1000);
	if ( (first_event_time_of_day.tv_usec + (tmp % 1000) * 1000) >= 1000000) {
		time_value->tv_sec++;
		time_value->tv_usec = first_event_time_of_day.tv_usec + (tmp % 1000) * 1000 - 1000000;
	} else {
		time_value->tv_usec = first_event_time_of_day.tv_usec + (tmp % 1000) * 1000;
	}		
}

static Time translate_timeval_time(const struct timeval* time_value) {
	Time tmp = first_event_xserver_time;
	tmp += (time_value->tv_sec - first_event_time_of_day.tv_sec) * 1000;
	tmp += (int)((time_value->tv_usec - first_event_time_of_day.tv_usec) / 1000);
	return tmp;
}

void handle_event(Time timestamp) {		
	pthread_mutex_lock(&event_mutex);
	
	static Bool first_event = True;
	Bool ignore = False;
	Time interested_damage = 0;
	
	if ( first_event == True) {
		first_event_xserver_time = timestamp;
		gettimeofday(&first_event_time_of_day, NULL);
		ignore = True;
		first_event = False;
	}
	
	if (unhandled_event == True) {
		if (latest_big_damage >= latest_event) {
			interested_damage = latest_big_damage;
		} else if (latest_damage >= latest_event ) {
			interested_damage = latest_damage;
		} else {
			ignore = True;
		}
	}
		
	if (ignore == False) {
		if (unhandled_event == True) {
			struct timeval report_timestamp, report_event, report_damage;
			gettimeofday(&report_timestamp, NULL);
			translate_xserver_time(latest_event, &report_event);
			translate_xserver_time(interested_damage, &report_damage);
			log_analysis_report(report_timestamp, report_event, report_damage, interested_damage - latest_event, "Got new event in the middle of update. It is possible that update time is not correct.");
		}
		unhandled_event = True;
		latest_event = timestamp;
	}
	
	pthread_mutex_unlock(&event_mutex);
}

void handle_damage(Time timestamp, XDamageNotifyEvent* dev) {
	pthread_mutex_lock(&damage_mutex);

	if (dev->area.width*dev->area.height > ignore_area) {
		latest_big_damage = timestamp;
	}
	latest_damage = timestamp;
	pthread_mutex_unlock(&damage_mutex);
}

void* run_analyser(void* init_data) {
	
	pthread_mutex_init(&event_mutex, NULL);
	pthread_mutex_init(&damage_mutex, NULL);
	
	AnalyserInitData* i_data = NULL;
	Time wait_time = 0;
	
	if (init_data != NULL) {
		// TODO: Check that init_data contains some sane values.
		i_data = (AnalyserInitData*) init_data;
		wait_time = i_data->wait_time;
		ignore_area = i_data->ignore_area;
	} else {
		wait_time = DEFAULT_WAIT_TIME;
		ignore_area = DEFAULT_IGNORE_AREA;
	}
	
	latest_big_damage = 0;
	latest_damage = 0;
	latest_event = 0;
	unhandled_event = False;
	end_thread = False;
	
	while(end_thread == False) {
		usleep(wait_time * 1000);
		if (unhandled_event == True) {
			
			pthread_mutex_lock(&event_mutex);
			
			struct timeval now, report_damage, report_event;
			gettimeofday(&now, NULL);
			translate_xserver_time(latest_event, &report_event);
			translate_xserver_time(latest_big_damage, &report_damage);
			if (latest_big_damage >= latest_event 
				&& get_millisecond_diff(report_damage, now) >= wait_time) {
				log_analysis_report(now, report_event, report_damage, latest_big_damage - latest_event, NULL);
				unhandled_event = False;
			}
			
			pthread_mutex_unlock(&event_mutex);
		} 
	}

	pthread_mutex_destroy(&event_mutex);
	pthread_mutex_destroy(&damage_mutex);
	
	pthread_exit(NULL);
}
