/**
 * @file ui_mainwindow.c some functions concerning the main window 
 *
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-helper.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-help.h>
#include <osso-log.h>
#include <langinfo.h>

#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/cache_handling.h>

#include "interface.h"
#include "support.h"
#include "conf.h"
#include "debug.h"
#include "callbacks.h"
#include "ui_feedlist.h"
#include "ui_mainwindow.h"
#include "ui_folder.h"
#include "ui_itemlist.h"
#include "update.h"
#include "htmlview.h"
#include "dbus.h"
#include "gtkhtml/gtkhtml_plugin.h"

#include "debug_new.h"

/* The Hildon AppView for the application */
GtkWidget *mainwindow;

/* A container for holding the feed list container and the search view */
GtkWidget *newsbox;

/* A GtkTreeView widget for displaying the feed list */
GtkWidget *feedlist;

/* A container for the feed list, used to show or hide the folder view */
GtkWidget *feedlist_container;

/* Application specific date */
extern AppData *app_data;

/* The number of unread posts */
gulong rss_unread;

/* The last refresh time of the news reader */
time_t rss_updated;

static GtkWidget *search_count = NULL;
gfloat zoom;

gboolean itemlist_mode = FALSE; /* TRUE means three pane, FALSE means two panes */

gboolean key_press_for_cancelling_dialog2(GtkWidget * widget,
                                          GdkEventKey * event, gpointer data);
void ui_create_toolbar(AppUIData * app_ui_data);

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

GtkWidget *ui_mainwindow_get_active_htmlview()
{
    return app_data->app_ui_data->html_scrollpane;
}

extern gchar *prev_find_text;

void ui_mainwindow_set_search(gboolean search)
{
    g_assert(NULL != search_count);
    g_assert(NULL != app_data->app_ui_data->html_scrollpane);

    if (prev_find_text) {
        g_free(prev_find_text);
        prev_find_text = NULL;
    }

    if (TRUE == search) {
        g_assert(app_data != NULL);
        g_assert(app_data->app_ui_data != NULL);
        ui_htmlview_clear(ui_mainwindow_get_active_htmlview());
        gtk_window_set_title(GTK_WINDOW(app_data->app_ui_data->main_view),
                             _("rss_ti_search"));
        debug0(DEBUG_GUI, "Setting search-view on.");
        gtk_widget_show(search_count);
    } else {
        debug0(DEBUG_GUI, "Setting search-view off.");
        gtk_widget_hide(search_count);
        gtk_widget_hide(app_data->app_ui_data->search_label);
    }
}

void ui_mainwindow_construct_search_view()
{
    gchar *str = NULL;
    gchar *t1_raw = NULL;
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (displayed_node != NULL)
        if (displayed_node->type != FST_VFOLDER)
            return;

    t1_raw = g_strdup_printf("<span weight=\"bold\">%s</span>",
                             ngettext("rss_ia_item_found",
                                      "rss_ia_items_found",
                                      app_data->app_ui_data->
                                      rss_search_found));
    str = g_strdup_printf(t1_raw, app_data->app_ui_data->rss_search_found);
    g_free(t1_raw);

    gtk_label_set_markup(GTK_LABEL(app_data->app_ui_data->search_label), str);

    gtk_widget_show(app_data->app_ui_data->search_label);

    g_free(str);
}

void ui_mainwindow_set_mode(gboolean threePane)
{
    debug_enter("ui_mainwindow_set_mode");
    debug1(DEBUG_GUI, "Setting threePane mode: %s", threePane ? "on" : "off");

    g_assert(NULL != newsbox);

    if (threePane == TRUE && (itemlist_mode == FALSE
                              || app_data->app_ui_data->html_scrollpane ==
                              NULL)) {
        ULOG_ERR("Three-pane view is not supported!");

    } else if (threePane == FALSE
               && (itemlist_mode == TRUE
                   || app_data->app_ui_data->html_scrollpane == NULL)) {
        gtk_widget_grab_focus(feedlist);
        /* Construct htmlview */
        if (app_data->app_ui_data->html_scrollpane != NULL)
            gtk_widget_destroy(app_data->app_ui_data->html_scrollpane);
        app_data->app_ui_data->html_scrollpane = NULL;
        ULOG_DEBUG("tvh ui_mainwindow: ui_htmlviw_new starting...");
        app_data->app_ui_data->html_scrollpane = ui_htmlview_new();

        g_signal_connect(G_OBJECT
                         (gtk_bin_get_child
                          (GTK_BIN(app_data->app_ui_data->html_scrollpane))),
                         "focus-in-event",
                         G_CALLBACK(on_html_scrollpane_focus_in), app_data);

        g_signal_connect(G_OBJECT (gtk_bin_get_child (GTK_BIN(app_data->app_ui_data->html_scrollpane))),
                         "expose-event",
                         G_CALLBACK(on_gtkhtml_expose), NULL);

        g_signal_connect(G_OBJECT (gtk_bin_get_child (GTK_BIN(app_data->app_ui_data->html_scrollpane))),
                         "key-press-event",
                         G_CALLBACK(on_gtkhtml_keyboard_event), app_data);

        ULOG_DEBUG("tvh ui_mainwindow: ui_htmlviw_new done");

        g_assert(app_data != NULL);
        g_assert(app_data->app_ui_data != NULL);

        /* Construct search-view */
        app_data->app_ui_data->search_label = gtk_label_new(NULL);
        ui_mainwindow_construct_search_view();
        g_object_set(G_OBJECT(app_data->app_ui_data->search_label),
                     "no-show-all", TRUE, NULL);

        gtk_misc_set_alignment(GTK_MISC
                               (app_data->app_ui_data->search_label), 0, 0.5);
        search_count = gtk_vbox_new(FALSE, HILDON_MARGIN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(search_count),
                           app_data->app_ui_data->search_label, FALSE,
                           FALSE, 0);
        GtkWidget *search_view = gtk_vbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(search_view), search_count, FALSE,
                           FALSE, 0);
        gtk_box_pack_end(GTK_BOX(search_view),
                         app_data->app_ui_data->html_scrollpane,
                         TRUE, TRUE, 0);
        gtk_widget_show_all(search_view);
        gtk_paned_pack2(GTK_PANED(newsbox), search_view, TRUE,  /*resize-able */
                        FALSE); /*shrink-able */
        gtk_widget_show_all(newsbox);

        ui_mainwindow_set_search(FALSE);
    }


    itemlist_mode = threePane;

    debug_exit("ui_mainwindow_set_mode");
    ULOG_DEBUG("tvh ui_mainwindow:  done");
}

void ui_mainwindow_finish()
{
    gchar *textbuffer = NULL;
    gchar *t1_raw = NULL;
    gchar *t1_format = NULL;
    gchar *t2_raw = NULL;
    gchar *t2_format = NULL;
    gchar *t3_raw = NULL;
    gchar *t3_format = NULL;
    gchar *t4_raw = NULL;
    gchar *t4_format = NULL;
    gint nr_feeds = 0;
    gint nr_unread = 0;
    gint nr_unread_feeds = 0;
    gint nr_marked = 0;
    glong last_poll_time = 0;
    gchar *update_time;
    gchar *update_date;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    ULOG_DEBUG
        ("tvh ui_mainwindow_finish:  calling ui_feedlist_find_mainwindow_data......");


    ui_feedlist_find_mainwindow_data(NULL, &nr_feeds, &nr_unread,
                                     &nr_unread_feeds, &nr_marked,
                                     &last_poll_time);

    rss_unread = (long) nr_unread;

    ULOG_DEBUG("rss_updated: %ld last_poll_time: %ld", rss_updated,
               last_poll_time);

    /*if(rss_updated < (time_t)last_poll_time)
     * { */
    ULOG_DEBUG("Setting rss_updated to %ld", last_poll_time);
    rss_updated = (time_t) last_poll_time;
    /*} */
    //    general_save();
    /* tvh: TODO commented this out due to bug 22893: Applet only gets signal when
     * refreshing....FOLDER read doesn't affect applet. DOES IT? */
    if (!app_data->app_ui_data->feed_displayed) {
        ui_mainwindow_set_search(FALSE);

        update_time = get_localized_time(rss_updated);
        update_date = get_localized_date(rss_updated, TRUE);
        //        if(nr_feeds > 0)
        {
            t1_raw = g_strdup_printf("<p>%s\n</p>",
                                     ngettext
                                     ("rss_ia_you_have_subscribed_to_n_feed",
                                      "rss_ia_you_have_subscribed_to_n_feeds",
                                      nr_feeds));
            t1_format = g_strdup_printf(t1_raw, nr_feeds);
            g_free(t1_raw);
        }

        if (rss_updated > 0) {
            t2_raw = g_strdup_printf("<p>%s\n</p>",
                                     _
                                     ("rss_ia_last_refresh_on_date_at_time"));
            t2_format = g_strdup_printf(t2_raw, update_date, update_time);
            g_free(t2_raw);
        } else
            t2_format =
                g_strdup_printf("<p>%s\n</p>",
                                _("rss_ia_feeds_never_refreshed"));

        //        if(nr_unread > 0)
        {
            t3_raw = g_strdup_printf("<p>%s\n</p>",
                                     ngettext
                                     ("rss_ia_there_are_n_unread_post",
                                      "rss_ia_there_are_n_unread_posts",
                                      nr_unread));
            t3_format = g_strdup_printf(t3_raw, nr_unread, nr_unread_feeds);
            g_free(t3_raw);
        }

        //        if(nr_marked > 0)
        {
            t4_raw = g_strdup_printf("<p>%s\n</p>",
                                     ngettext("rss_ia_you_have_saved_n_item",
                                              "rss_ia_you_have_saved_n_items",
                                              nr_marked));
            t4_format = g_strdup_printf(t4_raw, nr_marked);
            g_free(t4_raw);
        }
        //        else
        //           t4_format = g_strdup_printf("<p>%s\n</p>", _("rss_ia_no_saved_posts"));

        textbuffer =
            g_strconcat(t1_format, t2_format, t3_format, t4_format, NULL);

        ULOG_DEBUG
            ("tvh ui_mainwindow_finish:  writing general window now with textbuffer =%s",
             textbuffer);

        ui_htmlview_write(app_data->app_ui_data->html_scrollpane, textbuffer,
                          NULL, NULL);
        /* needed
         * Otherwise, after a refresh_all action, ui_mainwindow_finish will be called
         * if we don't set displayed_node to NULL, clicking on the previously selected feed wouldn't
         * display it 
         */
        displayed_node = NULL;

        if (GTK_WIDGET_VISIBLE(feedlist)) {
            gtk_widget_grab_focus(feedlist);
        }

        g_free(update_time);
        g_free(update_date);
        g_free(t1_format);
        g_free(t2_format);
        g_free(t3_format);
        g_free(t4_format);
        g_free(textbuffer);
    }
}

void on_unsubscribe_clicked(GtkButton * button, gpointer user_data)
{



    /* TODO: downloading images also uses SFM_REFRESH mode */
    if (SFM_REFRESH == app_data->app_ui_data->search_mode) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, _("rss_ib_unable_unsubscribe"));
        return;
    } else if (SFM_SEARCH == app_data->app_ui_data->search_mode) {
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL,
                                       _
                                       ("rss_ib_unable_unsubscribe_searching"));
        return;
    }

    nodePtr ptr;

    if (displayed_node && gtk_widget_is_focus
        (gtk_bin_get_child(GTK_BIN(app_data->app_ui_data->html_scrollpane))))
    {

        ptr = (nodePtr) displayed_node;

    } else {

        ptr = (nodePtr) ui_feedlist_get_selected();

    }



    if (ui_feedlist_check_factory_delete(TRUE))
        return;

    delete_feed(ptr);
}



void on_menu_open(GtkMenuItem * menuitem, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);
    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();

    if (ptr && ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type))) {
        ui_feedlist_load_selected(ptr);
    } else {
        // Should be on_feed_insensitive, but here for safety.
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, _("rss_ib_no_feed_selected"));
    }
}

gboolean delete_feed(nodePtr ptr)
{
    if ((ptr != NULL) && (ptr->type != FST_FOLDER)) {
        return ui_feedlist_delete(ptr);
    } else {
        // Should be on_feed_insensitive, but here for safety.
        ULOG_WARN("Feed selection was NULL, or FOLDER cannot delete.");
        hildon_banner_show_information(GTK_WIDGET
                                       (app_data->app_ui_data->main_view),
                                       NULL, _("rss_ib_no_feed_selected"));
    }
    return FALSE;
}

void on_menu_properties(GtkMenuItem * menuitem, gpointer user_data)
{
    nodePtr ptr;

    if (displayed_node && gtk_widget_is_focus
        (gtk_bin_get_child(GTK_BIN(app_data->app_ui_data->html_scrollpane))))
    {

        ptr = displayed_node;

    } else {

        ptr = (nodePtr) ui_feedlist_get_selected();

    }

    ULOG_DEBUG("on_menu_properties: CALLED \n\n\n\n\n\n\n\n\n\n\n\n");



    if ((ptr != NULL) && (FST_FEED == ptr->type)) {
        on_popup_prop_selected((gpointer) ptr, 0, NULL);
    }
}

gboolean on_close(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
    //TODO: Remove SHOW_TRAY_ICON
    if (getBooleanConfValue(SHOW_TRAY_ICON) == FALSE)
        return on_quit(TRUE);
    gtk_widget_hide(mainwindow);
    return TRUE;
}

void ui_help_cb()
{
    g_assert(app_data != NULL);
    g_assert(app_data != NULL);
    g_assert(app_data->osso);

    osso_return_t rc;

    rc = hildon_help_show(app_data->osso, RSS_NEWS_READER_HELP_TOPIC, 0);
    if (rc != OSSO_OK) {
        ULOG_ERR("Unable to show help: '%s' (errcode %d)",
                 RSS_NEWS_READER_HELP_TOPIC, (int) rc);
        return;
    }
}

gboolean on_mainwindow_window_state_event(GtkWidget * widget,
                                          GdkEvent * event,
                                          gpointer user_data)
{

    if (!GTK_WIDGET_VISIBLE(mainwindow))
        return FALSE;

    if ((event->type) == (GDK_WINDOW_STATE)) {
        setBooleanConfValue(LAST_WINDOW_MAXIMIZED, FALSE);
    }
    return FALSE;
}

struct file_chooser_tuple {
    GtkWidget *dialog;
    fileChoosenCallback func;
    gpointer user_data;
};


/* Create the toolbar */
void ui_create_toolbar(AppUIData * app_ui_data)
{
    /* Toolbar buttons */
    GtkToolItem *refresh_button,
        *new_feed_button,
        *feed_properties_button,
        *remove_feed_button, *toolitem, *search_button;
    GtkToolItem *separator;

    GtkToggleToolButton *folder_list_toggle_button;
    ULOG_DEBUG("ui_create_toolbar:");
    /* Paranoia */
    g_assert(app_ui_data);
    g_assert(!app_ui_data->main_toolbar);

    app_ui_data->main_toolbar = gtk_toolbar_new();


    app_ui_data->toolbar_temp_shown = FALSE;
    g_signal_connect(G_OBJECT(app_ui_data->main_view),
                     "key_press_event",
                     G_CALLBACK(key_press_with_hidden_toolbar), NULL);
    app_ui_data->toolbar_keypress_handler_blocked = FALSE;

    gtk_widget_show_all(app_ui_data->main_toolbar);

    gtk_toolbar_set_orientation(GTK_TOOLBAR(app_ui_data->main_toolbar),
                                GTK_ORIENTATION_HORIZONTAL);
    gtk_container_set_border_width(GTK_CONTAINER
                                   (app_ui_data->main_toolbar), 0);

    g_object_ref_sink(G_OBJECT(app_ui_data->main_toolbar));

    /*Refresh button */
    ADD_TOOLBAR_BUTTON(refresh_button, "qgn_toolb_gene_refresh",
                       on_refreshbtn_clicked)
        app_ui_data->refresh_button = GTK_TOOL_BUTTON(refresh_button);
    app_ui_data->button_image_refresh =
        gtk_tool_button_get_icon_widget(app_ui_data->refresh_button);
    g_object_ref(app_ui_data->button_image_refresh);

    /*Add feed button */
    ADD_TOOLBAR_BUTTON(new_feed_button, "qgn_toolb_rss_addfeed",
                       on_newbtn_clicked_cb)


        app_ui_data->new_feed_button = GTK_TOOL_BUTTON(new_feed_button);
    /* feed properties button */
    ADD_TOOLBAR_BUTTON(feed_properties_button, "qgn_toolb_rss_feedprop",
                       on_menu_properties)


        app_ui_data->feed_properties_button =
        GTK_TOOL_BUTTON(feed_properties_button);

    /* Delete feed button */
    ADD_TOOLBAR_BUTTON(remove_feed_button,
                       "qgn_toolb_rss_unsubscribefeed",
                       on_unsubscribe_clicked)
        app_ui_data->unsubscribe_button = GTK_TOOL_BUTTON(remove_feed_button);


    ADD_TOOLBAR_SEPARATOR
        folder_list_toggle_button =
        GTK_TOGGLE_TOOL_BUTTON(gtk_toggle_tool_button_new());

    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                                    (folder_list_toggle_button),
                                    GTK_WIDGET(gtk_image_new_from_icon_name
                                               ("qgn_toolb_rss_fldonoff",
                                                HILDON_ICON_SIZE_TOOLBAR)));

    gtk_widget_show_all(GTK_WIDGET(folder_list_toggle_button));
    gtk_toggle_tool_button_set_active(folder_list_toggle_button, TRUE);

    GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(folder_list_toggle_button),
                           GTK_CAN_FOCUS);
    g_signal_connect(G_OBJECT(folder_list_toggle_button), "clicked",
                     G_CALLBACK(on_folder_list_button_clicked), app_ui_data);
    gtk_toolbar_insert(GTK_TOOLBAR(app_ui_data->main_toolbar),
                       GTK_TOOL_ITEM(folder_list_toggle_button), -1);
    app_ui_data->folders_list_toggle_button =
        GTK_TOGGLE_TOOL_BUTTON(folder_list_toggle_button);

    ADD_TOOLBAR_SEPARATOR
        app_ui_data->var_field = gtk_alignment_new(0, 0.5, 1, 0);
    gtk_widget_show(app_ui_data->var_field);

    toolitem = gtk_tool_item_new();
    gtk_widget_show(GTK_WIDGET(toolitem));
    gtk_tool_item_set_expand(GTK_TOOL_ITEM(toolitem), TRUE);
    gtk_container_add(GTK_CONTAINER(toolitem), app_ui_data->var_field);

    app_ui_data->search_field = gtk_combo_box_entry_new_text();
    gtk_widget_show(app_ui_data->search_field);
    g_object_ref(app_ui_data->search_field);

    /* Add key press event to entry inside combobox */
    {
        GtkWidget *search_field_entry =
            GTK_WIDGET(GTK_BIN(app_ui_data->search_field)->child);
        g_assert(search_field_entry);
        g_signal_connect(G_OBJECT(search_field_entry), "key-press-event",
                         G_CALLBACK(on_search_field_key_press), app_ui_data);
        g_signal_connect(G_OBJECT(search_field_entry), "focus-out-event",
                         G_CALLBACK(on_search_field_focus_out), app_ui_data);
    }

    app_ui_data->progress_bar = gtk_progress_bar_new();
    gtk_widget_show(app_ui_data->progress_bar);
    g_object_ref(app_ui_data->progress_bar);
    gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
                      GTK_WIDGET(app_ui_data->search_field));
    gtk_toolbar_insert(GTK_TOOLBAR(app_ui_data->main_toolbar),
                       GTK_TOOL_ITEM(toolitem), -1);
    ADD_TOOLBAR_BUTTON(search_button, "qgn_toolb_browser_gobutton",
                       on_searchentry_activate)
        app_ui_data->search_button = GTK_TOOL_BUTTON(search_button);
    app_ui_data->button_image_search =
        gtk_tool_button_get_icon_widget(app_ui_data->search_button);
    g_object_ref(app_ui_data->button_image_search);

    /* Set toolbar properties */
    gtk_toolbar_set_style(GTK_TOOLBAR(app_ui_data->main_toolbar),
                          GTK_TOOLBAR_ICONS);

    hildon_window_add_toolbar(app_ui_data->main_view,
                                      GTK_TOOLBAR(app_ui_data->main_toolbar));

    app_ui_data->button_image_stop =
        gtk_image_new_from_icon_name("qgn_toolb_gene_stop",
                                     GTK_ICON_SIZE_LARGE_TOOLBAR);
    gtk_widget_show(app_ui_data->button_image_stop);
    g_object_ref(app_ui_data->button_image_stop);

    return;
}


/* Create the main window */
void ui_create_main_window(AppData * app_data)
{
    AppUIData *app_ui_data;
    DMSG("Entering %s\n", __FUNCTION__);

    /* Paranoia */
    g_assert(app_data);
    g_assert(app_data->app_ui_data);

    app_ui_data = app_data->app_ui_data;

    app_ui_data->app = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name(_("rss_ti_newsreader"));


    g_signal_connect(G_OBJECT(app_ui_data->app), "notify::is-topmost",
                     G_CALLBACK(top_changed), app_data);



    app_ui_data->main_view = HILDON_WINDOW(hildon_window_new());

    if (!app_data->app_ui_data->start_on_background)
        gtk_widget_show_all(GTK_WIDGET(app_ui_data->main_view));    /* show_all to make toolbar visible */

    /* Connect destroy signal */
    g_signal_connect(G_OBJECT(app_ui_data->main_view), "destroy",
                     G_CALLBACK(gtk_main_quit), NULL);

    gtk_widget_add_events(GTK_WIDGET(app_ui_data->main_view),
                          GDK_KEY_RELEASE_MASK | GDK_KEY_PRESS_MASK);

    g_signal_connect((gpointer) app_ui_data->main_view, "key-release-event",
                     G_CALLBACK(key_release), app_ui_data);

    g_signal_connect((gpointer) app_ui_data->main_view, "key-press-event",
                     G_CALLBACK(key_press), app_ui_data);
    
    g_signal_connect((gpointer) app_ui_data->main_view, "focus-out-event",
                     G_CALLBACK(window_focus_out), app_ui_data);


    hildon_program_add_window(app_ui_data->app, app_ui_data->main_view);

    ui_create_toolbar(app_ui_data);
    return;
}

void ui_set_folders_menu_item_state(AppUIData * app_ui_data, gboolean state)
{
    g_assert(app_ui_data != NULL);

    g_signal_handlers_block_by_func(G_OBJECT
                                    (app_ui_data->menuitem_folders),
                                    G_CALLBACK(toggle_folder_list), NULL);
    gtk_check_menu_item_set_active(app_ui_data->menuitem_folders, state);
    g_signal_handlers_unblock_by_func(G_OBJECT
                                      (app_ui_data->menuitem_folders),
                                      G_CALLBACK(toggle_folder_list), NULL);
}

void ui_set_fullscreen_menu_item_state(AppUIData * app_ui_data,
                                       gboolean state)
{
    g_assert(app_ui_data != NULL);

    g_signal_handlers_block_by_func(G_OBJECT
                                    (app_ui_data->menuitem_fullscreen),
                                    G_CALLBACK(on_toggle_fullscreen_mode),
                                    NULL);
    gtk_check_menu_item_set_active(app_ui_data->menuitem_fullscreen, state);
    g_signal_handlers_unblock_by_func(G_OBJECT
                                      (app_ui_data->menuitem_fullscreen),
                                      G_CALLBACK
                                      (on_toggle_fullscreen_mode), NULL);
}

void ui_set_toolbar_full_menu_item_state(AppUIData * app_ui_data,
                                         gboolean state)
{
    g_assert(app_ui_data != NULL);

    g_signal_handlers_block_by_func(G_OBJECT(app_ui_data->menuitem_full),
                                    G_CALLBACK(toggle_toolbar_cb), app_ui_data);
    gtk_check_menu_item_set_active(app_ui_data->menuitem_full, state);
    g_signal_handlers_unblock_by_func(G_OBJECT(app_ui_data->menuitem_full),
                                      G_CALLBACK(toggle_toolbar_cb), app_ui_data);
}

void ui_set_toolbar_normal_menu_item_state(AppUIData * app_ui_data,
                                           gboolean state)
{
    g_assert(app_ui_data != NULL);

    g_signal_handlers_block_by_func(G_OBJECT(app_ui_data->menuitem_normal),
                                    G_CALLBACK(toggle_toolbar_cb), app_ui_data);
    gtk_check_menu_item_set_active(app_ui_data->menuitem_normal, state);
    g_signal_handlers_unblock_by_func(G_OBJECT
                                      (app_ui_data->menuitem_normal),
                                      G_CALLBACK(toggle_toolbar_cb), app_ui_data);
}


void ui_set_folder_list_state(AppUIData * app_ui_data,
                              gboolean state, gboolean update_menu_item)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (state) {
        gtk_widget_show(feedlist_container);
    } else {
        gtk_widget_hide(feedlist_container);
    }
    if (update_menu_item) {
        ui_set_folders_menu_item_state(app_ui_data, state);
    }
}


void ui_set_fullscreen_state(AppUIData * app_ui_data,
                             gboolean state, gboolean update_menu_item)
{
    GtkWidget *toolbar;

    g_assert(app_ui_data != NULL);

    toolbar = app_ui_data->main_toolbar;

    if (state) {
        gtk_window_fullscreen(GTK_WINDOW(app_ui_data->main_view));
        app_ui_data->fullscreen = TRUE;
        ULOG_INFO("Application now in fullscreen mode");
    } else {
        gtk_window_unfullscreen(GTK_WINDOW(app_ui_data->main_view));
        app_ui_data->fullscreen = FALSE;
        ULOG_INFO("Application now out of fullscreen mode.");
    }
    toggle_toolbar_cb(NULL,app_ui_data);
    if (update_menu_item) {
        ui_set_fullscreen_menu_item_state(app_ui_data, state);
    }
}


gboolean key_press_for_cancelling_dialog2(GtkWidget * widget,
                                          GdkEventKey * event, gpointer data)
{
    (void) data;

    ULOG_DEBUG("Keypress for cancel");
    if (event == NULL)
        return FALSE;

    if (event->keyval == GDK_Escape) {
        gtk_dialog_response(GTK_DIALOG(widget), CONFRESP_CANCEL);
        return FALSE;
    }

    return FALSE;
}

ConfirmationResponse
ui_show_folder_delete_confirmation(AppUIData * app_ui_data, gint n_feeds)
{
    HildonNote *note = NULL;
    gint resp1 = CONFRESP_ERROR;
    ConfirmationResponse resp = CONFRESP_ERROR;

    note = HILDON_NOTE
        (hildon_note_new_confirmation_add_buttons
         (NULL,
          C_("ckct_nc_delete_folder"),
          _("rss_bd_ok"),
          CONFRESP_OK, _("rss_bd_cancel"), CONFRESP_CANCEL, NULL, NULL));
    if (note == NULL) {
        ULOG_CRIT("Unable to create HildonNote");
        return CONFRESP_ERROR;
    }

    /* show dialog */
    do {
        resp1 = gtk_dialog_run(GTK_DIALOG(note));
        if (resp1 == CONFRESP_OK && SFM_REFRESH == app_ui_data->search_mode) {
            hildon_banner_show_information(GTK_WIDGET(app_ui_data->main_view),
                                           NULL,
                                           dgettext
                                           (HILDON_COMMON_STRINGS_L10N_PACKAGE,
                                            _
                                            ("sfil_ib_unable_to_delete_selected_folder")));
        }
    }
    while (resp1 == CONFRESP_OK && SFM_REFRESH == app_ui_data->search_mode);


    g_signal_connect(G_OBJECT(note), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog2), NULL);
    gtk_widget_destroy(GTK_WIDGET(note));
    resp = (resp1 == CONFRESP_OK || resp1 == CONFRESP_CANCEL)
        ? resp1 : CONFRESP_ERROR;
    return resp;
}

ConfirmationResponse
ui_show_feed_delete_confirmation(AppUIData * app_ui_data, gchar * feed_title)
{
    HildonNote *note = NULL;
    gint resp1 = CONFRESP_ERROR;
    ConfirmationResponse resp = CONFRESP_ERROR;
    gchar *str = NULL, *str2 = NULL;
    GtkWidget *parent;

    str = g_strdup_printf("%s", _("rss_nc_are_you_sure_unsubscribe"));
    str2 = g_strdup_printf(str, feed_title ? feed_title : "");

    if (app_ui_data->propdialog) {
        ULOG_DEBUG("unsubs from propdialog");
        parent = app_ui_data->propdialog;
    } else {
        parent = GTK_WIDGET(app_ui_data->main_view);
    }

    note =
        HILDON_NOTE(hildon_note_new_confirmation_add_buttons
                    (GTK_WINDOW(parent), str2, _("rss_bd_ok"),
                     CONFRESP_OK, _("rss_bd_cancel"), CONFRESP_CANCEL, NULL,
                     NULL));
    app_ui_data->unsubscribe_dialog = GTK_WIDGET(note);
    if (note == NULL) {
        ULOG_CRIT("Unable to create HildonNote");
        g_free(str);
        g_free(str2);
        str = NULL;
        str2 = NULL;
        return CONFRESP_ERROR;
    }

    g_signal_connect(G_OBJECT(note), "key-press-event",
                     G_CALLBACK(key_press_for_cancelling_dialog2), NULL);

    /* show dialog */
    do {
        resp1 = gtk_dialog_run(GTK_DIALOG(note));
        if (resp1 == CONFRESP_OK && SFM_REFRESH == app_ui_data->search_mode) {
            hildon_banner_show_information(GTK_WIDGET(app_ui_data->main_view),
                                           NULL,
                                           _("rss_ib_unable_unsubscribe"));
        }
    }
    while (resp1 == CONFRESP_OK && SFM_REFRESH == app_ui_data->search_mode);

    if (app_ui_data->unsubscribe_dialog) {
            /* it can be already destroyed and set to NULL if 
             * OSSO_RSS_FEED_READER_ADD_SIGNAL was cought on dbus */
            gtk_widget_destroy(app_ui_data->unsubscribe_dialog);
            app_ui_data->unsubscribe_dialog = NULL;
    }

    g_free(str);
    g_free(str2);
    str = NULL;
    str2 = NULL;
    resp = (resp1 == CONFRESP_OK || resp1 == CONFRESP_CANCEL)
        ? resp1 : CONFRESP_ERROR;
    return resp;
}

void ui_mainwindow_show_statistics()
{
    ULOG_DEBUG("ui_mainwindow_show_statistics...........");
    gtkhtml_stop();

    ui_mainwindow_set_search(FALSE);
    ui_htmlview_clear(ui_mainwindow_get_active_htmlview());
    ui_itemlist_clear();
    ui_mainwindow_finish();

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    gtk_window_set_title(GTK_WINDOW(app_data->app_ui_data->main_view),
                         _("rss_ti_newsreader"));

}

void ui_mainwindow_save_search_entries()
{
    gchar *filename = NULL;
    gchar *entry_text = NULL;
    FILE *file = NULL;
    GtkComboBox *search_combo;
    int i = 0;
    int search_string_count = 0;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ULOG_DEBUG
        ("**********ui_mainwindow_save_search_entries: saving search entries ");

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "searchentries", g_get_home_dir());

    if (g_file_test(filename, G_FILE_TEST_EXISTS))
        g_unlink(filename);

    file = fopen(filename, "wb");

    if (file == NULL) {
        ULOG_DEBUG("Could not save the search entries");
        g_free(filename);
        return;
    }

    search_combo = GTK_COMBO_BOX(app_data->app_ui_data->search_field);

    if (search_combo == NULL) {
        g_free(filename);
        fclose(file);
        return;
    }

    search_string_count = app_data->app_ui_data->search_string_count;

    cache_write_int(file, search_string_count);

    for (i = 0; i < search_string_count; i++) {
        gtk_combo_box_set_active(search_combo, i);
        entry_text = gtk_combo_box_get_active_text(search_combo);
        if (cache_write_str(file, entry_text))
        {
            ULOG_DEBUG("Wrote: %s", entry_text);
        }
        g_free(entry_text);
    }
    fclose(file);
    g_free(filename);
}

void ui_mainwindow_load_search_entries()
{
    gchar *filename = NULL;
    FILE *file = NULL;
    GtkComboBox *search_combo;
    int i = 0;
    int search_string_count = 0;
    gchar *tmp;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ULOG_DEBUG("ui_mainwindow_load_search_entries");

    filename =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
                        G_DIR_SEPARATOR_S "searchentries", g_get_home_dir());

    if (!g_file_test(filename, G_FILE_TEST_EXISTS)) {
        ULOG_DEBUG("Search entry file not found");
        g_free(filename);
        return;
    }

    file = fopen(filename, "rb");

    if (file == NULL) {
        ULOG_DEBUG("Could not load the search entries");
        g_free(filename);
        return;
    }

    search_combo = GTK_COMBO_BOX(app_data->app_ui_data->search_field);

    if (search_combo == NULL) {
        g_free(filename);
        fclose(file);
        return;
    }

    search_string_count = cache_read_int(file);
    app_data->app_ui_data->search_string_count = search_string_count;

    ULOG_DEBUG("ui_mainwindow_load_search_entries: search_string_count = %d",
               search_string_count);
    for (i = 0; i < search_string_count; i++) {
        tmp = cache_read_str(file);
        gtk_combo_box_append_text(search_combo, tmp);
        g_free(tmp);
    }
    fclose(file);
    g_free(filename);
}

void ui_mainwindow_empty_search_field()
{
    GtkWidget *search_entry;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    search_entry =
        gtk_bin_get_child(GTK_BIN(app_data->app_ui_data->search_field));

    g_assert(NULL != search_entry);

    gtk_entry_set_text(GTK_ENTRY(search_entry), "");
}

void ui_mainwindow_set_feed_deatils_sensitive(AppData * app_data,
                                              gboolean sensitive)
{
    AppUIData *app_ui_data = NULL;
    g_assert(app_data);
    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_properties),
                             sensitive);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->feed_properties_button),
                             sensitive);
}

void ui_mainwindow_set_unsubscribe_sensitive(AppData * app_data,
                                             gboolean sensitive)
{
    AppUIData *app_ui_data = NULL;
    DMSG("Entering %s with %d\n", __FUNCTION__, sensitive);
    g_assert(app_data);
    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_unsubscribe),
                             sensitive);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button),
                             sensitive);
}
