/**
    @file download.c
	
    Download engine of rss reader.

    Copyright (c) 2004-2006 Nokia Corporation. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

#include <string.h>

#include "download.h"

#include "../debug_new.h"

#define BYTES_TO_PROCESS 40960
#define MAXIMUM_FILESIZE 102400

static download_data *current_ddata = NULL;

static void download_call_end(download_data * ddata);


static void download_free_ddata(download_data ** data_to_free)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (!*data_to_free)
        return;
    if ((*data_to_free)->data) {
        g_free((*data_to_free)->data);
        (*data_to_free)->data = NULL;
    }
    if ((*data_to_free)->buffer) {
        g_free((*data_to_free)->buffer);
        (*data_to_free)->buffer = NULL;
    }
    g_free(*data_to_free);
    *data_to_free = NULL;
    DMSG("Leaving %s\n", __FUNCTION__);
}

static void download_free_current_ddata(void)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (!current_ddata) {
        DMSG("Current ddata is NULL in %s\n", __FUNCTION__);
        return;
    } else {
        download_free_ddata(&current_ddata);
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

static void download_close_cb(GnomeVFSAsyncHandle * handle,
                              GnomeVFSResult result, gpointer callback_data)
{
    DMSG("Entering %s %d\n", __FUNCTION__, gnome_vfs_is_primary_thread());
    update_req_ended();
}

static void download_read_cb(GnomeVFSAsyncHandle * handle,
                             GnomeVFSResult result, gpointer buffer,
                             GnomeVFSFileSize bytes_requested,
                             GnomeVFSFileSize bytes_read,
                             gpointer callback_data)
{
    download_data *ddata = NULL;
    ddata = (download_data *) callback_data;
    if (!current_ddata) {
        DMSG("Current ddata is NULL in %s\n", __FUNCTION__);
        return;
    }
    DMSG("Entering %s\n", __FUNCTION__);
    if (result == GNOME_VFS_OK) {
        if (((ddata->data_pos + bytes_read) - ddata->data) > ddata->fsize) {    //somewhere the concatenation was wrong, or the server responded a wrong size???
            guint mempos = 0;
            guint new_memsize = 0;
            mempos = ddata->data_pos - ddata->data;
            new_memsize = (ddata->data_pos + bytes_read) - ddata->data;
            ddata->data = g_realloc(ddata->data, new_memsize);
            ddata->fsize = new_memsize;
            ddata->data_pos = ddata->data + mempos;
        }
        memcpy(ddata->data_pos, buffer, bytes_read);
        ddata->data_pos = ddata->data_pos + bytes_read;
        gnome_vfs_async_read(ddata->vfs_handle, ddata->buffer,
                             BYTES_TO_PROCESS, download_read_cb, ddata);
    } else {
        struct request *nrequest = NULL;

        nrequest = ddata->req_data;
        if (result == GNOME_VFS_ERROR_EOF) {
            if (nrequest) {
                if (ddata->data[ddata->fsize - 1] != 0) {
                    ddata->data =
                        g_realloc(ddata->data, ++(ddata->fsize));
                    ddata->data[ddata->fsize - 1] = 0;
                }
                nrequest->data = g_memdup(ddata->data, ddata->fsize);
                nrequest->httpstatus = 200;
                nrequest->size = ddata->fsize;
                nrequest->lastmodified = ddata->last_modified;
            }    
        } else {
            if (nrequest) {
                nrequest->httpstatus = 404;
            }
            g_free(ddata->data);
            ddata->data = NULL;
        }
        download_call_end(ddata);
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

static void download_open_cb(GnomeVFSAsyncHandle * handle,
                             GnomeVFSResult result, gpointer callback_data)
{
    download_data *ddata = NULL;
    ddata = (download_data *) callback_data;

    if (!current_ddata) {
        DMSG("Current ddata is NULL in %s\n", __FUNCTION__);
        return;
    }
    DMSG("Entering %s with %d %d\n", __FUNCTION__, result,
         gnome_vfs_is_primary_thread());

    if (result == GNOME_VFS_OK) {
        ddata->buffer = g_new0(gchar, BYTES_TO_PROCESS);
        gnome_vfs_async_read(ddata->vfs_handle, ddata->buffer,
                             BYTES_TO_PROCESS, download_read_cb, ddata);
    } else {                    //error during open
        download_call_end(ddata);
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

static void download_call_end(download_data * ddata)
{
    struct request *nrequest = NULL;
    DMSG("Entering %s\n", __FUNCTION__);
    if (!current_ddata) {
        DMSG("Current ddata is NULL in %s\n", __FUNCTION__);
        return;
    }
    nrequest = ddata->req_data;
    if (ddata->vfs_handle != NULL) {
        DMSG("Calling async close\n");
        gnome_vfs_async_close(ddata->vfs_handle, download_close_cb,
                              current_ddata);
    }
    download_free_current_ddata();
    if (nrequest) {
        // this is also redundant
        current_ddata = NULL;
        update_download_finished(nrequest);
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

//this will check only the first list element
static void download_info_cb(GnomeVFSAsyncHandle * handle, GList * results,
                             gpointer callback_data)
{
    download_data *ddata = NULL;
    DMSG("Entering %s %d\n", __FUNCTION__, gnome_vfs_is_primary_thread());
    ddata = (download_data *) callback_data;
    GnomeVFSGetFileInfoResult *finfo_res = NULL;
    GnomeVFSFileInfo *finfo = NULL;

    if (!current_ddata) {
        DMSG("Current ddata is NULL in %s\n", __FUNCTION__);
        return;
    }

    finfo_res = results->data;

    if (finfo_res->result == GNOME_VFS_OK) {
        finfo = finfo_res->file_info;
        if ((!finfo->mtime) || (finfo->mtime > ddata->last_modified)) { //file was changed... need to be dowloaded
            gint size_alloc = 0;
            if (MAXIMUM_FILESIZE < finfo->size) {
                size_alloc = MAXIMUM_FILESIZE;
            } else if (finfo->size == 0) {
                size_alloc = MAXIMUM_FILESIZE;
            } else {
                size_alloc = finfo->size;
            }
            ddata->data = g_new0(gchar, size_alloc);
            ddata->data_pos = ddata->data;
            ddata->fsize = size_alloc;
            ddata->last_modified = finfo->mtime;
            gnome_vfs_async_open_uri(&ddata->vfs_handle, finfo_res->uri,
                                     GNOME_VFS_OPEN_READ,
                                     GNOME_VFS_PRIORITY_DEFAULT,
                                     download_open_cb, ddata);
        } else {                //File was not changed
            g_message("Content not changed!!!! %s",ddata->req_data->source);
            ddata->vfs_handle = NULL;
            struct request *nrequest = NULL;
            nrequest = ddata->req_data;
            if (nrequest) {
                nrequest->httpstatus = 304;
            }
            download_call_end(ddata);
            update_req_ended();
        }
    } else {                    //error during information request
        struct request *nrequest = ddata->req_data;
        if (nrequest) {
            nrequest->httpstatus = 404;
        }
        /* we don't have to close this handle, because it is not opened */
        ddata->vfs_handle = NULL;
        download_call_end(ddata);
        update_req_ended();
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

void download_inform_for_free(struct request *request)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (current_ddata) {
        if ((current_ddata->req_data == request)) {
            DMSG("Canceling current request\n");
            download_cancel();
            update_req_ended();
        }
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}

//returns TRUE on BUSY
gboolean download_start_request(const gchar * url, time_t last_modified,
                                struct request *nrequest)
{
    download_data *ddata = NULL;
    GList *urilist = NULL;
    GnomeVFSURI *uri = NULL;
    DMSG("Entering %s with: %s\n", __FUNCTION__, url);
    if (current_ddata) {        //ONLY ONE REQUEST AT ONE TIME
        DMSG("Leaving %s 1\n", __FUNCTION__);
        return TRUE;
    }

    if (!nrequest) {
        return FALSE;
    }
    uri = gnome_vfs_uri_new(url);
    if (!uri) {
        g_message("URI is NULL!!!");
        ddata = g_new0(download_data, 1);
        ddata->req_data = nrequest;
        nrequest->httpstatus = 401;
        current_ddata = ddata;
        download_call_end(ddata);
        DMSG("Leaving %s 2\n", __FUNCTION__);
        return FALSE;
    }
    urilist = g_list_append(urilist, uri);
    ddata = g_new0(download_data, 1);
    ddata->last_modified = last_modified;
    ddata->req_data = nrequest;
    current_ddata = ddata;
    gnome_vfs_async_get_file_info(&ddata->vfs_handle, urilist,
                                  GNOME_VFS_FILE_INFO_DEFAULT,
                                  GNOME_VFS_PRIORITY_DEFAULT,
                                  download_info_cb, ddata);
    gnome_vfs_uri_list_free(urilist);
    DMSG("Leaving %s\n", __FUNCTION__);
    return FALSE;
}


void download_cancel(void)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (current_ddata) {
        DMSG("Entering %s 1 %s\n", __FUNCTION__,current_ddata->req_data->source);
        if (current_ddata->vfs_handle) {
            gnome_vfs_async_cancel(current_ddata->vfs_handle);
        }
        download_free_current_ddata();
    }
    DMSG("Leaving %s\n", __FUNCTION__);
}


gboolean download_begin(struct request *nrequest)
{
    DMSG("Entering %s\n", __FUNCTION__);
    if (nrequest) {
        DMSG("Leaving %s\n", __FUNCTION__);
        return download_start_request(nrequest->source,
                                      nrequest->lastmodified, nrequest);
    } else {                    //pointer is null??? this should not happen
        g_warning("Request pointer is NULL!!!!");
        return FALSE;
    }
}

gboolean download_is_in_progress(void)
{
  return current_ddata!=NULL;
}

struct request *download_get_current_request(void)
{
    if (current_ddata)
    {
        return current_ddata->req_data;
    }
    return NULL;
}
