/**
 * @file export.c OPML feedlist import&export
 *
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libxml/tree.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include "vfolder.h"
#include "conf.h"
#include "callbacks.h"
#include "osso-log.h"
#include "debug_new.h"
#include <hildon/hildon-note.h>
#include <osso-rss-feed-reader/cache_handling.h>

struct exportData {
    gboolean internal;     /**< Include all the extra Liferea-specific tags */
    xmlNodePtr cur;
};

extern AppData *app_data;
gboolean just_empty_items = FALSE;


static gboolean
feedTaggedForDelete(gint type)
{
    /* basicly this means that type can be any of the values:
     *      FST_INVALID_DELETED 
     *      FST_FOLDER_DELETED 
     *      FST_VFOLDER_DELETED
     *      FST_FEED_DELETED 
     *     
     *     since they all start from 32 
     */

    return (type >= 32);
}


/** Used for exporting, this adds a folder or feed's node to the XML tree
  * 
  * @param ptr node to append
  * @param userdata xml node
  * */
static void append_node_tag(nodePtr ptr, gpointer userdata)
{
    xmlNodePtr cur = ((struct exportData *) userdata)->cur;
    gboolean internal = ((struct exportData *) userdata)->internal;
    xmlNodePtr childNode = NULL, ruleNode = NULL;
    GSList *iter = NULL;
    rulePtr rule = NULL;

    if (FST_FOLDER == ptr->type) {
        folderPtr folder = (folderPtr) ptr;
        struct exportData data;
        /* We don't generate any XML code for the root node but we process
         * its children. */
        childNode = xmlNewChild(cur, NULL, BAD_CAST "outline", NULL);
        xmlNewProp(childNode, BAD_CAST "title",
                   BAD_CAST folder_get_title(folder));

        if (folder_get_nonremovable(folder))
            xmlNewProp(childNode, BAD_CAST "nonremovable", BAD_CAST "true");

        if (internal) {
            /* ID is stored for folders, too, if saving feedlist.opml. */
            xmlNewProp(childNode, BAD_CAST "id", BAD_CAST folder->id);

            if (folder->expanded)
                xmlNewProp(childNode, BAD_CAST "expanded", NULL);
            else
                xmlNewProp(childNode, BAD_CAST "collapsed", NULL);
        }
        debug1(DEBUG_CONF, "adding folder %s...", folder_get_title(folder));
        data.cur = childNode;
        data.internal = internal;
        ui_feedlist_do_for_all_data(ptr,
                                    ACTION_FILTER_CHILDREN,
                                    append_node_tag, (gpointer) & data);
    } else {
        feedPtr fp = (feedPtr) ptr;
        if (feedTaggedForDelete(fp->type) == FALSE) {
            const gchar *type = feed_type_fhp_to_str(feed_get_fhp(fp));
            gchar *interval =
                g_strdup_printf("%d", feed_get_update_interval(fp));
            gchar *cacheLimit = NULL;

            childNode = xmlNewChild(cur, NULL, BAD_CAST "outline", NULL);

            /* The OPML spec requires "text" */
            xmlNewProp(childNode, BAD_CAST "text",
                       BAD_CAST feed_get_title(fp));
            xmlNewProp(childNode, BAD_CAST "title",
                       BAD_CAST feed_get_title(fp));
            xmlNewProp(childNode, BAD_CAST "description",
                       BAD_CAST feed_get_title(fp));
            xmlNewProp(childNode, BAD_CAST "type", BAD_CAST type);
            if (feed_get_html_url(fp) != NULL) {
                xmlNewProp(childNode, BAD_CAST "htmlUrl",
                           BAD_CAST feed_get_html_url(fp));
            } else {
                xmlNewProp(childNode, BAD_CAST "htmlUrl", BAD_CAST "");
            }
            xmlNewProp(childNode, BAD_CAST "xmlUrl",
                       BAD_CAST feed_get_source(fp));
            xmlNewProp(childNode, BAD_CAST "updateInterval",
                       BAD_CAST interval);

            if (fp->cacheLimit >= 0) {
                cacheLimit = g_strdup_printf("%d", fp->cacheLimit);
            }
            if (fp->cacheLimit == CACHE_UNLIMITED) {
                cacheLimit = g_strdup("unlimited");
            }
            if (cacheLimit != NULL) {
                xmlNewProp(childNode, BAD_CAST "cacheLimit",
                           BAD_CAST cacheLimit);
            }

            if (feed_get_filter(fp) != NULL) {
                xmlNewProp(childNode, BAD_CAST "filtercmd",
                           BAD_CAST feed_get_filter(fp));
            }

            if (internal) {
                if (fp->noIncremental)
                    xmlNewProp(childNode, BAD_CAST "noIncremental",
                               BAD_CAST "true");

                xmlNewProp(childNode, BAD_CAST "id",
                           BAD_CAST feed_get_id(fp));
                if (!just_empty_items && fp->lastPoll.tv_sec > 0) {
                    gchar *lastPoll =
                        g_strdup_printf("%ld", fp->lastPoll.tv_sec);
                    xmlNewProp(childNode, BAD_CAST "lastPollTime",
                               BAD_CAST lastPoll);
                    g_free(lastPoll);
                }
                if (fp->lastFaviconPoll.tv_sec > 0) {
                    gchar *lastPoll =
                        g_strdup_printf("%ld", fp->lastFaviconPoll.tv_sec);
                    xmlNewProp(childNode, BAD_CAST "lastFaviconPollTime",
                               BAD_CAST lastPoll);
                    g_free(lastPoll);
                }
                if (fp->sortColumn == IS_TITLE)
                    xmlNewProp(childNode, BAD_CAST "sortColumn",
                               BAD_CAST "title");
                else if (fp->sortColumn == IS_TIME)
                    xmlNewProp(childNode, BAD_CAST "sortColumn",
                               BAD_CAST "time");
                if (fp->sortReversed)
                    xmlNewProp(childNode, BAD_CAST "sortReversed",
                               BAD_CAST "true");

                if (fp->nonremovable)
                    xmlNewProp(childNode, BAD_CAST "nonremovable",
                               BAD_CAST "true");

                if (!feed_get_download_images(fp))
                    xmlNewProp(childNode, BAD_CAST "hideImages",
                               BAD_CAST "true");
            }

            /* add vfolder rules */
            iter = fp->rules;
            while (NULL != iter) {
                rule = iter->data;
                ruleNode =
                    xmlNewChild(childNode, NULL, BAD_CAST "outline", NULL);
                xmlNewProp(ruleNode, BAD_CAST "type", BAD_CAST "rule");
                xmlNewProp(ruleNode, BAD_CAST "text",
                           BAD_CAST rule->ruleInfo->title);
                xmlNewProp(ruleNode, BAD_CAST "rule",
                           BAD_CAST rule->ruleInfo->ruleId);
                xmlNewProp(ruleNode, BAD_CAST "value", BAD_CAST rule->value);
                if (TRUE == rule->additive)
                    xmlNewProp(ruleNode, BAD_CAST "additive",
                               BAD_CAST "true");
                else
                    xmlNewProp(ruleNode, BAD_CAST "additive",
                               BAD_CAST "false");

                iter = g_slist_next(iter);
            }

            debug6(DEBUG_CONF,
                   "adding feed: title=%s type=%s source=%d id=%s interval=%s cacheLimit=%s",
                   feed_get_title(fp), type, feed_get_source(fp),
                   feed_get_id(fp), interval, cacheLimit);
            g_free(cacheLimit);
            g_free(interval);
        } else {
            g_warning("current feed has been tagged for deletion");
        }
    }
    // debug_exit("append_node_tag");
}

/** Parses an integer from a string
  *
  * @param str string to parse
  * @param def value to return in case of an error
  */
static int parse_integer(gchar * str, int def)
{
    int num = 0;
    if (str == NULL)
        return def;
    if (0 == (sscanf(str, "%d", &num)))
        num = def;

    return num;
}

/** Parses a long integer from a string
  *
  * @param str string to parse
  * @param def value to return in case of an error
  */
static long parse_long(gchar * str, long def)
{
    long num = 0;
    if (str == NULL)
        return def;
    if (0 == (sscanf(str, "%ld", &num)))
        num = def;

    return num;
}

/** Called by import_parse_outline to parse all children outline tags
 *  as vfolder rule descriptions 
 *
 * @param cur xml node
 * @param vp pointer to a feed
 */
static void import_parse_children_as_rules(xmlNodePtr cur, feedPtr vp)
{
    xmlChar *type = NULL, *ruleId = NULL, *value = NULL, *additive = NULL;
    /* rulePtr rp; */

    g_assert(cur != NULL);
    /* process any children */
    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
        if (!xmlStrcmp(cur->name, BAD_CAST "outline")) {
            type = xmlGetProp(cur, BAD_CAST "type");
            if (type != NULL && !xmlStrcmp(type, BAD_CAST "rule")) {

                ruleId = xmlGetProp(cur, BAD_CAST "rule");
                value = xmlGetProp(cur, BAD_CAST "value");
                additive = xmlGetProp(cur, BAD_CAST "additive");

                if ((NULL != ruleId) && (NULL != value)) {
                    debug3(DEBUG_CACHE,
                           "loading rule \"%s\" \"%s\" \"%s\"\n", type,
                           ruleId, value);

                    if (additive != NULL
                        && !xmlStrcmp(additive, BAD_CAST "true"))
                        vfolder_add_rule(vp, ruleId, value, TRUE);
                    else
                        vfolder_add_rule(vp, ruleId, value, FALSE);
                } else {
                    g_warning
                        ("ignoring invalid rule entry in feed list...\n");
                }

                xmlFree(ruleId);
                xmlFree(value);
                xmlFree(additive);
            }
            xmlFree(type);
        }
        cur = cur->next;
    }
}

/** Parses an outline tag from an xml document
  *
  * @param cur xml node
  * @param folder pointer to a folder structure
  * @param trusted set to FALSE for a safer mode of parsing */
static void import_parse_outline(xmlNodePtr cur, folderPtr folder,
                                 gboolean trusted, gint mode)
{
    gchar *cacheLimitStr = NULL, *filter = NULL, *intervalStr =
        NULL, *lastPollStr = NULL, *htmlUrlStr = NULL, *sortStr =
        NULL, *nonremovableStr = NULL;
    gchar *title = NULL, *source = NULL, *typeStr = NULL, *tmp = NULL;
    feedPtr fp = NULL;
    folderPtr child = NULL;
    gboolean dontParseChildren = FALSE;
    gint interval = 0;
    gchar *id = NULL;

    // ULOG_DEBUG("------------------import_parse_outline---------------------------");
    /* process the outline node */
    title = xmlGetProp(cur, BAD_CAST "title");
    if (title == NULL || !xmlStrcmp(title, BAD_CAST "")) {
        if (title != NULL)
            xmlFree(title);
        title = xmlGetProp(cur, BAD_CAST "text");
    }

    if (NULL == (source = xmlGetProp(cur, BAD_CAST "xmlUrl")))
        source = xmlGetProp(cur, BAD_CAST "xmlurl");    /* e.g. for
                                                         * AmphetaDesk */

    if (NULL != source) {       /* Reading a feed */
        filter = xmlGetProp(cur, BAD_CAST "filtercmd");

        if (!trusted && filter != NULL) {

            tmp = g_strdup_printf("unsafe command: %s", filter);
            xmlFree(filter);
            filter = tmp;
        }

        if (!trusted && source[0] == '|') {

            tmp = g_strdup_printf("unsafe command: %s", source);
            xmlFree(source);
            source = tmp;
        }

        intervalStr = xmlGetProp(cur, BAD_CAST "updateInterval");
        interval = parse_integer(intervalStr, -1);
        xmlFree(intervalStr);

        /* The id should only be used from feedlist.opml. Otherwise, it could 
         * cause corruption if the same id was imported multiple times. */
        if (trusted)
            id = xmlGetProp(cur, BAD_CAST "id");

        /* get type attribute and use it to assign a value to
         * fhp. fhp will default to NULL. */
        typeStr = xmlGetProp(cur, BAD_CAST "type");
        if ((NULL != typeStr) && (0 == strcmp("vfolder", typeStr))) {
            if (mode == RSS_CONFIG_MODE_FULL) {
                dontParseChildren = TRUE;
                fp = vfolder_new();
                import_parse_children_as_rules(cur, fp);
            }
        } else {
            if (mode == RSS_CONFIG_MODE_FULL) {
                fp = feed_new();
            } else if (mode == RSS_CONFIG_MODE_STATE) {
                gchar *sid = xmlGetProp(cur, BAD_CAST "id");
                if (id && sid) {
                    fp = (feedPtr) ui_feedlist_find_feed(NULL, id);
                    if (fp == NULL) {
                        xmlFree(sid);
                        goto contin;
                    }
                    xmlFree(sid);
                } else {
                    goto contin;
                }
            }
        }
        if (!fp) {
            goto contin;
        }

        fp->fhp = feed_type_str_to_fhp(typeStr);


        /* Set the cache limit */
        cacheLimitStr = xmlGetProp(cur, BAD_CAST "cacheLimit");
        if (cacheLimitStr != NULL && !xmlStrcmp(cacheLimitStr, "unlimited")) {
            fp->cacheLimit = CACHE_UNLIMITED;
        } else
            fp->cacheLimit = parse_integer(cacheLimitStr, CACHE_DEFAULT);
        xmlFree(cacheLimitStr);

        /* Obtain the htmlUrl */
        htmlUrlStr = xmlGetProp(cur, BAD_CAST "htmlUrl");
        if (htmlUrlStr != NULL && 0 != xmlStrcmp(htmlUrlStr, ""))
            feed_set_html_url(fp, htmlUrlStr);
        xmlFree(htmlUrlStr);

        tmp = xmlGetProp(cur, BAD_CAST "noIncremental");
        if (NULL != tmp && !xmlStrcmp(tmp, BAD_CAST "true"))
            fp->noIncremental = TRUE;
        xmlFree(tmp);

        /* Last poll time */
        lastPollStr = xmlGetProp(cur, BAD_CAST "lastPollTime");
        fp->lastPoll.tv_sec = parse_long(lastPollStr, 0L);
        fp->lastPoll.tv_usec = 0L;
        if (lastPollStr != NULL)
            xmlFree(lastPollStr);

        lastPollStr = xmlGetProp(cur, BAD_CAST "lastFaviconPollTime");
        fp->lastFaviconPoll.tv_sec = parse_long(lastPollStr, 0L);
        fp->lastFaviconPoll.tv_usec = 0L;
        if (lastPollStr != NULL)
            xmlFree(lastPollStr);

        /* nonremovable status */

        nonremovableStr = xmlGetProp(cur, BAD_CAST "nonremovable");
        if (nonremovableStr != NULL
            && !xmlStrcmp(nonremovableStr, BAD_CAST "true"))
            fp->nonremovable = TRUE;
        else
            fp->nonremovable = FALSE;

        if (nonremovableStr != NULL)
            xmlFree(nonremovableStr);

        /* show hide images */
        tmp = xmlGetProp(cur, BAD_CAST "hideImages");
        feed_set_download_images(fp, !tmp || xmlStrcmp(tmp, BAD_CAST "true"));
        if (tmp != NULL)
            xmlFree(tmp);

        /* sorting order */
        sortStr = xmlGetProp(cur, BAD_CAST "sortColumn");
        if (sortStr != NULL) {
            if (!xmlStrcmp(sortStr, "title"))
                fp->sortColumn = IS_TITLE;
            else if (!xmlStrcmp(sortStr, "time"))
                fp->sortColumn = IS_TIME;
            xmlFree(sortStr);
        }
        sortStr = xmlGetProp(cur, BAD_CAST "sortReversed");
        if (sortStr != NULL && !xmlStrcmp(sortStr, BAD_CAST "true"))
            fp->sortReversed = TRUE;
        if (sortStr != NULL)
            xmlFree(sortStr);

        /* set feed properties available from the OPML feed list they may be 
         * overwritten by the values of the cache file but we need them in
         * case the cache file loading fails */

        feed_set_source(fp, source);
        feed_set_filter(fp, filter);
        feed_set_title(fp, title);
        feed_set_added(fp, time(NULL));
        feed_set_update_interval(fp, interval);
        debug6(DEBUG_CONF,
               "loading feed: title=%s source=%s typeStr=%s id=%s interval=%d lastpoll=%ld",
               title, source, typeStr, id, interval, fp->lastPoll.tv_sec);

        if (mode == RSS_CONFIG_MODE_FULL) {
            if (id != NULL) {
                feed_set_id(fp, id);
                xmlFree(id);
                /* don't load here, because it's not sure that all vfolders are loaded */
            } else {
                id = conf_new_id();
                feed_set_id(fp, id);
                debug1(DEBUG_CONF,
                       "seems to be an import, setting new id: %s and doing first download...",
                       id);
                g_free(id);
            }
            id = NULL;
            ui_feedlist_add(folder, (nodePtr) fp, -1);
        }
        if (source != NULL)
            xmlFree(source);
        if (filter != NULL)
            xmlFree(filter);
        xmlFree(typeStr);

    } else {                    /* It is a folder */
        if (mode == RSS_CONFIG_MODE_FULL) {
            /* ID is loaded for folders, too, if reading
             * from feedlist.opml.
             */
            if (trusted) {
                id = xmlGetProp(cur, BAD_CAST "id");
            }
            debug2(DEBUG_CONF, "adding folder: title=%s id=%s", title, id);
            child = restore_folder(folder, title, id, FST_FOLDER);
            g_assert(NULL != child);
            if (id != NULL) {
                xmlFree(id);
                id = NULL;
            }
            ui_feedlist_add(folder, (nodePtr) child, -1);
            folder = child;

            nonremovableStr = xmlGetProp(cur, BAD_CAST "nonremovable");
            if (nonremovableStr != NULL
                && !xmlStrcmp(nonremovableStr, BAD_CAST "true"))
                folder_set_nonremovable(folder, TRUE);
            else
                folder_set_nonremovable(folder, FALSE);
            xmlFree(nonremovableStr);
        }
        else if (mode == RSS_CONFIG_MODE_STATE)
        {
            /* Folder expansion states are nomally only stored in the state
             * file. When loading statefile, no need to add folder to tree
             * again, just updateing the expansion state */
            if (trusted)
            {
                id = xmlGetProp(cur, BAD_CAST "id");
            }

            child = ui_folder_lookup_by_id(NULL, id);
            if ( NULL != child )
            {
                child->expanded = (NULL != xmlHasProp(cur, BAD_CAST "expanded"));
            }

            if (id != NULL)
            {
                xmlFree(id);
                id = NULL;
            }
        }
    }
  contin:
    if (id != NULL)
        xmlFree(id);
    if (title != NULL)
        xmlFree(title);

    if (!dontParseChildren) {
        /* process any children */
        cur = cur->xmlChildrenNode;
        while (cur != NULL) {
            if ((!xmlStrcmp(cur->name, BAD_CAST "outline")))
                import_parse_outline(cur, folder, trusted, mode);

            cur = cur->next;
        }
    }

}

/** Parses the body of an xml document
  *
  * @param xml node
  * @param parent pointer to a parent folder
  * @param trusted set to FALSE for a safer mode of parsing */
static void import_parse_body(xmlNodePtr n, folderPtr parent,
                              gboolean trusted, gint mode)
{
    xmlNodePtr cur = NULL;

    g_assert(n != NULL);
    ULOG_DEBUG("import_parse_body");
    cur = n->xmlChildrenNode;
    while (cur != NULL) {
        if ((!xmlStrcmp(cur->name, BAD_CAST "outline")))
            import_parse_outline(cur, parent, trusted, mode);
        cur = cur->next;
    }
}

/** Parses an OPML document
  *
  * @param xml node
  * @param parent pointer to a parent folder
  * @param trusted set to FALSE for a safer mode of parsing */
static void import_parse_OPML(xmlNodePtr n, folderPtr parent,
                              gboolean trusted, gint mode)
{
    xmlNodePtr cur = NULL;

    g_assert(n != NULL);

    cur = n->xmlChildrenNode;
    while (cur != NULL) {
        /* we ignore the head */
        if ((!xmlStrcmp(cur->name, BAD_CAST "body"))) {
            import_parse_body(cur, parent, trusted, mode);
        }
        cur = cur->next;
    }
}

/** Used to process feeds directly after feed list loading.
 *  Loads the given feed or requests a download. During feed
 *  loading its items are automatically checked against all 
 *  vfolder rules.
 *
 * @param fp feed to load
 */
static void import_initial_load_feed(feedPtr fp)
{

    if (!feed_load(fp))
        ULOG_INFO("Failed to load feed cache for %s", fp->title);
    /* feed_unload(fp); */
}

static void filter_expand_folders(feedPtr fp)
{
    folderPtr folder = (folderPtr) fp;

    if (folder->expanded)
        ui_folder_set_expansion(folder, TRUE);
}

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

void import_OPML_feedlist(const gchar * filename, folderPtr parent,
                          gboolean showErrors, gboolean trusted, gint mode)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL;

    ULOG_DEBUG("Importing OPML file: %s", filename);

    debug1(DEBUG_CONF, "Importing OPML file: %s", filename);

    /* read the feed list */
    if (NULL == (doc = xmlParseFile(filename))) {
        if (showErrors)
            ui_show_error_box
                ("XML error occurred while reading cache file. \"%s\" cannot be imported.",
                 filename);
        else
            g_warning
                ("XML error occurred while reading cache file. \"%s\" cannot be imported.",
                 filename);
    } else {
        if (NULL == (cur = xmlDocGetRootElement(doc))) {
            if (showErrors)
                ui_show_error_box
                    ("Empty document. OPML document \"%s\" should not be empty when importing.",
                     filename);
            else
                g_warning
                    ("Empty document. OPML document \"%s\" should not be empty when importing.",
                     filename);
        } else {
            while (cur != NULL) {
                if (!xmlIsBlankNode(cur)) {
                    if (!xmlStrcmp(cur->name, BAD_CAST "opml")) {
                        ULOG_DEBUG("import_parse_OPML");
                        import_parse_OPML(cur, parent, trusted, mode);
                    } else {
                        if (showErrors)
                            ui_show_error_box
                                ("\"%s\" is not a valid OPML document. Liferea cannot import this file.",
                                 filename);
                        else
                            g_warning
                                ("\"%s\" is not a valid OPML document. Liferea cannot import this file.",
                                 filename);
                    }
                }
                cur = cur->next;
            }
        }
        xmlFreeDoc(doc);
    }

    /* load all feeds and by doing so automatically load all vfolders */
    ULOG_DEBUG("import_OPML_feedlist: initial load of all feeds");
    ui_folder_check_if_empty();
    //    ui_feedlist_update();

    ui_feedlist_do_for_all(NULL, ACTION_FILTER_FOLDER, filter_expand_folders);
    ui_feedlist_do_for_all(NULL, ACTION_FILTER_FEED,
                           import_initial_load_feed);
    ui_feedlist_update();
}

int export_OPML_feedlist(const gchar * filename, gboolean internal)
{
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL, opmlNode = NULL;
    gint error = 0;

    if (NULL != (doc = xmlNewDoc("1.0"))) {
        if (NULL !=
            (opmlNode = xmlNewDocNode(doc, NULL, BAD_CAST "opml", NULL))) {
            xmlNewProp(opmlNode, BAD_CAST "version", BAD_CAST "1.0");
            /* create head */
            if (NULL !=
                (cur = xmlNewChild(opmlNode, NULL, BAD_CAST "head", NULL))) {
                xmlNewTextChild(cur, NULL, BAD_CAST "title",
                                BAD_CAST "Liferea Feed List Export");
            }

            /* create body with feed list */
            if (NULL != (cur = xmlNewChild(opmlNode, NULL,
                                           BAD_CAST "body", NULL))) {
                struct exportData data;
                data.internal = internal;
                data.cur = cur;
                ui_feedlist_do_for_all_data((nodePtr)
                                            ui_feedlist_get_root_folder(),
                                            ACTION_FILTER_CHILDREN,
                                            append_node_tag,
                                            (gpointer) & data);
            }
            xmlDocSetRootElement(doc, opmlNode);
        } else {
            g_warning
                ("could not create XML feed node for feed cache document!");
            error = 1;
        }
        if (-1 == xmlSaveFormatFileEnc(filename, doc, NULL, 1)) {
            g_warning("Could not export to OPML file!!");
            g_warning("no space left on device");
            app_data->app_ui_data->nospace = TRUE;
            error = 1;
        }
        if (0 != fflush(NULL)) {
            g_error("Could not flush files");
        }
        sync();
        xmlFreeDoc(doc);
    } else {
        g_warning("could not create XML document!");
        error = 1;
    }
    // debug_exit("export_OPML_feedlist");
    return error;
}
