/**
 * @file common.h common routines
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _COMMON_H
#define _COMMON_H

#include <hildon/hildon-defines.h>

#include <config.h>
#include <time.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <glib.h>
#define htmlToCDATA(buffer) g_strdup_printf("<![CDATA[%s]]>", buffer)

#define	TIMESTRLEN	30

#define EMPTEXTCOLOR "EmpTextColor"
#define DEFAULTTEXTCOLOR "DefaultTextColor"

#define MAX_PARSE_ERROR_LINES	10

#define TZ_SIZE 64              // maximum size of a timezone string
#define LOCALTIME_FILE	"/etc/localtime"
#define ZONEINFO_DIR	"/usr/share/zoneinfo/"

extern gboolean lifereaStarted;

struct folder;

typedef struct node {
    gint type;
    gpointer *ui_data;
} *nodePtr;

typedef struct {
    gchar *data;
    gint length;
} result_buffer;

/** used by bufferParseError to keep track of error messages */
typedef struct errorCtxt {
    gchar *buffer;
    gint errorCount;
} *errorCtxtPtr;

/** Conversion function which should be applied to all read XML strings, 
  * to ensure proper UTF8. doc points to the xml document and its encoding and
  * string is a xmlChar pointer to the read string. The result gchar
  * string is returned, the original XML string is freed. 
  *
  * @param string an xml string
  * @return a valid utf8 string */
gchar *utf8_fix(xmlChar * string);

/** converts a UTF-8 string to HTML (resolves XML entities) 
  *
  * @param string an UTF-8 string
  * @return a html string */
gchar *convertToHTML(gchar * string);

/** converts a UTF-8 string containing HTML tags to plain text 
  *
  * @param an UTF-8 string
  * @return plain text
  */
gchar *unhtmlize(gchar * string);

gchar *unxmlize(gchar * string);


gchar *remove_newlines(gchar * string);
gchar *trim_whitespaces(gchar * string);
gchar *remove_newlines_and_extra_spaces(gchar * string);

/** to extract not escaped XHTML from a node
 * @param only extract the children of the passed node */
gchar *extractHTMLNode(xmlNodePtr cur, gboolean children);

void addToHTMLBufferFast(gchar ** buffer, const gchar * string);
void addToHTMLBuffer(gchar ** buffer, const gchar * string);

/**
 * Common function to create a XML DOM object from a given
 * XML buffer. This function sets up a parser context,
 * enables recovery mode and sets up the error handler.
 * 
 * The function returns a XML document pointer or NULL
 * if the document could not be read. It also sets 
 * errormsg to the last error messages on parsing
 * errors. 
 *
 * @param data		XML source
 * @param dataLength the length of the data string in bytes
 * @param errormsg	error buffer
 *
 * @return XML document
 */
xmlDocPtr parseBuffer(gchar * data, size_t dataLength, gchar ** errormsg);

/** Registers custom encoding handlers
  */
void register_encoding_handlers();

time_t parseISO8601Date(gchar * date);

/**
 * Parses a RFC822 format date. This FAILS if a timezone string is
 * specified such as EDT or EST and that timezone is in daylight
 * savings time.
 *
 * @returns UNIX time (GMT, no daylight savings time)
 */
time_t parseRFC822Date(gchar * date);
gchar *createRFC822Date(const time_t * time);

/* FIXME: formatDate used by several functions not only
 * to format date column, don't use always date column format!!!
 * maybe gchar * formatDate(time_t, gchar *format) */
gchar *formatDate(time_t t);



/**
 * Encodes all non URI conformant characters in the passed
 * string to be included in a HTTP URI. The original string
 * is freed.
 *
 * @param string	string to be URI-escaped
 * @returns new string that can be inserted into a HTTP URI
 */
gchar *encode_uri_string(gchar * string);

/**
 * Encodes all non URI conformant characters in the passed
 * string and returns a valid HTTP URI. The original string
 * is freed.
 *
 * @param uri_string	string to be URI-escaped
 * @returns valid HTTP URI
 */
gchar *encode_uri(gchar * uri_string);

/** 
 * To correctly escape and expand URLs, does not touch the
 * passed strings.
 *
 * @param url		relative URL
 * @param baseURL	base URL
 * @returns resulting absolute URL
 */
xmlChar *common_build_url(const gchar * url, const gchar * baseURL);

/**
 * Filter the title. Note that the string is modified and not
 * duplicated!
 *
 * @param title		title string be be filtered
 * @returns the result string, which is nearly always the same as the value passed to it.
 */
gchar *filter_title(gchar * title);

#ifndef HAVE_STRSEP
char *strsep(char **stringp, const char *delim);
#endif

gchar *strreplace(const char *string, const char *delimiter,
                  const char *replacement);

/**
 * Get a Pango/HTML markup color-string for Hildon logical color name.
 * The returned string is stored in a static buffer, so g_strdup it if
 * you need it later.
 *
 * @param logical Hildon logical color name, for example EmpTextColor
 * @returns a # prefixed hexadecimal representation of the color for pango
 */
gchar *get_logical_color(gchar * logical);

/** Gets localized time
  *
  * @param time the time for which the time string is produced
  */
gchar *get_localized_time(time_t time);

/** Gets localized date
  *
  * @param time the time for which the date string is produced
  * @param years set to TRUE to display years
  */
gchar *get_localized_date(time_t time, gboolean years);

/** Paragraphizes the text in a nice way so that only one
  * newline character is allowed in succession. Also extra
  * whitespaces are removed from the beginning of each paragraph
  *
  * @param text the text to process
  * @return the text with the tags removed
  */
gchar *paragraphize(gchar * text);

/** Prepares content downloaded from the internet in xml format to
  * a human readable chunk of text which is displayed on the screen
  *
  * @param text raw text
  * @return the prepared text
  */
gchar *prepare_description(gchar * text);

/** Removes <pre> and </pre> tags from the text and replaces them with newlines
  *
  * @param the text to modify
  * @return the text without the tags
  */
gchar *remove_pre_tags(gchar * text);

gchar *strip_invalid_char_from_fn(gchar * fn);

gchar *feed_eliminate_end_sign(gchar * text);

/**
 * If application has been started, process pending GUI events.
 */
void ui_update(void);

#endif
