/**
    @file applet.c

    Copyright (c) 2004-2005 Nokia. All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "applet.h"
#include "debug.h"
#include "applet_cache.h"

/* Locale includes */
#include <libintl.h>
#include <locale.h>

#include <glib/gi18n-lib.h>

#include <tablet-browser-interface.h>

#include <libosso.h>

#include <hildon/hildon.h>

#include <string.h>

#include <gdk/gdk.h>

/* Settings dialog and dbus message constants */
#include <osso-rss-feed-reader/settings.h>
#include <osso-rss-feed-reader/dbus.h>


#define WAITTIME_BEFORE_SHOWN_NEW_OUTOFMEM_DIALOG 30*60

/* Global variables */
AppletInfo *rss_appl_inf;
gint htimeout;
gboolean save_nomem_dialog_visible = FALSE;
gboolean save_nodevicemem_dialog_visible = FALSE;


/* GConf - public */
static void _applet_gconf_init(AppletInfo * applet);
static void _applet_gconf_deinit(AppletInfo * applet);

/* DBus - public */
static DBusHandlerResult _applet_handle_signal_cb(DBusConnection * connection,
                                                  DBusMessage * signal,
                                                  gpointer user_data);
static gboolean _applet_dbus_display(AppletInfo * applet,
                                     gchar * feed, gchar * url,
                                     guint32 item);

/* UI / Callbacks - public */
static void realize(GtkWidget * widget, gpointer user_data);
static void Unselect_all(void);

static void _applet_refresh_completed(AppletInfo * applet);
static int _applet_create_temporary_file(gchar * file);
static void _applet_set_headlines(AppletInfo * applet);
static void _applet_refresh_button_cb(GtkWidget * widget, gpointer user_data);
static gboolean _applet_tree_view_cycle_up(GtkWidget * tr);
static gboolean _applet_tree_view_cycle_down(GtkWidget * tr);
static gboolean _applet_headline_clicked_cb(GtkTreeView       *tree_view,
                                            GtkTreePath       *path,
                                            GtkTreeViewColumn *column,
                                            gpointer           user_data);

static void on_menuitem_settings(GtkWidget * widget, gpointer user_data);
void ui_show_save_nomem_dialog(AppletInfo * applet);
void ui_show_save_nodevicemem_dialog(AppletInfo * applet);

static void redraw_refresh_button();
static void redraw_refresh_button_pressed();
static void up_button_pressed_cb();
static void up_button_released_cb();
static void up_button_clicked_cb(GtkWidget * widget, gpointer user_data);
static void down_button_pressed_cb();
static void down_button_released_cb();
static void down_button_clicked_cb(GtkWidget * widget, gpointer user_data);
void Launch_item(GtkTreeModel * model, gpointer user_data);

void create_common_view(AppletInfo * rss_appl_inf);
void create_csm(AppletInfo * rss_appl_inf);
static gboolean expose(GtkWidget *widget, GdkEventExpose *event);

/* Animation - public */
gboolean Scroll_Down_TW(gpointer data);
gboolean Start_TW_Scroll_again(gpointer data);
void stop_animation();
void restart_animation(gdouble timeout);

void applet_check_animation(AppletInfo * applet);


static void _applet_display_handler(osso_display_state_t state, gpointer data);

/* Continous scrolling - public */
/* Starts continous scrolling by calling do_cont_scrolling
 * @param data specifies the diredtion of scrolling (TRUE = up, FALSE = DOWN)*/
gboolean start_cont_scrolling(gpointer data);
/* Scrollins one headline up/down and adds a timeout to call itself again
 * @param data specifies the diredtion of scrolling (TRUE = up, FALSE = DOWN)*/
gboolean do_cont_scrolling(gpointer data);

/************************************************************************/
/* PUBLIC FUNCTIONS */
/************************************************************************/

/* Applet interface functions under here */


void *
hildon_home_applet_lib_initialize(void *state_data,
                                  gint * state_size, GtkWidget ** widget)
{
    ULOG_DEBUG(__FUNCTION__);
    gboolean ret = OSSO_OK;

    g_type_init();

    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);

    ULOG_DEBUG("Initializing News reader applet");

    rss_appl_inf = g_new0(AppletInfo, 1);
    rss_appl_inf->avg_char_width = 10;

    rss_appl_inf->backgrounded = FALSE;

    create_common_view(rss_appl_inf);

    /* Changes the order of the next two, and start and end changed */

    rss_appl_inf->state_data = state_data;
    rss_appl_inf->state_size = (int) &state_size;

    create_csm(rss_appl_inf);

    htimeout = g_timeout_add(5000, _setup_alarmd, NULL);


    hildon_helper_set_logical_color(rss_appl_inf->empty_label, GTK_RC_FG,
                                    GTK_STATE_NORMAL,
                                    "RssAppletFeedTextColor");
    hildon_helper_set_logical_color(rss_appl_inf->empty_label, GTK_RC_BG,
                                    GTK_STATE_NORMAL,
                                    "RssAppletBgColor");

    hildon_helper_set_logical_font(rss_appl_inf->rss_news, "SystemFont");
    hildon_helper_set_logical_color(rss_appl_inf->rss_news, GTK_RC_TEXT,
                                    GTK_STATE_NORMAL,
                                    "RssAppletFeedTextColor");
    hildon_helper_set_logical_color(rss_appl_inf->rss_news, GTK_RC_BG,
                                    GTK_STATE_NORMAL,
                                    "RssAppletBgColor");

    *widget = rss_appl_inf->applet_frame;

    ret = osso_hw_set_display_event_cb(rss_appl_inf->osso, _applet_display_handler , NULL);
    if( ret != OSSO_OK ) {
        g_warning( "Couldn't set hw event handler (%d)", ret );
    }

    return rss_appl_inf;
}

gint
_setup_alarmd(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);


    ULOG_ERR("ENTER alarm setup timeout\n");

    if (setup_alarm_event_now(FALSE) == FALSE)
    {
        ULOG_ERR("Must run again...\n");
        return TRUE;
    }
    else
    {
        ULOG_ERR("Operation finished\n");
        return FALSE;
    }

    ULOG_ERR("LEAVE alarm setup timeout'n");

}

void
hildon_home_applet_lib_deinitialize(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);

    AppletInfo *appl_inf = (AppletInfo *) applet_data;
    DBusError err;
    dbus_error_init(&err);

    if (appl_inf)
    {
        if (appl_inf->scroll_down_tw_timeout)
        {
            g_source_remove(appl_inf->scroll_down_tw_timeout);
            appl_inf->scroll_down_tw_timeout = 0;
        }
        if (appl_inf->scroll_down_tw_starter_timeout)
        {
            g_source_remove(appl_inf->scroll_down_tw_starter_timeout);
            appl_inf->scroll_down_tw_starter_timeout = 0;
        }
        if (rss_appl_inf->long_buttonpress_timeout)
        {
            g_source_remove(rss_appl_inf->long_buttonpress_timeout);
            rss_appl_inf->long_buttonpress_timeout = 0;
        }
        if (rss_appl_inf->cont_scroll_timeout)
        {
            g_source_remove(rss_appl_inf->cont_scroll_timeout);
            rss_appl_inf->cont_scroll_timeout = 0;
        }

        if (appl_inf->timeout > 0)
        {
            ULOG_INFO("Removing old RSS Refresh timeout");
            g_source_remove(appl_inf->timeout);
            appl_inf->timeout = 0;
        }
        close_rss_settings();


        if (htimeout != 0)
            gtk_timeout_remove(htimeout);


        _applet_gconf_deinit(appl_inf);

        g_object_unref(appl_inf->empty_label);
        gchar *rule = NULL;

        rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
                               RSS_SERVICE, 
                               RSS_INTERFACE);

        dbus_bus_remove_match(appl_inf->connection, rule, &err);
        if (dbus_error_is_set(&err))
        {
            ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
                     err.message);
            dbus_error_free(&err);
            g_free(rule);
            osso_hw_unset_event_cb(appl_inf->osso,NULL);
            osso_deinitialize(appl_inf->osso);
            return;
        }
        else
        {
            dbus_connection_remove_filter(appl_inf->connection,
                                          (DBusHandleMessageFunction)
                                          _applet_handle_signal_cb, appl_inf);
        }
        dbus_error_free(&err);
        g_free(rule);
        osso_hw_unset_event_cb(appl_inf->osso,NULL);
        osso_deinitialize(appl_inf->osso);

        if (appl_inf->state_data != NULL)
            g_free(appl_inf->state_data);
        if (appl_inf != NULL) ;
        g_free(appl_inf);
    }
}

void
hildon_home_applet_lib_background(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);

    if (rss_appl_inf != NULL)
    {
        rss_appl_inf->backgrounded = TRUE;
        stop_animation();
        ULOG_DEBUG("animation stopped");
    }
    else
        ULOG_DEBUG("rss_appl_inf NULL, cannot update applet info");

    return;
}

void
hildon_home_applet_lib_foreground(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);


    if (rss_appl_inf != NULL)
    {
        rss_appl_inf->backgrounded = FALSE;
        restart_animation(ANIMATION_TO_DOWN_TIMEOUT);
    }
    else
        ULOG_DEBUG("rss_appl_inf NULL, cannot update applet info");

    Unselect_all();

    return;
}

GtkWidget *hildon_home_applet_lib_settings(void *applet_data,
                                           GtkWindow *parent)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkWidget *menuitem = 
        gtk_menu_item_new_with_label(_("rss_ap_news_reader"));

    g_signal_connect(G_OBJECT(menuitem), "activate",
                     G_CALLBACK(on_menuitem_settings),
                     NULL);

    return menuitem;
}

/***************************************************************************/
/* PRIVATE FUNCTIONS */
/***************************************************************************/

/***************************************************************************/
/* Gconf handling */
/***************************************************************************/

/** Read data from gconf
  *
  * @param client gconf client
  * @param cnxn_id not used
  * @param entry not used
  * @param user_data the applet structure
  */
static void
_applet_gconf_read_cb(GConfClient * client, guint cnxn_id,
                      GConfEntry * entry, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    AppletInfo *applet;

    g_assert(client != NULL);
    g_assert(user_data != NULL);

    applet = (AppletInfo *) user_data;

    applet->update =
        gconf_client_get_bool(client, RSS_SETTINGS_AUTOMATIC_UPDATES, NULL);
    applet->interval =
        gconf_client_get_int(client, RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
                             NULL);
    applet->animation =
        gconf_client_get_bool(client, RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL);
    applet->open_feeds_to =
        gconf_client_get_int(client, RSS_SETTINGS_OPEN_FEEDS_TO, NULL);

    if (applet->interval < MIN_INTERVAL)
        applet->interval = MIN_INTERVAL;

    if (applet->animation)
    {
        restart_animation(ANIMATION_TO_DOWN_WAIT);
    }
    else
    {
        if (applet->scroll_down_tw_timeout)
        {
            g_source_remove(applet->scroll_down_tw_timeout);
            applet->scroll_down_tw_timeout = 0;
        }
        if (applet->scroll_down_tw_starter_timeout)
        {
            g_source_remove(applet->scroll_down_tw_starter_timeout);
            applet->scroll_down_tw_starter_timeout = 0;
        }
    }
}


/** Initialize gconf for reading the update related data
  *
  * @param applet the applet structure
  */
static void
_applet_gconf_init(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GError *error = NULL;

    GConfClient *client = gconf_client_get_default();
    g_assert(NULL != client);
    g_assert(NULL != applet);

    applet->client = client;
    applet->timeout = 0;

    gconf_client_add_dir(client, RSS_SETTINGS, GCONF_CLIENT_PRELOAD_NONE,
                         &error);
    if (NULL != error)
    {
        ULOG_ERR
            ("Could not add %s directory to GConf clients watch list,"
             "error: %s", RSS_SETTINGS, error->message);
        g_free(error);
    }

    if (NULL == error)
    {
        /* Should we save notification numbers and unregister later? */
        applet->gconf_notify_automatic_updates =
            gconf_client_notify_add(client, RSS_SETTINGS_AUTOMATIC_UPDATES,
                                    _applet_gconf_read_cb, applet, NULL,
                                    &error);
        if (NULL != error)
        {
            ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                     RSS_SETTINGS_AUTOMATIC_UPDATES, error->message);
            g_free(error);
        }

        applet->gconf_notify_updates_interval =
            gconf_client_notify_add(client,
                                    RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
                                    _applet_gconf_read_cb, applet, NULL,
                                    &error);
        if (NULL != error)
        {
            ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                     RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL, error->message);
            g_free(error);
        }

        applet->gconf_notify_automatic_scrolling =
            gconf_client_notify_add(client, RSS_SETTINGS_AUTOMATIC_SCROLLING,
                                    _applet_gconf_read_cb, applet, NULL,
                                    &error);
        if (NULL != error)
        {
            ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                     RSS_SETTINGS_AUTOMATIC_SCROLLING, error->message);
            g_free(error);
        }

        applet->gconf_notify_open_feeds_to =
            gconf_client_notify_add(client, RSS_SETTINGS_OPEN_FEEDS_TO,
                                    _applet_gconf_read_cb, applet, NULL,
                                    &error);
        if (NULL != error)
        {
            ULOG_ERR("Could not set notify for GConf key %s, error: %s",
                     RSS_SETTINGS_OPEN_FEEDS_TO, error->message);
            g_free(error);
        }
    }

    if (NULL != error)
        g_free(error);

    /* Try to install timeout even if we got errors setting up change
     * listeners */
    _applet_gconf_read_cb(client, 0, NULL, applet);
}

/** Deinitialize gconf
  *
  * @param applet the applet structure
  */
static void
_applet_gconf_deinit(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GError *error = NULL;

    g_assert(applet != NULL);
    if (applet->gconf_notify_automatic_updates > 0)
    {
        gconf_client_notify_remove(applet->client, 
                                   applet->gconf_notify_automatic_updates);
    }
    gconf_client_remove_dir(applet->client, RSS_SETTINGS, &error);
    if (applet->gconf_notify_updates_interval > 0)
    {
        gconf_client_notify_remove(applet->client, 
                                   applet->gconf_notify_updates_interval);
    }
    if (applet->gconf_notify_automatic_scrolling > 0)
    {
        gconf_client_notify_remove(applet->client,
                                   applet->gconf_notify_automatic_scrolling);
    }
    if (applet->gconf_notify_open_feeds_to > 0)
    {
        gconf_client_notify_remove(applet->client,
                                   applet->gconf_notify_open_feeds_to);
    }
    g_free(error);
    g_object_unref(applet->client);
}


/***************************************************************************/
/* DBus handling */
/***************************************************************************/

/* Depending on the state of hw, do something */
void
hw_event_handler(osso_hw_state_t * state, gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    (void) data;

    g_assert(state != NULL);

    ULOG_DEBUG("Hardware event received: %d %d %d %d", state->shutdown_ind,
               state->memory_low_ind, state->save_unsaved_data_ind,
               state->system_inactivity_ind);

    if (state->shutdown_ind)
    {

    }
    if (state->memory_low_ind)
    {
        /* Memory low */

        rss_appl_inf->startrss = TRUE;

    }
    else
    {

        rss_appl_inf->startrss = FALSE;

    }
    if (state->save_unsaved_data_ind)
    {
    }
}


static void _applet_display_handler(osso_display_state_t state, gpointer data)
{
    if (state == OSSO_DISPLAY_OFF)
    {
        ULOG_DEBUG("display off");
        rss_appl_inf->animation = FALSE;
        stop_animation();

    }
    else if( state == OSSO_DISPLAY_ON )
    {
        rss_appl_inf->animation =
            gconf_client_get_bool(rss_appl_inf->client,
                                  RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL);

        restart_animation(ANIMATION_TO_DOWN_TIMEOUT);
    }
}

/** The D-BUS messages are handled here
  *
  * @param connection the D-BUS connection
  * @param signal the incoming D-BUS message
  * @param user_data the applet structure
  * @return returns DBUS_HANDLER_RESULT_HANDLED on success and
  *         DBUS_HANDLER_RESULT_NOT_YET_HANDLED on failure
  */
static DBusHandlerResult
_applet_handle_signal_cb(DBusConnection *
                         connection, DBusMessage * signal, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    DBusHandlerResult msg_handled = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    AppletInfo *applet = (AppletInfo *) user_data;

    ULOG_DEBUG("DBus message: %s\n%s", dbus_message_get_member(signal),
               dbus_message_get_interface(signal));

    g_assert(NULL != connection);
    g_assert(NULL != signal);

    if (applet == NULL)
    {
        /* Why would this happen? */
        ULOG_DEBUG("Applet data is NULL");
        return msg_handled;
    }

    if (!dbus_message_is_method_call
        (signal, "com.nokia.statusbar", "statusbar_insensitive"))
    {
        if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                   OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL))
        {
            ULOG_DEBUG("signal: refresh_finished");

            _applet_refresh_completed(applet);
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
            applet->refreshing = FALSE;
        }
        else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                        OSSO_RSS_FEED_READER_REFRESH_INTERRUPTED_SIGNAL))
        {
            ULOG_DEBUG("signal: refresh_interrupted");
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
            if (applet->refreshing)
            {
                applet->refreshing = FALSE;
                redraw_refresh_button();
            }
        }
        else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                        OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL))
        {
            ULOG_DEBUG("signal: refresh_prevented");
            g_assert(NULL != applet->rss_image);
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
            if (applet->refreshing)
            {
                applet->refreshing = FALSE;
                redraw_refresh_button();
            }
        }
        else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                        OSSO_RSS_FEED_READER_FOLDER_READ_SIGNAL))
        {
            ULOG_DEBUG("signal: folder_read");
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
        }
        else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                        OSSO_RSS_FEED_READER_CONNECTING_STARTED))
        {
            ULOG_DEBUG("signal: connecting_started");
            applet->refreshing = TRUE;
            redraw_refresh_button();
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
        }
        else if (dbus_message_is_signal(signal, RSS_INTERFACE,
                                        OSSO_RSS_FEED_READER_REFRESH_STARTED))
        {
            ULOG_DEBUG("refresh started");
            applet->refreshing = TRUE;
            redraw_refresh_button();
            msg_handled = DBUS_HANDLER_RESULT_HANDLED;
        }
    }
    ULOG_DEBUG("end of _applet_handle_signal_cb");
    return msg_handled;
}

/** Gets a D-BUS connection
  * 
  * @param applet the applet structure
  * @param signal_handler the handler for the D-BUS connection
  * @return TRUE on success, FALSE on failure
  */
static gboolean
_applet_dbus_connect(AppletInfo * applet,
                     DBusHandleMessageFunction signal_handler)
{
    ULOG_DEBUG(__FUNCTION__);

    DBusError err;
    g_assert(applet);
    applet->osso = NULL;
    applet->connection = NULL;
    dbus_error_init(&err);

    if (applet != NULL && applet->connection == NULL)
    {
        applet->osso =
            osso_initialize("osso_rss_feed_reader_applet", PACKAGE_VERSION,
                            FALSE, NULL);
        g_assert(applet->osso != NULL);
        applet->connection =
            (DBusConnection *) osso_get_dbus_connection(applet->osso);
        g_assert(applet->connection != NULL);

        /* Set handling changes in HW states. Note: not tested */
        osso_return_t ret = osso_hw_set_event_cb(applet->osso,
                                                 NULL, hw_event_handler,
                                                 applet);

        if (ret != OSSO_OK)
        {
            ULOG_CRIT("Could not set callback for HW monitoring");
        }


        if (applet->connection == NULL || dbus_error_is_set(&err))
        {
            dbus_error_free(&err);
            return FALSE;
        }

        gchar *rule = NULL;

        rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
                               RSS_SERVICE, 
                               RSS_INTERFACE);

        dbus_bus_add_match(applet->connection, rule, &err);

        if (dbus_error_is_set(&err))
        {
            ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
                     err.message);
            dbus_error_free(&err);
            g_free(rule);
            return FALSE;
        }
        else
        {
            dbus_connection_add_filter(applet->connection,
                                       signal_handler, applet, NULL);
        }
        g_free(rule);

        /* Finally, register the connection handling to main loop */
        dbus_connection_setup_with_g_main(applet->connection, NULL);
    }

    return TRUE;
}

/** The applet is refreshed by calling this function
  *
  * @param applet the applet structure
  * @param timed if the refresh should be timed or not
  * @return TRUE on success, FALSE on failure
  */
static gboolean
_applet_dbus_refresh(AppletInfo * applet, gboolean timed)
{
    ULOG_DEBUG(__FUNCTION__);

    gboolean timing = timed;

    if (applet != NULL && applet->connection != NULL)
    {

        if (rss_appl_inf->startrss == TRUE)
        {

            ui_show_save_nomem_dialog(rss_appl_inf);

            return TRUE;

        }

        DBusMessage *message = NULL;
        message =
            dbus_message_new_method_call
            (RSS_SERVICE,
             RSS_OBJECT_PATH,
             RSS_INTERFACE, OSSO_RSS_FEED_READER_REFRESH_SIGNAL);
        if (message == NULL)
        {
            ULOG_ERR("Creating DBus refresh call failed");
            return FALSE;
        }
        dbus_message_set_auto_start(message, TRUE);
        if (!dbus_message_append_args(message,
                                      DBUS_TYPE_BOOLEAN, &timing,
                                      DBUS_TYPE_INVALID))
        {
            ULOG_ERR("Could not append args to message");
        }
        if (!_applet_create_temporary_file("background"))
        {
            ULOG_ERR("Could not create temporary file \"background\"");
            if (!applet->backgrounded)
            {
                ui_show_save_nodevicemem_dialog(rss_appl_inf);
            }
            //call the refresh completed function, to restore the 
            //refresh image
            _applet_refresh_completed(rss_appl_inf);
            dbus_message_unref(message);
            return TRUE;
        }
        if (!(dbus_connection_send(applet->connection, message, NULL)))
        {
            ULOG_ERR("Sending DBus refresh message failed");
            dbus_message_unref(message);
            return FALSE;
        }
        if (message != NULL)
        {
            dbus_message_unref(message);
        }
    }
    else
    {
        ULOG_ERR
            ("Refresh message failed because applet is"
             "not properly initialized");
    }
    return TRUE;
}

/** Calls the main application to display a selected feed
  *
  * @param applet the applet structure
  * @param feed the feed to display
  * @param item the item to display
  * @return TRUE on success, FALSE on failure
  */
static gboolean
_applet_dbus_display(AppletInfo * applet,
                     gchar * feed, gchar * url, guint32 item)
{
    ULOG_DEBUG(__FUNCTION__);


    g_assert(feed != NULL);

    if (applet != NULL && applet->connection != NULL)
    {
        if (OPEN_TO_RSS == rss_appl_inf->open_feeds_to)
        {
            if (rss_appl_inf->startrss == TRUE)
            {

                ui_show_save_nomem_dialog(rss_appl_inf);

                return TRUE;
            }

            if (OSSO_OK != osso_rpc_async_run(applet->osso,
                                              RSS_SERVICE,
                                              RSS_OBJECT_PATH,
                                              RSS_INTERFACE,
                                              OSSO_RSS_FEED_READER_DISPLAY_SIGNAL,
                                              NULL, NULL,
                                              DBUS_TYPE_STRING, feed,
                                              DBUS_TYPE_UINT32, item,
                                              DBUS_TYPE_INVALID, NULL))
            {
                ULOG_DEBUG("Could not send the display message");
            }
        }
        else                    // opening feed in browser
        {
            if (OSSO_OK != osso_rpc_run_with_defaults(applet->osso,
                                                      "osso_browser",
                                                      OSSO_BROWSER_LOAD_URL_REQ,
                                                      NULL, DBUS_TYPE_STRING,
                                                      url, DBUS_TYPE_INVALID))
            {
                ULOG_DEBUG("Could not send the display message");
            }
        }
    }
    else
    {
        ULOG_ERR
            ("Display message failed because applet"
             "is not properly initialized");
    }

    return TRUE;
}


/***************************************************************************/
/* UI / Callbacks handling */
/***************************************************************************/

/***************************************************************************/
/* UI / Callbacks handling -- helper functions */
/***************************************************************************/
void
update_font_width(AppletInfo * rss_appl_inf)
{
    ULOG_DEBUG(__FUNCTION__);

    PangoContext *context;
    PangoFontMetrics *metrics;
    PangoFontDescription *font_desc, *font2;

    font_desc =
        pango_font_description_copy(rss_appl_inf->rss_news->style->font_desc);
    g_object_get(rss_appl_inf->textRenderer, "font-desc", &font2, NULL);
    pango_font_description_merge(font_desc, font2, TRUE);

    context = gtk_widget_get_pango_context(rss_appl_inf->rss_news);

    metrics = pango_context_get_metrics(context,
                                        font_desc,
                                        pango_context_get_language(context));
    rss_appl_inf->line_height = (pango_font_metrics_get_ascent(metrics) +
                                 pango_font_metrics_get_descent(metrics)) /
        PANGO_SCALE + 2;
    pango_font_metrics_unref(metrics);

    ULOG_DEBUG("Character height: %d", rss_appl_inf->line_height);
    pango_font_description_set_size(font_desc,
                                    pango_font_description_get_size(font_desc)
                                    / (float) 1.2);

    metrics = pango_context_get_metrics(context,
                                        font_desc,
                                        pango_context_get_language(context));
    rss_appl_inf->avg_char_width =
        pango_font_metrics_get_approximate_char_width(metrics) / PANGO_SCALE +
        2;
    ULOG_DEBUG("Avarage character width: %d", rss_appl_inf->avg_char_width);
    pango_font_metrics_unref(metrics);
    pango_font_description_free(font_desc);
    pango_font_description_free(font2);
}

/* calculates which byte the text is needed to break at at wrapping */
gint
calc_breaking_position(AppletInfo *rss_appl_inf, gchar *text)
{
    PangoContext *context;
    PangoLayout *layout;
    gint offs;
    gint trailing;
    gint width;
    gchar *brk;

    ULOG_DEBUG(__FUNCTION__);

    width = (gtk_tree_view_column_get_width(rss_appl_inf->textColumn) >
                    HEADLINE_MIN_WIDTH)
            ? gtk_tree_view_column_get_width(rss_appl_inf->textColumn)
            : HEADLINE_MIN_WIDTH;
    context = gtk_widget_get_pango_context(rss_appl_inf->rss_news);
    layout = pango_layout_new(context);
    pango_layout_set_markup(layout, text, -1);
    ULOG_DEBUG("pango x: %d", width - rss_appl_inf->avg_char_width);
    pango_layout_xy_to_index(layout, 
            (width - 2*rss_appl_inf->avg_char_width) * PANGO_SCALE, 
            0, &offs, &trailing);
    g_object_unref(layout);
    ULOG_DEBUG("xy index %d trailing %d", offs, trailing);
    brk = g_utf8_offset_to_pointer(text + offs, trailing);
    ULOG_DEBUG("byte offset: %d", brk - text);
    return brk - text;
}

/* puts a newline at a space before offset (counted in bytes) or 
 * at offset it there's no space before that */
gchar *
put_newline(gchar *title, gint offset)
{
    gint i;

    ULOG_DEBUG(__FUNCTION__);

    if (NULL == title)
    {
        return NULL;
    }
    
    if (offset >= strlen(title))
    {
        return title;
    }

    i = offset;
    while (i > 0 && title[i] != ' ')
        --i;
    
    if (i > 0)
    {
        title[i] = '\n';
        return title;
    }

    /* there was no space inside. we wrap the text simply */
    gchar *part1 = title;
    gchar *part2 = g_strdup(title + offset);
    part1[offset] = '\n';
    part1[offset+1] = 0;
    title = g_strjoin(NULL, part1, part2, NULL);
    g_free(part1);
    g_free(part2);

    return title;
}



/** Creates an empty file
  *
  * @param file the filename to create
  */
static int
_applet_create_temporary_file(gchar * file)
{
    ULOG_DEBUG(__FUNCTION__);

    gchar *filename = NULL;
    FILE *tmpf;
    filename = g_strdup_printf("%s" G_DIR_SEPARATOR_S
                               ".osso_rss_feed_reader"
                               G_DIR_SEPARATOR_S "%s",
                               g_get_home_dir(), file);
    test_if_cache_directory_exists();
    tmpf = fopen(filename, "wb");
    if (tmpf == NULL)
    {
        g_free(filename);
        return 0;
    }
    fclose(tmpf);
    g_free(filename);
    return 1;
}


static void
Unselect_all(void)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreeSelection *selection;
    selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
    gtk_tree_selection_unselect_all(selection);
    ULOG_DEBUG("Unselect end");
}


gboolean
focus_lost(GtkWidget * widget, GdkEventFocus * event, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    Unselect_all();
    return TRUE;
}

gboolean
focus_lost2(GtkWidget * widget, GdkEventCrossing * event, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    ULOG_DEBUG("x: %f y: %f", event->x, event->y);
    gint xcord = 0, ycord = 0;
    gtk_widget_get_pointer(rss_appl_inf->applet_frame, &xcord, &ycord);
    ULOG_DEBUG("getcord:x: %d y: %d", xcord, ycord);

    update_font_width(rss_appl_inf);
    if (rss_appl_inf->textColumn)
        ULOG_DEBUG("Column: %d, max_chars: %d",
                   gtk_tree_view_column_get_width(rss_appl_inf->textColumn),
                   (gtk_tree_view_column_get_width(rss_appl_inf->textColumn) /
                    rss_appl_inf->avg_char_width));
    GtkAllocation sreq;
    sreq = rss_appl_inf->applet_frame->allocation;
    ULOG_DEBUG("widget:x: %d y: %d", sreq.width, sreq.height);
    if ((xcord - 1 <= 0) || (ycord - 1 <= 0) || ((gint) xcord >= sreq.width)
        || ((gint) ycord >= sreq.height))
    {
        ULOG_DEBUG("Entered...");

        Unselect_all();

        return TRUE;
    }
    ULOG_DEBUG("Omited...%d", xcord + 1 >= sreq.width);
    return FALSE;
}

static void
realize(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkAllocation sreq;
    /* HEADLINE_MIN_WIDTH is used, because at start, textColumn doesn't seem
     * to know it's size. Since applet has a minimum width, also the
     * textColumn will have one... */
    gint width =
        (gtk_tree_view_column_get_width(rss_appl_inf->textColumn) >
                HEADLINE_MIN_WIDTH)
        ? gtk_tree_view_column_get_width(rss_appl_inf->textColumn)
        : HEADLINE_MIN_WIDTH;
    sreq = rss_appl_inf->applet_frame->allocation;
    if ((rss_appl_inf->prev_height != sreq.height && rss_appl_inf->prev_width)
        || (rss_appl_inf->prev_width != sreq.width && width))
    {
        gint new_sw_height =
                sreq.height - BUTTON_HEIGHT - 2.5*HILDON_MARGIN_DOUBLE;
        gtk_widget_set_size_request(rss_appl_inf->scrolledwindow, 
                                    sreq.width, new_sw_height);
        gtk_widget_set_size_request(rss_appl_inf->empty_label, 
                                    sreq.width, new_sw_height);
        ULOG_DEBUG("new_sw_height: %d", new_sw_height);

        update_font_width(rss_appl_inf);
        rss_appl_inf->prev_width = sreq.width;
        rss_appl_inf->prev_height = sreq.height;

        rss_appl_inf->chars_per_line =
            width / rss_appl_inf->avg_char_width - 2;
        rss_appl_inf->displayed_lines =
            new_sw_height / rss_appl_inf->line_height;
        rss_appl_inf->displayed_headlines = new_sw_height / HEADLINE_HEIGHT;

        /* Calculating padding between rows */
        GValue *prop = g_new0(GValue, 1);
        g_value_init(prop, G_TYPE_INT);
        gtk_widget_style_get_property(rss_appl_inf->rss_news, "vertical-separator", prop);
        rss_appl_inf->space_between_rows = g_value_get_int(prop);
        g_free(prop);

        prop = g_new0(GValue, 1);
        g_value_init(prop, G_TYPE_INT);
        gtk_widget_style_get_property(rss_appl_inf->rss_news, "focus-line-width", prop);
        rss_appl_inf->space_between_rows += 2 * g_value_get_int(prop);
        g_free(prop);
        ULOG_DEBUG("space_between_rows: %d", rss_appl_inf->space_between_rows);

        rss_appl_inf->tree_height =
            (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows) * (MAX_HEADLINES) - 2;

        ULOG_DEBUG(">>Column: %d, max_chars: %d, lines: %d, headlines: %d",
                   width, rss_appl_inf->chars_per_line,
                   rss_appl_inf->displayed_lines,
                   rss_appl_inf->displayed_headlines);
        ULOG_DEBUG(">>widget:x: %d y: %d", sreq.width, sreq.height);
        ULOG_DEBUG(">>widget rss:y: %d", new_sw_height);
        ULOG_DEBUG(">>headline col width: %d", width);

        sreq = rss_appl_inf->applet_frame->allocation;
        ULOG_DEBUG("AFTER aca: >>widget:x: %d y: %d", sreq.width,
                   sreq.height);

        /* _applet_set_headlines does some resizing according to the number
         * of headlines; all initial resizing should come before it */
        _applet_set_headlines(rss_appl_inf);
        ULOG_DEBUG("AFTER: >>widget:x: %d y: %d", sreq.width, sreq.height);
        ULOG_DEBUG("AFTER: >>widget rss: y: %d", new_sw_height);

        applet_check_animation(rss_appl_inf);
    }
}


/***************************************************************************/
/* UI / Callbacks handling -- Callbacks */
/***************************************************************************/

gboolean
dialog_key_press(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    (void) data;

    if (event->state & (GDK_CONTROL_MASK |
                        GDK_SHIFT_MASK |
                        GDK_MOD1_MASK |
                        GDK_MOD3_MASK |
                        GDK_MOD4_MASK | 
                        GDK_MOD5_MASK))
    {
        return FALSE;
    }

    switch (event->keyval)
    {
        case GDK_Escape:
            return FALSE;
            break;
        default:
            break;
    }

    return FALSE;
}

static gboolean
_applet_key_release(GtkWidget * widget,
                    GdkEventKey * event, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    if (event == NULL)
        return FALSE;
    if (event->state & (GDK_CONTROL_MASK |
                        GDK_SHIFT_MASK |
                        GDK_MOD1_MASK |
                        GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK))
    {
        return TRUE;
    }


    switch (event->keyval)
    {
        case GDK_Escape:
            ULOG_DEBUG("Escape released");
            if (rss_appl_inf->refreshing)
            {
                _applet_refresh_button_cb(NULL, rss_appl_inf);
                rss_appl_inf->refreshing = FALSE;
            }
            break;

        default:
            break;
    }
    return TRUE;
}

/** A callback function that is called after the refresh button has been clicked
  *
  * @param widget not used
  * @param user_data the applet structure
  */
static void
_applet_refresh_button_cb(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    AppletInfo *applet = (AppletInfo *) user_data;

    g_assert(user_data != NULL);

    if (_applet_dbus_refresh(applet, FALSE) == FALSE)
    {
        ULOG_ERR("User requested refresh DBus message failed");
    }
}

/** A callback function to listen to key strokes to give focus to tree view
  *
  * @param widget the widget to listen to
  * @param event information about the key that has been hit
  * @param user_data the tree view to cycle
  * @return TRUE if the signal should not be propagated further, FALSE otherwise
  */
static gboolean
_applet_button_key_press(GtkWidget * widget,
                         GdkEventKey * event)
{
    ULOG_DEBUG(__FUNCTION__);

    if (event == NULL)
        return FALSE;

    if (GDK_KEY_PRESS == event->type)
    {
        switch (event->keyval)
        {
            case GDK_Return:
                /* When user presses select with up or down button active, we need
                 * to scroll, but we won't get the button press/button release
                 * signals, so emulating click, returning FALSE, to let the
                 * default handler redraw the button */
                if (widget == rss_appl_inf->up_button)
                {
                    up_button_clicked_cb(NULL, NULL);
                }
                else if (widget == rss_appl_inf->down_button)
                {
                    down_button_clicked_cb(NULL, NULL);
                }
                return FALSE;
            case GDK_Up:
            case GDK_Down:
            {
                GtkTreePath *path = NULL;
                gtk_tree_view_get_cursor(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                         &path, NULL);

                GtkTreeSelection *select = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
                gtk_tree_selection_select_path(select, path);

                if (GDK_Up == event->keyval)
                {
                    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                                 path, NULL, TRUE, 0.0, 0.0);
                }
                else
                {
                    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                                 path, NULL, TRUE, 1.0, 1.0);
                }
                gtk_widget_grab_focus(rss_appl_inf->rss_news);

                return TRUE;
            }
            default:
                return FALSE;
        }
    }

    return FALSE;
}

/** A callback function to listen to key strokes to enable cycling in the tree view
  *
  * @param widget the widget to listen to
  * @param event information about the key that has been hit
  * @param user_data the tree view to cycle
  * @return TRUE if the signal should not be propagated further, FALSE otherwise
  */
static gboolean
_applet_tree_view_key_release(GtkWidget * widget,
                              GdkEventKey * event, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    if (event == NULL)
        return FALSE;

    if (event->state & (GDK_CONTROL_MASK |
                        GDK_SHIFT_MASK |
                        GDK_MOD1_MASK |
                        GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK))
    {
        return TRUE;
    }

    if (user_data == NULL)
        return FALSE;


    if (GDK_KEY_PRESS == event->type)
    {
        switch (event->keyval)
        {
            case GDK_Up:
                return _applet_tree_view_cycle_up(rss_appl_inf->rss_news);
                break;

            case GDK_Down:
                return _applet_tree_view_cycle_down(rss_appl_inf->rss_news);
                break;

            case GDK_Left:
            case GDK_Right:
                return TRUE;
            case GDK_Return:
            {
                ULOG_DEBUG("Return released");
                Launch_item(gtk_tree_view_get_model
                            (GTK_TREE_VIEW(rss_appl_inf->rss_news)),
                            user_data);
                return TRUE;
            }

            default:
                break;
        }
    }
    return TRUE;
}

/** A callback function that is called after a headline is clicked
  *
  * @param widget the tree view containing the headlines
  * @param event not used
  * @param user_data the applet structure
  * @return always FALSE
  */
static gboolean
_applet_headline_clicked_cb(GtkTreeView       *tree_view,
                            GtkTreePath       *path,
                            GtkTreeViewColumn *column,
                            gpointer           user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreeModel *model;
    GtkTreeSelection *selection;
    GtkTreeIter iter;

    g_assert(NULL != tree_view);
    g_assert(NULL != path);
    g_assert(NULL != column);
    g_assert(NULL != user_data);

    stop_animation();

    selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));

    if (gtk_tree_selection_get_selected(selection, &model, &iter))
        Launch_item(model, user_data);

    restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);

    return FALSE;
}

static void
refresh_clicked_cb(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    rss_appl_inf->refreshing = !rss_appl_inf->refreshing;

    redraw_refresh_button();

    if (rss_appl_inf->refreshing)
    {
        ULOG_DEBUG("refreshing is true");
        _applet_refresh_button_cb(NULL, rss_appl_inf);
    }
}

static void
up_button_clicked_cb(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);


    GdkRectangle *visible_rect = g_new0(GdkRectangle, 1);
    gtk_tree_view_get_visible_rect(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                   visible_rect);

    stop_animation();

    gint item_nr =
        ((gint) visible_rect->y / (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows) % MAX_HEADLINES) + 1;
    gint new_scroll_pos =
        ((gint) ((item_nr - 1) * (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows))) %
        rss_appl_inf->tree_height;


    /* When we are at the top of a headline, prev_headline shouldn't be the
     * current headline, shifting one headline up */
    new_scroll_pos = ((visible_rect->y == new_scroll_pos)
                     ? new_scroll_pos - HEADLINE_HEIGHT - rss_appl_inf->space_between_rows
                     : new_scroll_pos);
    /* When we are at the top of the list, prev_headline should be the top of 
     * last item */
    new_scroll_pos = (new_scroll_pos < 0)
        ? (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows) * (rss_appl_inf->num_of_headlines - 1)
        : new_scroll_pos;

    gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                  -1, new_scroll_pos);

    restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);
    g_free(visible_rect);
}


static void
down_button_clicked_cb(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    GdkRectangle *visible_rect = g_new0(GdkRectangle, 1);
    gtk_tree_view_get_visible_rect(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                   visible_rect);

    stop_animation();

    gint item_nr =
        ((gint) visible_rect->y / (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows) % MAX_HEADLINES) + 1;

    gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(rss_appl_inf->rss_news), -1,
                             ((gint) ((item_nr) * (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows))) %
                             rss_appl_inf->tree_height);

    restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);
    g_free(visible_rect);
}


static void
on_menuitem_settings(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    execute_rss_settings(rss_appl_inf->osso, NULL, TRUE, TRUE);
}

void
on_menuitem_close(GtkWidget * widget, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    gtk_widget_destroy(GTK_WIDGET(user_data));
}


/***************************************************************************/
/* UI / Callbacks handling -- UI creation */
/***************************************************************************/


void
create_headline_display(AppletInfo * rss_appl_inf)
{
    ULOG_DEBUG(__FUNCTION__);

    /* Creating tree store for displaying headlines */
    GtkTreeStore *headlines = gtk_tree_store_new(N_COLUMNS,
                                                 GDK_TYPE_PIXBUF,
                                                 G_TYPE_STRING,
                                                 G_TYPE_STRING,
                                                 G_TYPE_STRING,
                                                 G_TYPE_STRING,
                                                 G_TYPE_INT);
    rss_appl_inf->headlines = headlines;
    GtkWidget *rss_news =
        gtk_tree_view_new_with_model(GTK_TREE_MODEL(headlines));
    rss_appl_inf->rss_news = rss_news;

    GtkTreeSelection *select;

    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_news));
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);

    gtk_container_add(GTK_CONTAINER(rss_appl_inf->scrolledwindow), rss_news);
    g_signal_connect(G_OBJECT(rss_news), "row-activated",
                     G_CALLBACK(_applet_headline_clicked_cb), rss_appl_inf);
    g_signal_connect(G_OBJECT(rss_news), "key-press-event",
                     G_CALLBACK(_applet_tree_view_key_release), rss_appl_inf);
    g_signal_connect(G_OBJECT(rss_news), "focus-out-event",
                     G_CALLBACK(focus_lost), NULL);
    g_signal_connect(G_OBJECT(rss_news), "leave-notify-event",
                     G_CALLBACK(focus_lost2), NULL);
    gtk_widget_show(rss_news);

    /* Adding scrollbar to indicate where we are in the headline list */
    GtkWidget *scrollbar =
        gtk_vscrollbar_new(gtk_scrolled_window_get_vadjustment
                           (GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow)));
    gtk_widget_set_name(scrollbar, "osso-rss-feed-reader-scrollbar");
    rss_appl_inf->scrollbar = scrollbar;
    gtk_widget_set_size_request(GTK_WIDGET(scrollbar), SCROLLBAR_WIDTH, -1);
    gtk_box_pack_start(GTK_BOX(rss_appl_inf->news_hbox),
                       scrollbar, FALSE, FALSE, 0);

    /* Adding columns to treeview */
    GtkCellRenderer *iconRenderer = gtk_cell_renderer_pixbuf_new();
    GtkCellRenderer *textRenderer = gtk_cell_renderer_text_new();
    rss_appl_inf->textRenderer = (GtkCellRendererText *) textRenderer;
    gtk_cell_renderer_set_fixed_size(textRenderer, -1, HEADLINE_HEIGHT);

    GtkTreeViewColumn *column = gtk_tree_view_column_new();
    gtk_tree_view_append_column(GTK_TREE_VIEW(rss_news), column);

    gtk_tree_view_column_pack_start(column, iconRenderer, FALSE);
    gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
                                       ICON_COLUMN);
    gtk_tree_view_column_set_expand(column, FALSE);

    rss_appl_inf->textColumn = column = gtk_tree_view_column_new();
    gtk_tree_view_append_column(GTK_TREE_VIEW(rss_news), column);
    gtk_tree_view_column_pack_start(column, textRenderer, TRUE);
    gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
                    RENDER_COLUMN);
    g_object_set(textRenderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL);

    GtkTreeSelection *rss_select =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_news));
    gtk_tree_selection_set_mode(rss_select, GTK_SELECTION_SINGLE);
}

void
create_buttons(AppletInfo * rss_appl_inf)
{
    ULOG_DEBUG(__FUNCTION__);

    /* Creating box contatining refresh/up/down buttons */
    GtkWidget *button_hbox = gtk_hbox_new(FALSE, 0);
    g_object_set(button_hbox,
                 "homogeneous", FALSE, "spacing", 0, "border-width", 0, NULL);
    rss_appl_inf->button_hbox = button_hbox;

    gtk_box_pack_end(GTK_BOX(rss_appl_inf->applet_vbox), 
                               button_hbox, FALSE, FALSE, 
                               0);
    gtk_widget_show(button_hbox);

    /* Creating toolbar buttons */
    /* Refresh button (with two pixmaps: refresh/stop) */
    GtkWidget *rss_refresh_button = gtk_button_new();
    gtk_widget_set_name(rss_refresh_button, "osso-rss-feed-reader-button");
    rss_appl_inf->rss_refresh_button = rss_refresh_button;
    gtk_button_set_relief(GTK_BUTTON(rss_refresh_button), GTK_RELIEF_NONE);
    gtk_widget_set_size_request(rss_appl_inf->rss_refresh_button,
                                BUTTON_WIDTH, BUTTON_HEIGHT);
    gtk_box_pack_start(GTK_BOX(button_hbox), rss_refresh_button, FALSE, FALSE,
                       0);
    redraw_refresh_button();
    g_signal_connect(G_OBJECT(rss_refresh_button), "clicked",
                     G_CALLBACK(refresh_clicked_cb), NULL);
    g_signal_connect(G_OBJECT(rss_refresh_button), "pressed",
                     G_CALLBACK(redraw_refresh_button_pressed), NULL);
    g_signal_connect(G_OBJECT(rss_refresh_button), "key-press-event",
                     G_CALLBACK(_applet_button_key_press), rss_appl_inf);


    /* Scroll up button */
    GtkWidget *up_button = gtk_button_new();
    gtk_widget_set_name(up_button, "osso-rss-feed-reader-button");
    rss_appl_inf->up_button = up_button;
    gtk_button_set_relief(GTK_BUTTON(up_button), GTK_RELIEF_NONE);
    gtk_widget_set_size_request(up_button, BUTTON_WIDTH, BUTTON_HEIGHT);
    gtk_box_pack_start(GTK_BOX(button_hbox), up_button, FALSE, FALSE, 0);
    rss_appl_inf->up_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_up",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->up_image), ICON_SIZE);
    gtk_button_set_image(GTK_BUTTON(up_button), rss_appl_inf->up_image);
    g_signal_connect(G_OBJECT(up_button), "pressed",
                     G_CALLBACK(up_button_pressed_cb), NULL);
    g_signal_connect(G_OBJECT(up_button), "released",
                     G_CALLBACK(up_button_released_cb), NULL);
    g_signal_connect(G_OBJECT(up_button), "key-press-event",
                     G_CALLBACK(_applet_button_key_press), rss_appl_inf);

    /* Scroll down button */
    GtkWidget *down_button = gtk_button_new();
    gtk_widget_set_name(down_button, "osso-rss-feed-reader-button");
    rss_appl_inf->down_button = down_button;
    gtk_button_set_relief(GTK_BUTTON(down_button), GTK_RELIEF_NONE);
    gtk_widget_set_size_request(down_button, BUTTON_WIDTH, BUTTON_HEIGHT);
    gtk_box_pack_start(GTK_BOX(button_hbox), down_button, FALSE, FALSE, 0);
    rss_appl_inf->down_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_down",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->down_image), ICON_SIZE);

    gtk_button_set_image(GTK_BUTTON(down_button), rss_appl_inf->down_image);
    g_signal_connect(G_OBJECT(down_button), "pressed",
                     G_CALLBACK(down_button_pressed_cb), NULL);
    g_signal_connect(G_OBJECT(down_button), "released",
                     G_CALLBACK(down_button_released_cb), NULL);
    g_signal_connect(G_OBJECT(down_button), "key-press-event",
                     G_CALLBACK(_applet_button_key_press), rss_appl_inf);

}

void
create_common_view(AppletInfo * rss_appl_inf)
{
    ULOG_DEBUG(__FUNCTION__);

    /* Creating main applet frame */
    GtkWidget *applet_frame = gtk_widget_new(GTK_TYPE_FRAME,
                                             "visible", TRUE,
                                             "name", "osso-rss-feed-reader",
                                             "shadow-type", GTK_SHADOW_NONE,
                                             "border-width",
                                             0,
                                             "can-focus", FALSE,
                                             NULL);
    GtkWidget *applet_alignment = gtk_widget_new (GTK_TYPE_ALIGNMENT,
                                                  "bottom-padding",
                                                  HILDON_MARGIN_DOUBLE,
                                                  "top-padding",
                                                  HILDON_MARGIN_DOUBLE,
                                                  "left-padding",
                                                  HILDON_MARGIN_DOUBLE,
                                                  "right-padding",
                                                  HILDON_MARGIN_DOUBLE);
    gtk_container_add(GTK_CONTAINER(applet_frame), applet_alignment);
    rss_appl_inf->applet_frame = applet_frame;

    /* Creating main vbox */
    GtkWidget *applet_vbox = gtk_vbox_new(FALSE, 0);
    g_object_set(applet_vbox,
                 "homogeneous", FALSE,
                 "spacing", HILDON_MARGIN_DOUBLE, "border-width", 0, NULL);
    rss_appl_inf->applet_vbox = applet_vbox;
    gtk_container_add(GTK_CONTAINER(applet_alignment), applet_vbox);
    g_signal_connect(G_OBJECT(applet_vbox), "key-release-event",
                     G_CALLBACK(_applet_key_release), NULL);

    /* Creating news_hbox, containing headlines and scrollbar */
    GtkWidget *news_hbox = gtk_hbox_new(FALSE, 0);
    g_object_set(news_hbox,
                 "homogeneous", FALSE,
                 "spacing", 0, "border-width", 0, NULL);
    rss_appl_inf->news_hbox = news_hbox;

    gtk_box_pack_start(GTK_BOX(applet_vbox), news_hbox, FALSE, FALSE, 0);
    gtk_widget_show(news_hbox);

    /* Creating scrolled window for displaying news items */
    GtkWidget *scrolledwindow;
    scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
    rss_appl_inf->scrolledwindow = scrolledwindow;
    gtk_box_pack_start(GTK_BOX(news_hbox), scrolledwindow, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
                                   GTK_POLICY_NEVER, GTK_POLICY_NEVER);


    /* Creation empty label (displayed when no headlines are present) */
    GtkWidget *empty_label = gtk_label_new(_("rss_ia_no_posts"));
    rss_appl_inf->empty_label = empty_label;
    g_object_ref(empty_label);
    gtk_box_pack_start(GTK_BOX(news_hbox), empty_label, TRUE, TRUE, 0);

    create_headline_display(rss_appl_inf);
    create_buttons(rss_appl_inf);

    g_signal_connect(applet_frame, "expose-event",
                     G_CALLBACK (expose),
                     NULL);

    gtk_widget_show_all(applet_frame);

    rss_appl_inf->displayed_lines = 1;
    _applet_gconf_init(rss_appl_inf);

    {
        /* Check the gconf value at startup in case the applet was
         * started during the refresh operation.
         */
        gint refresh_state =
            gconf_client_get_int(rss_appl_inf->client, REFRESH_STATE, NULL);
        rss_appl_inf->refreshing = (refresh_state == 1);
        redraw_refresh_button();
    }

    _applet_dbus_connect(rss_appl_inf,
                         (DBusHandleMessageFunction)
                         _applet_handle_signal_cb);

    Unselect_all();

    g_signal_connect(G_OBJECT(applet_frame), "size-allocate",
                     G_CALLBACK(realize), rss_appl_inf);
    gtk_widget_set_size_request(applet_frame, RSS_APPLET_WIDTH,
                                RSS_APPLET_HEIGHT);

    update_font_width(rss_appl_inf);
    applet_check_animation(rss_appl_inf);
}

void
create_csm(AppletInfo * rss_appl_inf)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkMenu *context_menu = GTK_MENU(gtk_menu_new());

    GtkWidget *menuitem = gtk_menu_item_new_with_label(_("rss_me_settings"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem);
    g_signal_connect((gpointer) menuitem, "activate",
                     G_CALLBACK(on_menuitem_settings),
                     rss_appl_inf->applet_frame);

    GtkWidget *menuitem_separator = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_separator);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem_separator);
    gtk_widget_set_sensitive(menuitem_separator, FALSE);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_close"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem);
    g_signal_connect((gpointer) menuitem, "activate",
                     G_CALLBACK(on_menuitem_close),
                     rss_appl_inf->applet_frame);

    gtk_widget_tap_and_hold_setup(GTK_WIDGET(rss_appl_inf->rss_news),
                                  GTK_WIDGET(context_menu),
                                  NULL,  0);
}

void
redraw_refresh_button()
{
    ULOG_DEBUG(__FUNCTION__);

    if (rss_appl_inf->refreshing)
    {
        /* This souldn't be needed -- rss_image/rss_image_stop seems to be
         * overwritten by set_image */
        rss_appl_inf->rss_image_stop =
            gtk_image_new_from_icon_name("qgn_toolb_gene_stop",
                                         GTK_ICON_SIZE_INVALID);
        gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->rss_image_stop), ICON_SIZE);
        gtk_button_set_image(GTK_BUTTON(rss_appl_inf->rss_refresh_button),
                             rss_appl_inf->rss_image_stop);
    }
    else
    {
        /* This souldn't be needed -- rss_image/rss_image_stop seems to be
         * overwritten by set_image */
        rss_appl_inf->rss_image =
            gtk_image_new_from_icon_name("qgn_toolb_gene_refresh",
                                         GTK_ICON_SIZE_INVALID);
        gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->rss_image), ICON_SIZE);
        gtk_button_set_image(GTK_BUTTON(rss_appl_inf->rss_refresh_button),
                             rss_appl_inf->rss_image);
    }
}

void
redraw_refresh_button_pressed()
{
    ULOG_DEBUG(__FUNCTION__);

    if (rss_appl_inf->refreshing)
    {
        /* This souldn't be needed -- rss_image/rss_image_stop seems to be
         * overwritten by set_image */
        rss_appl_inf->rss_image_stop =
            gtk_image_new_from_icon_name("qgn_toolb_gene_stop_pressed",
                                         GTK_ICON_SIZE_INVALID);
        gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->rss_image_stop), ICON_SIZE);
        gtk_button_set_image(GTK_BUTTON(rss_appl_inf->rss_refresh_button),
                             rss_appl_inf->rss_image_stop);
    }
    else
    {
        /* This souldn't be needed -- rss_image/rss_image_stop seems to be
         * overwritten by set_image */
        rss_appl_inf->rss_image =
            gtk_image_new_from_icon_name("qgn_toolb_gene_refresh_pressed",
                                         GTK_ICON_SIZE_INVALID);
        gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->rss_image), ICON_SIZE);
        gtk_button_set_image(GTK_BUTTON(rss_appl_inf->rss_refresh_button),
                             rss_appl_inf->rss_image);
    }
}

void
up_button_pressed_cb()
{
    ULOG_DEBUG(__FUNCTION__);
    rss_appl_inf->up_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_up_pressed",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->up_image), ICON_SIZE);
    gtk_button_set_image(GTK_BUTTON(rss_appl_inf->up_button),
                         rss_appl_inf->up_image);

    start_cont_scrolling((gpointer)TRUE);
}

void
up_button_released_cb()
{
    ULOG_DEBUG(__FUNCTION__);
    rss_appl_inf->up_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_up",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->up_image), ICON_SIZE);
    gtk_button_set_image(GTK_BUTTON(rss_appl_inf->up_button),
                         rss_appl_inf->up_image);

    /* If long_buttonpress_timeout is present, continous scrolling is not
     * started yet, killing timer, and emulating click */
    if (rss_appl_inf->long_buttonpress_timeout)
    {
        g_source_remove(rss_appl_inf->long_buttonpress_timeout);
        rss_appl_inf->long_buttonpress_timeout = 0;
        up_button_clicked_cb(NULL, NULL);
    }

    /* If continous scrolling is going on, killing it */
    if (rss_appl_inf->cont_scroll_timeout)
    {
        g_source_remove(rss_appl_inf->cont_scroll_timeout);
        rss_appl_inf->cont_scroll_timeout = 0;
    }

}

void
down_button_pressed_cb()
{
    ULOG_DEBUG(__FUNCTION__);
    rss_appl_inf->down_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_down_pressed",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->down_image), ICON_SIZE);
    gtk_button_set_image(GTK_BUTTON(rss_appl_inf->down_button),
                         rss_appl_inf->down_image);

    start_cont_scrolling((gpointer)FALSE);
}

void
down_button_released_cb()
{
    ULOG_DEBUG(__FUNCTION__);
    rss_appl_inf->down_image =
        gtk_image_new_from_icon_name("qgn_applet_rss_applet_down",
                                     GTK_ICON_SIZE_INVALID);
    gtk_image_set_pixel_size(GTK_IMAGE(rss_appl_inf->down_image), ICON_SIZE);
    gtk_button_set_image(GTK_BUTTON(rss_appl_inf->down_button),
                         rss_appl_inf->down_image);

    /* If long_buttonpress_timeout is present, continous scrolling is not
     * started yet, killing timer, and emulating click */
    if (rss_appl_inf->long_buttonpress_timeout)
    {
        g_source_remove(rss_appl_inf->long_buttonpress_timeout);
        rss_appl_inf->long_buttonpress_timeout = 0;
        down_button_clicked_cb(NULL, NULL);
    }

    /* If continous scrolling is going on, killing it */
    if (rss_appl_inf->cont_scroll_timeout)
    {
        g_source_remove(rss_appl_inf->cont_scroll_timeout);
        rss_appl_inf->cont_scroll_timeout = 0;
    }
}

void
ui_show_save_nomem_dialog(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (save_nomem_dialog_visible==TRUE) {
        return;
    } else {
        HildonNote *note = HILDON_NOTE(hildon_note_new_information (GTK_WINDOW (gtk_widget_get_ancestor (GTK_WIDGET(applet->applet_frame), GTK_TYPE_WINDOW)),
                 dgettext("hildon-common-strings", "sfil_ni_not_enough_memory")));
        hildon_note_set_button_text(note, dgettext("hildon-common-strings", "sfil_ni_not_enough_memory_ok"));
        g_signal_connect(G_OBJECT(note), "key_press_event", G_CALLBACK(dialog_key_press), NULL);
        save_nomem_dialog_visible = TRUE; 
        gtk_dialog_run(GTK_DIALOG(note));
        gtk_widget_destroy(GTK_WIDGET(note));
        save_nomem_dialog_visible = FALSE; 
    }

}

void
ui_show_save_nodevicemem_dialog(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (save_nodevicemem_dialog_visible==TRUE) {
        return;
    } else {
        HildonNote *note = HILDON_NOTE(hildon_note_new_information (GTK_WINDOW (gtk_widget_get_ancestor (GTK_WIDGET(applet->applet_frame), GTK_TYPE_WINDOW)),
                 dgettext("ke-recv", "cerm_device_memory_full")));
        hildon_note_set_button_text(note, dgettext("hildon-common-strings", "sfil_ni_not_enough_memory_ok"));
        g_signal_connect(G_OBJECT(note), "key_press_event",
                     G_CALLBACK(dialog_key_press), NULL);
        save_nodevicemem_dialog_visible = TRUE; 
        gtk_dialog_run(GTK_DIALOG(note));
        gtk_widget_destroy(GTK_WIDGET(note));
        save_nodevicemem_dialog_visible = FALSE; 
    }
}

/***************************************************************************/
/* UI / Callbacks handling -- Headline list handling */
/***************************************************************************/

void
Launch_item(GtkTreeModel * model, gpointer user_data)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreeIter iter;
    GtkTreeSelection *selection = NULL;
    gchar *feed = NULL;
    gchar *url = NULL;
    guint32 itemnr = 0;
    selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
    if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
        gtk_tree_model_get(model, &iter, FEED_COLUMN, &feed,
                           ITEMNR_COLUMN, &itemnr, -1);
        gtk_tree_model_get(model, &iter, URL_COLUMN, &url,
                           ITEMNR_COLUMN, &itemnr, -1);
        if (_applet_dbus_display(user_data, feed, url, itemnr) == FALSE)
        {
            ULOG_ERR("Sending DBus message to display news item failed");
        }
        g_free(feed);
    }
    else
    {
        ULOG_DEBUG("None is selected");
    }
}

/** Shows or hides the empty label or the news.
  * @param show if the empty label should be shown or not
  */
static void
_applet_show_empty_label(AppletInfo * applet, gboolean show)
{
    ULOG_DEBUG(__FUNCTION__);

    if (show)
    {
        gtk_widget_hide(applet->scrolledwindow);
        gtk_widget_show(applet->empty_label);
    }
    else
    {
        gtk_widget_hide(applet->empty_label);
        gtk_widget_show(applet->scrolledwindow);

        if (GTK_WIDGET_REALIZED(rss_appl_inf->rss_news))
        {
            gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                          0, 0);
        }
    }
}

/** Set the list of feed headlines for the applet
  *
  * @param applet the applet structure
  */
static void
_applet_set_headlines(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    GList *items = NULL;
    GList *list_iter = NULL;
    GtkTreeIter iter;

    GdkPixbuf *pixbuf = NULL;
    GdkPixbuf *default_pixbuf = NULL;
    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();

    HeadLine *headline = NULL;
    gchar *item_title = NULL;

    int num = 0;
    rss_appl_inf->num_of_headlines = 0;

    g_assert(NULL != applet);
    g_assert(NULL != applet->headlines);

    items = feed_load_all(&(applet->updated));

    default_pixbuf = gtk_icon_theme_load_icon(icon_theme, RSS_ICON_ITEM,
                                              HILDON_ICON_PIXEL_SIZE_SMALL, 0,
                                              NULL);

    gtk_tree_store_clear(applet->headlines);

    if (NULL != items) list_iter = g_list_first(items);

    while (NULL != list_iter)
    {
        headline = (HeadLine *) list_iter->data;
        if (headline)
        {
            num++;
            if (num <= MAX_HEADLINES)
            {
               /* Processing headline information */
                if (headline->title)
                {
                    item_title = g_markup_escape_text(headline->title, -1);
                    item_title = g_strstrip(item_title);
                    item_title = g_strdelimit(item_title, "\r\n\t", ' ');
                    gint brk = calc_breaking_position(applet, item_title);
                    ULOG_DEBUG("The string: %s", item_title);
                    item_title = put_newline(item_title, brk);
                    ULOG_DEBUG("wrapped as:\n>%s<", item_title);
                }
                else
                {
                    item_title = g_strdup(" ");
                }

                if (headline->icon == NULL)
                    pixbuf = default_pixbuf;
                else
                    pixbuf = headline->icon;

                /* Adding the title */
                ULOG_DEBUG("adding title %d %s", num, item_title);

                /* If displayed_headlines <= MAX_HEADLINES, animation needed. 
                 * Appending first few items to end of list, to enable smooth
                 * wraparound */
                if (num <= applet->displayed_headlines + 1
                    && MAX_HEADLINES >= applet->displayed_headlines)
                {
                    gtk_tree_store_append(applet->headlines, &iter, NULL);
                    gtk_tree_store_set(applet->headlines, &iter,
                                       ICON_COLUMN, pixbuf,
                                       URL_COLUMN, headline->url,
                                       RENDER_COLUMN, item_title,
                                       TITLE_COLUMN, item_title,
                                       FEED_COLUMN, headline->feed,
                                       ITEMNR_COLUMN, headline->nr, -1);
                }

                /* Inserting headline to list; appended headlines will
                 * appear in list twice */
                gtk_tree_store_insert(applet->headlines, &iter, NULL,
                                      num - 1);
                gtk_tree_store_set(applet->headlines, &iter,
                                   ICON_COLUMN, pixbuf,
                                   URL_COLUMN, headline->url,
                                   RENDER_COLUMN, item_title,
                                   TITLE_COLUMN, item_title,
                                   FEED_COLUMN, headline->feed,
                                   ITEMNR_COLUMN, headline->nr, -1);
                g_free(item_title);

                rss_appl_inf->num_of_headlines = num;
            }
        }
        list_iter = g_list_next(list_iter);
    }

    /* All the items where leaking memory. This propably fixes those leaks  */

    if (items != NULL ) list_iter = g_list_first(items);
    while (NULL != list_iter)
    {
        headline = (HeadLine *) list_iter->data;
        if (headline)
        {
            freeHeadLine(headline);
        }
        list_iter = g_list_next(list_iter);
    }

    g_list_free(items);

    if (NULL != default_pixbuf) g_object_unref(default_pixbuf);

    /* Showing empty label or list */
    _applet_show_empty_label(rss_appl_inf,
                             0 == rss_appl_inf->num_of_headlines);
}

/** This is called after the refresh is completed in the main application
  *
  * @param applet the applet structure
  */
static void
_applet_refresh_completed(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(applet != NULL);
    g_assert(applet->rss_image != NULL);

    if (applet->refreshing)
    {
        applet->refreshing = FALSE;
        redraw_refresh_button();
    }

     _applet_set_headlines(applet);

     applet_check_animation(applet);
}

GtkTreePath *get_tree_view_last_node_path(GtkTreeView *treeview)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreeIter iter;
    GtkTreeModel *model = gtk_tree_view_get_model(treeview);
    GtkTreePath *path = NULL;

    gtk_tree_model_iter_nth_child(model, &iter, NULL, 
                                  rss_appl_inf->num_of_headlines +
                                    rss_appl_inf->displayed_headlines-1);
    path = gtk_tree_model_get_path(model, &iter);

    return path;
}

/** A function to handle cycling up through the tree view with the keyboard
  * @param tr the tree view to cycle
  * @return TRUE on success, FALSE on failure
  */
static gboolean
_applet_tree_view_cycle_up(GtkWidget * tr)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreeIter iter;
    GtkTreePath *path = NULL, *root_path = NULL;
    GtkTreeModel *model =
        gtk_tree_view_get_model(GTK_TREE_VIEW(rss_appl_inf->rss_news));

    if (tr == NULL)
    return FALSE;

    gtk_tree_view_get_cursor(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                             &path, NULL);

    if (path == NULL)
    return FALSE;

    stop_animation();

    root_path = gtk_tree_path_new_root();
    if (gtk_tree_path_compare(path, root_path) == 0) 
    {
        gtk_tree_path_free(root_path);
        gtk_tree_path_free(path);
        gtk_tree_model_iter_nth_child(model, &iter, NULL, 
                                      rss_appl_inf->num_of_headlines - 1);
        path = gtk_tree_model_get_path(model, &iter);

        gtk_tree_view_set_cursor(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                               path, NULL, FALSE);
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                     path, NULL, TRUE, 0.0, 0.0);

        gtk_tree_path_free(path);
        restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);
        return TRUE;
    }

    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(rss_appl_inf->rss_news), path,
                                 NULL, FALSE, 0.0, 0.0);

    restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);

    return FALSE;
}


/** A function to handle cycling down through the tree view with the keyboard
  * @param tr the tree view to cycle
  * @return TRUE on success, FALSE on failure
  */
static gboolean
_applet_tree_view_cycle_down(GtkWidget * tr)
{
    ULOG_DEBUG(__FUNCTION__);

    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    GtkTreeModel *model = 
        gtk_tree_view_get_model(GTK_TREE_VIEW(rss_appl_inf->rss_news));

    if (tr == NULL)
    return FALSE;

    gtk_tree_view_get_cursor(GTK_TREE_VIEW(tr), &path, NULL);
    ULOG_DEBUG("get_cursor returned: %s", gtk_tree_path_to_string(path));

    if (path == NULL)
    return FALSE;

    stop_animation();

    if (gtk_tree_path_compare(path, 
            get_tree_view_last_node_path(GTK_TREE_VIEW(tr))) == 0)
    {
        ULOG_DEBUG("looping needed");
        gtk_tree_model_iter_nth_child(model, &iter, NULL, 
                                      rss_appl_inf->displayed_headlines);
        path = gtk_tree_model_get_path(model, &iter);
        gtk_tree_view_set_cursor(GTK_TREE_VIEW(tr), path, NULL, FALSE);
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(tr), path,
                                     NULL, TRUE, 1.0, 1.0);
        gtk_tree_path_free(path);

        restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);
        return TRUE;
    }
    ULOG_DEBUG("no looping needed");

    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(tr), path,
                                 NULL, FALSE, 0.0, 0.0);

    restart_animation(ANIMATION_TO_DOWN_WAIT_ON_USER_ACTION);

    return FALSE;
}

static gboolean
expose(GtkWidget *widget, GdkEventExpose *event)
{
    ULOG_DEBUG(__FUNCTION__);

    if (GTK_WIDGET_VISIBLE(widget))
    {
        gtk_paint_box(widget->style,
                      widget->window,
                      GTK_STATE_NORMAL,
                      GTK_SHADOW_NONE,
                      &event->area,
                      widget,
                      NULL,
                      widget->allocation.x,
                      widget->allocation.y,
                      widget->allocation.width,
                      widget->allocation.height);

        GTK_WIDGET_CLASS(
            g_type_class_peek_parent(
                GTK_FRAME_GET_CLASS(widget)))->expose_event(widget, event);
        return TRUE;
    }

    return FALSE;
}


/***************************************************************************/
/* Animation handling */
/***************************************************************************/


gboolean
Start_TW_Scroll_again(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    Scroll_Down_TW(NULL);
    if (!rss_appl_inf->scroll_down_tw_timeout && rss_appl_inf->animation)
    {
        rss_appl_inf->scroll_down_tw_timeout =
            g_timeout_add(ANIMATION_TO_DOWN_TIMEOUT, Scroll_Down_TW, NULL);
    }
    rss_appl_inf->scroll_down_tw_starter_timeout = 0;
    return FALSE;
}

gboolean
Scroll_Down_TW(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    if (!GTK_WIDGET_REALIZED(rss_appl_inf->rss_news))
    {
        rss_appl_inf->scroll_down_tw_timeout = 0;
        return FALSE;
    }

    GdkRectangle *visible_rect = g_new0(GdkRectangle, 1);
    gtk_tree_view_get_visible_rect(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                   visible_rect);

    visible_rect->y = 
        ((visible_rect->y/2)*2 + ANIMATION_TO_DOWN) 
            % rss_appl_inf->tree_height;
    gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(rss_appl_inf->rss_news),
                                  -1, visible_rect->y);

    if (!rss_appl_inf->scroll_down_tw_starter_timeout
        && rss_appl_inf->animation)
    {
        /* End of the list; jumping back to top, stopping */
        if (visible_rect->y % rss_appl_inf->tree_height == 0)
        {
            rss_appl_inf->scroll_down_tw_starter_timeout =
                g_timeout_add(ANIMATION_TO_DOWN_WAIT_ON_END,
                              Start_TW_Scroll_again, NULL);
            rss_appl_inf->scroll_down_tw_timeout = 0;

            g_free(visible_rect);

            return FALSE;
        }
        /* New headline; stopping */
        if (visible_rect->y % (HEADLINE_HEIGHT + rss_appl_inf->space_between_rows) == 0)
        {
            rss_appl_inf->scroll_down_tw_starter_timeout =
                g_timeout_add(ANIMATION_TO_DOWN_WAIT, Start_TW_Scroll_again,
                              NULL);
            rss_appl_inf->scroll_down_tw_timeout = 0;

            g_free(visible_rect);

            return FALSE;
        }

        g_free(visible_rect);
        return TRUE;
    }
    else
    {
        g_free(visible_rect);
        return TRUE;
    }
}

void
stop_animation()
{
    ULOG_DEBUG(__FUNCTION__);

    if (rss_appl_inf->scroll_down_tw_timeout)
    {
        g_source_remove(rss_appl_inf->scroll_down_tw_timeout);
        rss_appl_inf->scroll_down_tw_timeout = 0;
    }
    if (rss_appl_inf->scroll_down_tw_starter_timeout)
    {
        g_source_remove(rss_appl_inf->scroll_down_tw_starter_timeout);
        rss_appl_inf->scroll_down_tw_starter_timeout = 0;
    }
}

void
restart_animation(gdouble timeout)
{
    ULOG_DEBUG(__FUNCTION__);

    ULOG_DEBUG("restart_animation called, applet backgrounded: %d "
               "timeout: %d, animation: %d", 
               rss_appl_inf->backgrounded,
               rss_appl_inf->scroll_down_tw_starter_timeout,
               rss_appl_inf->animation);

    if (!rss_appl_inf->scroll_down_tw_starter_timeout
        && rss_appl_inf->animation && !rss_appl_inf->backgrounded && GTK_WIDGET_REALIZED(rss_appl_inf->rss_news))
    {
        rss_appl_inf->scroll_down_tw_starter_timeout =
            g_timeout_add(timeout,
                          Start_TW_Scroll_again, NULL);
    }
}

void
applet_check_animation(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    if (rss_appl_inf->num_of_headlines >= rss_appl_inf->displayed_headlines)
    {
        ULOG_DEBUG("scrollbar visible");
        gtk_widget_show(rss_appl_inf->scrollbar);

        gtk_widget_set_sensitive(rss_appl_inf->down_button, TRUE);
        gtk_widget_set_sensitive(rss_appl_inf->up_button, TRUE);

        /* Animation needs to be restarted but only if it's switched on in
         * setup */
        rss_appl_inf->animation =
            gconf_client_get_bool(rss_appl_inf->client,
                                  RSS_SETTINGS_AUTOMATIC_SCROLLING, NULL);
    }
    else
    {
        ULOG_DEBUG("invisible");
        gtk_widget_hide(rss_appl_inf->scrollbar);

        gtk_widget_set_sensitive(rss_appl_inf->down_button, FALSE);
        gtk_widget_set_sensitive(rss_appl_inf->up_button, FALSE);

        rss_appl_inf->animation = FALSE;
    }

    if (rss_appl_inf->animation)
    {
        ULOG_DEBUG("animation on -- restarting");
        restart_animation(ANIMATION_TO_DOWN_WAIT);
    }
    else
    {
        ULOG_DEBUG("animation on -- stopping");
        stop_animation();
    }
}

/**************************************************************************
 * Continous scrolling
 **************************************************************************/

gboolean start_cont_scrolling(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    stop_animation();

    if (!rss_appl_inf->long_buttonpress_timeout)
    {
        rss_appl_inf->long_buttonpress_timeout =
            g_timeout_add(LONG_BUTTON_PRESS_TIMEOUT_INTERVAL,
                          do_cont_scrolling, data);
    }

    return FALSE;
}

gboolean do_cont_scrolling(gpointer data)
{
    ULOG_DEBUG(__FUNCTION__);

    gboolean direction = (gboolean)data;

    if (rss_appl_inf->long_buttonpress_timeout)
    {
        g_source_remove(rss_appl_inf->long_buttonpress_timeout);
        rss_appl_inf->long_buttonpress_timeout = 0;
    }

    if (rss_appl_inf->cont_scroll_timeout)
    {
        g_source_remove(rss_appl_inf->cont_scroll_timeout);
        rss_appl_inf->cont_scroll_timeout = 0;
    }

    if (direction)
    {
        up_button_clicked_cb(NULL, NULL);
    }
    else
    {
        down_button_clicked_cb(NULL, NULL);
    }

    rss_appl_inf->cont_scroll_timeout =
        g_timeout_add(CONT_SCROLL_TIMEOUT_INTERVAL,
                      do_cont_scrolling, data);

    return FALSE;
}
