/**
  @file util.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "util.h"

#ifdef DEBUG
char *bytestr(const unsigned char *data, int len)
{
    int i;
    char *str = g_malloc((len << 1) + 1);

    for (i = 0; i < len; i++)
        sprintf(str + (2*i), "%02X", data[i]);

    return str;
}

char *chomp(char *str)
{
    char *nl;

    nl = strchr(str, '\n');
    if (nl)
        *nl = '\0';

    return str;
}
#endif

/* From Imendio's GnomeVFS OBEX module (om-utils.c) */
time_t parse_iso8601(const gchar *str, int len)
{
    gchar    *tstr;
    struct tm tm;
    gint      nr;
    gchar     tz;
    time_t    time;
    time_t    tz_offset = 0;

    memset (&tm, 0, sizeof (struct tm));

    /* According to spec the time doesn't have to be null terminated */
    if (str[len - 1] != '\0') {
        tstr = g_malloc(len + 1);
        strncpy(tstr, str, len);
        tstr[len] = '\0';
    }
    else
        tstr = g_strdup(str);

    nr = sscanf (tstr, "%04u%02u%02uT%02u%02u%02u%c",
            &tm.tm_year, &tm.tm_mon, &tm.tm_mday,
            &tm.tm_hour, &tm.tm_min, &tm.tm_sec,
            &tz);

    g_free(tstr);

    /* Fixup the tm values */
    tm.tm_year -= 1900;       /* Year since 1900 */
    tm.tm_mon--;              /* Months since January, values 0-11 */
    tm.tm_isdst = -1;         /* Daylight savings information not avail */

    if (nr < 6) {
        /* Invalid time format */
        return -1;
    }

    time = mktime (&tm);

#if defined(HAVE_TM_GMTOFF)
    tz_offset = tm.tm_gmtoff;
#elif defined(HAVE_TIMEZONE)
    tz_offset = -timezone;
    if (tm.tm_isdst > 0) {
        tz_offset += 3600;
    }
#endif

    if (nr == 7) { /* Date/Time was in localtime (to remote device)
                    * already. Since we don't know anything about the
                    * timezone on that one we won't try to apply UTC offset
                    */
        time += tz_offset;
    }

    return time;
}

char *utf16_to_utf8(const unsigned char *utf16, size_t len)
{
    return g_convert((char*)utf16, len, "UTF8", "UTF16BE", NULL, NULL, NULL);
}

