/**
  @file client.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>

#include <openobex/obex.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "dbus.h"
#include "obex.h"
#include "client.h"

#define OBEXSRV_RX_MTU 32767
#define OBEXSRV_TX_MTU 32767

typedef struct {
    client_dc_cb_t func;
    gpointer       data;
} ClientDisconnectData;

struct client_connection {
    ServerContext       *server;
                        
    obex_t              *obex;

    GIOChannel          *io;
    guint                io_id;
    struct sockaddr_rc   addr;
                        
    char                 addr_str[18];
                        
    GSList              *dc_listeners;
                        
    uint16_t             tx_max;
    uint16_t             rx_max;
                        
    ObjectTransfer      *xfer;

    client_auth_cb_t     auth_cb;
    gpointer             auth_data;
    GFreeFunc            auth_data_free;

    client_progr_cb_t    progr_cb;
    gpointer             progr_data;
    GFreeFunc            progr_data_free;

    client_xfer_cb_t     xfer_cb;
    gpointer             xfer_data;
    GFreeFunc            xfer_data_free;
};

static gboolean client_callback(GIOChannel *io, GIOCondition cond,
                                ClientConnection *client)
{
    if ((cond & G_IO_ERR) || (cond & G_IO_HUP)) { 
        GSList *l;

        report("Disconnected from %s", client->addr_str);

	client_transfer_set_success(client, FALSE);
	client_transfer_complete(client);

        for (l = client->dc_listeners; l != NULL; l = l->next) {
            ClientDisconnectData *data = l->data;
            data->func(client, data->data);
        }

        client_close(client);

        return FALSE;
    }

    /* Sanity check */
    if (!(cond & G_IO_IN)) {
        debug("no error or hup, but also no input!?");
        return TRUE;
    }

    OBEX_HandleInput(client->obex, 0);

    return TRUE;
}

static gboolean client_root_progr(ClientConnection *client,
                                ObjectTransfer   *xfer,
                                gpointer          user_data)
{
    debug("Progress: %d/%d", xfer->counter, xfer->length);
    return TRUE;
}

static void client_root_xfer(ClientConnection *client,
                             ObjectTransfer   *xfer,
                             gboolean          started,
                             gboolean          success,
                             gpointer          user_data)
{
    debug("Transfer: %s %s", success ? "successful" : "failed",
            started ? "start" : "completion");
}


static char *client_root_auth(ClientConnection *client,
                              ObjectTransfer   *xfer,
                              char             *root)
{
    if (!xfer->name)
        return NULL;

    return g_strdup_printf("%s/%s", root, xfer->name);
}

void client_add_disconnect_callback(ClientConnection *client,
                                    client_dc_cb_t cb, gpointer user_data)
{
    ClientDisconnectData *data;

    data = g_new(ClientDisconnectData, 1);

    data->func = cb;
    data->data = user_data;

    client->dc_listeners = g_slist_append(client->dc_listeners, data);
}

void client_remove_disconnect_callback(ClientConnection *client,
                                       client_dc_cb_t cb, gpointer user_data)
{
    GSList *l;
    ClientDisconnectData *match = NULL;

    for (l = client->dc_listeners; l != NULL; l = l->next) {
        ClientDisconnectData *d = l->data;

        if (d->func == cb && d->data == user_data) {
            match = d;
            break;
        }
    }

    if (match) {
        client->dc_listeners = g_slist_remove(client->dc_listeners, match);
        g_free(match);
    }
}

void client_transfer_start(ClientConnection *client)
{
    ObjectTransfer *xfer = client->xfer;

    g_assert(xfer);

    if (client->xfer_cb)
        client->xfer_cb(client, xfer, TRUE, xfer->success, client->xfer_data);
}

void client_transfer_complete(ClientConnection *client)
{
    ObjectTransfer *xfer = client->xfer;

    if (!xfer)
        return;

    if (client->xfer_cb && xfer->filename)
        client->xfer_cb(client, xfer, FALSE, xfer->success, client->xfer_data);

    if (client->xfer->fd >= 0)
        close(client->xfer->fd);

    g_free(client->xfer->name);
    g_free(client->xfer->type);
    g_free(client->xfer->dscr);
    g_free(client->xfer->buf);
    g_free(client->xfer->filename);
    g_free(client->xfer);
    
    client->xfer = NULL;
}

char *client_authorize(ClientConnection *client)
{
    ObjectTransfer *xfer = client->xfer;

    g_assert(xfer);

    if (!client->auth_cb)
        return NULL;

    return client->auth_cb(client, xfer, client->auth_data);
}

gboolean client_progress(ClientConnection *client)
{
    ObjectTransfer *xfer = client->xfer;

    if (!xfer)
        return TRUE;

    if (!client->progr_cb)
        return TRUE;

    if (!xfer->filename)
        return TRUE;

    return client->progr_cb(client, xfer, client->progr_data);
}

void client_set_tx_max(ClientConnection *client, uint16_t size)
{
    client->tx_max = size;
}

void client_transfer_set_success(ClientConnection *client, gboolean success)
{
    if (!client->xfer)
        return;

    client->xfer->success = success;
}

ObjectTransfer *client_get_transfer(ClientConnection *client, uint8_t cmd)
{
    if (client->xfer) {
        if (client->xfer->cmd != cmd)
            return NULL;
        return client->xfer;
    }

    client->xfer = g_new0(ObjectTransfer, 1);

    switch (cmd) {
        case OBEX_CMD_GET:
            client->xfer->buf_size = client->tx_max;
            break;
        case OBEX_CMD_PUT:
            client->xfer->buf_size = client->rx_max;
            /* Assume delete until we have gotten a body header */
            client->xfer->is_delete = TRUE;
            break;
        default:
            g_assert_not_reached();
    }

    client->xfer->cmd = cmd;
    client->xfer->buf = g_malloc(client->xfer->buf_size);
    client->xfer->fd = -1;
    client->xfer->success = TRUE;

    return client->xfer;
}

gboolean client_allow_target(ClientConnection *client, const unsigned char *target,
                                unsigned int length)
{
    return server_has_target(client->server, target, length);
}

const char *client_get_bt_address(ClientConnection *client)
{
    return client->addr_str;
}

ClientConnection *client_new(ServerContext *server, int fd, struct sockaddr_rc *addr,
                                const char *root)
{
    ClientConnection *client;

    client = g_new0(ClientConnection, 1);
    client->io = g_io_channel_unix_new(fd);
    g_io_channel_set_close_on_unref(client->io, TRUE);

    ba2str(&addr->rc_bdaddr, client->addr_str);
    report("New connection from %s", client->addr_str);

    memcpy(&client->addr, addr, sizeof(struct sockaddr_rc));

    client->obex = OBEX_Init(OBEX_TRANS_FD, obex_event_handler, 0);
    if (client->obex == NULL) {
        error("OBEX_Init() failed");
        goto disconnect;
    }

    OBEX_SetCustomData(client->obex, client);

    (void) OBEX_SetTransportMTU(client->obex, OBEXSRV_RX_MTU, OBEXSRV_TX_MTU);

    if (FdOBEX_TransportSetup(client->obex, fd, fd, 0) < 0) {
        error("OBEX can't setup transport\n");
        OBEX_Cleanup(client->obex);
        goto disconnect;
    }       

    client->rx_max = OBEXSRV_RX_MTU;

    client->server = server;

    if (root) {
        client->auth_cb = (client_auth_cb_t)client_root_auth;
        client->auth_data = g_strdup(root);
        client->auth_data_free = g_free;
        client->progr_cb = (client_progr_cb_t)client_root_progr;
        client->xfer_cb = (client_xfer_cb_t)client_root_xfer;
    }
    else {
        client->auth_cb = (client_auth_cb_t)srv_dbus_auth;
        client->progr_cb = (client_progr_cb_t)srv_dbus_progr;
        client->xfer_cb = (client_xfer_cb_t)srv_dbus_xfer;
    }

    client->io_id = g_io_add_watch(client->io, G_IO_IN | G_IO_ERR | G_IO_HUP,
                                   (GIOFunc)client_callback, client);

    return client;

disconnect:
    g_io_channel_unref(client->io);
    g_free(client);
    return NULL;
}

gboolean client_close(ClientConnection *client)
{
    OBEX_TransportDisconnect(client->obex);
    OBEX_Cleanup(client->obex);

    g_source_remove(client->io_id);
    g_io_channel_shutdown(client->io, FALSE, NULL);
    g_io_channel_unref(client->io);

    if (client->auth_data && client->auth_data_free)
        client->auth_data_free(client->auth_data);

    if (client->progr_data && client->progr_data_free)
        client->progr_data_free(client->progr_data);

    g_free(client);

    return TRUE;
}

