/* Function for controlling osso gps. This file is here so that
 * we avoid circular dependency with libgpsbt.
 */

/*
GPS control functions from gpsctrl.c. Put here so that we
can avoid circular package dependency.

Copyright (C) 2007 Nokia Corporation. All rights reserved.

Author: Jukka Rissanen <jukka.rissanen@nokia.com>

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
The name of the author may not be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include <stdio.h>
#include <errno.h>
#include <errno.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "osso-gps.h"

#define MAX_REPLY_LEN 255
#define MAX_SMALL_MSG 32
#define MSG_REPORTING_INTERVAL "I"
#define MSG_FAILURE            "ERROR"
#define GPSCTRL_CONTROL_SOCKET "/var/lib/gps/gps_driver_ctrl"
#define MIN(a,b) (((a)>(b)) ? (b) : (a))

static int debug_level = 1;
int *osso_debug_level = &debug_level;

//#define DEBUG

#ifdef DEBUG
#if (__GNUC__ > 2) && ((__GNUC__ > 3) || (__GNUC_MINOR__ > 2))
#define PDEBUG(fmt...)							\
	do {								\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#else
#define PDEBUG(fmt...)							\
	do {								\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(##fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#endif
#else
#define PDEBUG(fmt...)
#endif



/* ----------------------------------------------------------------------- */
static int gpsctrl_check_internal_chip(void)
{
	/* Check from /sys that we are having a GPS chip */

#ifdef TEST_GPSCTRL
	return internal_chip_status;
#else
	int st = 0;
	struct stat buf;
	char *file = "/dev/pgps";

	if (stat(file, &buf)<0) {
		if (errno == EACCES) {
			/* no access but dev is there */
			st = 1;
		} else {
			st = 0;
		}
	} else {
		if (S_ISCHR(buf.st_mode))
			st = 1;
		else
			st = 0;
	}

	return st;
#endif
}


/* ----------------------------------------------------------------------- */
static inline int open_ctrl_socket(char *file)
{
	int fd, st;
	struct sockaddr_un srvaddr;

	fd = socket(AF_LOCAL, SOCK_STREAM, 0);
	if (fd<0) {
		PDEBUG("Cannot create socket for %s [%s, %d]\n", file, strerror(errno), errno);
		return -1;
	}

	memset(&srvaddr, 0, sizeof(srvaddr));
	srvaddr.sun_family = AF_LOCAL;
	strncpy(srvaddr.sun_path, file, sizeof(srvaddr.sun_path));

	if (connect(fd, (struct sockaddr *)&srvaddr, sizeof(srvaddr))<0) {
		PDEBUG("Cannot connect to %s [%s, %d]\n", file, strerror(errno), errno);
		st = -1;
	} else
		st = fd;

	return st;
}


static inline int close_ctrl_socket(int fd)
{
	return close(fd);
}


/* ----------------------------------------------------------------------- */
static int send_msg(char *msg, char *replybuf, int replylen)
{
	char *file = GPSCTRL_CONTROL_SOCKET;
	int fd, st = 0;

	if ((fd = open_ctrl_socket(file))<0) {
		PDEBUG("Cannot send message \"%s\" to %s [%s, %d]\n", msg, file, strerror(errno), errno);
		st = -1;
	} else {
		int len = strlen(msg);
		int ret;

		struct timeval tv;
		fd_set fdwset;

#define MAX_CTRL_WRITE_TIMEOUT 2
		tv.tv_sec = MAX_CTRL_WRITE_TIMEOUT;
		tv.tv_usec = 0;

		FD_ZERO(&fdwset);
		FD_SET(fd, &fdwset);

		ret = select(fd+1, NULL, &fdwset, NULL, &tv);
		if (ret < 0) {
			perror("select(write)");
			st = -1;
			goto OUT;
		} else if (ret==0) {
			/* timeout */
			st = -1;
			goto OUT;
		}

		if (!FD_ISSET(fd, &fdwset)) {
			/* not writable, error */
			perror("FD_ISSET(writable)");
			st = -1;
			goto OUT;
		}

		ret = write(fd, msg, len);

		//PDEBUG("Wrote %d bytes \"%s\"\n", ret, msg);

		if (ret<len) {
			PDEBUG("Cannot write all bytes (%d<%d) to %s [%s, %d]\n", ret, len, file, strerror(errno), errno);
			st = -1;
			goto OUT;
		} else {
			/* wait for answer but only for timeout seconds */
#define MAX_REPLY_LEN 255
			char reply[MAX_REPLY_LEN+1];
			fd_set fdrset;

#define MAX_CTRL_READ_TIMEOUT 5
			tv.tv_sec = MAX_CTRL_READ_TIMEOUT;
			tv.tv_usec = 0;

			FD_ZERO(&fdrset);
			FD_SET(fd, &fdrset);

			//PDEBUG("Waiting in select\n");

			ret = select(fd+1, &fdrset, NULL, NULL, &tv);
			if (ret < 0) {
				perror("select(read)");
				st = -1;
				goto OUT;
			} else if (ret==0) {
				/* timeout */
				st = -1;
				goto OUT;
			}

			if (!FD_ISSET(fd, &fdrset)) {
				/* not readable, error */
				perror("FD_ISSET(readable)");
				st = -1;
				goto OUT;
			}

			len = read(fd, reply, MAX_REPLY_LEN);

			//PDEBUG("Read %d bytes\n", len);

			if (len <= 0) {
				/* nothing could be read */
				st = -1;
				goto OUT;
			} else {
				//PDEBUG("Received \"%s\"\n", reply);
			}

			if (replybuf && replylen>0) {
				strncpy(replybuf, reply, MIN(replylen, len));
			}
		}
	}

OUT:
	if (fd>=0)
		close_ctrl_socket(fd);

	return st;
}


/* ----------------------------------------------------------------------- */
/* Only to be called from gpsd! */
extern int gpsctrl_for_gpsd_set_reporting_interval(int *val)
{
	char reply[MAX_REPLY_LEN+1];
	char msg[MAX_SMALL_MSG+1];
	int st = 0, ret, value;
	int reply_ival = -1;

	if (!val)
		return -1;

	value = *val;
	if (value < 0)
		return -1;

	if (gpsctrl_check_internal_chip()<=0) {
		/* no internal chip found */
		printf("Internal GPS chip not found.");
		return -1;
	}

	memset(reply, 0, MAX_REPLY_LEN);
	snprintf(msg, MAX_SMALL_MSG, MSG_REPORTING_INTERVAL " %d\n", value);

	ret = send_msg(msg, reply, MAX_REPLY_LEN);
	if (ret<0) {
		PDEBUG("Cannot set reporting interval of the chip [%s, %d]\n", strerror(errno), errno);
		st = -1;
		goto OUT;
	}

	/* parse the msg */
	if (sscanf(reply, MSG_REPORTING_INTERVAL " %d\n", &reply_ival)==1) {
		if (value != reply_ival) {
			PDEBUG("Invalid reporting interval (%d) returned from gps driver (should be %d)\n", reply_ival, value);
			st = -1;
			goto OUT;
		}
	} else {
		/* error */
		printf("%s\n", reply);
		st = -1;
	}

OUT:
	*val = value;
	return st;
}


