/* Copyright (c) 2006, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define DBUS_API_SUBJECT_TO_CHANGE

#include <libmodest-dbus-client/libmodest-dbus-client.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <string.h>
#include <log-functions.h>
#include "osso-log.h"
#include "ossoemailinterface.h"


/**
   This function will send a top D-Bus message to the email
   application with a list of gnomeVFS URI's as argutments.
 
   @param osso The library context as returned by #osso_initialize.
   @param list A GSList of strings -- URI's in GnomeVFS format.
   
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/
osso_return_t osso_email_files_email(osso_context_t * osso, GSList * list)
{
	gchar *args = NULL;
	gboolean ret = FALSE;

	d_log(LOG_D, "send as files to email  application \n");

	if ((osso == NULL) || (list == NULL)) {
		return OSSO_INVALID;
	}

	ret = libmodest_dbus_client_compose_mail (osso,
					       NULL, /*to*/
					       NULL, /*cc*/ 
					       NULL, /*bcc*/
					       NULL, /*subject*/
					       NULL, /*body*/
					       list  /*attachments*/);
	g_free(args);

	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}

osso_return_t osso_email_url_email(osso_context_t * osso, const gchar * url)
{
	gboolean ret = FALSE;
	d_log(LOG_D, "send url link to email application \n");


	if ((osso == NULL) || (url == NULL)) {
		return OSSO_INVALID;
	}

	ret = libmodest_dbus_client_compose_mail (osso, NULL /* to */, 
	NULL /* cc */, NULL /* bcc */, NULL /* subject */, url /* body */, 
	NULL /* attachments */);
	printf ("%s: ret = %d\n", __FUNCTION__, ret);
	
	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}


/**
   This function will cause the email application to be topped
   with an empty email to the specified recipients opened. This is
   implemented with the top D-BUS message to the email
   application.
 
   @param osso The library context as returned by #osso_initialize.
   @param to A GSList of mailto URI strings.
 
   @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
    #OSSO_INVALID if some parameter is invalid, #OSSO_RPC_ERROR if an error
    occured while sending the dbus message.
*/

osso_return_t osso_email_emailto_email(osso_context_t * osso, GSList * to)
{
	GSList *next = NULL;
	gchar *args = NULL;
	gchar *tmp = NULL;
	gboolean ret = FALSE;

	d_log(LOG_D, "send mailto parameters to email application \n");

	if ((osso == NULL) || (to == NULL) || (to->data == NULL)) {
		return OSSO_INVALID;
	}
/*
	if (osso_get_dbus_connection(osso) == NULL) {
		DLOG_OPEN("libosso");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return OSSO_INVALID;
	}
*/
	if (strncmp(to->data, "mailto:", strlen("mailto:")) != 0) {
		d_log(LOG_D, "Input doesn't start with 'mailto' string \n");
		return OSSO_INVALID;
	}
/*
	args = g_strconcat("--to=", (gchar *) (to->data), NULL);
*/
	args = g_strdup((gchar *) (to->data));
	
	if (args == NULL) {
		return OSSO_ERROR;
	}

	for (next = g_slist_next(to); next != NULL; next = g_slist_next(next)) {
		tmp = g_strconcat(args, ",", (gchar *) (next->data), NULL);
		g_free(args);
		args = tmp;
		if (args == NULL) {
			return OSSO_ERROR;
		}
	}

	ret = libmodest_dbus_client_mail_to (osso, args);

	g_free(args);
	
	if(ret) {
		return OSSO_OK;
	} else {
		return OSSO_ERROR;
	}
}


/**
 *  This function is called to get the email folder list to other applications
 *  (like Global search etc) 
 *
 *  @param osso The library context as returned by #osso_initialize.
 *  @return GSList of folder names (gchar*)or NULL in case of error. The items 
 *  should be freed with g_free() and the list should be freed with g_slist_free().
*/
GSList *osso_email_get_folder_list(osso_context_t * ctx)
{
	GSList *result = NULL;
	GList *iter = NULL;
	
	GList *modest_folders = NULL;
	gboolean ret = libmodest_dbus_client_get_folders (ctx,
					    &modest_folders);

	if (ret == FALSE) {
		osso_log (LOG_INFO, "Error during osso_email_get_folder_list");
		return NULL;
	}

	/*
	printf ("DEBUG: %s: GList=%p\n", __FUNCTION__, modest_folders);
	printf ("DEBUG: %s: GList length=%d\n", __FUNCTION__, g_list_length (modest_folders));
	*/

	for (iter = modest_folders; iter; iter = iter->next) {
		ModestFolderResult *item = (ModestFolderResult *) iter->data;
		if (item) {
			const gchar *folder_name = item->folder_name;
	
			if (folder_name) {
				/* printf ("DEBUG: %s: folder_name=%s\n", __FUNCTION__, folder_name); */
				result = g_slist_append (result, g_strdup (folder_name));
			}
			else {
				g_warning ("%s: ModestFolderResult::folder_name was NULL\n", __FUNCTION__);
			}
		}
	}
	
	modest_folder_result_list_free (modest_folders);

	return result;
}

