/*
 * @file nitro-upload.c
 *   
 * This file implements the upload operation to NITRO post-processing server.
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <glib.h>
#include <curl/curl.h>
#include <libosso.h>
#include <log-functions.h>
#include <osso-log.h>
#include <unistd.h>
#include <errno.h>

#include "nitro-upload.h"
#include "nitro-settings-file.h"

static gint http_resp_code = 0;

/**
  This function is a callback to read while uploading
  raw data to server

  @param void pointer is a file pointer
  @param size_t is the no of the bytes read from the buffer
  @param void pointer to the stream

*/
static size_t read_callback(void *ptr, size_t size, size_t nmemb, void *stream)
{
	size_t retcode;

	retcode = fread(ptr, size, nmemb, stream);

	return retcode;
}

#define SERVER_RESPONSE_SZ (2048)
gchar server_response[SERVER_RESPONSE_SZ];
static gint server_response_sz = 0;
/**
  Function to write the http response.

  @param buffer the buffer that needs to be written.
  @param size of the buffer to be written.
  @param nmemb size.
  @param userdata passed by fn callback.
  @return actual num of bytes written to file.
  */
static size_t nitro_parse_http_response(char *buffer, size_t size,
					size_t nmemb, void *userdata)
{
    size_t len = size*nmemb;

    if(http_resp_code == HTTP_RESP_OTHER_ERROR) {
	if(g_strrstr_len(buffer, len, HTTP_OK) != NULL) {
	    osso_log(LOG_DEBUG,"[%s]: HTTP header response is OK!\n",__FUNCTION__);
	    http_resp_code = HTTP_RESP_OK;
	}
	else if(g_strrstr_len(buffer, len, HTTP_UNAUTH) != NULL) {
	    osso_log(LOG_DEBUG,"[%s]: HTTP autherization failure!\n",__FUNCTION__);
	    http_resp_code = HTTP_RESP_AUTH_FAIL;
	}
    }
    if ((len > 2) && (server_response_sz + len < SERVER_RESPONSE_SZ - 1)) {
	strncpy(server_response + server_response_sz, buffer, len);
	server_response_sz += len;
    }
    return len;
}

#ifdef use_progress_callback_this_is_currently_not_set
/*
 * if visual feedback/progressbar thingy needed, we should have this.
 * But in my first trial, I uploaded 5MB file and never saw
 * that log msg, so I am not sure if/how this even works...
 * Leaving it not activated for now
 * --Olev--
 */
static int progress_callback(void *clientp,
			     double dltotal,
			     double dlnow,
			     double ultotal,
			     double ulnow)
{
    osso_log(LOG_DEBUG, "[%s]: clientp=%s dltotal=%ld dlnow=%ld ultotal=%ld ulnow=%ld", __FUNCTION__, clientp,
			     dltotal,
			     dlnow,
			     ultotal,
			     ulnow);

}
#endif

/**
  Function to upload raw data to server

  @param gchar * is the pointer to upload file name
  @param gchar * is the pointer to server url
  @param gchar * is the pointer to server user id
  @param gchar * is the pointer to server password
  @param ssl is TRUE if ssl is to be used else FALSE

  @return gint - returns the https response code
*/
  
gint upload(gchar * file, gchar * url, gchar * user_name, gchar * passwd, gboolean ssl)
{
	CURL *curl = NULL;
	CURLcode res = 0;
	FILE *hd_src;
	struct stat file_info;
	gchar *credentials = NULL;

	hd_src = fopen(file, "rb");
	if (hd_src == NULL) {
	    osso_log(LOG_DEBUG, "[%s]:can not open file %s", __FUNCTION__, file);
	    return res;
	}
	fstat(fileno(hd_src), &file_info);
	curl_global_init(CURL_GLOBAL_ALL);

	curl = curl_easy_init();
	if (curl) {
	        credentials = g_strdup_printf("%s:%s", user_name, passwd);
		curl_easy_setopt(curl, CURLOPT_READFUNCTION, read_callback);
		curl_easy_setopt(curl, CURLOPT_UPLOAD, 1);
		curl_easy_setopt(curl, CURLOPT_PUT, 1);
		curl_easy_setopt(curl, CURLOPT_URL, url);
		curl_easy_setopt(curl, CURLOPT_USERPWD, credentials);
		curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
		curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 60);
		curl_easy_setopt(curl, CURLOPT_TIMEOUT, 180);
#ifdef use_progress_callback_this_is_currently_not_set
		curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progress_callback);
		curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, file);
#endif
		if(ssl == TRUE)
		{
			 curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
  	                 curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);
		}

		curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);
		curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

		curl_easy_setopt(curl, CURLOPT_INFILE, hd_src);
		curl_easy_setopt(curl, CURLOPT_INFILESIZE, file_info.st_size);

		/*
		 * initial value is OTHER_ERROR. Parse gets called for pieces of response,
		 * and if OK or AUHT_FAIL is seen, we record status in http_resp_code and stop noting.
		 * If none of them was seen, it was something else, i.e OTHER_ERROR remains
		 */	
		curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, nitro_parse_http_response);

		osso_log(LOG_DEBUG, "[%s]: Start curl upload of %s size %d B", __FUNCTION__, file, file_info.st_size);
		http_resp_code = HTTP_RESP_OTHER_ERROR;
		server_response_sz = 0;
		server_response[0] = 0;
		res = curl_easy_perform(curl);
		osso_log(LOG_DEBUG, "[%s]: Curl result:%d, response parse:%d server_response_sz=%d\n", __FUNCTION__, 
			 res, http_resp_code, server_response_sz);
		curl_easy_cleanup(curl);
		g_free(credentials);
	} else {
	    osso_log(LOG_DEBUG, "[%s]:can not initialize curl", __FUNCTION__);
	}
	if(hd_src!=NULL)
		fclose(hd_src);

	if (res == 0) {
	    res = http_resp_code;
	}

	curl_global_cleanup();
	return res;
}
