/**
 * @file nitro-daemon-main.c
 *  
 * This file contains the main function for NITRO device daemon
 * and dynamically check for the availability
 * of internal/external memory cards sets monitors appropriately
 *
 * This file is part of nitro
 *
 * Copyright (C) 2007-2008 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "nitro-daemon-defs.h"
#include "nitro-daemon-monitor.h"

static gboolean mmc_change_tmo_handler(void * context);
static gint tmo_id = 0;

/*
 * Check if path is part of rootfs or is there other fs mounted
 * return value 0: part of rootfs
 * return value 1: not part of rootfs
 */
static int 
is_mounted(const char * path)
{
    struct stat st;
    dev_t path_dev, root_dev;

    bzero(&st, sizeof(st));
    if(stat(path, &st)) {
	return 0;
    }
    path_dev = st.st_dev;
    
    bzero(&st, sizeof(st));
    if(stat(ROOTFS_LOCATION, &st)) {
	return 0;
    }
    root_dev = st.st_dev;

    return (path_dev != root_dev);
}

/*
 * Check if directory exists, create if does not.
 * (re-)activate directory monitoring.
 */
static void
mmc_check_dir(const gchar * mmc_path, gchar * core_path)
{
    GDir *dir = NULL;
    if (is_mounted(mmc_path)) {
	dir = g_dir_open(core_path, 0, NULL);
	if (dir == NULL) {
	    if(g_mkdir(core_path, FILE_PERMISSION)) {
		osso_log(LOG_ERR, "[%s] Error while creating directory %s",__FUNCTION__, core_path);
	    } else {
		osso_log(LOG_INFO, "[%s] Created directory %s",__FUNCTION__, core_path);
	    }
	}
	else {
	    g_dir_close(dir);
	}
	if(!set_directory_monitor(core_path)) {
	    osso_log(LOG_ERR, "[%s] Could not set directory monitor for %s",__FUNCTION__, core_path);
	}

    } else {
	osso_log(LOG_INFO, "[%s]: %s is not mounted", __FUNCTION__, mmc_path);
    }
}

/**
  This function is a callback to detect when the device is about to
  shut down. When that occurs, it's time for the Nitro to go down.
  
  @param hw_state contains the monitored hardware state changes
  @param data contains a pointer to main loop.

*/
static void hw_state_event_cb(osso_hw_state_t *hw_state, gpointer data)
{
	if (hw_state->shutdown_ind == TRUE) {
		osso_log(LOG_DEBUG, "[%s]: Shutting down\n", __FUNCTION__);
		/* Should we also request shutdown of nitro-ui here? */
		g_main_loop_quit((GMainLoop *)data);
		return;
	}
}

/**
  This function is the callback called when ext/int MMC removal/insertion 
  occurs. The directory monitors are reset in such case.

  @param GConfClent for notification
  @param connection ID from gconf_client_notify_add(). 
  @param GConfEntry
  @param user_data
  @return void
  */
static void 
mmc_notifier(GConfClient * gconf_client,
	     guint cnxn_id, GConfEntry * entry,
	     gpointer user_data)
{
    if (entry) {
	/*
	 * called with entry != NULL when gconf values change;
	 * Order timed execution (overwriting previous timer if any)
	 */
	osso_log(LOG_DEBUG, "[%s]: Mmc status changed:key=[%s]",__FUNCTION__, entry->key);
	if (tmo_id) {
	    g_source_remove(tmo_id);
	}
	tmo_id = g_timeout_add(5000, mmc_change_tmo_handler, gconf_client);
    } else {
	/*
	 * called with entry==NULL in these cases:
	 * 1. explicitly from main() with when nitro_daemon starts
	 * 2. from self-ordered callback few seconds after latest MMC change;
	 * Check and start monitoring directories.
	 */
	mmc_check_dir(MMC1_LOC, MMC1_CORES_LOCATION);
	mmc_check_dir(MMC2_LOC, MMC2_CORES_LOCATION);
    }
}

/*
 * timer based callback after latest change to MMC status, plus some more time.
 * This is needed because gconf change events come before mounting
 * but our actions need to be carried out after mounting.
 */
static gboolean 
mmc_change_tmo_handler(void * context)
{
    GConfClient * gconf_client = (GConfClient *)context;

    osso_log(LOG_DEBUG, "[%s]: Mmc status change timeout",__FUNCTION__);
    mmc_notifier(gconf_client, 0, NULL, NULL);	
    tmo_id = 0;
    return FALSE; /* stop timer calling this */
}

/**
  main:

  @param argc is a number of command line arguments.
  @param argv is Command line argument strings.
  @return gint
*/
gint main(int argc, char *argv[])
{
	GConfClient *gconf_client = NULL;
	GMainLoop *loop = NULL;
	osso_hw_state_t hw_state = {TRUE, FALSE, FALSE, FALSE, 0};

	osso_log(LOG_DEBUG, "nitro_daemon[%d] starts", getpid());
	/* Initialization functions */
#ifdef  MAEMO_CHANGES	
	g_thread_init(NULL);
#endif
	gnome_vfs_init();
	g_type_init();

	/* context for libosso */
	context = osso_initialize("nitro_daemon", "0.1", TRUE, NULL);
	g_return_val_if_fail(context != NULL, 0);


	gconf_client = gconf_client_get_default();
	gconf_client_add_dir(gconf_client, GCONF_DIR,
			     GCONF_CLIENT_PRELOAD_NONE, NULL);

	mmc_notifier(gconf_client, 0, NULL, NULL);	
	check_for_cores();	

	/* Order notification for some mmc events */
	gconf_client_notify_add(gconf_client, EXTERNAL_COVER_OPEN,
				mmc_notifier, NULL, NULL, NULL);
	gconf_client_notify_add(gconf_client, EXTERNAL_MMC_PRESENT,
				mmc_notifier, NULL, NULL, NULL);
	gconf_client_notify_add(gconf_client, INTERNAL_MMC_PRESENT,
				mmc_notifier, NULL, NULL, NULL);
	gconf_client_notify_add(gconf_client, MMC_CORRUPTED,
				mmc_notifier, NULL, NULL, NULL);
	gconf_client_notify_add(gconf_client, EXTERNAL_MMC_USED_USB,
				mmc_notifier, NULL, NULL, NULL);
	gconf_client_notify_add(gconf_client, INTERNAL_MMC_USED_USB,
				mmc_notifier, NULL, NULL, NULL);

	loop = g_main_loop_new(NULL, FALSE);
	osso_hw_set_event_cb(context, &hw_state, hw_state_event_cb, loop);
        g_main_loop_run(loop);
	osso_deinitialize(context);
	gnome_vfs_shutdown();
	
	return 0;
}
