/* Copyright (c) 2006, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <config.h>

#include <string.h>
#include <time.h>
#include <unistd.h>

#include <glib-object.h>
#include <glib.h>

#include <libosso.h>
#include <log-functions.h>

#include <libogs/ogs.h>
#include <libogs/ogs-text-searcher.h>
#include <libogs/ogs-file-system.h>
#include <libogs/ogs-email-hit.h>

#include <libmodest-dbus-client/libmodest-dbus-client.h>

#define MAX_TRANSLATED_FOLDER_NAME_LEN 255

#define OGS_EMAIL_SEARCH_PLUGIN         "ogs_email_search_plugin"
#define OGS_EMAIL_SEARCH_PLUGIN_VERSION "0.1"



#define OGS_TYPE_EMAIL_PLUGIN   	(email_plugin_get_type)
#define OGS_EMAIL_PLUGIN(obj)   	(G_TYPE_CHECK_INSTANCE_CAST ((obj), \
                                 	OGS_TYPE_EMAIL_PLUGIN, \
                                 	OgsEmailPlugin))
#define EMAIL_PLUGIN_CLASS(k)   	(G_TYPE_CHECK_CLASS_CAST((k), \
                                 	OGS_TYPE_EMAIL_PLUGIN, \
                                 	OgsEmailPluginClass))
#define OGS_IS_EMAIL_PLUGIN(obj)  	(G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
	                           	OGS_TYPE_EMAIL_PLUGIN))
#define OGS_IS_EMAIL_PLUGIN_CLASS(k)  	(G_TYPE_CHECK_CLASS_TYPE((k), \
		                     	OGS_TYPE_EMAIL_PLUGIN))
#define OGS_EMAIL_PLUGIN_GET_CLASS(o) 	(G_TYPE_INSTANCE_GET_CLASS ((o), \
                                      	OGS_TYPE_EMAIL_PLUGIN, \
                                    	OgsEmailPluginClass))
                                    	
#define DGETTEXT_GS(String) dgettext("osso-global-search",("srch_ia_no_subject"))

typedef struct _OgsEmailPlugin      OgsEmailPlugin;
typedef struct _OgsEmailPluginClass OgsEmailPluginClass;

struct _OgsEmailPlugin {
	OgsPlugin parent;
};

struct _OgsEmailPluginClass {
	OgsPluginClass parent_class;
};

static OgsPluginClass *parent_class = NULL;
static osso_context_t *email_plugin_ctx = NULL;

static GType    email_plugin_get_type   (GTypeModule *module);
static void     email_plugin_class_init (OgsEmailPluginClass *klass);
static void     email_plugin_finalize   (GObject *object);

static void     email_plugin_init       (OgsEmailPlugin *plugin);
static void     email_plugin_query      (OgsPlugin         *plugin,
										 const gchar       *query,
										 OgsSearchContext  *context,
										 OgsFileSystem     *fs,
										 OgsSearchCategory  category,
										 OgsSearchOptions  *options);

static gboolean email_plugin_delete_hit (OgsPlugin      *plugin,
										 OgsSearchHit   *hit,
										 GError        **error);
static gboolean email_plugin_open_hit   (OgsPlugin      *plugin,
										 OgsSearchHit   *hit,
										 GError       **error);
/**
	Email Plugin Module is required to implemenet this function. 
	This function  will be called by ligogs library at the time 
	of module loading.

	@param module is added with email plugin type module (Email)
	@return nothing
*/

G_MODULE_EXPORT void
ogs_module_load (OgsModule *module)
{
	email_plugin_get_type (G_TYPE_MODULE (module));
}

/**
	Email Plugin Module is required to implemenet this function. 
	This function  will be called by ligogs library at the time 
	of module loading.

	@param module is OgsModule for which email plugin type registered. 
	@return nothing
*/

G_MODULE_EXPORT
void ogs_module_query (OgsModule *module)
{
	ogs_search_module_register_plugin (OGS_SEARCH_MODULE (module),
					   OGS_SEARCH_CATEGORY_EMAIL,
					   email_plugin_get_type ((GTypeModule*)module));
}

/**
	Email Plugin Module is required to implemenet this function. 
	This function will be called by ligogs library at the time of 
	unloading the email plugin.

	@param module contains OgsType module
	@return nothing
*/

G_MODULE_EXPORT void
ogs_module_unload (OgsModule *module)
{
/* Nothing can be done here */
}

/**
	This method will be called to register email plugin info.  
	@param module Email plugin module is populated with email search plugin data
	@return  the email plugin type  
*/

static GType
email_plugin_get_type (GTypeModule *module)
{
	static GType type = 0;
	
	if (!module) {
		osso_log (LOG_INFO, "GType module is null \n");
	}
	
	if (!type) {
		static const GTypeInfo plugin_info = {
			sizeof( OgsEmailPluginClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) email_plugin_class_init,
			NULL,	/* class_finalize */
			NULL,	/* class_data     */
			sizeof (OgsEmailPlugin),
			0,	/* n_preallocs    */
			(GInstanceInitFunc) email_plugin_init
		};

		type = g_type_module_register_type (module,
											OGS_TYPE_PLUGIN,
											"EmailPlugin",
											&plugin_info,
											0);
	}
	
	return type;
}

/**
	This method is called to initialse email plugin class and populate
	the query, open, delete methods for Email search plugin.
	@param Email Plugin Class 
	@return nothing 
*/

static void
email_plugin_class_init (OgsEmailPluginClass *klass)
{
	OgsPluginClass *plugin_class;
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);
	
	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = email_plugin_finalize;

	plugin_class = OGS_PLUGIN_CLASS (klass);

	plugin_class->query        = email_plugin_query;
	plugin_class->options_type = OGS_TYPE_SEARCH_OPTIONS;
	plugin_class->hit_type     = OGS_TYPE_EMAIL_HIT;
	plugin_class->open_hit     = email_plugin_open_hit;
	plugin_class->open_hits    = NULL;
	plugin_class->delete_hit   = email_plugin_delete_hit;
}

/**
	This method is called to initialse email plugin object categaory 
	and create osso context for email search plugin 
	@param plugin  Email Search plugin 
*/

static void
email_plugin_init (OgsEmailPlugin *plugin)
{
	g_object_set (plugin, "id", "email-plugin",
				  "category", OGS_SEARCH_CATEGORY_EMAIL,
				  NULL);
				  
				  
	/* Create a osso EMail plugin context */
	email_plugin_ctx = osso_initialize (OGS_EMAIL_SEARCH_PLUGIN,
					    OGS_EMAIL_SEARCH_PLUGIN_VERSION,
					    TRUE,
					    NULL);
					   
	if (email_plugin_ctx == NULL) {
		osso_log (LOG_ERR, "Email Search context creation Error \n");
		return;
	}
}

/**
	This method is called to un intialise email search plugin osso context
       	and free the static date allocated.
	@return nothing 
*/

static void email_plugin_finalize (GObject *object)
{
	if (email_plugin_ctx != NULL) {
		osso_deinitialize (email_plugin_ctx);
	}
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

/* ************************* */
/* vfunc implementation */

/* this utility function converts a GDate to a time_t normalizing it.
 * If @date is lower than 01-01-1970, we set time_t to 0. If @date is
 * bigger 01-01-2038 we convert it to 01-01-2038 */
static time_t
g_date_to_time_t (const GDate *date)
{
	GDate tmp_date;
	struct tm tm;
	tmp_date = *date;
	if (g_date_get_year (&tmp_date) < 1970)
		return 0;

	if (g_date_get_year (&tmp_date) >= 2038)
		g_date_set_dmy (&tmp_date, 1, 1, 2038);
	
	g_date_to_struct_tm (&tmp_date,
			     &tm);
	
	return mktime (&tm);
}

static void
email_plugin_query (OgsPlugin         *plugin,
		    const gchar       *query,
		    OgsSearchContext  *context,
		    OgsFileSystem     *fs,
		    OgsSearchCategory  category,
		    OgsSearchOptions  *options)
{
	ModestDBusSearchFlags  flags;
	OgsSearchFields    fields;
	OgsSearchHit      *hit;
	const gchar       *folder;
	time_t             start_date;
	time_t             end_date;
	gboolean           ret;
	GList             *hits;
	GList             *iter;
	gint               scope;
	guint32            size;

	osso_log (LOG_INFO, "Email Plugin Query Starts here %s \n", query);

	if (!ogs_plugin_supports_category (plugin, category)) {
		osso_log (LOG_INFO, "Invalid Category\n");
		return;
	}

	start_date = 0;
	end_date   = 0;
	size       = 0;
	flags      = 0;
	hits       = NULL;

	fields = ogs_search_options_get_valid_fields (options);

	scope = ogs_search_options_get_scope (options);

	switch (scope) {

	case OGS_SEARCH_SCOPE_ALL:	        /* Entire body and attachment */
		flags |= MODEST_DBUS_SEARCH_SUBJECT |
			     MODEST_DBUS_SEARCH_SENDER |
				 MODEST_DBUS_SEARCH_RECIPIENT |
				 MODEST_DBUS_SEARCH_BODY;
		break;

	case OGS_SEARCH_SCOPE_ADDRESSES:	/* From, To, CC fileds */
		flags |= MODEST_DBUS_SEARCH_SENDER |
			MODEST_DBUS_SEARCH_RECIPIENT;
		break;

	case OGS_SEARCH_SCOPE_TITLES:	    /* Titles, subjects only */
		flags |= MODEST_DBUS_SEARCH_SUBJECT;
		break;

	case OGS_SEARCH_SCOPE_CONTENTS:	    /* message bodies */
		flags |= MODEST_DBUS_SEARCH_BODY;
		break;

	default:
		osso_log (LOG_INFO, "Invalid Search option = %d \n", scope);
		return;
	}

	folder = ogs_search_options_get_folder (options);

	if (fields & OGS_SEARCH_OPTIONS_MIN_SIZE) {
		size = ogs_search_options_get_min_size (options);
	}

	if (fields & OGS_SEARCH_OPTIONS_DATE_START) {
		const GDate *date;
		
		date = ogs_search_options_get_date_start (options);

		if (date != NULL) {
			start_date = g_date_to_time_t (date);
		}
	}

	if (fields & OGS_SEARCH_OPTIONS_DATE_END) {
		const GDate *ogs_date;
		GDate date;
		ogs_date = ogs_search_options_get_date_end (options);
	
		if (ogs_date != NULL) {
			/* We have to add 1 day - 1 second to the end date, to
			   properly create the range */
			date = *ogs_date;
			g_date_add_days (&date, 1);
			end_date = g_date_to_time_t (&date);
			if (end_date != ((time_t) -1))
				end_date --;
		}
	}

	ret = libmodest_dbus_client_search (email_plugin_ctx,
					    query,
					    folder,
					    start_date,
					    end_date,
					    size,
					    flags,
					    &hits);

	if (ret == FALSE) {
		osso_log (LOG_INFO, "Error during search");
		return;
	}


	for (iter = hits; iter; iter = iter->next) {
		ModestSearchHit *mh = (ModestSearchHit *) iter->data;
				
		hit = g_object_new (OGS_TYPE_EMAIL_HIT, NULL);
		if (!hit) {
			g_warning ("%s: failed to create hit", __FUNCTION__);
			continue;
		}

		ogs_search_hit_set_category  (hit, OGS_SEARCH_CATEGORY_EMAIL);
		ogs_search_hit_set_mime_type (hit, "application/x-modest");

		ogs_email_hit_set_contact        (OGS_EMAIL_HIT(hit),  mh->sender);
		ogs_email_hit_set_has_attachment (OGS_EMAIL_HIT(hit),  mh->has_attachment);
		ogs_email_hit_set_is_unread      (OGS_EMAIL_HIT(hit),  mh->is_unread);
		ogs_email_hit_set_msgid          (OGS_EMAIL_HIT(hit),  mh->msgid);
		ogs_search_hit_set_size          (OGS_SEARCH_HIT (hit), mh->msize);
		ogs_search_hit_set_timestamp     (OGS_SEARCH_HIT (hit), mh->timestamp);
		ogs_search_hit_set_folder        (OGS_SEARCH_HIT (hit), mh->folder);
		ogs_search_hit_set_name          (OGS_SEARCH_HIT (hit), mh->subject);

		ogs_plugin_new_hit (plugin, hit, context);
		g_object_unref (hit);
	}

	modest_search_hit_list_free (hits);
}


static gboolean
email_plugin_delete_hit (OgsPlugin *plugin, OgsSearchHit *hit, GError **error)
{
	OgsEmailHit *email_hit;
	gboolean      ret;
	const gchar *msgid = NULL;

	email_hit = OGS_EMAIL_HIT (hit);
	msgid = ogs_email_hit_get_msgid (email_hit);

	if (msgid == NULL) {
		g_set_error (error, 0, 0, "Could not delete mail");
		return FALSE;
	}

	ret = libmodest_dbus_client_delete_message (email_plugin_ctx,
												msgid);

	if (ret == FALSE) {
		g_set_error (error, 0, 0, "Could not delete mail");
		return FALSE;
	}
	
	return TRUE;
}


static gboolean
email_plugin_open_hit (OgsPlugin *plugin, OgsSearchHit *hit, GError **error)
{
	OgsEmailHit *email_hit;
	gboolean      ret;
	const gchar *msgid = NULL;

	email_hit = OGS_EMAIL_HIT (hit);
	msgid = ogs_email_hit_get_msgid (email_hit);

	if (msgid == NULL) {
		g_set_error (error, 0, 0, "Could not open mail");
		return FALSE;
	}

	ret = libmodest_dbus_client_open_message (email_plugin_ctx,
											  msgid);

	if (ret == FALSE) {
		g_set_error (error, 0, 0, "Could not open mail");
		return FALSE;
	}

	return TRUE;

}


/* vim: ts=4 sw=4  */
