/**
    @file texts.c

    Maemo-Blocks text printing functions

  This file is part of Maemo Blocks

  Copyright (C) 2006 Nokia Corporation
 
  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2.1 of
  the License, or (at your option) any later version.
 
  This software is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with this software; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  02110-1301 USA
*/


#include <libintl.h>
#include <locale.h>
#include <stdio.h>
#include <unistd.h>
#include <glib.h>
#include <math.h>
#include "texts.h"
#include "i18n.h"

#define BD_X1 0
#define BD_Y1 1
#define BD_X2 2
#define BD_Y2 3
/** The color of the font on the main window*/
#define FONT_COLOR 0x58, 0x48, 0x30

/** The color of the to font on the game over dialog*/
#define FONT_COLOR_GAME_OVER 0x58, 0x3A, 0x1A

/** Freetype Face */
FT_Face ft_fnt;

/** Freetype Library */
FT_Library ft_lib;

/** The vertical length of the string*/
int leny = 0;


/**
 Convert the degree into radian
 @param degree The degree which needs to be converted
 @return The radian number
*/
float
convert_degree_to_radian(float degree)
{
    return ((degree / 360.0) * M_PI * 2.0);
}

int Text_Print(FT_Face face, SDL_Surface * dest, int x, int y,
               int size, char *string, unsigned char r,
               unsigned char g, unsigned char b, int center);
int Text_Print_Wrap(FT_Face face, SDL_Surface * dest, int x, int y,
                    int size, char *string, unsigned char r,
                    unsigned char g, unsigned char b, int center,
                    int maxw, int maxh);
int *Text_Get_Sizes(FT_Face face, int size, char *string);
gchar **Text_Lines_To_Wrap(FT_Face face, char *string, int *lines, int size,
                           int maxw, int maxh);

/**
 Initialize the freetype2 library
 @return 0 on success, 1 otherwise
*/
int
Text_Init(void)
{

    if (FT_Init_FreeType(&ft_lib))
    {
        printf("cannot init FreeType\n");
        return 1;
    }
    if (FT_New_Face(ft_lib, "/usr/share/fonts/NtmBR4nh.ttf", 0, &ft_fnt))
    {
        printf("cannot load font\n");
        return 1;
    }
    // font size
    return 0;
}

int *
Text_Get_Sizes(FT_Face face, int size, char *string)
{
    int w = 0;
    int maxw = 0;
    int *sizes = NULL;
    sizes = (int *) calloc(2, sizeof(int));
    FT_ULong c = 0;
    if (!face)
    {
        free(sizes);
        return NULL;
    }
    if (FT_Set_Pixel_Sizes(face, 0, size))
    {
        free(sizes);
        return NULL;
    }
    for (; *string; string = g_utf8_next_char(string))
    {
        c = g_utf8_get_char(string);
        if (*string == '\n')
        {
            if (w > maxw)
                maxw = w;
            w = 0;
        }
        else if (FT_Load_Char(face, c, 0))
        {
        }
        else
        {
            w += (face->glyph->advance.x >> 6);
        }
    }
    if (abs(w) > abs(maxw))
    {
        sizes[0] = abs(w);
    }
    else
    {
        sizes[0] = abs(maxw);
    }
    sizes[1] = size;
    return sizes;
}

int
Text_Print(FT_Face face, SDL_Surface * dest, int x, int y,
           int size, char *string, unsigned char r,
           unsigned char g, unsigned char b, int center)
{
    int *sizes;
    SDL_Surface *new_surface = NULL;
    unsigned short *sptr16 = NULL, *tptr16 = NULL;
    int rasterpos = 0, yp = 0, xp = 0;
    int i = 0, j = 0;
    int tr, tg, tb, ta;
    int xn = 0.0, yn = 0.0;

    FT_ULong c = 0;
    if (!face)
    {
        return 1;
    }
    sizes = Text_Get_Sizes(face, size, string);
    if (sizes == NULL)
    {
        return 1;
    }
    if ((sizes[0] <= 0) || (sizes[1] <= 0))
    {
        return 1;
    }
    if (center)
    {
        y -= sizes[0] / 2;
    }
    new_surface = dest;
    sptr16 = new_surface->pixels;
    if (FT_Set_Pixel_Sizes(face, 0, size))
    {
        free(sizes);
        return 1;
    }
    SDL_LockSurface(new_surface);
    for (; *string; string = g_utf8_next_char(string))
    {
        c = g_utf8_get_char(string);
        if (FT_Load_Char(face, c, 0))
        {
            SDL_UnlockSurface(new_surface);
            SDL_FreeSurface(new_surface);
            free(sizes);
            return 1;
        }

        if (FT_Render_Glyph(face->glyph, FT_RENDER_MODE_NORMAL))
        {
            SDL_UnlockSurface(new_surface);
            SDL_FreeSurface(new_surface);
            free(sizes);
            return 1;
        }
        yp = (size) - face->glyph->bitmap_top;
        xp = rasterpos + face->glyph->bitmap_left;

        for (i = 0; i < face->glyph->bitmap.rows; i++)
        {
            for (j = 0; j < face->glyph->bitmap.width; j++)
            {
                xn = yp + i;
                xn = sizes[1] - xn;
                yn = xp + j;
                /* xn=(double)cos(radian)*((double)xp+(double)j-(double)sizes[0]/2.0)-(double)sin(radian)*((double)yp+(double)i-(double)sizes[0]/2.0)+(double)sizes[0]/2.0+(double)x;
                 * yn=(double)sin(radian)*((double)xp+(double)j-(double)sizes[0]/2.0)+(double)cos(radian)*((double)yp+(double)i- (double)sizes[1]/2.0)+(double)sizes[1]/2.0+(double)y; */

                if ((xn + x) > 0 && (yn + y) > 0 && (yn + y) < new_surface->h
                    && (xn + x) < new_surface->w
                    && face->glyph->bitmap.buffer[i *
                                                  (face->glyph->bitmap.
                                                   width) + j])
                {
                    tptr16 =
                        sptr16 + (((yn + y) * new_surface->pitch) >> 1) +
                        (xn + x);
                    ta = face->glyph->bitmap.buffer[i *
                                                    (face->glyph->bitmap.
                                                     width) + j];
                    tr = (((*tptr16) >> 11) & 31) << 3;
                    tg = (((*tptr16) >> 5) & 63) << 2;
                    tb = ((*tptr16) & 31) << 3;
                    *tptr16 =
                        ((((((ta) * (r - tr)) >> 8) +
                           tr) << 8) & 63488) | ((((((ta) * (g - tg)) >> 8) +
                                                   tg) << 3) & 2016) |
                        ((((((ta) * (b - tb)) >> 8) + tb) >> 3) & 31);
            /**tptr16 = 31;*/
            /**(sptr16 + ((yp + i)*dest->pitch)) = 63488;*/
                }

            }
        }

        rasterpos += (face->glyph->advance.x >> 6);
    }
    SDL_UnlockSurface(new_surface);
    free(sizes);
    return 0;
}

gchar **
Text_Lines_To_Wrap(FT_Face face, char *string, int *lines, int size, int maxw,
                   int maxh)
{
    char *last_start = NULL, *last_space = NULL, *string_start = NULL;
    int size_from_lastspace = 0;
    gchar **str_list = NULL;
    int current_size = 0;
    str_list = g_new0(gchar *, 10);
    int linen_index = 0;
    FT_ULong c = 0;
    if (!string)
    {
        return NULL;
    }
    if (FT_Set_Pixel_Sizes(face, 0, size))
    {
        return NULL;
    }
    string_start = string;
    last_start = string;
    for (; *string && (linen_index < 10); string = g_utf8_next_char(string))
    {
        c = g_utf8_get_char(string);
        if ((g_unichar_break_type(c) == G_UNICODE_BREAK_LINE_FEED)
            || (g_unichar_break_type(c) == G_UNICODE_BREAK_CARRIAGE_RETURN))
        {
            str_list[linen_index] =
                g_strndup(last_start, string - last_start);
            last_start = string + 1;
            last_space = NULL;
            linen_index++;
            current_size = 0;
            size_from_lastspace = 0;

        }
        else
        {
            if (g_unichar_break_type(c) == G_UNICODE_BREAK_SPACE)
            {
                last_space = string;
                size_from_lastspace = 0;
            }
            if (FT_Load_Char(face, c, 0))
            {
            }
            else
            {
                current_size += (face->glyph->advance.x >> 6);
                size_from_lastspace += (face->glyph->advance.x >> 6);
            }
            if (current_size >= maxw)
            {
                if (last_space)
                {
                    str_list[linen_index] =
                        g_strndup(last_start, last_space - last_start);
                    last_start = last_space + 1;
                    linen_index++;
                    current_size = size_from_lastspace;
                }
                else
                {
                    str_list[linen_index] =
                        g_strndup(last_start, string - last_start - 1);
                    linen_index++;
                    size_from_lastspace = current_size =
                        (face->glyph->advance.x >> 6);

                }
            }
        }
    }
    if (linen_index < 10)
    {
        str_list[linen_index] = g_strndup(last_start, string - last_start);
    }
    return str_list;

}

int
Text_Print_Wrap(FT_Face face, SDL_Surface * dest, int x, int y,
                int size, char *string, unsigned char r,
                unsigned char g, unsigned char b, int center,
                int maxw, int maxh)
{
    gchar **string_list = NULL;
    int current_x = x, x_jump;
    int lines = 0;
    int i = 0;
    int error = 0;
    if (!string)
    {
        return 1;
    }
    string_list = Text_Lines_To_Wrap(face, string, &lines, size, maxw, maxh);
    while (string_list[i])
    {
        i++;
    }
    x_jump = maxh / (i + 1);
    current_x = x + (maxh / 2) - x_jump;
    if (string_list)
    {
        for (i = 0; (error == 0) && (string_list[i]); i++)
        {
            error =
                Text_Print(face, dest, current_x, y, size, string_list[i], r,
                           g, b, center);
            g_free(string_list[i]);
            current_x -= size;
        }
        g_free(string_list);
    }
    return error;
}

/**
 Prints the restart string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_restart(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 52, 113, 21,
                      _("game_bd_blocks_main_restart"), FONT_COLOR, 1);
}

/**
 Prints the exit to menu string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_exit_to_menu(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 47, 356, 21,
                      _("game_bd_blocks_main_exit_to_menu"), FONT_COLOR, 1);
}

/**
 Prints the next string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_next(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 698, 346, 25,
                      _("game_fi_blocks_main_next"), FONT_COLOR, 0);
}

/**
 Prints the level string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_level(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 458, 346, 25,
                      _("game_fi_blocks_main_level"), FONT_COLOR, 0);
}

/**
 Prints the scrore string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_score(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 335, 346, 25,
                      _("game_fi_blocks_main_score"), FONT_COLOR, 0);
}

/**
 Prints the lines string to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_lines(SDL_Surface * destination)
{
    return Text_Print(ft_fnt, destination, 212, 346, 25,
                      _("game_fi_blocks_main_lines"), FONT_COLOR, 0);
}

/**
 Prints the game over strings to the destination
 @param destination The destination surface
 @return 0 on succes, non-0 on fail
*/
int
Print_game_over(SDL_Surface * destination)
{
    int error = 0;
    error =
        Text_Print_Wrap(ft_fnt, destination, 35, 195, 23,
                        _("game_fi_blocks_end_game_description"), FONT_COLOR,
                        1, 390, 70);
    if (!error)
    {
        return (Text_Print
                (ft_fnt, destination, 70, 195, 25,
                 _("game_ti_blocks_end_game_title"), FONT_COLOR, 1));
    }
    return error;
}
