/**
    @file main.c

    Maemo-Blocks main file.

  This file is part of Maemo Blocks

  Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
  http://www.indt.org/maemo
 
  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2.1 of
  the License, or (at your option) any later version.
 
  This software is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with this software; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  02110-1301 USA
*/


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <locale.h>
#include <SDL/SDL.h>
#include <glib.h>
#include <libosso.h>
#include <osso-log.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "menu.h"
#include "game.h"
#include "images.h"
#include "texts.h"
#include "i18n.h"
#if SAPP_FUNC
#include <shadowapp.h>
sapp_data *sapp_context = NULL;
#endif
#include "callbacks.h"
#include <math.h>
#include <signal.h>

/** This gconf value will be true while the games startup is running */
#define BLOCKS_RUNNING	"/apps/indt/maemoblocks/running"

/* Max move distance of tap (in pixels) */
#define BLOCKS_TAP_DISTANCE 15
/* Max time of time (in milliseconds) */
#define BLOCKS_TAP_TIME 250

unsigned int delay;
int difficulty_level, noise_level, noise_height;
int block_selected = false, last_x = 0, last_y = 0;
osso_context_t *osso;
int home_pressed = 0;
GConfClient *gcc = NULL;
int terminate_game = 0;


static void sig_handler(int signum);
void osso_hw_state(osso_hw_state_t * state, gpointer data);
static inline void gmainloop(void);
static void game_loop(void);
static int position_have_block(int x, int y);

static void
sig_handler(int signum)
{
        if(signum == SIGTERM)
        {
                exit_callback(0);
        }
}

void
osso_hw_state(osso_hw_state_t * state, gpointer data)
{
    (void) data;
    static SDL_Event ev;
    if (state->memory_low_ind)
    {
        /* 
         * Memory low 
         */
        quit_callback(1);
    }
    if (state->system_inactivity_ind)
    {
        ev.type = SDL_USEREVENT;
        SDL_PushEvent(&ev);
    }
    if (state->save_unsaved_data_ind)
    {
        exit_callback(1);
    }
}

/**
  Call this function in time consuming loops, because we want to run
  osso (dbus) for hw-events.
*/
static inline void
gmainloop(void)
{
    while (g_main_context_iteration(g_main_context_default(), FALSE)) ;
}

static int
position_have_block(int x, int y)
{
	int dx, dy;
	dx = abs( x - cur_x );
	dy = abs( y - cur_y );

	if( dx < 4 && dy < 4 ) {
		return true;
 	} else {
 		return false;
    }

	/* Why it was like this?...
    int i, j, mask = 1 << 15;

    for (i = 0; i < 4; i++) {
        for (j = 0; j < 4; j++)
        {
            if (blocks[cur_block][cur_frame] & mask && cur_x + j == x
                && cur_y + i == y)
                return true;
            mask >>= 1;
        }
    }

    return false;
	*/
}

/**
  Main loop
*/
static void
game_loop(void)
{
    SDL_Event event;
    int flip = 0, x, y, dropbonus;

    /* Values used to get a tap */
    int x_down = 0, y_down = 0;
    int change = 0;
    Uint32 time_down = 0;

    /* Prototype of hw key rotate event, send by tap event */
    SDL_Event rotate_event;
    rotate_event.type = SDL_KEYDOWN;
    rotate_event.key.keysym.sym = SDLK_RETURN;
    rotate_event.key.state = SDL_RELEASED;

    delay = SDL_GetTicks();

    while (!terminate_game)
    {
        gmainloop();
        while (SDL_PollEvent(&event))
        {

            // Key pressed
            if (event.type == SDL_KEYDOWN)
            {
            
            	// Big Quit with Ctrl+Q
/*            	if( event.key.keysym.sym == SDLK_q &&
            		event.key.keysym.mod & ( KMOD_LCTRL | KMOD_RCTRL ) ) {
            		printf( "Shutdown whole game\n" );
            		exit_callback(0);
            		return;
            	}

                else */
                // Quit
		if (event.key.keysym.sym == SDLK_ESCAPE ||
                    event.key.keysym.sym == SDLK_F4 ||
                    event.key.keysym.sym == SDLK_F5 ||
                    event.key.keysym.sym == SDLK_F6 ||
                    event.key.keysym.sym == SDLK_F10 ) {

                    exit_callback(0);
                    return;

                }
                else if (!game_over)
                {
                    switch ((int) event.key.keysym.sym)
                    {
                        case SDLK_DOWN:
                            flip = move_block(1, 0);
                            break;

                        case SDLK_UP:
                            flip = move_block(-1, 0);
                            break;

                        case SDLK_RIGHT:
                        case SDLK_RETURN:
                            flip = change_frame();
                            break;

                        case SDLK_LEFT:
                            dropbonus = 0;
                            while (move_down())
                                dropbonus++;
                            current_score +=
                                dropbonus * (current_level + 1) / 8;
                            update_game_values(current_lines, current_level,
                                               current_score);
                    }
                    if (flip && block_selected)
                    {
                        SDL_GetMouseState(&x, &y);
                        y = MAX_Y - (x - BOARDOFFSETX2) / BLOCK_WIDTH - 1;
                        x = (y - BOARDOFFSETY2) / BLOCK_HEIGHT;
                        
                        block_selected = position_have_block(x, y);
                    }
                }
                // Button
            }
            else if (event.type == SDL_MOUSEBUTTONDOWN)
            {
                /* Mark start point */
                x_down = event.button.x;
                y_down = event.button.y;
                change = 0.0;
                time_down = SDL_GetTicks();

                if (event.button.x >= NEWGAME_OFFSETX &&
                    event.button.x < NEWGAME_OFFSETX2 &&
                    event.button.y >= NEWGAME_OFFSETY
                    && event.button.y < NEWGAME_OFFSETY2)
                {
                    game_new_wrapper(difficulty_level, noise_level,
                                     noise_height);
                    draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
                               BOARDOFFSETY + cur_y * BLOCK_WIDTH,
                               cur_block, cur_frame);
                    draw_next_block();
                    SDL_Flip(screen);
                    delay = SDL_GetTicks();
                }
                else if (event.button.x >= EXIT_OFFSETX &&
                         event.button.x < EXIT_OFFSETX2 &&
                         event.button.y >= EXIT_OFFSETY
                         && event.button.y < EXIT_OFFSETY2)
                {
                    exit_callback(0);
                    return;
                }
                else
                {
                    y = MAX_Y - (event.button.x -
                                 BOARDOFFSETX2) / BLOCK_WIDTH - 1;
                    x = (event.button.y - BOARDOFFSETY2) / BLOCK_HEIGHT;
                    last_x = x;
                    last_y = y;
                    block_selected = position_have_block(x, y);
                }
            }
            else if (event.type == SDL_MOUSEBUTTONUP)
            {

                /* Check if to send a rotate event */
                if (block_selected &&
                    (SDL_GetTicks() - time_down) < BLOCKS_TAP_TIME &&
                    change < (BLOCKS_TAP_DISTANCE * BLOCKS_TAP_DISTANCE))
                {
                    SDL_PushEvent(&rotate_event);
                }

                block_selected = false;

            }
            else if (event.type == SDL_MOUSEMOTION)
            {
                if (block_selected)
                {
                    int current_change = (x_down - event.button.x) *
                        (x_down - event.button.x) + (y_down -
                                                     event.button.y) *
                        (y_down - event.button.y);
                    if (current_change > change)
                    {
                        change = current_change;
                    }

                    x = (event.button.y - BOARDOFFSETY2) / BLOCK_HEIGHT;
                    y = MAX_Y - (event.button.x -
                                 BOARDOFFSETX2) / BLOCK_WIDTH - 1;
                    if (y < last_y)
                        y = last_y;
                    if (x != last_x || y != last_y)
                    {
                        move_block(x - last_x, y - last_y);
                        last_x = x;
                        last_y = y;
                        flip = 1;
                    }
                }
            }
            else if (event.type == SDL_ACTIVEEVENT)
            {
                if (event.active.gain == 0)
                {
                    exit_callback(0);
                    return;
                }
            }
            else if (event.type == SDL_QUIT)
            {
                quit_callback(0);
            }
        }

        // Game loop
        if (!game_over)
        {
            if (SDL_GetTicks() - delay >= level_speeds[current_level])
            {
                delay += level_speeds[current_level];
                flip = move_down();
            }
            if (flip)
            {
                flip = 0;
                SDL_UpdateRect(screen, BOARDOFFSETX2, BOARDOFFSETY2,
                               BOARDWIDTH, BOARDHEIGHT);
            }
        } else {
        	SDL_Delay(10);
        }
    }
}

/**
  Main function
*/
int
main(int argc, char **argv)
{
    FILE *han;
    char aux[64];
    unsigned char mask = 0;
    osso_hw_state_t hw_state;
#ifdef ENABLE_NLS
    /* 
     * Initialize localization 
     */
    /* 
     * Getttext does not seem to work properly without
     * the following function call 
     */
    setlocale(LC_ALL, "");

    bindtextdomain(GETTEXT_PACKAGE, BLOCKSLOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
    g_type_init();
    gcc = gconf_client_get_default();

#endif /* ENABLE_NLS */
    osso = osso_initialize(PACKAGE_NAME, PACKAGE_VERSION, FALSE, NULL);
    if (osso == NULL)
    {
        printf("osso initialize failed!\n");
        return -1;
    }

    hw_state.shutdown_ind = TRUE;
    hw_state.save_unsaved_data_ind = FALSE;
    hw_state.memory_low_ind = FALSE;
    hw_state.system_inactivity_ind = TRUE;
    hw_state.sig_device_mode_ind = FALSE;
    if (osso_hw_set_event_cb(osso, NULL, osso_hw_state, NULL) != OSSO_OK)
    {
        printf("osso_hw_set_event_cb failes!\n");
        return -1;
    }
    // if startup kills us, we want to save the state
    signal(SIGTERM, sig_handler);

    // Initialize libshadow
#if SAPP_FUNC
    sapp_context = sapp_initialize(argc, argv);
    if (!sapp_context)
    {
        fprintf(stderr, "Cannot init libshadow!\n");
        return 0;
    }
    sapp_set_msg_cb_exit(sapp_context, exit_callback);
    sapp_set_msg_cb_quit(sapp_context, quit_callback);
    sapp_set_msg_cb_flush(sapp_context, flush_callback);
    if (!sapp_check(sapp_context))
        return 0;

    /* 
     * Shadow app part 
     */
    // sapp_context->smain (sapp_context, 0);

    sapp_mainloop(sapp_context, 0);
    /* Why? usleep (100); */
#endif
    if (Text_Init())
    {
        printf("Error while initializing fonts\n");
        exit(1);
    }
    // Initialize SDL
    g_debug("%s: Init SDL", __FUNCTION__);
    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0)
    {
        fprintf(stderr, "Couldn't initialize SDL: %s\n", SDL_GetError());
        return 0;
    }
    sprintf(aux, _("game_ap_blocks_name"));
    SDL_WM_SetCaption(aux, 0);
    // Initialize video
    if (!gconf_client_get_bool(gcc, BLOCKS_RUNNING, NULL))
    {
        osso_deinitialize(osso);
#if SAPP_FUNC
        if (!game_over)
        {
            sapp_deinitialize(sapp_context);
        }
#endif
        return 0;
    }

    g_debug("%s: Init screen", __FUNCTION__);

    screen =
        SDL_SetVideoMode(800, 480, 16,
                         SDL_DOUBLEBUF | SDL_HWSURFACE | SDL_FULLSCREEN);
    if (screen == NULL)
    {
        g_debug("%s: Unable to set 800x480 video: %s", __FUNCTION__,
                SDL_GetError());
        fprintf(stderr, "Unable to set 800x480 video: %s\n", SDL_GetError());
        return 0;
    }
    SDL_SetCursor(SDL_CreateCursor(&mask, &mask, 8, 1, 0, 0));
    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY,
                        SDL_DEFAULT_REPEAT_INTERVAL);

    g_debug("%s: Load images", __FUNCTION__);

    // Load images
    if (!load_images())
    {
#if SAPP_FUNC
        // sapp_context->deinit (sapp_context);
        sapp_deinitialize(sapp_context);
#endif
        free_images();
        SDL_Quit();
        return 0;
    }
    // Read configuration
    g_debug("%s: Read configuration", __FUNCTION__);
    difficulty_level = gconf_client_get_int(gcc, DIFFICULTY_LEVEL, NULL);
    noise_level = gconf_client_get_int(gcc, NOISE_LEVEL, NULL);
    noise_height = gconf_client_get_int(gcc, NOISE_HEIGHT, NULL);

    // init game values
    g_debug("%s: Init game", __FUNCTION__);
    game_init();
    srandom(time(NULL));

    // Continue game
    han = fopen("/tmp/.maemoblocks-save", "rb");
    if (han)
    {
        fread(matrix, sizeof(int), MAX_X * MAX_Y, han);
        fread(&cur_x, sizeof(int), 1, han);
        fread(&cur_y, sizeof(int), 1, han);
        fread(&cur_block, sizeof(int), 1, han);
        fread(&cur_frame, sizeof(int), 1, han);
        fread(&current_score, sizeof(int), 1, han);
        fread(&current_level, sizeof(int), 1, han);
        fread(&current_lines, sizeof(int), 1, han);
        fread(&next_block, sizeof(int), 1, han);
        fread(&next_frame, sizeof(int), 1, han);
        fclose(han);
        remove("/tmp/.maemoblocks-save");
        SDL_BlitSurface(background, NULL, screen, NULL);
        draw_board();
        update_game_values(current_lines, current_level, current_score);
        game_over = false;


        // New game
    }
    else
    {
        game_new_wrapper(difficulty_level, noise_level, noise_height);
    }
    if (!gconf_client_get_bool(gcc, BLOCKS_RUNNING, NULL))
    {
        osso_deinitialize(osso);
#if SAPP_FUNC
        if (!game_over)
        {
            sapp_deinitialize(sapp_context);
        }
#endif
        return 0;
    }
    draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
               BOARDOFFSETY + cur_y * BLOCK_WIDTH, cur_block, cur_frame);
    draw_next_block();
    if (Print_restart(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_exit_to_menu(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_next(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_level(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_score(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_lines(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    SDL_Flip(screen);
    // Main loop
    g_debug("%s: Star game loop", __FUNCTION__);
    game_loop();
    osso_deinitialize(osso);
    free_images();
    SDL_Quit();
#if SAPP_FUNC
    if (!game_over)
    {
        sapp_deinitialize(sapp_context);
    }
    else
    {
        hgw_context_destroy(sapp_context, HGW_BYE_INACTIVE);
    }
#endif
    return 0;
}
