/**
    @file game.c

    Maemo-Blocks game routines.

  This file is part of Maemo Blocks

  Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
  http://www.indt.org/maemo
 
  This software is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2.1 of
  the License, or (at your option) any later version.
 
  This software is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with this software; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  02110-1301 USA
*/

#include <stdlib.h>
#include <string.h>
#include <SDL.h>
#include "game.h"
#include "images.h"
#include "texts.h"

int matrix[MAX_Y][MAX_X];
int next_block = 0, next_frame = 0;
int lines_score[4] = { 10, 20, 40, 80 };
int cur_x, cur_y, cur_block, cur_frame;
int num_blocks[BLOCKS] = { 1, 2, 2, 2, 4, 4, 4 };
int current_score, current_level, current_lines, game_over;
int level_speeds[NUM_LEVELS] =
    { 1000, 886, 785, 695, 616, 546, 483, 428, 379, 336, 298, 264, 234, 207,
183, 162, 144, 127, 113, 100 };
int blocks[BLOCKS][4] = { {0xCC00},
{0x4444, 0x0F00},
{0x6C00, 0x8C40},
{0x4C80, 0xC600},
{0x6440, 0x0E20, 0x44C0, 0x8E00},
{0xC440, 0x2E00, 0x4460, 0x0E80},
{0x4E00, 0x4640, 0x0E40, 0x4C40}
};

/* static function definitions */
static int clean_lines(void);
static int check_position(int x, int y, int block, int frame);
static void init_game_over(void);
static void new_block(void);
static inline void fix_block(void);
static void make_noise(int level, int height);


/**
   Clean horizontal lines
   @return the cleared lines
*/
static int
clean_lines(void)
{
    int i, j, lines = 0, clean;

    for (i = 0; i < MAX_Y; i++)
    {
        clean = true;
        for (j = 0; j < MAX_X; j++)
            if (!matrix[i][j])
            {
                clean = false;
                break;
            }
        if (clean)
        {
            lines++;
            for (j = i; j > 0; j--)
                memcpy(matrix[j], matrix[j - 1], sizeof(matrix[0]));
            memset(matrix[0], 0, sizeof(matrix[0]));
            draw_board();
        }
    }

    return lines;
}

/**
 Validate the new position of the block
*/
static int
check_position(int x, int y, int block, int frame)
{
    int i, j, mask = 1 << 15;

    for (i = 0; i < 4; i++)
        for (j = 0; j < 4; j++)
        {
            if (blocks[block][frame] & mask && (x + j < 0 || x + j >= MAX_X ||
                                                y + i < 0 || y + i >= MAX_Y
                                                || matrix[y + i][x + j]))
                return false;
            mask >>= 1;
        }

    return true;
}

/**
  Show game over message
*/
static void
init_game_over(void)
{
    SDL_Rect dest;

    game_over = true;
    dest.x = 350;
    dest.y = 40;
    dest.w = dest.h = 0;
    SDL_BlitSurface(loose, NULL, screen, &dest);
    SDL_Flip(screen);
}

/**
  Generate a new block
*/
static void
new_block(void)
{
    cur_block = next_block;
    cur_frame = next_frame;
    next_block = rand() % BLOCKS;
    next_frame = rand() % num_blocks[next_block];
    cur_x = (MAX_X / 2) - 1;
    if (check_position(cur_x, -2, cur_block, cur_frame))
        cur_y = -2;
    else if (check_position(cur_x, -1, cur_block, cur_frame))
        cur_y = -1;
    else if (check_position(cur_x, 0, cur_block, cur_frame))
        cur_y = 0;
    else
    {
        init_game_over();
        return;
    }

    block_selected = false;
    draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
               BOARDOFFSETY + cur_y * BLOCK_WIDTH, cur_block, cur_frame);
    draw_next_block();
    delay = SDL_GetTicks();
}

/**
  Move block horizontally
*/
int
move_block(int dx, int dy)
{
    if (check_position(cur_x + dx, cur_y + dy, cur_block, cur_frame))
    {
        draw_board();
        cur_x += dx;
        cur_y += dy;
        draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
                   BOARDOFFSETY + cur_y * BLOCK_WIDTH, cur_block, cur_frame);
        return true;
    }

    return false;
}

/**
  Change block's frame
*/
int
change_frame(void)
{
    int nx_frame = (cur_frame + 1) % num_blocks[cur_block];

    if (check_position(cur_x, cur_y, cur_block, nx_frame))
    {
        draw_board();
        cur_frame = nx_frame;
        draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
                   BOARDOFFSETY + cur_y * BLOCK_WIDTH, cur_block, cur_frame);
        return true;
    }

    return false;
}

/**
  Fix the block in current position
*/
static inline void
fix_block(void)
{
    int i, j, mask = 1 << 15;

    for (i = 0; i < 4; i++)
        for (j = 0; j < 4; j++)
        {
            if (blocks[cur_block][cur_frame] & mask)
                matrix[cur_y + i][cur_x + j] = cur_block + 1;
            mask >>= 1;
        }
}

/**
  Knock down the block
*/
int
move_down(void)
{
    int lines;

    if (check_position(cur_x, cur_y + 1, cur_block, cur_frame))
    {
        cur_y++;
        draw_board();
        draw_block(BOARDOFFSETX + cur_x * BLOCK_HEIGHT,
                   BOARDOFFSETY + cur_y * BLOCK_WIDTH, cur_block, cur_frame);
        return true;
    }
    else
    {
        fix_block();
        lines = clean_lines();
        if (lines)
        {
            current_lines += lines;
            if (current_lines / 15 > current_level)
                current_level = current_lines / 15;
            if (current_level > 19)
                current_level = 19;
            current_score += lines_score[lines - 1] * (current_level + 1);
            update_game_values(current_lines, current_level, current_score);
        }
        new_block();
        SDL_Flip(screen);
        return false;
    }
}

/**
  Init game
*/
inline void
game_init(void)
{
    game_over = true;
    next_block = next_frame = 0;
    current_score = current_level = current_lines = 0;
    memset(matrix, 0, sizeof(matrix));
    new_block();

}

/**
  Make difficulty level
*/
static void
make_noise(int level, int height)
{
    int i, j;
    const int k = MAX_X - level;
    int select[MAX_X]; /* select will contain the indexes of blocks to be
                          blackened; only the first level items used; the
                          algoritm generates black block with uniform
                          distribution */

    if (!level || !height)
        return;

    for (i = MAX_Y - height; i < MAX_Y; i++)
    {
        for (j = 0; j < MAX_X; j++)
            matrix[i][j] = (rand() % BLOCKS) + 1;

        for (j = 0; j < k; j++)   select[j] = j;
        for (j = k; j < MAX_X; j++)
        {
            if (rand() % j < k)   select[rand() % k] = j;
        }
        for (j = 0; j < k; j++)
            matrix[i][select[j]] = 0;
    }
}

/**
  Init game screen
*/
void
game_new_wrapper(int new_level, int new_noise_level, int new_noise_height)
{
    game_init();

    make_noise(new_noise_level, new_noise_height);
    SDL_BlitSurface(background, NULL, screen, NULL);
    draw_board();

    current_level = new_level;
    update_game_values(0, current_level, 0);
    game_over = false;
    if (Print_restart(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_exit_to_menu(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_next(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_level(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_score(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
    if (Print_lines(screen))
    {
        printf("Error while printing text\n");
        exit(1);
    }
}
