/* test-cases.c - test cases for libtelepathy
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#include <dbus/dbus-glib.h>

#include <tp-chan.h>
#include <tp-connmgr.h>
#include <tp-conn.h>
#include <tp-interfaces.h>

#include <outo.h>

#define ACCOUNT_USER "jrautio@jabbernet.dk"
#define ACCOUNT_UNAME "jrautio"
#define ACCOUNT_PWD "kukukuu6"

#define CONTACT_NAME "myrjola@jabbernet.dk"

#define CM_BUS_NAME "org.freedesktop.Telepathy.ConnectionManager.gabble"
#define CM_OBJECT_PATH "/org/freedesktop/Telepathy/ConnectionManager/gabble"

#define TEXT_CHANNEL "org.freedesktop.Telepathy.Channel.Type.Text"

#define PROTO "jabber"


/******************* Glib async testing helper funcs **********************/

static DBusGConnection *connection;
static GMainLoop *mainloop;


/* timeout callback for the async response waiting */
gboolean test_wait_timeout_callback (gpointer data)
{
  int *condition = data;

  *condition = -1;
  g_print("test_wait_timeout_callback: timeout!\n");

  return FALSE;
}

/* the wait func for async callback */
int test_wait_g_main_until (int *condition, guint timeout)
{
  g_timeout_add (timeout, test_wait_timeout_callback, condition);

  while (!*condition)
    g_main_context_iteration(NULL, TRUE);
  
  return *condition;
}

/* Initialize test case */
void init_test (void)
{
  GError *error = NULL;

  g_type_init();
  
  mainloop = g_main_loop_new(NULL, FALSE);

  /* Initialize DBus connection */
  connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  if (connection == NULL)
  {
    g_printerr("Failed to open connection to bus: %s", error->message);
    g_error_free(error);
    return;
  }
  
  return;
}

/* End test case */
void post_test (void)
{
}



/******************* Test Cases **********************/

/*
 * 1. org.freedesktop.Telepathy.ConnectionManager
 */

int connmgr_new (void)
{
  TpConnMgr *connmgr;
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  assert(connmgr);
  
  /*
  check that bus name, obj path and interface name are correct
  */
 
  g_object_unref(connmgr);
  
  return 1;
}

void init_parameters (GHashTable *connection_parameters)
{
  static GValue val_acc, val_pwd;

  /* Setting "g_type" is a hack since GValue is broken */
  val_acc.g_type = 0;
  val_pwd.g_type = 0;
  g_value_init(&val_acc, G_TYPE_STRING);
  g_value_init(&val_pwd, G_TYPE_STRING);

  /* Connection parameters are dummy: fill in suitable replacements */

  g_value_set_string(&val_acc, ACCOUNT_USER);
  g_value_set_string(&val_pwd, ACCOUNT_PWD);

  g_hash_table_insert(connection_parameters, "account",
                      (gpointer) &val_acc);

  g_hash_table_insert(connection_parameters, "password",
                      (gpointer) &val_pwd);  
}


static void newconnection_handler(DBusGProxy *proxy, const char *s1,
                                  const char *o, const char *s2,
                                  gpointer user_data)

{
  int *condition = (int *)user_data;

  g_print("****newconnection_handler() got called: %s %s %s\n", s1, o, s2);

    /* proper tests */

    /* close connection */

    /* all fine, set result ok */
  *condition = 1;
}

static gboolean status_change_handler(DBusGProxy *proxy,
  guint status, guint reason,
  gpointer user_data)
{
  int *condition = (int *) user_data;

  g_print("****status_change_handler() got called: %d %d\n", 
	  status, reason);

  *condition = 1;

  return TRUE;
}

int connmgr_new_connection(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;

  int condition = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);
  
  conn = tp_connmgr_new_connection(connmgr,
				   connection_parameters,
				   PROTO);
  
  g_assert(conn);

  test_wait_g_main_until (&condition, 500);

  assert(condition == 1);
  
  g_object_unref(connmgr);

  return 1;
}

int conn_new(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;
  GError *error = NULL;

  char *bus_name = NULL, *object_path = NULL;

  int condition = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);

  if (!tp_connmgr_connect(DBUS_G_PROXY(connmgr), PROTO,
			  connection_parameters, &bus_name,
			  &object_path, &error))
  {
    g_print("Error - could not connect! %s\n", error -> message);
    g_assert_not_reached();
  }

  g_assert(bus_name);
  g_assert(object_path);

  conn = tp_conn_new(connection, bus_name, object_path);

  g_assert(conn);

  return 1;
}
    
int conn_new_channel(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;
  TpChan *chan1;

  GError *error;

  guint conn_status = TP_CONN_STATUS_DISCONNECTED;
  guint handle = 0;

  int condition = 0, condition2 = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);
  
  conn = tp_connmgr_new_connection(connmgr,
				   connection_parameters,
				   PROTO);
  
  g_assert(conn);

  test_wait_g_main_until (&condition, 500);

  assert(condition == 1);

  condition2 = 0;

  dbus_g_proxy_connect_signal(DBUS_G_PROXY(conn),
			      "StatusChanged",
			      G_CALLBACK(status_change_handler),
			      &condition2, NULL);

  if (!tp_conn_get_status(DBUS_G_PROXY(conn), &conn_status,
			  &error))
    {
      g_warning("GetStatus() failed: %s\n", error->message);
      g_assert_not_reached();
    }

  if (conn_status != TP_CONN_STATUS_CONNECTED)
    test_wait_g_main_until (&condition2, 500);

  g_assert(condition2 == 1);

  if (!tp_conn_request_handle(DBUS_G_PROXY(conn), 
			      TP_CONN_HANDLE_TYPE_CONTACT,
			      CONTACT_NAME, &handle, &error))
  {
    g_warning("No handle: %s\n", error->message);
    g_assert_not_reached();
  }

  chan1 = tp_conn_new_channel(connection, conn, CM_BUS_NAME, TEXT_CHANNEL, 
			      TP_CONN_HANDLE_TYPE_CONTACT, handle, TRUE);

  assert(chan1);
  
  return 1;
}

/* test both set and get interfaces */
int conn_local_set_interfaces(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;
  GError *error = NULL;
  DBusGProxy *iface;
  gchar **interfaces;
  guint conn_status = TP_CONN_STATUS_DISCONNECTED;

  int condition = 0, condition2 = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  g_assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);
  
  conn = tp_connmgr_new_connection(connmgr,
				   connection_parameters,
				   PROTO);
  
  g_assert(conn);

  test_wait_g_main_until (&condition, 500);

  g_assert(condition == 1);

  condition2 = 0;

  dbus_g_proxy_connect_signal(DBUS_G_PROXY(conn),
			      "StatusChanged",
			      G_CALLBACK(status_change_handler),
			      &condition2, NULL);

  if (!tp_conn_get_status(DBUS_G_PROXY(conn), &conn_status,
			  &error))
    {
      g_warning("GetStatus() failed: %s\n", error->message);
      g_assert_not_reached();
    }

  if (conn_status != TP_CONN_STATUS_CONNECTED)
    test_wait_g_main_until (&condition2, 500);

  g_assert(condition2 == 1);


  if (!tp_conn_get_interfaces(DBUS_G_PROXY(conn),
			      &interfaces, &error))
    {
      g_warning("GetStatus() failed: %s, dbus err: %s\n", 
		error->message, dbus_g_error_get_name(error));
      g_assert_not_reached();
    }

  iface = tp_conn_get_interface(conn, 
				TELEPATHY_CONN_IFACE_PRESENCE_QUARK);

  g_assert(iface);

  return 1;
}

int chan_new(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;
  TpChan *chan1;

  GError *error;

  gchar *chan_object_path;

  guint conn_status = TP_CONN_STATUS_DISCONNECTED;
  guint handle = 0;

  int condition = 0, condition2 = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  g_assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);
  
  conn = tp_connmgr_new_connection(connmgr,
				   connection_parameters,
				   PROTO);
  
  g_assert(conn);

  test_wait_g_main_until (&condition, 500);

  assert(condition == 1);

  condition2 = 0;

  dbus_g_proxy_connect_signal(DBUS_G_PROXY(conn),
			      "StatusChanged",
			      G_CALLBACK(status_change_handler),
			      &condition2, NULL);

  if (!tp_conn_get_status(DBUS_G_PROXY(conn), &conn_status,
			  &error))
    {
      g_warning("GetStatus() failed: %s\n", error->message);
      g_assert_not_reached();
    }

  if (conn_status != TP_CONN_STATUS_CONNECTED)
    test_wait_g_main_until (&condition2, 500);

  g_assert(condition2 == 1);

  if (!tp_conn_request_handle(DBUS_G_PROXY(conn), 
			      TP_CONN_HANDLE_TYPE_CONTACT,
			      CONTACT_NAME, &handle, &error))
  {
    g_warning("No handle: %s\n", error->message);
    g_assert_not_reached();
  }

  if (!tp_conn_request_channel(DBUS_G_PROXY(conn), TEXT_CHANNEL,
			       TP_CONN_HANDLE_TYPE_CONTACT, 
			       handle, TRUE,
			       &chan_object_path, &error))
  {
    printf("Error! %s\n", error -> message);
    g_assert_not_reached();
  }

  chan1 = tp_chan_new(connection, CM_BUS_NAME, chan_object_path,
		      TEXT_CHANNEL, TP_CONN_HANDLE_TYPE_CONTACT,
		      handle);

  assert(chan1);
  
  return 1;
}

/* test both set and get interfaces */
int chan_local_set_interfaces(void)
{
  TpConnMgr *connmgr;
  TpConn *conn;
  TpChan *chan1;

  GError *error;
  DBusGProxy *iface;
  gchar **interfaces;

  guint conn_status = TP_CONN_STATUS_DISCONNECTED;
  guint handle = 0;

  int condition = 0, condition2 = 0;

  GHashTable *connection_parameters = g_hash_table_new(g_str_hash, 
						       g_str_equal); 
  
  connmgr = tp_connmgr_new(connection, CM_BUS_NAME, CM_OBJECT_PATH,
			   TP_IFACE_CONN_MGR_INTERFACE);

  assert(connmgr);
 
  dbus_g_proxy_connect_signal(DBUS_G_PROXY(connmgr), "NewConnection",
                              G_CALLBACK(newconnection_handler),
                              &condition, NULL); 

  init_parameters(connection_parameters);
  
  conn = tp_connmgr_new_connection(connmgr,
				   connection_parameters,
				   PROTO);
  
  g_assert(conn);

  test_wait_g_main_until (&condition, 500);

  assert(condition == 1);

  condition2 = 0;

  dbus_g_proxy_connect_signal(DBUS_G_PROXY(conn),
			      "StatusChanged",
			      G_CALLBACK(status_change_handler),
			      &condition2, NULL);

  if (!tp_conn_get_status(DBUS_G_PROXY(conn), &conn_status,
			  &error))
    {
      g_warning("GetStatus() failed: %s\n", error->message);
      g_assert_not_reached();
    }

  if (conn_status != TP_CONN_STATUS_CONNECTED)
    test_wait_g_main_until (&condition2, 500);

  g_assert(condition2 == 1);

  if (!tp_conn_request_handle(DBUS_G_PROXY(conn), 
			      TP_CONN_HANDLE_TYPE_CONTACT,
			      CONTACT_NAME, &handle, &error))
  {
    g_warning("No handle: %s\n", error->message);
    g_assert_not_reached();
  }

  chan1 = tp_conn_new_channel(connection, conn, CM_BUS_NAME, TEXT_CHANNEL, 
			      TP_CONN_HANDLE_TYPE_CONTACT, handle, TRUE);

  assert(chan1);

  if (!tp_conn_get_interfaces(DBUS_G_PROXY(conn),
			      &interfaces, &error))
    {
      g_warning("GetStatus() failed: %s, dbus err: %s\n", 
		error->message, dbus_g_error_get_name(error));
      g_assert_not_reached();
    }

  iface = tp_chan_get_interface(chan1, 
				TELEPATHY_CHAN_IFACE_TEXT_QUARK);

  g_assert(iface);  

  return 1;
}

/******************* List of Tests   **********************/

/* 
 * List of test case functions.
 * Each test case will be run in it's own process
 */

/* result options are EXPECT_OK or EXPECT_ASSERT */

testcase cases[] = {
  {*connmgr_new, "connmgr_new", EXPECT_OK},  
  {*connmgr_new_connection, "connmgr_new_connection", EXPECT_OK}, 
  {*conn_new, "conn_new", EXPECT_OK},
  {*conn_new_channel, "conn_new_channel", EXPECT_OK},
  {*conn_local_set_interfaces, "conn_local_set_interfaces", EXPECT_OK},
  {*chan_new, "chan_new", EXPECT_OK},
  {*chan_local_set_interfaces, "chan_local_set_interfaces", EXPECT_OK}
};


/*
 * return test cases for outo
 */
testcase* get_tests (void)
{
  /*  This could be set also in the init test func */
  g_log_set_always_fatal (G_LOG_LEVEL_ERROR | 
			  G_LOG_LEVEL_CRITICAL | G_LOG_LEVEL_WARNING);
  
  return cases;
}
