/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 tw=80 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Oleg Romashin <romaxa@gmail.com>
 *
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include <stdio.h>
#include "gtkmozembed_common2.h"
#include "nsIGenericFactory.h"
#include "nsIComponentRegistrar.h"
#include "nsIComponentManager.h"
#include "nsEmbedCID.h"
#include "nsCOMPtr.h"

#include "gtkmozembed.h"
#include "gtkmozembed_internal.h"

#include "GtkPrompterService.h"
#include "EmbedPrivate.h"

#include "EmbedDownloadMgr.h"
#include "EmbedMicrobObserver.h"

// shistory
#include "nsIHistoryEntry.h"
#include "nsIURI.h"

// certs
#include "nsIDocShell.h"
#include "nsISSLStatus.h"
#include "nsISSLStatusProvider.h"
#include "nsIX509Cert.h"
#include "nsISecureBrowserUI.h"

#include "nsIClipboardCommands.h"

#include "gtkmozembed_maemo.h"

NS_GENERIC_FACTORY_CONSTRUCTOR (GtkPrompterService)

static const nsModuleComponentInfo microbAppComps[] =
{
  {
    "GtkPrompter Service",
    GTK_EMBED_PROMPTER_SERVICE_CID,
    NS_PROMPTSERVICE_CONTRACTID,
    GtkPrompterServiceConstructor
  },
  {
    "GtkPrompter Cookie Service",
    NS_COOKIEPROMPTSERVICE_CID,
    NS_COOKIEPROMPTSERVICE_CONTRACTID,
    GtkPrompterServiceConstructor
  },
  {
    "GtkPrompter Nonblocking Alert Service",
    NS_NONBLOCKINGALERTSERVICE_CID,
    "@mozilla.org/embedcomp/nbalert-service;1",
    GtkPrompterServiceConstructor
  },
  { "EmbedMicrobObserver",
    NS_EMBED_MICROB_OBSERVER_CID,
    NS_EMBED_CLIENT_OBSERVER_CONTRACTID,
    EmbedMicrobObserver::Create,
    EmbedMicrobObserver::RegisterSelf,
    EmbedMicrobObserver::UnregisterSelf
  },
#ifdef MOZ_RDF
  { "EmbedDownloadObserver",
    NS_EMBED_DOWNLOAD_CID,
    NS_EMBED_DOWNLOAD_CONTRACTID,
    EmbedDownloadObserver::Create,
    EmbedDownloadObserver::RegisterSelf,
    EmbedDownloadObserver::UnregisterSelf
  }
#endif
};

static const nsModuleComponentInfo *sMicrobAppComps = microbAppComps;
static int   sMicrobNumAppComps = sizeof(microbAppComps) / sizeof(nsModuleComponentInfo);

gboolean
gtk_moz_common_microb_init (void)
{
#if 0
  // Call gtk_moz_common_microb_init before first gtk_moz_embed_push_startup call
  // It will override all internal GtkMozEmbed components
  gtk_moz_embed_set_app_components(sMicrobAppComps, sMicrobNumAppComps);

#else
  // Call gtk_moz_common_microb_init after gtk_moz_embed_push_startup call
  // It will add microb components
  // Whould be better to create something like gtk_moz_embed_register_app_components(aAppComps, aNumAppComps);
  nsCOMPtr<nsIComponentRegistrar> cr;
  nsresult rv = NS_GetComponentRegistrar(getter_AddRefs(cr));
  NS_ENSURE_SUCCESS(rv, rv);

  nsCOMPtr<nsIComponentManager> cm;
  rv = NS_GetComponentManager (getter_AddRefs (cm));
  NS_ENSURE_SUCCESS (rv, rv);

  for (int i = 0; i < sMicrobNumAppComps; ++i) {
    nsCOMPtr<nsIGenericFactory> componentFactory;
    rv = NS_NewGenericFactory(getter_AddRefs(componentFactory),
                              &(sMicrobAppComps[i]));
    if (NS_FAILED(rv)) {
      NS_WARNING("Unable to create factory for component");
      continue;  // don't abort registering other components
    }

    rv = cr->RegisterFactory(sMicrobAppComps[i].mCID, sMicrobAppComps[i].mDescription,
                             sMicrobAppComps[i].mContractID, componentFactory);
    NS_ASSERTION(NS_SUCCEEDED(rv), "Unable to register factory for component");

    // Call the registration hook of the component, if any
    if (sMicrobAppComps[i].mRegisterSelfProc) {
      rv = sMicrobAppComps[i].mRegisterSelfProc(cm, nsnull, nsnull, nsnull,
                                                &(sMicrobAppComps[i]));
      NS_ASSERTION(NS_SUCCEEDED(rv), "Unable to self-register component");
    }
  }
#endif

  EmbedMaemoRunMemoryHandling();

  return TRUE;
}

gboolean
gtk_moz_common_microb_stop (void)
{
  EmbedMaemoStopMemoryHandling();
  
  return TRUE;
}

gint
gtk_moz_common_get_shistory_list(GtkMozEmbed *embed, GtkMozHistoryItem **GtkHI,
                                 guint type)
{
  g_return_val_if_fail ((embed != NULL), 0);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), 0);

  EmbedPrivate *embedPrivate;
  embedPrivate = (EmbedPrivate *)embed->data;
  g_return_val_if_fail (embedPrivate && embedPrivate->mSessionHistory, 0);

  PRInt32 curIndex = 0, totalCount = 0, navIndex = 0, maxItems = 0;

  //Get the current index at session History
  embedPrivate->mSessionHistory->GetIndex(&curIndex);
  //Gets the number of toplevel documents available in session history.
  embedPrivate->mSessionHistory->GetCount(&totalCount);

  if (type == GTK_MOZ_COMMON_BACK_SHISTORY) {
    navIndex = curIndex - 1;
    maxItems = curIndex;
  } else if (type == GTK_MOZ_COMMON_FORWARD_SHISTORY) {
    navIndex = curIndex + 1;
    maxItems = totalCount - navIndex;
  }

  if (maxItems <= 0)
    return 0;

  *GtkHI = g_new0(GtkMozHistoryItem, maxItems);
  GtkMozHistoryItem * item = (GtkMozHistoryItem *)*GtkHI;

  nsresult rv;
  for (PRInt32 numItems = 0; numItems < maxItems; numItems++) {
    // Get the HistoryEntry at a given index.
    nsCOMPtr<nsIHistoryEntry> curEntry;
    rv = embedPrivate->mSessionHistory->GetEntryAtIndex((navIndex), PR_FALSE,
                                                        getter_AddRefs(curEntry));
    if (NS_FAILED(rv) || (!curEntry))
      continue;

    // Get the URI of the HistoryEntry
    nsCOMPtr<nsIURI> uri;
    rv = curEntry->GetURI(getter_AddRefs(uri));
    if (NS_FAILED(rv) || (!uri))
      continue;

    nsCString uriString;
    rv = uri->GetSpec(uriString);
    if (NS_FAILED(rv) || uriString.IsEmpty())
      continue;

    // Get the title of the HistoryEntry
    PRUnichar* title;
    rv = curEntry->GetTitle (&title);
    if (NS_FAILED(rv) || (!title))
      continue;

    item[numItems].url = strdup(uriString.get());
    item[numItems].title = strdup(NS_ConvertUTF16toUTF8(title).get());
    item[numItems].accessed = 0;

    if (type == GTK_MOZ_COMMON_BACK_SHISTORY)
      navIndex--;
    else if (type == GTK_MOZ_COMMON_FORWARD_SHISTORY)
      navIndex++;
  }

  return (gint)maxItems;
}

gint
gtk_moz_common_get_shistory_index(GtkMozEmbed *embed)
{
  g_return_val_if_fail ((embed != NULL), -1);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), -1);

  PRInt32 curIndex = -1;
  EmbedPrivate *embedPrivate;

  embedPrivate = (EmbedPrivate *)embed->data;
  if (embedPrivate->mSessionHistory)
    embedPrivate->mSessionHistory->GetIndex(&curIndex);

  return (gint)curIndex;
}

void
gtk_moz_common_shistory_goto_index(GtkMozEmbed *embed, gint index)
{
  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  EmbedPrivate *embedPrivate;

  embedPrivate = (EmbedPrivate *)embed->data;
  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->GotoIndex(index);
}

gboolean
gtk_moz_common_get_server_cert(GtkMozEmbed *embed, gpointer *cert, gpointer ctx)
{
  g_return_val_if_fail(embed != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);

  nsIWebBrowser *webBrowser = nsnull;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), &webBrowser);
  NS_ENSURE_TRUE(webBrowser, FALSE);

  nsCOMPtr<nsIDocShell> docShell(do_GetInterface((nsISupports*)webBrowser));
  NS_ENSURE_TRUE(docShell, FALSE);

  nsCOMPtr<nsISecureBrowserUI> mSecureUI;
  docShell->GetSecurityUI(getter_AddRefs(mSecureUI));
  NS_ENSURE_TRUE(mSecureUI, FALSE);

  nsCOMPtr<nsISSLStatusProvider> mSecureProvider = do_QueryInterface(mSecureUI);
  NS_ENSURE_TRUE(mSecureProvider, FALSE);

  nsCOMPtr<nsISSLStatus> SSLStatus;
  mSecureProvider->GetSSLStatus(getter_AddRefs(SSLStatus));
  NS_ENSURE_TRUE(SSLStatus, FALSE);

  nsCOMPtr<nsIX509Cert> serverCert;
  SSLStatus->GetServerCert(getter_AddRefs(serverCert));
  NS_ENSURE_TRUE(serverCert, FALSE);

  *cert = serverCert;
  return TRUE;
}

gboolean
gtk_moz_common_clipboard(GtkMozEmbed *embed, guint action)
{
  g_return_val_if_fail(embed != NULL, FALSE);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);

  nsIWebBrowser *webBrowser = nsnull;
  gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed), &webBrowser);
  NS_ENSURE_TRUE(webBrowser, FALSE);

  nsresult rv = NS_OK;
  PRBool canDo = PR_TRUE;
  nsCOMPtr<nsIClipboardCommands> clipboard(do_GetInterface(webBrowser));
  NS_ENSURE_TRUE(webBrowser, FALSE);
  
    switch (action) {
    case GTK_MOZ_CLIPBOARD_SELECT_ALL:
    {
        rv = clipboard->SelectAll();
        break;
    }
    case GTK_MOZ_CLIPBOARD_CUT:
    {
        rv = clipboard->CutSelection();
        break;
    }
    case GTK_MOZ_CLIPBOARD_COPY:
    {
        rv = clipboard->CopySelection();
        break;
    }
    case GTK_MOZ_CLIPBOARD_PASTE:
    {
        rv = clipboard->Paste();
        break;
    }
    case GTK_MOZ_CLIPBOARD_CAN_CUT:
    {
        rv = clipboard->CanCutSelection(&canDo);
        break;
    }
    case GTK_MOZ_CLIPBOARD_CAN_PASTE:
    {
        rv = clipboard->CanPaste(&canDo);
        break;
    }
    case GTK_MOZ_CLIPBOARD_CAN_COPY:
    {
        rv = clipboard->CanCopySelection(&canDo);
        break;
    }
    default:
        break;
    }
    NS_ENSURE_SUCCESS(rv, FALSE);
    return canDo?TRUE:FALSE;
}


#include "nsIImageLoadingContent.h"
#include "nsCOMPtr.h"

gboolean
gtk_moz_common_load_image(gpointer node, const char *url)
{
  nsresult rv;
  nsCOMPtr<nsIImageLoadingContent> imgContent = do_QueryInterface(static_cast<nsISupports*>(node), &rv);
  nsCOMPtr<nsIURI> imgUri;
  NS_ENSURE_TRUE(imgContent, FALSE);
  rv = imgContent->GetCurrentURI(getter_AddRefs(imgUri));
  if (NS_FAILED(rv) || !imgUri)
      return FALSE;

  nsCAutoString spec;
  rv = imgUri->GetSpec(spec);
  if (NS_FAILED(rv))
      return FALSE;

  if (spec.Equals(url)) {
    gint tmp_value = 1, cur_value;
    gtk_moz_embed_common_get_pref(G_TYPE_INT, "permissions.default.image", &cur_value);
    if (cur_value != 1) {
      gtk_moz_embed_common_set_pref(G_TYPE_INT, "permissions.default.image", &tmp_value);
      imgContent->ForceReload();
      gtk_moz_embed_common_set_pref(G_TYPE_INT, "permissions.default.image", &cur_value);
    }
    else
      imgContent->ForceReload();
  }
  return TRUE;
}
