/*
 *
 * Copyright (C) 2007 Nokia Corporation.
 *
 * Contact: Igor Stoppa <igor.stoppa@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __NOTIFICATIONS_H
#define __NOTIFICATIONS_H

#include <linux/kthread.h>
#include <linux/list.h>
#include <linux/notifier.h>

/*
 * Every notification is associated to 2 driver states: waiting and done.
 * The driver state is used to detect, in case of timeout, which drivers have
 * failed to ACK the notification sent.
 */
enum dvfs_driver_state {
	STATE_WAITING_FOR_PRE = 0,
	STATE_DONE_PRE,
	STATE_WAITING_FOR_POST,
	STATE_DONE_POST,
	NUM_DVFS_DRIVER_STATES
};

enum dvfs_notifications {
	DVFS_PRE_NOTIFICATION = STATE_DONE_PRE,
	DVFS_POST_NOTIFICATION = STATE_DONE_POST,
	NUM_DVFS_NOTIFICATIONS
};

enum dvfs_threaded {
	DVFS_CHAINED_NOTIF = 0,
	DVFS_THREADED_NOTIF,
};

struct dvfs_notif {
	struct list_head list;
	struct task_struct *task;
	struct notifier_block nb;
	const char * name;
	int err;
	enum dvfs_driver_state state;
	enum dvfs_threaded threaded;
	unsigned int timeout;
};

#ifdef CONFIG_MACH_OMAP2420_DVFS

#define DVFS_NOTIF(n) container_of(n, struct dvfs_notif, nb)

int dvfs_run_notifications(enum dvfs_notifications event);
int dvfs_notif_set(struct dvfs_notif *target, const char *name,
		   int (*callback)(struct notifier_block *,
		   unsigned long, void *),
		   enum dvfs_threaded threaded, unsigned long timeout);
void dvfs_client_notification_cb(struct dvfs_notif *n);
int dvfs_register_notifier(struct dvfs_notif *n);
int dvfs_unregister_notifier(struct dvfs_notif *n);

#else

#define DVFS_NOTIF(n) NULL

static inline int dvfs_run_notifications(enum dvfs_notifications event)
{
	return 0;
}

static inline int dvfs_notif_set(struct dvfs_notif *target, const char *name,
				 int (*callback)(struct notifier_block *,
				 unsigned long, void *),
				 enum dvfs_threaded threaded,
				 unsigned long timeout)
{
	return 0;
}
static inline void dvfs_client_notification_cb(struct dvfs_notif *n) { }
static inline int dvfs_register_notifier(struct dvfs_notif *n) { return 0;}
static inline int dvfs_unregister_notifier(struct dvfs_notif *n) { return 0;}
#endif /* CONFIG_MACH_OMAP2420_DVFS */

#endif /* __NOTIFICATIONS_H */
