/*
 * Optional support for a non-standard TUSB6010 loopback test
 *
 * Copyright (C) 2006 Nokia Corporation
 * Tony Lindgren <tony@atomide.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * In order for the test to work, the dongle musb connect DM to DP over a 1k
 * resistor when VBUS is enabled. By using direct OTG PHY register access
 * DM line is raised by the test function, which then also raises DP line if
 * the dongle is connected.
 *
 * To run the test, connect the loopback dongle to the USB interface on the
 * board and type the following on the board:
 *
 * # cat /sys/devices/platform/musb_hdrc/looptest
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/irq.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <asm/mach-types.h>

#include "musb_core.h"
#include "tusb6010.h"

#define TUSB_TEST_PASSED		0
#define TUSB_TEST_FAILED		-1
#define TUSB_TEST_INIT_ERROR		-2

#define TUSB_PIN_TEST_MASK	(TUSB_DEV_OTG_STAT_DP_ENABLE |	\
					TUSB_DEV_OTG_STAT_DM_ENABLE)

#define test_otg()		(!machine_is_nokia_n800())

static struct musb *the_musb;

static inline void tusb_test_debug(u32 otg_stat, char *msg)
{
	DBG(2, "%s %s %s otg_stat: 0x%08x %s\n",
		otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS ? "ID" : "  ",
		otg_stat & TUSB_DEV_OTG_STAT_DP_ENABLE ? "D+" : "  ",
		otg_stat & TUSB_DEV_OTG_STAT_DM_ENABLE ? "D-" : "  ",
		otg_stat, msg);
}

static inline void tusb_test_set_mode(struct musb *musb, u8 musb_mode)
{
	void __iomem	*mbase = musb->mregs;
	void __iomem	*tbase = musb->ctrl_base;
	u32		dev_conf, otg_stat;
	u8		devctl;

	dev_conf = musb_readl(tbase, TUSB_DEV_CONF);
	devctl = musb_readb(mbase, MUSB_DEVCTL);
	switch (musb_mode) {
	case MUSB_HOST:
		otg_stat = musb_readl(tbase, TUSB_DEV_OTG_STAT);
		tusb_test_debug(otg_stat, "Forcing VBUS on");
		musb_platform_set_mode(the_musb, MUSB_HOST);
		dev_conf |= TUSB_DEV_CONF_USB_HOST_MODE;
		devctl |= MUSB_DEVCTL_SESSION;
		break;
	case MUSB_PERIPHERAL:
		musb_platform_set_mode(the_musb, MUSB_PERIPHERAL);
		devctl &= ~MUSB_DEVCTL_SESSION;
		break;
	case MUSB_OTG:	/* Use OTG for ID pin, turn session on */
		musb_platform_set_mode(the_musb, MUSB_OTG);
		dev_conf &= ~TUSB_DEV_CONF_USB_HOST_MODE;
		devctl |= MUSB_DEVCTL_SESSION;
		break;
	}
	musb_writeb(mbase, MUSB_DEVCTL, devctl);
	musb_writel(tbase, TUSB_DEV_CONF, dev_conf);
}

/*
 * Runs a loopback test if a test device is connected.
 * REVISIT: Change mdelay() to msleep() once GPIO level interrupt
 * masking works. Note that we cannot currently mask tusb6010
 * wake-up interrupt in the device.
 */
static int tusb_do_test(void)
{
	void __iomem	*base = the_musb->ctrl_base;
	unsigned long	flags;
	u32		int_mask, phy_otg_ena, phy_otg_ctrl, prcm_conf, otg_stat;
	int		retries;
	int		ret = 0;

	if (the_musb == NULL)
		return TUSB_TEST_INIT_ERROR;

	spin_lock_irqsave(&the_musb->lock, flags);
	int_mask = musb_readl(base, TUSB_INT_MASK);
	musb_writel(base, TUSB_INT_MASK, ~TUSB_INT_MASK_RESERVED_BITS);
	set_irq_type(the_musb->nIrq, IRQ_TYPE_NONE);

	tusb_allow_idle(the_musb, 0);

	/* Save registers */
	phy_otg_ena = musb_readl(base, TUSB_PHY_OTG_CTRL_ENABLE);
	phy_otg_ctrl = musb_readl(base, TUSB_PHY_OTG_CTRL);
	prcm_conf = musb_readl(base, TUSB_PRCM_CONF);

	/* Enable access to PHY OTG registers */
	musb_writel(base, TUSB_PHY_OTG_CTRL_ENABLE, TUSB_PHY_OTG_CTRL_WRPROTECT
			| phy_otg_ena | TUSB_PHY_OTG_CTRL_OTG_ID_PULLUP
			| TUSB_PHY_OTG_CTRL_DM_PULLDOWN);

	/* Check ID line. It should be up at this point */
	otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	if (test_otg() && !(otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS)) {
		tusb_test_debug(otg_stat, "ID down before VBUS is on");
		ret = TUSB_TEST_INIT_ERROR;
		goto restore;
	}

	/* Pull down data lines */
	musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT
			 | (phy_otg_ctrl | TUSB_PHY_OTG_CTRL_DM_PULLDOWN
				| TUSB_PHY_OTG_CTRL_DP_PULLDOWN));

	/* Turn on VBUS as the test device needs it to ground ID pin */
	tusb_test_set_mode(the_musb, MUSB_HOST);
	mdelay(500);

	/* Wait for the test device to ground ID pin */
	retries = 30;
	tusb_test_set_mode(the_musb, MUSB_OTG);
	otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	while (test_otg() && (otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS)) {
		tusb_test_debug(otg_stat, "ID still high after VBUS");
		if (retries-- < 1) {
			tusb_test_debug(otg_stat, "Timeout waiting ID ground");
			ret = TUSB_TEST_FAILED;
			goto restore;
		}
		tusb_test_set_mode(the_musb, MUSB_HOST);
		musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT
			 | (phy_otg_ctrl | TUSB_PHY_OTG_CTRL_DM_PULLDOWN));
		mdelay(100);
		tusb_test_set_mode(the_musb, MUSB_OTG);
		otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	}

	/* Check lines. DM and DP lines should be now down */
	otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	if (otg_stat & TUSB_PIN_TEST_MASK) {
		tusb_test_debug(otg_stat, "Line(s) up after VBUS is on");
		ret = TUSB_TEST_INIT_ERROR;
		goto restore;
	}

	/*
	 * Try to raise DM line. If the loopback dongle is connected, DP
	 * line should go up as well.
	 */
	for (retries = 0; retries < 30; retries++) {
		static int	connected = 0;

		tusb_test_set_mode(the_musb, MUSB_HOST);
		mdelay(100);

		musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT
			| (phy_otg_ctrl & ~TUSB_PHY_OTG_CTRL_DM_PULLDOWN));

		otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);

		if (otg_stat & TUSB_DEV_OTG_STAT_DP_ENABLE)
			connected++;
		else
			connected = 0;

		if (connected)
			tusb_test_debug(otg_stat, "D- & D+ connected");
		else
			tusb_test_debug(otg_stat, "D- & D+ disconnected");

		if (connected >= 5) {
			ret = TUSB_TEST_PASSED;
			goto restore;
		}
		mdelay(100);
	}

	/* Failed to raise DP line */
	tusb_test_debug(otg_stat, "DP down after test");
	ret = TUSB_TEST_FAILED;

restore:
	tusb_test_set_mode(the_musb, MUSB_PERIPHERAL);

	musb_writel(base, TUSB_PHY_OTG_CTRL_ENABLE, phy_otg_ena);
	musb_writel(base, TUSB_PHY_OTG_CTRL, phy_otg_ctrl);
	musb_writel(base, TUSB_PRCM_CONF, prcm_conf);

	musb_platform_try_idle(the_musb, 0);

	musb_writel(base, TUSB_INT_SRC_CLEAR,
			0xffffffff & ~TUSB_INT_MASK_RESERVED_BITS);
	musb_writel(base, TUSB_INT_MASK, int_mask);
	spin_unlock_irqrestore(&the_musb->lock, flags);
	set_irq_type(the_musb->nIrq, IRQ_TYPE_LEVEL_LOW);

	return ret;
}

static ssize_t tusb_show_result(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	char		*s;
	int		len;

	switch (tusb_do_test()) {
	case TUSB_TEST_PASSED:
		s = "passed";
		break;
	case TUSB_TEST_FAILED:
		s = "failed";
		break;
	case TUSB_TEST_INIT_ERROR:
		s = "init error";
		break;
	default:
		s = "unknown error";
		break;
	}

	len = sprintf(buf, "%s\n", s);

	return len;
}

static DEVICE_ATTR(looptest, 0440, tusb_show_result, NULL);

int __devinit tusb_test_init(struct musb *musb)
{
	struct platform_device	*pdev;
	struct device		*dev;

	the_musb = musb;

	pdev = to_platform_device(musb->controller);
	dev = &pdev->dev;
	return device_create_file(dev, &dev_attr_looptest);
}

void __devexit tusb_test_release(void)
{
	struct platform_device	*pdev;
	struct device		*dev;

	if (the_musb == NULL)
		return;

	pdev = to_platform_device(the_musb->controller);
	dev = &pdev->dev;
	device_remove_file(dev, &dev_attr_looptest);
}
