/*
 * TSC2005 touchscreen driver
 *
 * Copyright (C) 2006-2007 Nokia Corporation
 *
 * Written by Timo O. Karjalainen <timo.o.karjalainen@nokia.com>
 * based on TSC2301 driver by Jarkko Oikarinen, Imre Deak and Juha Yrjola
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/spi/spi.h>
#include <asm/arch/dvfs_notif.h>

#ifdef CONFIG_ARCH_OMAP
#include <asm/arch/gpio.h>
#endif

#include <linux/spi/tsc2005.h>

/**
 * The touchscreen interface operates as follows:
 *
 * Initialize:
 *    Request access to GPIO103 (DAV)
 *    tsc2005_dav_irq_handler will trigger when DAV line goes down
 *
 *  1) Pen is pressed against touchscreeen
 *  2) TSC2005 performs AD conversion
 *  3) After the conversion is done TSC2005 drives DAV line down
 *  4) GPIO IRQ is received and tsc2005_dav_irq_handler is called
 *  5) If gpio irq is enabled, irq is disabld and handler calls spi_async()
 *  6) SPI framework calls tsc2005_ts_rx after the coordinates are read
 *  7) tsc2005_ts_rx() reports coordinates to input layer and
 *     sets up tsc2005_ts_timer() to be called after TSC2005_TS_SCAN_TIME
 *  8) If the pen is down, tsc2005_tx_timer() calls spi_async(), causing
 *     steps 6 to 8 to repeat.
 *  9) if tsc2005_tx_timer() notices that the pen has been lifted, the lift
 *     event is sent, and irq is again enabled.
 */

#define TSC2005_HZ	(14000000)

#define TSC2005_CMD	(0x80)
#define TSC2005_REG	(0x00)

#define TSC2005_CMD_STOP	(1)
#define TSC2005_CMD_10BIT	(0 << 2)
#define TSC2005_CMD_12BIT	(1 << 2)

#define TSC2005_CMD_SCAN_XYZZ	(0 << 3)
#define TSC2005_CMD_SCAN_XY	(1 << 3)
#define TSC2005_CMD_SCAN_X	(2 << 3)
#define TSC2005_CMD_SCAN_Y	(3 << 3)
#define TSC2005_CMD_SCAN_ZZ	(4 << 3)
#define TSC2005_CMD_AUX_SINGLE	(5 << 3)
#define TSC2005_CMD_TEMP1	(6 << 3)
#define TSC2005_CMD_TEMP2	(7 << 3)
#define TSC2005_CMD_AUX_CONT	(8 << 3)
#define TSC2005_CMD_TEST_X_CONN	(9 << 3)
#define TSC2005_CMD_TEST_Y_CONN	(10 << 3)
/* command 11 reserved */
#define TSC2005_CMD_TEST_SHORT	(12 << 3)
#define TSC2005_CMD_DRIVE_XX	(13 << 3)
#define TSC2005_CMD_DRIVE_YY	(14 << 3)
#define TSC2005_CMD_DRIVE_YX	(15 << 3)

#define TSC2005_REG_X		(0 << 3)
#define TSC2005_REG_Y		(1 << 3)
#define TSC2005_REG_Z1		(2 << 3)
#define TSC2005_REG_Z2		(3 << 3)
#define TSC2005_REG_AUX		(4 << 3)
#define TSC2005_REG_TEMP1	(5 << 3)
#define TSC2005_REG_TEMP2	(6 << 3)
#define TSC2005_REG_STATUS	(7 << 3)
#define TSC2005_REG_AUX_HIGH	(8 << 3)
#define TSC2005_REG_AUX_LOW	(9 << 3)
#define TSC2005_REG_TEMP_HIGH	(10 << 3)
#define TSC2005_REG_TEMP_LOW	(11 << 3)
#define TSC2005_REG_CFR0	(12 << 3)
#define TSC2005_REG_CFR1	(13 << 3)
#define TSC2005_REG_CFR2	(14 << 3)
#define TSC2005_REG_FUNCTION	(15 << 3)

#define TSC2005_REG_READ	(0x01)
#define TSC2005_REG_WRITE	(0x00)


#define TSC2005_CFR0_LONGSAMPLING	(0x0001)
#define TSC2005_CFR0_DETECTINWAIT	(0x0002)
#define TSC2005_CFR0_SENSETIME_32US	(0x0000)
#define TSC2005_CFR0_SENSETIME_96US	(0x0004)
#define TSC2005_CFR0_SENSETIME_544US	(0x0008)
#define TSC2005_CFR0_SENSETIME_2080US	(0x0010)
#define TSC2005_CFR0_SENSETIME_2656US	(0x001C)
#define TSC2005_CFR0_PRECHARGE_20US	(0x0000)
#define TSC2005_CFR0_PRECHARGE_84US	(0x0020)
#define TSC2005_CFR0_PRECHARGE_276US	(0x0040)
#define TSC2005_CFR0_PRECHARGE_1044US	(0x0080)
#define TSC2005_CFR0_PRECHARGE_1364US	(0x00E0)
#define TSC2005_CFR0_STABTIME_0US	(0x0000)
#define TSC2005_CFR0_STABTIME_100US	(0x0100)
#define TSC2005_CFR0_STABTIME_500US	(0x0200)
#define TSC2005_CFR0_STABTIME_1MS	(0x0300)
#define TSC2005_CFR0_STABTIME_5MS	(0x0400)
#define TSC2005_CFR0_STABTIME_100MS	(0x0700)
#define TSC2005_CFR0_CLOCK_4MHZ		(0x0000)
#define TSC2005_CFR0_CLOCK_2MHZ		(0x0800)
#define TSC2005_CFR0_CLOCK_1MHZ		(0x1000)
#define TSC2005_CFR0_RESOLUTION12	(0x2000)
#define TSC2005_CFR0_STATUS		(0x4000)
#define TSC2005_CFR0_PENMODE		(0x8000)

#define TSC2005_CFR0_INITVALUE	(TSC2005_CFR0_DETECTINWAIT |	\
				 TSC2005_CFR0_STABTIME_1MS  |	\
				 TSC2005_CFR0_CLOCK_1MHZ    |	\
				 TSC2005_CFR0_RESOLUTION12  |	\
				 TSC2005_CFR0_PENMODE)

#define TSC2005_CFR1_BATCHDELAY_0MS	(0x0000)
#define TSC2005_CFR1_BATCHDELAY_1MS	(0x0001)
#define TSC2005_CFR1_BATCHDELAY_2MS	(0x0002)
#define TSC2005_CFR1_BATCHDELAY_4MS	(0x0003)
#define TSC2005_CFR1_BATCHDELAY_10MS	(0x0004)
#define TSC2005_CFR1_BATCHDELAY_20MS	(0x0005)
#define TSC2005_CFR1_BATCHDELAY_40MS	(0x0006)
#define TSC2005_CFR1_BATCHDELAY_100MS	(0x0007)

#define TSC2005_CFR1_INITVALUE	(TSC2005_CFR1_BATCHDELAY_1MS)

#define TSC2005_CFR2_MAVE_TEMP	(0x0001)
#define TSC2005_CFR2_MAVE_AUX	(0x0002)
#define TSC2005_CFR2_MAVE_Z	(0x0004)
#define TSC2005_CFR2_MAVE_Y	(0x0008)
#define TSC2005_CFR2_MAVE_X	(0x0010)
#define TSC2005_CFR2_AVG_1	(0x0000)
#define TSC2005_CFR2_AVG_3	(0x0400)
#define TSC2005_CFR2_AVG_7	(0x0800)
#define TSC2005_CFR2_MEDIUM_1	(0x0000)
#define TSC2005_CFR2_MEDIUM_3	(0x1000)
#define TSC2005_CFR2_MEDIUM_7	(0x2000)
#define TSC2005_CFR2_MEDIUM_15	(0x3000)

#define TSC2005_CFR2_IRQ_DAV	(0x4000)
#define TSC2005_CFR2_IRQ_PEN	(0x8000)
#define TSC2005_CFR2_IRQ_PENDAV	(0x0000)

#define TSC2005_CFR2_INITVALUE	(TSC2005_CFR2_IRQ_DAV  |	\
				 TSC2005_CFR2_MAVE_X    |	\
				 TSC2005_CFR2_MAVE_Y    |	\
				 TSC2005_CFR2_MAVE_Z    |	\
				 TSC2005_CFR2_MEDIUM_15 |	\
				 TSC2005_CFR2_AVG_7)

/*
  #define TSC2301_TOUCHSCREEN_PRODUCT_ID      		0x0052
  #define TSC2301_TOUCHSCREEN_PRODUCT_VERSION 		0x0001
*/

#define TSC2005_TS_SCAN_TIME		     		6

#define MAX_12BIT					((1 << 12) - 1)


static const u32 tsc2005_read_reg[] = {
	(0x100 | TSC2005_REG | TSC2005_REG_X | TSC2005_REG_READ) << 16,
	(0x100 | TSC2005_REG | TSC2005_REG_Y | TSC2005_REG_READ) << 16,
	(0x100 | TSC2005_REG | TSC2005_REG_Z1 | TSC2005_REG_READ) << 16,
	(0x100 | TSC2005_REG | TSC2005_REG_Z2 | TSC2005_REG_READ) << 16,
	(0x100 | TSC2005_REG | TSC2005_REG_STATUS | TSC2005_REG_READ) << 16
};
#define NUM_READ_REGS	(sizeof(tsc2005_read_reg)/sizeof(tsc2005_read_reg[0]))

struct tsc2005 {
	struct spi_device	*spi;

	s16			reset_gpio;
	u16			config2_shadow;

	struct input_dev	*idev;
	char			phys[32];
	struct timer_list	timer;
	spinlock_t		lock;

	struct spi_message	read_msg;
	struct spi_transfer	read_xfer[NUM_READ_REGS];
	u32                     data[NUM_READ_REGS];

	int			hw_avg_max;
	u16			x;
	u16			y;
	u16			p;
	int			sample_cnt;

	int			ignore_last : 1;
	u16			x_plate_ohm;
	int			stab_time;
	int			max_pressure;
	int			touch_pressure;
	int			pressure_limit;

	u16			irq_enabled:1;
	u16			pen_down:1;
	u16			disabled:1;
	u16			pending:1;
	u16			pnd:1;	/* power not down */

	int			hw_flags;

	s16			dav_gpio;
	int			irq;

#ifdef CONFIG_MACH_OMAP2420_DVFS
	struct dvfs_notif	notifier;
	u16			dvfs_pause:1;
	u16			sampling:1;
#endif
};

static inline void tsc2005_pen_lifted(struct tsc2005 *ts);
static inline void tsc2005_continue_sampling(struct tsc2005 *ts);

/* TSC2005 needs a 1-bit followed by 8 bits of command. */
static void tsc2005_cmd(struct tsc2005 *ts, u8 cmd)
{
	u16 data = (1 << 8) | TSC2005_CMD | TSC2005_CMD_12BIT | cmd;
	struct spi_message msg;
	struct spi_transfer xfer = { 0 };

	spi_message_init(&msg);
	msg.spi = ts->spi;
	xfer.tx_buf = &data;
	xfer.rx_buf = NULL;
	xfer.len = 2;
	xfer.bits_per_word = 9;

	spi_message_add_tail(&xfer, &msg);
	spi_sync(ts->spi, &msg);
}

/*
 *  To write a register, we send a 1-bit + 8 bit command + 16 bits of new
 * register value.
 */
static void tsc2005_write(struct tsc2005 *ts, u8 reg, u16 value)
{
	u32 tx;
	struct spi_message msg;
	struct spi_transfer xfer = { 0 };

	tx = 1 << (3 * 8);
	tx |= (TSC2005_REG | reg | (ts->pnd << 1) |
	       TSC2005_REG_WRITE) << (2 * 8);
	tx |= value;

	spi_message_init(&msg);
	msg.spi = ts->spi;
	xfer.tx_buf = &tx;
	xfer.rx_buf = NULL;
	xfer.len = 4;
	xfer.bits_per_word = 1 + 3 * 8;

	spi_message_add_tail(&xfer, &msg);
	spi_sync(ts->spi, &msg);
}

static int tsc2005_ts_check_config(struct tsc2005 *ts, int *hw_flags)
{
	int flags;

	flags = TSC2005_CMD_SCAN_XYZZ;

	*hw_flags = flags;
	return 0;
}

static int device_suspended(struct device *dev)
{
	struct tsc2005 *tsc = dev_get_drvdata(dev);
	return dev->power.power_state.event != PM_EVENT_ON || tsc->disabled;
}

static void update_pen_state(struct tsc2005 *ts, int x, int y, int pressure)
{
	int sync = 0;

	if (pressure) {
		input_report_abs(ts->idev, ABS_X, x);
		input_report_abs(ts->idev, ABS_Y, y);
		input_report_abs(ts->idev, ABS_PRESSURE, pressure);
		if (!ts->pen_down)
			input_report_key(ts->idev, BTN_TOUCH, 1);
		sync = 1;
	} else if (ts->pen_down) {
		input_report_abs(ts->idev, ABS_PRESSURE, 0);
		input_report_key(ts->idev, BTN_TOUCH, 0);
		sync = 1;
	}

	if (sync)
		input_sync(ts->idev);

	ts->pen_down = pressure ? 1 : 0;

#ifdef VERBOSE
	dev_dbg(&tsc->spi->dev, "x %4d y %4d p %4d\n", x, y, pressure);
#endif
}

/*
 * This procedure is called by the SPI framework after the coordinates
 * have been read from TSC2005
 */
static void tsc2005_ts_rx(void *arg)
{
	struct tsc2005 *ts = arg;
	unsigned int x, y, z1, z2, pressure;

	x  = ts->data[0];
	y  = ts->data[1];
	z1 = ts->data[2];
	z2 = ts->data[3];

	if (z1) {
		pressure = ts->x_plate_ohm * x;
		pressure /= 4096;
		pressure *= z2 - z1;
		pressure /= z1;
	} else
		pressure = 0;

	/*
	 * If pressure value is above a preset limit (pen is barely
	 * touching the screen) we can't trust the coordinate values.
	 */
	if (pressure < ts->pressure_limit && x < MAX_12BIT && y < MAX_12BIT) {
		ts->pressure_limit = ts->max_pressure;
		if (ts->ignore_last) {
			if (ts->sample_cnt)
				update_pen_state(ts, ts->x, ts->y, ts->p);
			ts->x = x;
			ts->y = y;
			ts->p = pressure;
		} else
			update_pen_state(ts, x, y, pressure);
		ts->sample_cnt++;
	}

	mod_timer(&ts->timer, jiffies +
		  msecs_to_jiffies(TSC2005_TS_SCAN_TIME));
}

static int is_pen_down(struct tsc2005 *ts)
{
	return ts->pen_down;
}


/*
 * Timer is called every TSC2005_TS_SCAN_TIME when the pen is down
 */
static void tsc2005_ts_timer(unsigned long arg)
{
	struct tsc2005 *ts = (void *) arg;
	unsigned long flags;
	int ndav;

	spin_lock_irqsave(&ts->lock, flags);
	ndav = omap_get_gpio_datain(ts->dav_gpio);
	if (ndav || device_suspended(&ts->spi->dev)) {
		tsc2005_pen_lifted(ts);
		spin_unlock_irqrestore(&ts->lock, flags);
	} else {
		ts->pen_down = 1;
		spin_unlock_irqrestore(&ts->lock, flags);
		tsc2005_continue_sampling(ts);
	}
}

/*
 * This interrupt is called when pen is down and first coordinates are
 * available. That is indicated by a falling edge on DEV line.  IRQ is
 * disabled here because while the pen is down the coordinates are
 * read by a timer.
 */
static irqreturn_t tsc2005_ts_irq_handler(int irq, void *dev_id)
{
	struct tsc2005 *ts = dev_id;
	unsigned long flags;

	spin_lock_irqsave(&ts->lock, flags);

	if (ts->irq_enabled) {
		ts->irq_enabled = 0;
		disable_irq(ts->irq);
		ts->pending = 1;
		ts->pressure_limit = ts->touch_pressure;
		ts->sample_cnt = 0;
		spi_async(ts->spi, &ts->read_msg);
	}
	spin_unlock_irqrestore(&ts->lock, flags);

	return IRQ_HANDLED;
}


void tsc2005_ts_prep_for_clk_stop(struct tsc2005 *tsc)
{
}

void tsc2005_ts_cont_after_clk_stop(struct tsc2005 *tsc)
{
}

static void tsc2005_ts_setup_spi_xfer(struct tsc2005 *ts)
{
	struct spi_message *m = &ts->read_msg;
	struct spi_transfer *x = &ts->read_xfer[0];
	int i;

	spi_message_init(m);
	m->spi = ts->spi;

	for (i = 0; i < NUM_READ_REGS; i++, x++ ) {
		x->tx_buf = &tsc2005_read_reg[i];
		x->rx_buf = &ts->data[i];
		x->len = 4;
		x->bits_per_word = 25;
		x->cs_change = i < (NUM_READ_REGS - 1);
		spi_message_add_tail(x, m);
	}

	m->complete = tsc2005_ts_rx;
	m->context = ts;
}

static ssize_t tsc2005_ts_pen_down_show(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct tsc2005 *tsc = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", is_pen_down(tsc));
}

static DEVICE_ATTR(pen_down, S_IRUGO, tsc2005_ts_pen_down_show, NULL);

static int tsc2005_configure(struct tsc2005 *tsc, int flags)
{
	tsc2005_write(tsc, TSC2005_REG_CFR0, TSC2005_CFR0_INITVALUE);
	tsc2005_write(tsc, TSC2005_REG_CFR1, TSC2005_CFR1_INITVALUE);
	tsc2005_write(tsc, TSC2005_REG_CFR2, TSC2005_CFR2_INITVALUE);
	tsc2005_cmd(tsc, flags);

	return 0;
}

static void tsc2005_start_scan(struct tsc2005 *tsc)
{
	tsc2005_configure(tsc, tsc->hw_flags);
}

static void tsc2005_stop_scan(struct tsc2005 *tsc)
{
	tsc2005_configure(tsc, TSC2005_CMD_STOP);
}

/* Must be called with ts->lock held */
static void tsc2005_disable(struct tsc2005 *ts)
{
	if (ts->disabled)
		return;

	ts->disabled = 1;
	if (!ts->pending) {
		ts->irq_enabled = 0;
		disable_irq(ts->irq);
	} else {
		while (ts->pending) {
			spin_unlock_irq(&ts->lock);
			msleep(1);
			spin_lock_irq(&ts->lock);
		}
	}

	spin_unlock_irq(&ts->lock);
	tsc2005_stop_scan(ts);
	spin_lock_irq(&ts->lock);
}

static void tsc2005_enable(struct tsc2005 *ts)
{
	if (!ts->disabled)
		return;

	ts->disabled = 0;
	ts->irq_enabled = 1;
	enable_irq(ts->irq);

	spin_unlock_irq(&ts->lock);
	tsc2005_start_scan(ts);
	spin_lock_irq(&ts->lock);
}

#ifdef CONFIG_MACH_OMAP2420_DVFS

#define DVFS_TIMEOUT (6 * 11 / 10)

static inline void tsc2005_pen_lifted(struct tsc2005 *ts)
{
	if (!device_suspended(&ts->spi->dev)) {
		if (ts->dvfs_pause) {
			ts->sampling = 0;
		} else {
			ts->irq_enabled = 1;
			enable_irq(ts->irq);
		}
	}
	update_pen_state(ts, 0, 0, 0);
	ts->pending = 0;
	if (ts->dvfs_pause)
		dvfs_client_notification_cb(&ts->notifier);
}

static inline void tsc2005_continue_sampling(struct tsc2005 *ts)
{
	if (ts->dvfs_pause)
		dvfs_client_notification_cb(&ts->notifier);
	else if (spi_async(ts->spi, &ts->read_msg))
		dev_err(&ts->spi->dev, "ts: spi_async() failed");
}

static void tsc2005_pause(struct tsc2005 *ts)
{
	unsigned long flags;

	spin_lock_irqsave(&ts->lock, flags);
	ts->dvfs_pause = 1;

	if (ts->disabled) {
		spin_unlock_irqrestore(&ts->lock, flags);
		return;
	}

	if (ts->irq_enabled) {
		disable_irq(ts->irq);
		ts->irq_enabled = 0;
		ts->sampling = 0;
		spin_unlock_irqrestore(&ts->lock, flags);
		dvfs_client_notification_cb(&ts->notifier);
	} else {
		ts->sampling = 1;
		spin_unlock_irqrestore(&ts->lock, flags);
	}
}

static void tsc2005_continue(struct tsc2005 *ts)
{
	unsigned long flags;

	spin_lock_irqsave(&ts->lock, flags);
	ts->dvfs_pause = 0;

	if (ts->disabled) {
		spin_unlock_irqrestore(&ts->lock, flags);
		return;
	}

	if (ts->sampling) {
		spin_unlock_irqrestore(&ts->lock, flags);
		dvfs_client_notification_cb(&ts->notifier);
		spi_async(ts->spi, &ts->read_msg);
	} else {
		ts->irq_enabled = 1;
		enable_irq(ts->irq);
		spin_unlock_irqrestore(&ts->lock, flags);
		dvfs_client_notification_cb(&ts->notifier);
	}
}

static int tsc2005_dvfs(struct notifier_block *nb,
               unsigned long state, void *p)
{
	struct tsc2005 *ts;

	ts = container_of(DVFS_NOTIF(nb), struct tsc2005, notifier);

	if (state == DVFS_PRE_NOTIFICATION)
		tsc2005_pause(ts);
	else if (state == DVFS_POST_NOTIFICATION)
		tsc2005_continue(ts);


	return NOTIFY_DONE;
}

static void tsc2005_dvfs_init(struct tsc2005 *ts)
{
	ts->dvfs_pause = 0;
	ts->sampling = 0;
	dvfs_notif_set(&ts->notifier, "tsc2005", tsc2005_dvfs,
		       DVFS_THREADED_NOTIF, DVFS_TIMEOUT);
	dvfs_register_notifier(&ts->notifier);
}

static void inline tsc2005_dvfs_done(struct tsc2005 *ts)
{
	dvfs_unregister_notifier(&ts->notifier);
}
#else

static void tsc2005_dvfs_init(struct tsc2005 *ts) {}
static void inline tsc2005_dvfs_done(struct tsc2005 *ts) {}

static inline void tsc2005_pen_lifted(struct tsc2005 *ts)
{
	if (!device_suspended(&ts->spi->dev)) {
		ts->irq_enabled = 1;
		enable_irq(ts->irq);
	}
	update_pen_state(ts, 0, 0, 0);
	ts->pending = 0;
}

static inline void tsc2005_continue_sampling(struct tsc2005 *ts)
{
	if (spi_async(ts->spi, &ts->read_msg))
		dev_err(&ts->spi->dev, "ts: spi_async() failed");
}
#endif

static ssize_t tsc2005_disable_show(struct device *dev,
				    struct device_attribute *attr, char *buf)
{
	struct tsc2005 *ts = dev_get_drvdata(dev);

	return sprintf(buf, "%u\n", ts->disabled);
}

static ssize_t tsc2005_disable_store(struct device *dev,
				     struct device_attribute *attr,
				     const char *buf, size_t count)
{
	struct tsc2005		*tsc = dev_get_drvdata(dev);
	char *endp;
	int i;

	i = simple_strtoul(buf, &endp, 10);
	spin_lock_irq(&tsc->lock);

	if (i)
		tsc2005_disable(tsc);
	else
		tsc2005_enable(tsc);

	spin_unlock_irq(&tsc->lock);

	return count;
}

static DEVICE_ATTR(disable_ts, 0664, tsc2005_disable_show,
		   tsc2005_disable_store);


static int __devinit tsc2005_ts_init(struct tsc2005 *ts,
				     struct tsc2005_platform_data *pdata)
{
	struct input_dev *idev;
	int dav_gpio, r;

	if (pdata->dav_gpio < 0) {
		dev_err(&ts->spi->dev, "need DAV GPIO");
		return -EINVAL;
	}
	dav_gpio = pdata->dav_gpio;
	ts->dav_gpio = dav_gpio;
	dev_dbg(&ts->spi->dev, "TSC2005: DAV GPIO = %d\n", dav_gpio);

#ifdef CONFIG_ARCH_OMAP
	r = omap_request_gpio(dav_gpio);
	if (r < 0) {
		dev_err(&ts->spi->dev, "unable to get DAV GPIO");
		goto err1;
	}
	omap_set_gpio_direction(dav_gpio, 1);
	ts->irq = OMAP_GPIO_IRQ(dav_gpio);
	dev_dbg(&ts->spi->dev, "TSC2005: DAV IRQ = %d\n", ts->irq);
#endif
	init_timer(&ts->timer);
	ts->timer.data = (unsigned long) ts;
	ts->timer.function = tsc2005_ts_timer;

	spin_lock_init(&ts->lock);

	ts->x_plate_ohm	= pdata->ts_x_plate_ohm ? : 280;
	ts->hw_avg_max	= pdata->ts_hw_avg;
	ts->max_pressure= pdata->ts_max_pressure ? : MAX_12BIT;
	ts->touch_pressure = pdata->ts_touch_pressure ? : ts->max_pressure;
	ts->ignore_last	= pdata->ts_ignore_last;
	ts->stab_time	= pdata->ts_stab_time;

	if ((r = tsc2005_ts_check_config(ts, &ts->hw_flags))) {
		dev_err(&ts->spi->dev, "invalid configuration\n");
		goto err2;
	}

	idev = input_allocate_device();
	if (idev == NULL) {
		r = -ENOMEM;
		goto err2;
	}

	/*
	 * TODO: should be "TSC2005 touchscreen", but X has hardcoded these
	 * strings and doesn't accept TSC2005 yet...
	 */
	idev->name = "TSC2301 touchscreen";
	snprintf(ts->phys, sizeof(ts->phys), "%s/input-ts",
		 ts->spi->dev.bus_id);
	idev->phys = ts->phys;

	idev->evbit[0] = BIT(EV_ABS) | BIT(EV_KEY);
	idev->absbit[0] = BIT(ABS_X) | BIT(ABS_Y) | BIT(ABS_PRESSURE);
	ts->idev = idev;

	tsc2005_ts_setup_spi_xfer(ts);

	/* These parameters should perhaps be configurable? */
	input_set_abs_params(idev, ABS_X, 0, 4096, 4, 0);
	input_set_abs_params(idev, ABS_Y, 0, 4096, 7, 0);
	input_set_abs_params(idev, ABS_PRESSURE, 0, 1024, 2, 0);

	tsc2005_start_scan(ts);

	ts->irq_enabled = 1;
	r = request_irq(ts->irq, tsc2005_ts_irq_handler,
			IRQF_TRIGGER_FALLING | IRQF_DISABLED |
			IRQF_SAMPLE_RANDOM, "tsc2005", ts);
	if (r < 0) {
		dev_err(&ts->spi->dev, "unable to get DAV IRQ");
		goto err3;
	}

	set_irq_wake(ts->irq, 1);

	r = input_register_device(idev);
	if (r < 0) {
		dev_err(&ts->spi->dev, "can't register touchscreen device\n");
		goto err4;
	}

	/* We can tolerate these failing */
	if (device_create_file(&ts->spi->dev, &dev_attr_pen_down));
	if (device_create_file(&ts->spi->dev, &dev_attr_disable_ts));

	return 0;
err4:
	free_irq(ts->irq, ts);
err3:
	tsc2005_stop_scan(ts);
	input_free_device(idev);
err2:
#ifdef CONFIG_ARCH_OMAP
	omap_free_gpio(dav_gpio);
#endif
err1:
	return r;
}

static int __devinit tsc2005_probe(struct spi_device *spi)
{
	struct tsc2005			*tsc;
	struct tsc2005_platform_data	*pdata = spi->dev.platform_data;
	int r;

	if (!pdata) {
		dev_dbg(&spi->dev, "no platform data?\n");
		return -ENODEV;
	}

	tsc = kzalloc(sizeof(*tsc), GFP_KERNEL);
	if (tsc == NULL)
		return -ENOMEM;

	dev_set_drvdata(&spi->dev, tsc);
	tsc->spi = spi;
	spi->dev.power.power_state = PMSG_ON;

	if (pdata->reset_gpio >= 0) {
		tsc->reset_gpio = pdata->reset_gpio;
#ifdef CONFIG_ARCH_OMAP
		r = omap_request_gpio(tsc->reset_gpio);
		if (r < 0)
			goto err1;
#endif
	} else
		tsc->reset_gpio = -1;

	spi->mode = SPI_MODE_3;
	spi->bits_per_word = 8;
	/* The max speed might've been defined by the board-specific
	 * struct */
	if (!spi->max_speed_hz)
		spi->max_speed_hz = TSC2005_HZ;

	spi_setup(spi);

	r = tsc2005_ts_init(tsc, pdata);
	if (r)
		goto err1;
	tsc2005_dvfs_init(tsc);
	return 0;

err1:
	kfree(tsc);
	return r;
}

static int __devexit tsc2005_remove(struct spi_device *spi)
{
	struct tsc2005 *ts = dev_get_drvdata(&spi->dev);
	unsigned long flags;

	spin_lock_irqsave(&ts->lock, flags);
	tsc2005_disable(ts);
	spin_unlock_irqrestore(&ts->lock, flags);

	device_remove_file(&ts->spi->dev, &dev_attr_disable_ts);
	device_remove_file(&ts->spi->dev, &dev_attr_pen_down);

	tsc2005_dvfs_done(ts);

	free_irq(ts->irq, ts);
	input_unregister_device(ts->idev);

#ifdef CONFIG_ARCH_OMAP
	omap_free_gpio(ts->dav_gpio);
#endif
	kfree(ts);

	return 0;
}

#ifdef CONFIG_PM
static int tsc2005_suspend(struct spi_device *spi, pm_message_t mesg)
{
	struct tsc2005 *ts = dev_get_drvdata(&spi->dev);

	spin_lock_irq(&ts->lock);
	tsc2005_disable(ts);
	spin_unlock_irq(&ts->lock);

	return 0;
}

static int tsc2005_resume(struct spi_device *spi)
{
	struct tsc2005 *ts = dev_get_drvdata(&spi->dev);

	spin_lock_irq(&ts->lock);
	tsc2005_enable(ts);
	spin_unlock_irq(&ts->lock);

	return 0;
}
#endif

static struct spi_driver tsc2005_driver = {
	.driver = {
		.name = "tsc2005",
		.bus = &spi_bus_type,
		.owner = THIS_MODULE,
	},
#ifdef CONFIG_PM
	.suspend = tsc2005_suspend,
	.resume = tsc2005_resume,
#endif
	.probe = tsc2005_probe,
	.remove = __devexit_p(tsc2005_remove),
};

static int __init tsc2005_init(void)
{
	printk("TSC2005 driver initializing\n");

	return spi_register_driver(&tsc2005_driver);
}
module_init(tsc2005_init);

static void __exit tsc2005_exit(void)
{
	spi_unregister_driver(&tsc2005_driver);
}
module_exit(tsc2005_exit);

MODULE_AUTHOR("Timo O. Karjalainen <timo.o.karjalainen@nokia.com>");
MODULE_LICENSE("GPL");
